/* 
 * This file part of sdcv - console version of Stardict program
 * http://sdcv.sourceforge.net
 * Copyright (C) 2003-2005 Evgeniy <dushistov@mail.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <cstdlib>
#include <cstdio>
#include <clocale>
#include <glib.h>
#include <glib/gi18n.h>
#include <getopt.h>
#include <string>
#include <vector>
#include <map>
#include <memory>
#include <sys/stat.h>
#include <sys/types.h>

#include "libwrapper.hpp"
#include "readline.hpp"
#include "utils.hpp"


const char gVersion[] = VERSION;

class sdcv_pager {
public:
	sdcv_pager()
	{
		output=stdout;

		const gchar *pager=g_getenv("SDCV_PAGER");
		if (pager) 
			if ((output=popen(pager, "w"))==NULL) {
				perror(_("popen failed"));
				output=stdout;
			}
	}
	~sdcv_pager()
	{
		if (output!=stdout)
			fclose(output);
	}
	FILE *get_stream() { return output; }
private:
	FILE *output;
};


static string utf8_to_locale_ign_err(const string& utf8_str)
{
  gsize bytes_read, bytes_written;
  GError *err=NULL;
  string res;
  
  const char * charset;
  if(g_get_charset(&charset))
    res=utf8_str;
  else{
    gchar *tmp=g_convert_with_fallback(utf8_str.c_str(), -1, charset, "UTF-8", NULL, 
																			 &bytes_read, &bytes_written, &err);
    if(NULL==tmp){
      fprintf(stderr, _("Can not convert %s to current locale.\n"), utf8_str.c_str());
      fprintf(stderr, "%s\n", err->message);
      g_error_free(err);
      exit(EXIT_FAILURE);
    }
		res=tmp;
		g_free(tmp);
  }

  return res;
}

static gchar *locale_to_utf8(const gchar *loc_str)
{
  if(NULL==loc_str)
    return NULL;
  gsize bytes_read;
  gsize bytes_written;
  GError *err=NULL;
  gchar *str=NULL;
  str=g_locale_to_utf8(loc_str, -1, &bytes_read, &bytes_written, &err);
  if(NULL==str){
    fprintf(stderr, _("Can not convert %s to utf8.\n"), loc_str);
    fprintf(stderr, "%s\n", err->message);
    g_error_free(err);
    return str;
  }

  return str;
}

static void print_search_result(FILE *out, const TSearchResult & res, bool utf8_output)
{
  string loc_bookname, loc_def, loc_exp;
  if(!utf8_output){
    loc_bookname=utf8_to_locale_ign_err(res.bookname);
    loc_def=utf8_to_locale_ign_err(res.def);
    loc_exp=utf8_to_locale_ign_err(res.exp);
  }

			
  fprintf(out, "-->%s\n-->%s\n%s\n\n",
					utf8_output ? res.bookname.c_str() : loc_bookname.c_str(), 
					utf8_output ? res.def.c_str() : loc_def.c_str(), 
					utf8_output ? res.exp.c_str() : loc_exp.c_str()); 
}


static void process_phrase(const char *loc_str, Library& lib, 
													 bool utf8_input, bool utf8_output, read_line *io, bool force=false)
{
  if(NULL==loc_str)
    return;
{
  const char *hist=loc_str;
  if (hist && hist[0]=='/')
    ++hist;

  if (hist && *hist!='\0')
		io->add_to_history(hist);
}


  gsize bytes_read;
  gsize bytes_written;
  GError *err=NULL;
  char *str=NULL;
  if (!utf8_input)
    str=g_locale_to_utf8(loc_str, -1, &bytes_read, &bytes_written, &err);
  else
    str=g_strdup(loc_str);
  if (NULL==str) {
    fprintf(stderr, _("Can not convert %s to utf8.\n"), loc_str);
    fprintf(stderr, "%s\n", err->message);
    g_error_free(err);
    return;
  }

  if(str[0]=='\0')
    return;

  
  TSearchResultList res_list;

  if (str[0]=='/')
    lib.LookupWithFuzzy(str+1, res_list);
  else if (bContainRule(str))
    lib.LookupWithRule(str, res_list);
  else {
    lib.SimpleLookup(str, res_list);
    if (res_list.empty())
      lib.LookupWithFuzzy(str, res_list);
  }

  if (!res_list.empty()) {    
    /* try to be more clever, if there are
       one or zero results per dictionary show all
    */
    bool show_all_results=true;
    typedef std::map< string, int, std::less<string> > DictResMap;
    if (!force) {
      DictResMap res_per_dict;
      for(TSearchResultList::iterator ptr=res_list.begin(); ptr!=res_list.end(); ++ptr){
				std::pair<DictResMap::iterator, DictResMap::iterator> r=res_per_dict.equal_range(ptr->bookname);
				DictResMap tmp(r.first, r.second);
				if (tmp.empty()) //there are no yet such bookname in map
					res_per_dict.insert(DictResMap::value_type(ptr->bookname, 1));
				else {
					++((tmp.begin())->second);
					if (tmp.begin()->second>1) {
						show_all_results=false;
						break;
					}
				}
      }
    }//if (!force)

    if (!show_all_results && !force) {
			printf(_("Found %d items, similar to %s.\n"), res_list.size(), utf8_output ? str : utf8_to_locale_ign_err(str).c_str());
			for (size_t i=0; i<res_list.size(); ++i) {
				string loc_bookname, loc_def;
				loc_bookname=utf8_to_locale_ign_err(res_list[i].bookname);
				loc_def=utf8_to_locale_ign_err(res_list[i].def);
				printf("%d)%s-->%s\n", i,
							 utf8_output ?  res_list[i].bookname.c_str() : loc_bookname.c_str(),
							 utf8_output ? res_list[i].def.c_str() : loc_def.c_str());
			}
			int choise;
			for (;;) {
				string str_choise;
				printf(_("Your choice: "));
				
				if(!stdio_getline(stdin, str_choise)){
					putchar('\n');
					exit(EXIT_SUCCESS);
				}
				sscanf(str_choise.c_str(), "%d", &choise);
				if (choise>=0 && choise<int(res_list.size())) { 
					sdcv_pager pager;
					print_search_result(pager.get_stream(), res_list[choise], utf8_output);
					break;
				} else 
					printf(_("Invalid choise.\nIt must be from 0 to %d.\n"), res_list.size()-1);	  
			}		
    } else {
			sdcv_pager pager;
			fprintf(pager.get_stream(), _("Found %d items, similar to %s.\n"), 
							res_list.size(), utf8_output ? str : utf8_to_locale_ign_err(str).c_str());
      for(PSearchResult ptr=res_list.begin(); ptr!=res_list.end(); ++ptr)
				print_search_result(pager.get_stream(), *ptr, utf8_output);
		}
    
  } else {
    string loc_str;
    if (!utf8_output)
      loc_str=utf8_to_locale_ign_err(str);
    
    printf(_("Nothing similar to %s, sorry :(\n"), utf8_output ? str : loc_str.c_str());
  }
  g_free(str);
}


struct option longopts[] ={
  {"version", no_argument, NULL, 'v' },
  {"help", no_argument, NULL, 'h' },
  {"list-dicts", no_argument, NULL, 'l'},
  {"use-dict", required_argument, NULL, 'u'},
  {"non-interactive", no_argument, NULL, 'n'},
  {"utf8-output", no_argument, NULL, 0},
  {"utf8-input", no_argument, NULL, 1},
  {"data-dir", required_argument, NULL, 2},
  { NULL, 0, NULL, 0 }
};

struct PrintDictInfo : public Dicts::FileHandler {
  void operator()(const std::string& filename, bool disabled)
  {
    DictInfo dict_info;
    if (dict_info.load_from_ifo_file(filename)) {
      string bookname=utf8_to_locale_ign_err(dict_info.bookname);
      printf("%s    %d\n", bookname.c_str(), dict_info.wordcount);
    }
  }
};

struct CreateDisableList : public Dicts::FileHandler {
  CreateDisableList(const TStringsList& enable_list_, TStringsList& disable_list_) :
    enable_list(enable_list_), disable_list(disable_list_)
  {
  }
  void operator()(const std::string& filename, bool disabled)
  {
    DictInfo dict_info;
    if (dict_info.load_from_ifo_file(filename)) {
      if (std::find(enable_list.begin(), enable_list.end(), dict_info.bookname)==enable_list.end())
	disable_list.push_back(dict_info.ifo_file_name);
    }
  }
private:
  const TStringsList& enable_list;
  TStringsList& disable_list;
};

int main(int argc, char *argv[])
{
  setlocale(LC_ALL, "");
#if ENABLE_NLS
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif	 
  int optc;
  bool h = false, v = false, show_list_dicts=false, 
    use_book_name=false, non_interactive=false, 
    utf8_output=false, utf8_input=false;
  TStringsList enable_list;
  string data_dir;
  int option_index = 0;
  while((optc = getopt_long (argc, argv, "hvu:ln", longopts, 
                             &option_index))!=-1)
    switch (optc){
    case 0:
      utf8_output=true;
      break;
    case 1:   
      utf8_input=true;
      break;
    case 2:
      data_dir=optarg;
      break;
    case 'v':
      v = true;
      break;
    case 'h':
      h = true;
      break;
    case 'l':
      show_list_dicts=true;
      break;
    case 'u':
      use_book_name=true;
      enable_list.push_back(locale_to_utf8(optarg));
      break;
    case 'n':
      non_interactive=true;
      break;
    case '?':
      fprintf(stderr, _("Unknown option.\nTry '%s --help' for more information.\n"), argv[0]);
      return EXIT_FAILURE;
    }
  
  if (h) {
    printf("sdcv - console version of StarDict.\n");
    printf(_("Usage: %s [OPTIONS] words\n"), argv[0]);
    printf(_("-h, --help               display this help and exit\n"));
    printf(_("-v, --version            display version information and exit\n"));
    printf(_("-l, --list-dicts         display list of available dictionaries and exit\n"));
    printf(_("-u, --use-dict bookname  for search use only dictionary with this bookname\n"));
    printf(_("-n, --non-interactive    for use in scripts\n"));
    printf(_("--utf8-output            output must be in utf8\n"));
    printf(_("--utf8-input             input of sdcv in utf8\n"));
    printf(_("--data-dir path/to/dir   use this directory as path to stardict data directory\n"));

    return EXIT_SUCCESS;
  }

  if (v) {
    printf(_("Console version of Stardict, version %s\n"), gVersion);
    return EXIT_SUCCESS;
  }

  const gchar *stardict_data_dir=g_getenv("STARDICT_DATA_DIR");
  if (data_dir.empty() && stardict_data_dir)
    data_dir=stardict_data_dir;



  TStringsList dicts_dir_list;

  dicts_dir_list.push_back(std::string(g_get_home_dir())+G_DIR_SEPARATOR+".stardict"+G_DIR_SEPARATOR+"dic");
  dicts_dir_list.push_back("/usr/share/stardict/dic");

  if (data_dir.empty() && 
      std::find(dicts_dir_list.begin(), dicts_dir_list.end(), data_dir)==dicts_dir_list.end())
    dicts_dir_list.push_back(data_dir);
    

  if (show_list_dicts) {
    printf(_("bookname     wordcount\n"));
    PrintDictInfo print_dict_info;
    TStringsList order_list, disable_list;
    Dicts::Load(dicts_dir_list, order_list, 
		disable_list, print_dict_info); 
    
    return EXIT_SUCCESS;
  }

  TStringsList disable_list;
  DictInfoList  dict_info_list;
  
  if (use_book_name) {
    TStringsList empty_list;
    CreateDisableList create_disable_list(enable_list, disable_list);
    Dicts::Load(dicts_dir_list, empty_list, empty_list, create_disable_list);
  }

    
  string conf_dir = string(g_get_home_dir())+G_DIR_SEPARATOR+".stardict";
  mkdir(conf_dir.c_str(), S_IRWXU);

  
  Library lib;
  TStringsList empty_list;
  lib.Load(dicts_dir_list, empty_list, disable_list);


	std::auto_ptr<read_line> io(create_readline_object());
  if (optind < argc) {
    for(int i=optind; i<argc; ++i)
      process_phrase(argv[i], lib, utf8_input, utf8_output, io.get(), non_interactive);
  } else if(!non_interactive) {

		string phrase;
		while (io->read(_("Enter word or phrase: "), phrase)) {
      process_phrase(phrase.c_str(), lib, utf8_input, utf8_output, io.get());
			phrase.clear();
		}

    putchar('\n');
  } else {
    fprintf(stderr, _("There are no words/phrases to translate.\n"));
  }
    
  return EXIT_SUCCESS;
}

