/*
 *  $Id: sctp_asconf.c,v 1.12 2003/06/06 13:12:35 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at www.sctp.de which should be used for
 * any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This file implements the handling of the adding/deleting
 *          IP addresses to established SCTP associations, and setting
 *          the peers primary address, as described in
 *          draft-ietf-tsvwg-stewart-addip-05.txt (work in progress)
 */

/*
 * general initialization function - not sure if we actually need this
 */

#include <glib.h>
#include <assert.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "sctp.h"
#include "distribution.h"
#include "sctp_asconf.h"
#include "bundling.h"
#include "pathmanagement.h"
#include "adaptation.h"
#include "auxiliary.h"


typedef struct _asc_data {
    gboolean        peer_supports_addip;
    gboolean        peer_supports_delip;
    gboolean        peer_supports_setPrimary;

    gboolean        i_support_addip;
    gboolean        i_support_delip;
    gboolean        i_support_setPrimary;

    guint32         my_serial_number;
    guint32         peer_serial_number;
    guint32         currentCorrelationId;
    gboolean        asconf_outstanding;
    guint32         lastAddressKeyUsed;
    TimerID         asconf_timer;
    GList*          queued_asconf_requests;
    GList*          sent_asconf_requests;
    SCTP_asconf*              lastRequestSent;
    SCTP_asconf_ack_chunk*    lastReplySent;
    unsigned long       crcOfLastRequestReceived;
}asc_data;



typedef struct ASCONF_ADDDELSETPARAM
{
    guint16         pType;
    guint16         pLength;
    guint32         correlationId;
    SCTP_ip_address address;
}
asconf_AddDelSetParam;


/**
 *  this can be:
 *   - an error cause indication, which is followed by an error cause
 *     (that would be an error cause code, length and TLV that caused
 *      the error)
 *   - a success indication for one correlation id (where the data is NULL)
 *
 */
typedef struct ASCONF_RESULT
{
    guint16         pType;
    guint16         pLength;
    guint32         correlationId;
    guchar          data[MAX_SCTP_PDU];
}
asconf_Result;


int asc_init(void)
{
    return 0;
}

/*
 * asc_new: Create a new AS_CONF instance and returns a pointer to its data.
 */
void *asc_new(Association* asok, guint32 initial_serial, guint32 peer_initial_serial)
{
    /* Alloc new bundling_instance data struct */
    asc_data *ptr;

    ptr = malloc(sizeof(asc_data));
    if (!ptr) {
        error_log(ERROR_FATAL, "Malloc failed in asc_new()");
        return NULL;
    }
    ptr->peer_supports_addip = TRUE;
    ptr->peer_supports_delip = TRUE;
    ptr->peer_supports_setPrimary = TRUE;

    ptr->i_support_addip = TRUE;
    ptr->i_support_delip = TRUE;
    ptr->i_support_setPrimary = TRUE;

    ptr->my_serial_number = initial_serial;
    
    /* section 4.2.C1) */
    ptr->peer_serial_number = peer_initial_serial - 1;
    ptr->currentCorrelationId = 1;

    ptr->asconf_outstanding = FALSE;
    ptr->asconf_timer = 0;
    ptr->lastAddressKeyUsed = 0;

    ptr->queued_asconf_requests = NULL;
    ptr->sent_asconf_requests = NULL;
    ptr->lastRequestSent = NULL;
    ptr->lastReplySent = NULL;
    ptr->crcOfLastRequestReceived = 0;

    return ptr;

}

/*
 * asc_delete: Deletes a AS_CONF instance
 * Params: Pointer/handle which was returned by asc_new()
 */
int asc_delete(Association* asok)
{
    asc_data *ascd = NULL;
    asconf_AddDelSetParam* param = NULL;

    assert(asok);
    ascd = (asc_data *) mdi_readASCONF((Association*)asok);

    if (asok->sctp_asconf == NULL) return -1;
    assert(ascd);

    if (ascd->lastRequestSent != NULL) free (ascd->lastRequestSent);
    if (ascd->lastReplySent != NULL) free (ascd->lastReplySent);

    while(g_list_length(ascd->queued_asconf_requests) > 0) {
        param = g_list_nth_data(ascd->queued_asconf_requests, 0);
        ascd->queued_asconf_requests = g_list_remove(ascd->queued_asconf_requests,
                                                     g_list_nth_data(ascd->queued_asconf_requests, 0));
        free(param);
    }
    g_list_free(ascd->queued_asconf_requests);
    ascd->queued_asconf_requests = NULL;

    while(g_list_length(ascd->sent_asconf_requests) > 0) {
        param = g_list_nth_data(ascd->sent_asconf_requests, 0);
        ascd->sent_asconf_requests = g_list_remove(ascd->sent_asconf_requests,
                                                   g_list_nth_data(ascd->sent_asconf_requests, 0));
        free(param);
    }
    g_list_free(ascd->sent_asconf_requests);
    ascd->queued_asconf_requests = NULL;

    free(asok->sctp_asconf);
    asok->sctp_asconf = NULL;
    return 0;
}


/*
 * function to copy the address from a received chunk into the sockunion
 * struct pointed to by the passed pointer
 * @return SCTP_SUCCESS if all is okay, else error code
 */
int asc_getAdressFromChunk(SCTP_simple_chunk * asc_chunk, union sockunion* addr)
{
    struct sockaddr_in* si;
    
#ifdef HAVE_IPV6
    struct sockaddr_in6* si6;
#endif
    SCTP_asconf* asc = (SCTP_asconf*) asc_chunk;
    
    switch (ntohs(asc->addressParameterHeader.param_type)) {
        case VLPARAM_IPV4_ADDRESS:
            addr->sa.sa_family = AF_INET;
            si = (struct sockaddr_in*)asc->variableParams;
            memcpy(&(addr->sin.sin_addr.s_addr), si, sizeof(struct sockaddr_in));
            break;
#ifdef HAVE_IPV6
        case VLPARAM_IPV6_ADDRESS:
            addr->sa.sa_family = AF_INET6;
            si6 = (struct sockaddr_in6*) asc->variableParams;
            memcpy(&(addr->sin6.sin6_addr), si6, sizeof(struct sockaddr_in6));
            break;
#endif           
        default:
            error_log(ERROR_FATAL, "Unknown adress parameter found in ASCONF chunk, Aborting here!");
            return SCTP_PARAMETER_PROBLEM;
            break;

    }
    return SCTP_SUCCESS;
}

void asc_stop_timer(Association* asok)
{
    asc_data *ascd = NULL;
    ascd = (asc_data *) mdi_readASCONF(asok);
    if (!ascd) {
        error_log(ERROR_MAJOR, "ASCONF instance not set in asc_stop_timer() !");
        return;
    }
    event_log(VVERBOSE, "asc_stop_timer: Stopping ASCONF Timer....");
    sctp_stopTimer(ascd->asconf_timer);
    event_log(VVERBOSE, "....DONE (stopping ASCONF Timer) ");
    ascd->asconf_timer = 0;
    return;
}

void asc_timer_cb(TimerID tid, void *assoc, void *dummy)
{
    asc_data *ascd = NULL;
    gboolean removedAssoc = FALSE;
    unsigned int nextPathKey = 0;
    int rto_time = 0, pathIndex, count = 0, result;

    event_log(INTERNAL_EVENT_0, "----------------> asc_timer_cb <------------------");
    assert(assoc);
    ascd = (asc_data *) mdi_readASCONF((Association*)assoc);
    assert(ascd);
    /* increase error counters of last address/assoc */
    removedAssoc =  pm_chunksRetransmitted((Association*)assoc, ascd->lastAddressKeyUsed);

    if (removedAssoc == FALSE) {    
        /* select next active address */
        pathIndex = mdi_getIndexForKey((Association*)assoc, ascd->lastAddressKeyUsed);
        while (count < ((Association*)assoc)->noOfDestAddresses) {
            pathIndex = (pathIndex+1)% ((Association*)assoc)->noOfDestAddresses;
            count++;
            nextPathKey = mdi_getKeyForIndex((Association*)assoc, pathIndex);
            if (pm_readState((Association*)assoc, nextPathKey) == PM_ACTIVE) break;
        }
        /* retransmit last chunk */
        ascd->lastAddressKeyUsed = nextPathKey;
        bu_put_Ctrl_Chunk((Association*)assoc, (SCTP_simple_chunk*) ascd->lastRequestSent, nextPathKey);
        result = bu_sendAllChunks((Association*)assoc, nextPathKey);
        /* and restart timer */
        rto_time = pm_readRTO((Association*)assoc, nextPathKey);
        ascd->asconf_timer = adl_startTimer(rto_time,
                                            &asc_timer_cb,
                                            TIMER_TYPE_ASCONF,
                                            assoc,
                                            NULL);
    }
    return;
}

/*
 *  asc_recv_asconf_chunk gets a pointer to an AS_CONF chunk and decodes it
 *  accordingly....
 *  @return  error code, 0 for success, less than one for error
 */
int asc_recv_asconf_chunk(Association* asok, SCTP_simple_chunk * asc_chunk, unsigned int fromPathKey)
{
    asc_data *ascd = NULL;
    SCTP_asconf_ack_chunk* ascAck = NULL;
    SCTP_asconf* ascChunk = (SCTP_asconf*)asc_chunk;
    unsigned char* posPtr= NULL;
    unsigned char* startAscPtr = NULL;
    unsigned char* startAscAckPtr = NULL;
    
    asconf_AddDelSetParam* ascParPtr = NULL;
    asconf_Result* ascResultPtr = NULL;
    SCTP_ip_address* addressPtr = NULL;
    int  chunkLen = 0;
    
    int len = 0, result=0, chunkPos=0;
    unsigned int peerSerial, crc32, pathKey=0, tmpKey=0;
    union sockunion receivedAddress;
    int gotValidAddress = FALSE;

    ascd = (asc_data *) mdi_readASCONF(asok);
    if (!ascd) {
        error_log(ERROR_MAJOR, "ASCONF instance not set !");
        return (SCTP_MODULE_NOT_FOUND);
    }
    /*   Section 4.2 - C1  */
    peerSerial = ntohl(ascChunk->serial_number);
    if (peerSerial == ascd->peer_serial_number + 1) {
        ascd->crcOfLastRequestReceived = generate_crc32c((unsigned char*)asc_chunk, ntohs(ascChunk->chunk_header.chunk_length));
        /*    Section 4.2 - C2, got a new one  */
        free(ascd->lastReplySent);
        /* this will be assigned a new chunk pointer below */
        ascAck = malloc(sizeof(SCTP_asconf_ack_chunk));
        ascAck->chunk_header.chunk_id = CHUNK_ASCONF_ACK;
        ascAck->chunk_header.chunk_flags = 0;
        ascAck->serial_number = ascChunk->serial_number;

        startAscPtr = (unsigned char*)ascChunk;
        startAscAckPtr = (unsigned char*)ascAck;

        chunkLen = ntohs(ascChunk->chunk_header.chunk_length);
        len = sizeof(SCTP_chunk_header) + sizeof(unsigned int);
        /* TODO: scan individual TLV parameters, and build errors and success reports */
        /* for now, we understand them all !!!!!  :-) */
        addressPtr = (SCTP_ip_address*)&(ascChunk->addressParameterHeader);
        chunkPos = sizeof(SCTP_chunk_header) + sizeof(guint32) + ntohs(ascChunk->addressParameterHeader.param_length);

        while (chunkPos <  chunkLen) {
            posPtr = &startAscPtr[chunkPos];
            ascParPtr = (asconf_AddDelSetParam*) posPtr;
            /*   Section 4.2.C2.V1  */
            /*   Section 4.2.C2.V2  */
            /*   Section 4.2.C2.V3  */

            switch (ntohs(ascParPtr->address.vlparam_header.param_type)) {
                case VLPARAM_IPV4_ADDRESS:
                    receivedAddress.sa.sa_family = AF_INET;
                    receivedAddress.sin.sin_port = htons(0);
                    receivedAddress.sin.sin_addr.s_addr = ascParPtr->address.dest_addr.sctp_ipv4;
                    gotValidAddress = TRUE;
                    break;
#ifdef HAVE_IPV6
                case VLPARAM_IPV6_ADDRESS:
                    receivedAddress.sa.sa_family = AF_INET6;
                    receivedAddress.sin6.sin6_port = htons(0);
                    memcpy(receivedAddress.sin6.sin6_addr.s6_addr,
                           ascParPtr->address.dest_addr.sctp_ipv6,
                           sizeof(struct sockaddr_in6));
                    gotValidAddress = TRUE;
#endif
                default :
                    /* invalid address parameter */
                    gotValidAddress = FALSE;
                    event_log(VERBOSE, "Received an invalid address parameter");
                    break;
            }

            switch (ntohs(ascParPtr->pType)) {
                case VLPARAM_ADDIP:
                    /* check if we are out of resources, and if so, add Error Indication */
                    event_log(VERBOSE, "Received an ASCONF ADD-IP........");
                    tmpKey = mdi_addDestinationAddressToAssoc(asok, &receivedAddress);
                    mdi_networkStatusChangeNotif(asok, tmpKey, SCTP_PATH_ADDED);
                    break;
                case VLPARAM_DELIP:
                    /* check if last address is deleted, of peer removes an address that */
                    /* is not there, and if so, add correct Error Indication */
                    event_log(VERBOSE, "Received an ASCONF DEL-IP........");
                    result = mdi_delDestinationAddressFromAssoc(asok, &receivedAddress);
                    mdi_networkStatusChangeNotif(asok, 0, SCTP_PATH_REMOVED);
                    break;
                case VLPARAM_SET_PRIMARY:
                    /* check if this is an address that belongs to the Assoc */
                    /* else ignore */
                    event_log(VERBOSE, "Received an ASCONF SET PRIMARY........do it....");
                    if (gotValidAddress == TRUE) {
                        pathKey = mdi_getKeyForAddress(asok, &receivedAddress);
                        if (pathKey != 0) {
                            /* set new primary here */
                            result = pm_setPrimaryPathKey(asok, pathKey);
                        }
                    }
                    break;
                default:
                    event_log(ERROR_FATAL, "Received an invalid ASCONF parameter, not handled yet, should ABORT");
                    exit(1);
                    break;
            }
            event_logi(VERBOSE, ".........done...(result==%d)................", result);
            /* if no error ocurred, add a success indication */
            ascResultPtr = (asconf_Result*)&startAscAckPtr[len];
            ascResultPtr->pType     =  htons(VLPARAM_SUCCESS_REPORT);
            ascResultPtr->pLength   =  htons(8); /* fixed size */
            /* copy this straight from the received parameter */
            ascResultPtr->correlationId = ascParPtr->correlationId;
            len += 8;
            /* move on to the next parameter (if there is more) */
            chunkPos += ntohs(ascParPtr->pLength);
            while ((chunkPos%4) !=0) chunkPos++;
        }
        ascAck->chunk_header.chunk_length = htons(len);
        /* finally, update to new peer serial : 4.2.C2.V5 */
        ascd->peer_serial_number = peerSerial;
        /* ...and send it out ! */
        ascd->lastReplySent = ascAck;
    } else if (peerSerial == ascd->peer_serial_number) {
        /*   Section 4.2 - C3, got a retransmission  */
        if (ascd->lastReplySent == NULL) {
            event_log(EXTERNAL_EVENT, "Got an old ASCONF chunk, but no reply stored, peer error ?!");
            return SCTP_PEER_ERROR;
        }
        crc32 = generate_crc32c((unsigned char*)asc_chunk, ntohs(ascChunk->chunk_header.chunk_length));
        if (crc32 != ascd->crcOfLastRequestReceived) {
            event_log(EXTERNAL_EVENT, "Got an old ASCONF chunk which is different from last old one -- peer error ?!");
            return SCTP_PEER_ERROR;
        }
        return SCTP_SUCCESS;
    } else {
        /*   Section 4.2 - C4, got a false one, discard  */
        event_log(EXTERNAL_EVENT, "Got a stale ASCONF chunk, maybe an attack ?");
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }
    /* return old response to the old address */
    bu_put_Ctrl_Chunk(asok, (SCTP_simple_chunk*)ascd->lastReplySent, fromPathKey);
    result = bu_sendAllChunks(asok, fromPathKey);
    return SCTP_SUCCESS;
}


int asc_sendRequest(Association* asok, asc_data *ascd)
{
    int len = 0, pos = 0, numOfParams = 0, currentRTO, result, count=0;
    unsigned int primaryPathKey = 0;
    asconf_AddDelSetParam* param = NULL;
    union sockunion* anAddress;
    gboolean addressFound = FALSE;
    SCTP_asconf* ascChunk = malloc(sizeof(SCTP_asconf));

    assert(ascd->asconf_outstanding == FALSE);

    /* we have data queued, now build ASCONF REQUEST chunk */
    ascChunk->chunk_header.chunk_id = CHUNK_ASCONF;
    ascChunk->chunk_header.chunk_flags = 0;
    len += sizeof(SCTP_chunk_header);
    ascChunk->serial_number = htonl(ascd->my_serial_number);
    len += sizeof(guint32);
    ascd->my_serial_number++;

    while (count < asok->noOfLocalAddresses) {
        anAddress = &g_array_index(asok->localAddresses, union sockunion, count);
        switch(sockunion_family(anAddress)) {
            case AF_INET:
                event_logi(VVERBOSE, "asc_sendRequest: checking address %x", ntohl(sock2ip(anAddress)));
                if (ntohl(sock2ip(anAddress)) != INADDR_LOOPBACK) {
                    event_log(VVERBOSE, "Found an IPv4 address different from loopback ! ");
                    addressFound = TRUE;
                    break;
                 }
                 break;
#ifdef HAVE_IPV6
            case AF_INET6:
                /*
                 * returns TRUE if address is not filtered, else FALSE if address is filtered by mask
                 */
                if (adl_filterInetAddress(anAddress, flag_Default|flag_HideLocal) == TRUE) {
                    event_log(VVERBOSE, "Found an IPv6 different from loopback ! ");
                    addressFound = TRUE;
                    /* FIXME: pay attention to scoping rules here ! */
                    break;
                }
                break;
#endif
            default:
                break;
        }
        if (addressFound == TRUE) break;
        count++;
    }
    /* make sure we got at least one valid address */
    if (count == asok->noOfLocalAddresses) count = 0;
    event_logii(VVERBOSE, "asc_sendRequest: count is finally %d, noOfLocalAddresses=%d", count, asok->noOfLocalAddresses);

    /* add my preferred address */
    switch (sockunion_family(&g_array_index(asok->localAddresses, union sockunion, count))) {
        case AF_INET:
            ascChunk->addressParameterHeader.param_type = htons(VLPARAM_IPV4_ADDRESS);
            ascChunk->addressParameterHeader.param_length = htons(8);
            memcpy(ascChunk->variableParams,
                   &(sock2ip(&g_array_index(asok->localAddresses, union sockunion, count))),
                   sizeof(guint32));
            len += 8;
            pos = 4;
            break;
#ifdef HAVE_IPV6
        case AF_INET6:
            ascChunk->addressParameterHeader.param_type = htons(VLPARAM_IPV6_ADDRESS);
            ascChunk->addressParameterHeader.param_length = htons(20);
            memcpy(ascChunk->variableParams,
                   &(sock2ip6(&g_array_index(asok->localAddresses, union sockunion, count))),
                   sizeof(struct in6_addr));
            len += 20;
            pos = 16;
            break;
#endif
        default:
            error_log(ERROR_FATAL, "Address Error in asc_sendRequest !");
            break;
    }

    /* add all (well, 5) parameters we have around, and delete them, since they are queued  */
    while(((param = g_list_nth_data(ascd->queued_asconf_requests, 0)) != NULL) && numOfParams < 5) {
        numOfParams++;
        memcpy(&ascChunk->variableParams[pos], param, ntohs(param->pLength));
        ascd->queued_asconf_requests = g_list_remove(ascd->queued_asconf_requests,
                                                     g_list_nth_data(ascd->queued_asconf_requests, 0));
                                                     
        ascd->sent_asconf_requests = g_list_append(ascd->sent_asconf_requests, param);
        pos += ntohs(param->pLength);
        len += ntohs(param->pLength);
        event_logii(VERBOSE, "asc_sendRequest: pos == %d, len == %d", pos, len);
    }
    /* that was all, insert len in network byte order */
    ascChunk->chunk_header.chunk_length = htons(len);
    /* send it out to primary/next address */
    primaryPathKey = pm_getPrimaryPathKey(asok);
    ascd->lastAddressKeyUsed = primaryPathKey;

    event_logii(VERBOSE, "asc_sendRequest: created %d byte ASCONF chunk, send it to pathKey=%u !", len, primaryPathKey);
                      
    bu_put_Ctrl_Chunk(asok, (SCTP_simple_chunk*) ascChunk, primaryPathKey);
    result = bu_sendAllChunks(asok, primaryPathKey);
    /* and store the pointer to the chunk for retransmission */
    ascd->asconf_outstanding = TRUE;
    if (ascd->lastRequestSent != NULL) free(ascd->lastRequestSent);
    ascd->lastRequestSent = ascChunk;
    /* start timer with used address rto  */
    currentRTO = pm_readRTO(asok, primaryPathKey);
    ascd->asconf_timer = adl_startTimer(currentRTO,
                                        &asc_timer_cb,
                                        TIMER_TYPE_ASCONF,
                                        asok,
                                        NULL);

    return 0;
}



/*
 *  asc_recv_asconf_ack gets a pointer to an AS_CONF_ACK chunk and decodes it
 *  accordingly....
 *  @return  error code, 0 for success, less than one for error
 */
int asc_recv_asconf_ack(Association* asok, SCTP_simple_chunk * asc_ack, unsigned int fromPathKey)
{
    asc_data *ascd = NULL;
    SCTP_asconf_ack_chunk* ascAck = (SCTP_asconf_ack_chunk*)asc_ack;
    unsigned char* posPtr = (unsigned char*)asc_ack;
    asconf_Result* resultParam = NULL;
    /* asconf_AddDelSetParam* asconfParam = NULL; */
    asconf_AddDelSetParam* queuedParam = NULL;
    
    unsigned int serialAcked;
    int result=0, position=0, totalLen=0;

    ascd = (asc_data *) mdi_readASCONF(asok);
    if (!ascd) {
        error_log(ERROR_MAJOR, "ASCONF instance not set !");
        return (SCTP_MODULE_NOT_FOUND);
    }
    /* section 4.3.D0 - check if no ASCONF was outstanding -> ABORT */
    if (ascd->asconf_outstanding == FALSE) {
        scu_abort(asok, ECC_UNRECOGNIZED_CHUNKTYPE, 0, NULL);
        return STATE_STOP_PARSING_REMOVED;
    }  
    /* section 4.1.A5 - check serial number, compare with the serial last sent !!! */
    serialAcked = ntohl(ascAck->serial_number);
    event_logi(EXTERNAL_EVENT, "Peer sent ASCONF-ACK with Serial Number %u", serialAcked);
    if (serialAcked == ascd->my_serial_number - 1) {
        /* section 4.1.A5 - stop timers, clear counter */
        event_log(VVERBOSE, "Stopping ASCONF Timer....");
        sctp_stopTimer(ascd->asconf_timer);
        event_log(VVERBOSE, "....DONE (stopping ASCONF Timer) ");
        
        ascd->asconf_timer = 0;
        ascd->asconf_outstanding = FALSE;
        event_logi(VVERBOSE, "Calling pm_chunksAcked(key=%u)", fromPathKey);
        pm_chunksAcked(asok, fromPathKey, 0L);
    }
    /* section 4.1.A6 - process TLVs and store peer capabilities */
    position = sizeof(SCTP_chunk_header) + sizeof(guint32);
    totalLen = ntohs    (ascAck->chunk_header.chunk_length);

    event_logii(VVERBOSE, "Checking Reply, totalLen==%d, position==%d", totalLen, position);

    if (totalLen == position) {
        /* peer acked the complete packet. dequeue all sent asconf chunks, sent packets */
        /* that will be done below anyway, so we don't do anything here */
        result = 0;
    } else {
        while(position < totalLen) {
            event_logii(VVERBOSE, "Loop thru reply, totalLen==%d, position==%d", totalLen, position);
            /* section 4.3.D1,D2,D4 - ADDIP address usage */
            /* section 4.3.D3a   - ADDIP/DELIP error -> record */
            /* section 4.3.D3b   - SET PRIMARY error -> record */
            resultParam = (asconf_Result*) &posPtr[position];
            switch (ntohs(resultParam->pType)) {
                case  VLPARAM_SUCCESS_REPORT:
                    /* we got an successful ack for one parameter. */
                    event_logi(EXTERNAL_EVENT, "Peer ASCONF-ACKED the correlation id %u",ntohl(resultParam->correlationId));
                    mdi_networkStatusChangeNotif(asok, 0, SCTP_ASCONF_SUCCEEDED);
                    break;
                case  VLPARAM_ERROR_CAUSE_INDICATION:
                    /* get type of error that occurred */
                    /* get parameter that
                    ECC_ASCONF_DELETE_LAST_IP_FAILED
                    ECC_ASCONF_OP_REFUSED_NO_RESOURCES     
                    ECC_ASCONF_DELETE_SOURCE_ADDRESS       
                    ECC_ILLEGAL_REQUEST                 */
                    /* something failed...stop sending ASCONF-chunks */
                    event_logi(EXTERNAL_EVENT, "Peer ERROR-CAUSED the correlation id %u",ntohl(resultParam->correlationId));
                    ascd->peer_supports_addip = FALSE;
                    ascd->peer_supports_delip = FALSE;
                    ascd->peer_supports_setPrimary = FALSE;
                    mdi_networkStatusChangeNotif(asok, 0, SCTP_ASCONF_FAILED);
                    result = -1;
                    break;
                default:
                    /* got an illegal AS-CONF-ACK */
                    scu_abort(asok, ECC_UNRECOGNIZED_PARAMS, 0, NULL);
                    return STATE_STOP_PARSING_REMOVED;
                    break;
            }
            /* section 4.1.A7 - skip TLVs after an error -> unsuccessful */
            if (result == -1) break;
        
            position += ntohs(resultParam->pLength);
            /* padding */
            while ((position % 4) != 0) position++;
        } /* end  while(position < totalLen) { */
    }

    /***********************************************************************************************/    
    /* section 4.1.A8 - no errors -> all were successful -> if error, we will take no action _yet_ */
    /* later we will have to deal with ADD-IPs that have failed (or DEL-IPs)                       */
    /***********************************************************************************************/
    while((queuedParam = g_list_nth_data(ascd->sent_asconf_requests, 0)) != NULL) {
        free(queuedParam);
        ascd->sent_asconf_requests = g_list_remove(ascd->sent_asconf_requests,
                                                   g_list_nth_data(ascd->sent_asconf_requests, 0));
        g_list_free(ascd->sent_asconf_requests);
        ascd->sent_asconf_requests = NULL;
    }
    
    if (result==0) {
        /* check if we can send another one now, that one has been acked */
        if (g_list_length(ascd->queued_asconf_requests) > 0) {
            if (ascd->asconf_outstanding == FALSE) {
                result = asc_sendRequest(asok, ascd);
            }
        }
    } else {
        /* dequeue old requests, they will be ignored, needs to be refined later */
        while((queuedParam = g_list_nth_data(ascd->queued_asconf_requests, 0)) != NULL) {
            ascd->queued_asconf_requests = g_list_remove(ascd->queued_asconf_requests,
                                                         g_list_nth_data(ascd->queued_asconf_requests, 0));
            free(queuedParam);                                                         
        }
        g_list_free(ascd->queued_asconf_requests);
        ascd->queued_asconf_requests = NULL;
        
    }
    return STATE_OK;
}


unsigned char* asc_buildParam(unsigned int corrId, unsigned short ptype, union sockunion* theIP)
{
    asconf_AddDelSetParam* param;
    int addressLen;

    param = malloc(sizeof(asconf_AddDelSetParam));
    if (param == NULL) return NULL;

    event_logii(VERBOSE, "asc_buildParam: corrId: %u, Type: %u",corrId, ptype);

    param->correlationId = htonl(corrId);
    param->pType = htons(ptype);
    switch (sockunion_family(theIP)) {
    case AF_INET:
        addressLen = 8;
        param->address.vlparam_header.param_type = htons(VLPARAM_IPV4_ADDRESS);
        param->address.vlparam_header.param_length = htons(8);
        param->address.dest_addr.sctp_ipv4 = sock2ip(theIP);
        break;
#ifdef HAVE_IPV6
    case AF_INET6:
        addressLen = 20;
        param->address.vlparam_header.param_type = htons(VLPARAM_IPV6_ADDRESS);
        param->address.vlparam_header.param_length = htons(20);
        memcpy(param->address.dest_addr.sctp_ipv6, &(sock2ip6(theIP)), sizeof(struct in6_addr));
        break;
#endif
    default:
        error_logi(ERROR_MAJOR, "Address family %d not supported", sockunion_family(theIP));
        free(param);
        return NULL;
        break;
    }
    param->pLength = htons(addressLen+8);
    return ((unsigned char*)param);
}



int asc_addIP(Association* asok, union sockunion* addedIP, unsigned int* corrId)
{
    asc_data *ascd;
    int result = SCTP_SUCCESS;
    asconf_AddDelSetParam* param;

    ascd = (asc_data *) mdi_readASCONF(asok);
    if (!ascd) {
        error_log(ERROR_MAJOR, "ASCONF instance not set !");
        return (SCTP_MODULE_NOT_FOUND);
    }
    if (ascd->peer_supports_addip == FALSE) return SCTP_NOT_SUPPORTED;

    param = (asconf_AddDelSetParam*)asc_buildParam(ascd->currentCorrelationId, VLPARAM_ADDIP, addedIP);
    if (param != NULL) {
        *corrId = ascd->currentCorrelationId++;

        /* queue it */
        ascd->queued_asconf_requests = g_list_append(ascd->queued_asconf_requests,
                                                 param);

        /* if none oustanding then send it */
        if (ascd->asconf_outstanding == FALSE)  result = asc_sendRequest(asok, ascd);

    } else {
        result =  SCTP_PARAMETER_PROBLEM;
    }
    return result;
}

int asc_deleteIP(Association* asok, union sockunion* deletedIP, unsigned int* corrId)
{
    asc_data *ascd;
    int result = SCTP_SUCCESS;
    asconf_AddDelSetParam* param;

    ascd = (asc_data *) mdi_readASCONF(asok);
    if (!ascd) {
        error_log(ERROR_MAJOR, "ASCONF instance not set !");
        return (SCTP_MODULE_NOT_FOUND);
    }
    if (ascd->peer_supports_delip == FALSE) return SCTP_NOT_SUPPORTED;

    param = (asconf_AddDelSetParam*)asc_buildParam(ascd->currentCorrelationId, VLPARAM_DELIP, deletedIP);

    if (param != NULL) {
        *corrId = ascd->currentCorrelationId++;

        /* queue it */
        ascd->queued_asconf_requests = g_list_append(ascd->queued_asconf_requests,
                                                 param);

        /* if none oustanding then send it */
        if (ascd->asconf_outstanding == FALSE)  result = asc_sendRequest(asok, ascd);

    } else {
        result =  SCTP_PARAMETER_PROBLEM;
    }
    return result;
}

int asc_setRemotePrimary(Association* asok, union sockunion* theIP)
{
    asc_data *ascd;
    int result = SCTP_SUCCESS;
    asconf_AddDelSetParam* param;

    ascd = (asc_data *) mdi_readASCONF(asok);
    if (!ascd) {
        error_log(ERROR_MAJOR, "ASCONF instance not set !");
        return (SCTP_MODULE_NOT_FOUND);
    }
    if (ascd->peer_supports_setPrimary == FALSE) return SCTP_NOT_SUPPORTED;

    param = (asconf_AddDelSetParam*)asc_buildParam(ascd->currentCorrelationId, VLPARAM_SET_PRIMARY, theIP);
    if (param != NULL) {
        ascd->currentCorrelationId++;

        /* queue it */
        ascd->queued_asconf_requests = g_list_append(ascd->queued_asconf_requests,
                                                     param);

        /* if none oustanding then send it */
        if (ascd->asconf_outstanding == FALSE)  result = asc_sendRequest(asok, ascd);
    } else {
        result =  SCTP_PARAMETER_PROBLEM;
    }
    return result;
}

