/*
 *  $Id: flowcontrol.c,v 1.15 2004/01/13 16:13:14 ajung Exp $
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * This module implements most parts of the flow control mechanisms
 *
 */

#include "flowcontrol.h"
#include "bundling.h"
#include "adaptation.h"
#include "recvctrl.h"
#include "streamengine.h"

#include <assert.h>
#include <stdio.h>
#include <glib.h>


/**
 * this struct contains all relevant congestion control parameters for
 * one PATH to the destination/association peer endpoint
 */
typedef struct flowcontrol_path_parameter
{
    unsigned int pathKey;
    unsigned int* pathKeyPtr;
    /** one timer may be running per destination address */
    TimerID T3_timer;
    unsigned int cwnd;
    unsigned int cwnd2;
    unsigned int partial_bytes_acked;
    unsigned int ssthresh;
    struct timeval time_of_cwnd_adjustment;
    struct timeval last_send_time;
    /* @} */
} fcPathParam;

typedef struct flowcontrol_struct
{
    /* @{  */
    unsigned int outstanding_bytes;
    unsigned int announced_rwnd;
    unsigned int number_of_addresses;
    /** pointer to array of congestion window parameters */
    GArray*  pathParams;        /* FROM: cparm *cparams; */
    unsigned int current_tsn;
    GList *chunk_list;
    unsigned int list_length;
    Association* myAssociation;
    boolean waiting_for_sack;
    gboolean t3_retransmission_sent;
    gboolean one_packet_inflight;
    gboolean doing_retransmission;
    unsigned int maxQueueLen;
    /* @} */
} fc_data;


/* ---------------  Function Prototypes -----------------------------*/
int fc_send_all_you_can(Association* asok, gboolean doInitialRetransmit);
/* ---------------  Function Prototypes -----------------------------*/


/**
 * Creates new instance of flowcontrol module and returns pointer to it
 * TODO : get and update MTU (guessed values ?) per destination address
 * @param  peer_rwnd receiver window that peer allowed us when setting up the association
 * @param  my_iTSN my initial TSN value
 * @param  number_of_destination_addresses the number of paths to the association peer
 * @return  pointer to the new fc_data instance
*/
void *fc_new_flowcontrol(Association* asok,
                         unsigned int peer_rwnd,
                         unsigned int my_iTSN,
                         unsigned int number_of_destination_addresses,
                         unsigned int maxQueueLen)
{
    fc_data *tmp;
    fcPathParam fcpp;
    int count;

    assert(asok);

    tmp = malloc(sizeof(fc_data));
    if (!tmp) error_log(ERROR_FATAL, "Malloc failed");

    tmp->current_tsn = my_iTSN;

    event_logi(VERBOSE,  "Flowcontrol: ===== Num of number_of_destination_addresses = %u ",
               number_of_destination_addresses);

    tmp->pathParams = g_array_new(FALSE, TRUE, sizeof(fcPathParam));

    for (count = 0; count < number_of_destination_addresses; count++) {
        fcpp.T3_timer   = 0;          /* i.e. timer not running */
        fcpp.cwnd       = 2 * MAX_MTU_SIZE;
        fcpp.cwnd2      = 0L;
        fcpp.partial_bytes_acked = 0L;
        fcpp.ssthresh   = peer_rwnd;
        fcpp.pathKey    = mdi_getKeyForIndex(asok, count);
        fcpp.pathKeyPtr = malloc(sizeof(unsigned int));
        *(fcpp.pathKeyPtr) = fcpp.pathKey;
        adl_gettime(&(fcpp.time_of_cwnd_adjustment));
        timerclear(&(fcpp.last_send_time));
        tmp->pathParams = g_array_append_val(tmp->pathParams, fcpp);
    }
    tmp->outstanding_bytes      = 0;
    tmp->announced_rwnd         = peer_rwnd;
    tmp->number_of_addresses    = number_of_destination_addresses;
    tmp->myAssociation          = asok;

    tmp->waiting_for_sack       = FALSE;
    tmp->doing_retransmission   = FALSE;
    tmp->t3_retransmission_sent = FALSE;
    tmp->one_packet_inflight    = FALSE;
    tmp->chunk_list             = NULL;
    tmp->maxQueueLen            = maxQueueLen;
    tmp->list_length            = 0;

    rtx_set_remote_receiver_window(asok, peer_rwnd);

    event_logi(VVERBOSE, "Created FlowControl for Association with ID== %u", asok->assocId);
    return tmp;
}

/**
 * this function stops all currently running timers, and may be called when
 * the shutdown is imminent
 * @param  new_rwnd new receiver window of the association peer
 */
void fc_restart(Association* asok, guint32 new_rwnd, unsigned int iTSN, unsigned int maxQueueLen)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    int count;

    fc = (fc_data *) mdi_readFlowControl(asok);
    event_log(INTERNAL_EVENT_0, "fc_restart()... ");
    assert(fc);

    fc_stop_timers(asok);

    for (count = 0; count < fc->number_of_addresses; count++) {
        fcpp = &g_array_index(fc->pathParams ,fcPathParam, count);
        assert(fcpp);
        fcpp->cwnd = 2 * MAX_MTU_SIZE;
        fcpp->cwnd2 = 0L;
        fcpp->partial_bytes_acked = 0L;
        fcpp->ssthresh = new_rwnd;
        adl_gettime(&(fcpp->time_of_cwnd_adjustment));
        timerclear(&(fcpp->last_send_time));
    }
    fc->outstanding_bytes       = 0;
    fc->announced_rwnd          = new_rwnd;
    fc->waiting_for_sack        = FALSE;
    fc->t3_retransmission_sent  = FALSE;
    fc->doing_retransmission   = FALSE;
    fc->one_packet_inflight     = FALSE;
    fc->current_tsn             = iTSN;
    fc->maxQueueLen             = maxQueueLen;

    rtx_set_remote_receiver_window(asok, new_rwnd);

    /* we leave the chunk list as it is !!!!!!!! */
    if ((fc->chunk_list) != NULL) {
        error_log(ERROR_MINOR, "FLOWCONTROL RESTART: List is deleted with chunks still queued...");
        g_list_foreach(fc->chunk_list, &free_list_element, GINT_TO_POINTER(1));
    }
    g_list_free(fc->chunk_list);
    fc->chunk_list = NULL;
    return;
}

/**
 * Deletes data occupied by a flow_control data structure
 * @param asok pointer to the association
 */
void fc_delete_flowcontrol(Association* asok)
{
    int count;
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
 
    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);

    event_log(INTERNAL_EVENT_0, "fc_delete_flowcontrol(): stop timers and delete flowcontrol data");
    fc_stop_timers(asok);
    for (count = 0; count < fc->number_of_addresses; count++) {
        fcpp = &g_array_index(fc->pathParams ,fcPathParam, count);
        assert(fcpp);
        free(fcpp->pathKeyPtr);
    }
    g_array_free(fc->pathParams, TRUE);

    if ((fc->chunk_list) != NULL) {
        error_log(ERROR_MINOR, "FLOWCONTROL : List is deleted with chunks still queued...");
        g_list_foreach(fc->chunk_list, &free_list_element, GINT_TO_POINTER(1));
    }
    g_list_free(fc->chunk_list);
    fc->chunk_list = NULL;
    free(fc);
}

/**
 * function to print debug data (flow control parameters of all paths)
 *  @param event_log_level  INTERNAL_EVENT_0 INTERNAL_EVENT_1 EXTERNAL_EVENT_X EXTERNAL_EVENT
 */
void fc_debug_cparams(Association* asok, short event_log_level)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    int count;


    if (event_log_level <= Current_event_log_) {
        fc = (fc_data *) mdi_readFlowControl(asok);
        assert(fc);

        event_log(event_log_level,   "===========================================================================");
        event_log(event_log_level,   "Debug-output for Congestion Control Parameters ! ");
        event_logii(event_log_level, "outstanding_bytes == %u; current_tsn == %u; ",
                    fc->outstanding_bytes, fc->current_tsn);
        event_logi(event_log_level,  "chunks queued in flowcontrol== %lu", fc->list_length);
        event_logi(event_log_level, "waiting_for_sack == %s", ((fc->waiting_for_sack == TRUE) ? "TRUE" : "FALSE"));
        event_logi(event_log_level,  "t3_retransmission_sent == %s ",
                   ((fc->t3_retransmission_sent == TRUE) ? "TRUE" : "FALSE"));

        for (count = 0; count < fc->number_of_addresses; count++) {
            fcpp = &g_array_index(fc->pathParams ,fcPathParam, count);
            assert(fcpp);

            event_log(event_log_level,"----------------------------------------------------------------------");
            event_logiii(event_log_level,   "%u: CWND=%u   SSTHRESH=%u",count, fcpp->cwnd, fcpp->ssthresh);
            event_logiiiii(event_log_level, "%u :  mtu from pm=%u   T3=%u   cwnd2=%u   pb_acked=%u",
                           count, pm_readPathSCTPMTU(asok, fcpp->pathKey),fcpp->T3_timer,fcpp->cwnd2,fcpp->partial_bytes_acked);
        }
        event_log(event_log_level,"============================================================================");
    }
    return;
}


/**
 * this function stops all currently running timers of the flowcontrol module
 * and may be called when the shutdown is imminent
 */
void fc_stop_timers(Association* asok)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    int result;
    unsigned int count;

    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);

    event_log(INTERNAL_EVENT_0, "fc_stop_timers()... ");
    if (!fc) {
        error_log(ERROR_MINOR, "fc_data instance not set !");
        return;
    }

    for (count = 0; count < fc->number_of_addresses; count++) {
        fcpp = &g_array_index(fc->pathParams ,fcPathParam, count);
        assert(fcpp);
        if (fcpp->T3_timer != 0) {
            result = sctp_stopTimer(fcpp->T3_timer);
            fcpp->T3_timer = 0;
            if (result == 1) error_log(ERROR_MINOR, "Timer not correctly reset to 0 !");
            event_logii(VVERBOSE, "Stopping T3-Timer(%d) = %d ", count, result);
        }
    }
}


/**
 *  timer controlled callback function, that reduces cwnd, if data is not sent within a certain time.
 *  As all timer callbacks, it takes three arguments, the timerID, and two pointers to relevant data
 *  @param  tid the id of the timer that has gone off
 *  @param  assoc  pointer to the association structure, where cwnd needs to be reduced
 *  @param  data2  currently unused == NULL
 */
int fc_reset_cwnd(Association* asok, unsigned int pathKey)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    int addressIndex;
    struct timeval now, resetTime;
    unsigned int rto;

    assert(asok);
    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);

    /* try opening up, if possible -- we may send again..... */
    if (fc->outstanding_bytes == 0) {
        fc->one_packet_inflight = FALSE;
    }

    addressIndex = mdi_getIndexForKey(asok, pathKey);
    if (addressIndex < 0) {
        event_logi(VERBOSE, "Invalid address key %u - returning", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    adl_gettime(&now);
    rto = pm_readRTO(asok, pathKey);
    resetTime = fcpp->last_send_time;
    adl_add_msecs_totime(&resetTime, rto);
    if (timercmp(&now, &resetTime, >)) {
        event_logii(INTERNAL_EVENT_0, "--------------> fc_reset_cwnd(path=%u, rto=%u) <------------------", pathKey, rto);
        /* path has been idle for at least an RTO worth of time */
        fcpp->cwnd = 2* MAX_MTU_SIZE;
        adl_gettime(&fcpp->last_send_time);
        event_logii(INTERNAL_EVENT_0, "--------------> fc_reset_cwnd(cwnd[%u] now: %u) <------------------", pathKey, fcpp->cwnd);
    }
    return SCTP_SUCCESS;
}

unsigned int fc_getNextActivePathKey(fc_data* fc, unsigned int start)
{
    unsigned int pathKey = 0;
    int count = 0, pathIndex;
    assert(fc);

    pathIndex = mdi_getIndexForKey(fc->myAssociation, start);

    while (count < fc->number_of_addresses) {
        pathIndex = (pathIndex+1)%fc->number_of_addresses;
        count++;
        pathKey = mdi_getKeyForIndex(fc->myAssociation, pathIndex);
        if (pm_readState(fc->myAssociation, pathKey) == PM_ACTIVE &&
            pm_pathConfirmed(fc->myAssociation, pathKey) == TRUE)
        {
             return pathKey;
        }
    }
    /* last resort */
    return pathKey;
}


/**
 * function that selects destination index for data chunks when they are sent,
 * or possibly new address when they are retransmitted.
 * @param  fc   pointer to the flow control structure
 * @param  dat  pointer to the data chunk that is to be sent
 * @param  data_retransmitted   has the chunk already been transmitted ?
 * @param  old_destination      if so, we pass a pointer to the index of the last address used, else NULL
 * @return index of the address where we should send this chunk to, now
 */
unsigned int fc_select_destination(fc_data * fc, chunk_data * dat,
                      boolean data_retransmitted, unsigned int *old_destination)
{
    unsigned int next = pm_getPrimaryPathKey(fc->myAssociation);

    event_logiii(VVERBOSE,
                 "fc_select_destination: chunk-tsn=%u, retrans=%s, primary path=%d ",
                 dat->chunk_tsn, ((data_retransmitted == TRUE) ? "TRUE" : "FALSE"), next);

    if (old_destination) {
        event_logi(VERBOSE, "fc_select_destination: old_dest = %u\n", *old_destination);
    } else {
        event_log(VERBOSE, "fc_select_destination: old_dest = NULL Pointer \n");
    }

    /* 1. return  a value that is equal to old_destination, if possible */
    if (old_destination)
    {
        if (pm_readState(fc->myAssociation, *old_destination) == PM_ACTIVE &&
            pm_pathConfirmed(fc->myAssociation, *old_destination) == TRUE)
        {
            return *old_destination;
        } else {
            return (fc_getNextActivePathKey(fc, *old_destination));
        }
    }
    /*
     * 2. try user selected address, else primary. We have checked before that the
     * user selected address is a valid path key
     */
    if (dat->initialDestinationKey != 0) {
        next = dat->initialDestinationKey;
    }

    /* 3. else try the primary */
    if ((data_retransmitted == FALSE) && (pm_readState(fc->myAssociation, next) == PM_ACTIVE) &&
        (pm_pathConfirmed(fc->myAssociation, next) == TRUE))
    {
        return next;
    }
    /* send retransmitted chunks to the next possible address */
    if (data_retransmitted == TRUE) next = dat->lastDestinationKey;

    return (fc_getNextActivePathKey(fc, next));
}


/**
 *  timer controlled callback function, called when T3 timer expires and data must be retransmitted
 *  This timer also adjusts the slow start threshold and cwnd values
 *  As all timer callbacks, it takes three arguments, the timerID, and two pointers to relevant data
 *  @param  tid the id of the timer that has gone off
 *  @param  assoc  pointer to the association structure to which this T3 timer belongs
 *  @param  data2  pointer to the index of the address where  T3 timer had been running
 */
void fc_timer_cb_t3_timeout(TimerID tid, void *assoc, void *addressKeyPtr)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    chunk_data **chunks;

    gboolean removed_association = FALSE;
    unsigned int /* retransmitted_bytes = 0,*/ oldListLen=0;
    unsigned int aKey = *((unsigned int*)addressKeyPtr);
    int addressIndex, num_of_chunks, count;
    int pathMTU;

    assert(assoc);
    assert(addressKeyPtr);

    fc = (fc_data *) mdi_readFlowControl((Association*)assoc);
    assert(fc);

    addressIndex = mdi_getIndexForKey((Association*)assoc, aKey);

    event_logi(INTERNAL_EVENT_0, "===============> fc_timer_cb_t3_timeout(address=%u) <========", aKey);

    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    fcpp->T3_timer = 0;

    num_of_chunks = rtx_readNumberOfUnackedChunks((Association*)assoc);

    event_logii(INTERNAL_EVENT_0, "Address-Index : %d, Number of Chunks==%d", addressIndex, num_of_chunks);

    if (num_of_chunks == 0) {
        event_log(VERBOSE, "Number of Chunks was 0 BEFORE calling rtx_t3_timeout - returning");
        return;
    }

    chunks = malloc(num_of_chunks * sizeof(chunk_data *));
    pathMTU = pm_readPathSCTPMTU(assoc, fcpp->pathKey);

    num_of_chunks = rtx_t3_timeout((Association*)assoc, aKey, pathMTU, chunks);

    if (num_of_chunks <= 0 && g_list_length(fc->chunk_list) == 0) {
        event_log(VERBOSE, "No Chunks to (re-)transmit - AFTER calling rtx_t3_timeout - returning");
        free(chunks);
        return;
    }

    oldListLen = fc->list_length;

    if (rtx_is_in_fast_recovery((Association*)assoc) == FALSE) {
        /* adjust ssthresh, cwnd - section 6.3.3.E1, respectively 7.2.3) */
        /* basically we halve the ssthresh, and set cwnd = mtu */
        fcpp->ssthresh = max(fcpp->cwnd / 2, 2 * pathMTU);
        fcpp->cwnd = pathMTU;
        /* see imp guide */
        fcpp->partial_bytes_acked = 0;
    }

    /* insert chunks to be retransmitted at the beginning of the list */
    /* make sure, that they are unique in this list ! */

    for (count = num_of_chunks - 1; count >= 0; count--) {
        if (g_list_find(fc->chunk_list, chunks[count]) == NULL)
        {
            if (chunks[count]->hasBeenAcked == FALSE)
            {
                 fc->chunk_list = g_list_insert_sorted(fc->chunk_list, chunks[count], (GCompareFunc) sort_tsn);
                 /* these chunks will not be counted, until they are actually sent again */
                 chunks[count]->counts_as_outstanding = FALSE;
                 fc->list_length++;
            }
        } else {
            event_logi(VERBOSE, "Chunk number %u already in list, skipped adding it", chunks[count]->chunk_tsn);
        }
    }
    event_log(VVERBOSE, "\n=========================================================================");
    event_log(VVERBOSE, "-----FlowControl (T3 timeout): Chunklist after reinserting chunks -------");
    chunk_list_debug(VVERBOSE, fc->chunk_list);
    fc_debug_cparams((Association*)assoc, VVERBOSE);
    event_log(VVERBOSE, "----------FlowControl (T3 timeout): Debug Output End ----------------");
    event_log(VVERBOSE, "=====================================================================\n");
    free(chunks);

    /* section 7.2.3 : assure that only one data packet is in flight, until a new sack is received */
    fc->waiting_for_sack        = TRUE;
    fc->t3_retransmission_sent  = FALSE; /* we may again send one packet ! */
    fc->one_packet_inflight     = FALSE;

    if (num_of_chunks > 0) {
        removed_association = pm_chunksRetransmitted((Association*)assoc, aKey);
    }

    if (removed_association) {
        event_log(INTERNAL_EVENT_0, "fc_timer_cb_t3_timeout: Association was TERMINATED by pm_chunksRetransmitted()");
        return;
    }
    pm_rto_backoff((Association*)assoc, aKey);

    fc_send_all_you_can((Association*)assoc, TRUE);

    return;
}

/**
 * function increases chunk's number of transmissions, stores used destination, updates counts per addresses
 */
void fc_update_chunk_data(fc_data * fc, chunk_data * dat, unsigned int destination)
{
    unsigned int rwnd;

    assert(fc);
    rwnd = rtx_read_remote_receiver_window(fc->myAssociation);

    dat->num_of_transmissions++;

    event_logiii(VERBOSE, "fc_update_chunk_data(TSN=%u, num_of_transmissions: %u, dest.: %u)",
                 dat->chunk_tsn, dat->num_of_transmissions, destination);

    if (dat->num_of_transmissions >= MAX_DEST) {
        error_log(ERROR_MINOR, "Maximum number of assumed transmissions exceeded ");
        dat->num_of_transmissions = MAX_DEST - 1;
    } else if (dat->num_of_transmissions < 1) {
        error_log(ERROR_FATAL, "Somehow dat->num_of_transmissions became less than 1 !");
        return;
    }

    /* this time we will send dat to destination */
    dat->lastDestinationKey = destination;
    /* this chunk must be counted as outstanding */
    dat->counts_as_outstanding = TRUE;

    /* section 6.2.1.B */
    /* leave peers arwnd untouched for retransmitted data !!!!!!!!! */
    if (dat->num_of_transmissions == 1) {
        fc->outstanding_bytes += dat->chunk_len;
        if (dat->chunk_len >= rwnd)
            rtx_set_remote_receiver_window(fc->myAssociation, 0);
        else
            rtx_set_remote_receiver_window(fc->myAssociation, rwnd - dat->chunk_len);
    }
    event_logi(VERBOSE, "outstanding_bytes overall: %u", fc->outstanding_bytes);
    return;
}

gboolean fc_send_okay(Association* asok,
                      fc_data* fc,
                      fcPathParam* fcpp,
                      chunk_data* nextChunk,
                      unsigned int totalSize,
                      unsigned int obpa,
                      int pathMTU)
{
    if (nextChunk == NULL) return FALSE;

    if (fc->doing_retransmission == TRUE) {
        if (totalSize + nextChunk->chunk_len > pathMTU) {
            fc->doing_retransmission = FALSE;
        } else {
            /* we must send at least on MTU worth of data without paying */
            /* attention to the CWND */
            return TRUE;
        }
    }

    if ((totalSize + obpa < (fcpp->cwnd + pathMTU-1)) &&
        (
         ((nextChunk->num_of_transmissions==0)&&(rtx_read_remote_receiver_window(fc->myAssociation) > nextChunk->chunk_len)) ||
          (fc->one_packet_inflight == FALSE) ||
          (nextChunk->num_of_transmissions > 0)) ) {
        return TRUE;
    }

    return FALSE;
}

/*
 * if a chunk is in the fc->chunk_list, returns the first chunk pointer in there,
 * else gets a message from streamengine, and queues all chunks in the fc->chunk_list,
 * and returns a pointer. If no chunk and no message are available, returns NULL.
 */
chunk_data* fc_get_chunk(Association* asok)
{
    fc_data* fc = NULL;
    chunk_data *dat = NULL;
    message_vector* mVec = NULL;
    SCTP_data_chunk* s_chunk = NULL;
    int count;
    
    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);
    /* ================================================================================= */
    /* check fc chunk_list, and if chunk in there, take it  */
    if (fc->chunk_list != NULL) {
        dat = g_list_nth_data(fc->chunk_list, 0);
        assert(dat);
        return dat;
    }
    /* else put one message (== 1..n chunks) in this queue (schedule from se) and  */
    /* take first one . Also: assign tsn's here                                     */
    mVec = se_schedule_next_message(asok);
    if (mVec == NULL) return NULL;
    event_logii(VVERBOSE, "Scheduling message (%u chunks, len: %u)", mVec->numberOfChunks, mVec->totalLength);
    for (count = 0; count < mVec->numberOfChunks; count++) {
        s_chunk = (SCTP_data_chunk*)mVec->cdat[count]->data;
        mVec->cdat[count]->chunk_tsn = fc->current_tsn++;
        mVec->cdat[count]->counts_as_outstanding = FALSE;
        s_chunk->tsn        =  htonl(mVec->cdat[count]->chunk_tsn);
        
        event_logiiii(VVERBOSE, "FREE: Queueing chunk #%d (tsn=%u, len=%u, address=%x)", count, mVec->cdat[count]->chunk_tsn,mVec->cdat[count]->chunk_len, mVec->cdat[count]);
        /* insert chunk at the list's tail */
        fc->chunk_list = g_list_append(fc->chunk_list, mVec->cdat[count]);
        fc->list_length++;
    }
    se_dequeue_message(asok, mVec);
    
    dat = g_list_nth_data(fc->chunk_list, 0);
    assert(dat);
    /* ================================================================================= */
    return dat;
}

/**
 *  function that checks whether we may transmit data that is currently in the send queue.
 *  Any time that some data chunk is added to the send queue, we must check, whether we can send
 *  the chunk, or must wait until cwnd opens up.
 *  @param fc_instance  pointer to the flowcontrol instance used here
 *  @return  0 for successful send event, -1 for error, 1 if nothing was sent
 */
int fc_send_all_you_can(Association* asok, gboolean doInitialRetransmit)
{
    fc_data* fc = NULL;
    chunk_data *dat = NULL;
    fcPathParam *fcpp = NULL;

    int addressIndex, result;
    unsigned int destination, oldDestination, peer_rwnd, rto_time, obpa;
    unsigned int queueLen = 0, total_size = 0, oldListLen = 0;
    int pathMTU;

    
    gboolean data_is_retransmitted = FALSE;
    gboolean lowest_tsn_is_retransmitted = FALSE;
    gboolean data_is_submitted = FALSE;

    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);
    peer_rwnd = rtx_read_remote_receiver_window(asok);
    oldListLen = fc->list_length + se_getQueuedChunksSendQueue(asok);
    
    event_logi(INTERNAL_EVENT_0, "Entering fc_send_all_you_can(rwnd=%u)... ", peer_rwnd);

    dat = fc_get_chunk(asok);

    if (dat == NULL) return -1;

    if (dat->num_of_transmissions >= 1)  data_is_retransmitted = TRUE;
    destination = fc_select_destination(fc, dat, data_is_retransmitted, NULL);

    addressIndex = mdi_getIndexForKey(fc->myAssociation, destination);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);
    pathMTU = pm_readPathSCTPMTU(asok, fcpp->pathKey);

    /* ---------------------------------- DEBUGGING ----------------------------------------------- */
    event_logiii(VERBOSE, "Called fc_select_destination == %d, chunk_len=%u, chunk_tsn=%u",
                             destination, dat->chunk_len, dat->chunk_tsn);
    event_logiiii(VERBOSE, "cwnd(%u) == %u, mtu == %u, MAX_MTU = %d ",
                              destination, fcpp->cwnd, pathMTU, MAX_MTU_SIZE);
    /* ---------------------------------- DEBUGGING ----------------------------------------------- */

    if (peer_rwnd == 0 && fc->one_packet_inflight == TRUE) {    /* section 6.1.A */
        event_log(VERBOSE, " NOT SENDING (peer rwnd == 0 and already one packet in flight) ");
        event_log(VERBOSE, "############# -> Returned in fc_send_all_you_can <- #############");
        return 1;
    }

    obpa = rtx_get_obpa(fc->myAssociation, destination, &fc->outstanding_bytes);
    if (obpa < 0) {
        error_log(ERROR_MAJOR, "Error in rtx_get_obpa called from fc_send_all_you_can()");
        return -1;
    }
    if (!doInitialRetransmit) {
        if (fcpp->cwnd <= obpa) {
            event_logiii(VERBOSE, " NOT SENDING (cwnd=%u, oustanding[%u]=%u) ",fcpp->cwnd, destination, obpa);
            event_log(VERBOSE, "############# -> Returned in fc_send_all_you_can <- #############");
            return 1;
        }
    }

    if (fc->waiting_for_sack == TRUE && data_is_retransmitted == TRUE) {
        /* make sure we send only one retransmission after T3 timeout */
        if (fc->t3_retransmission_sent == TRUE) {
            event_log(VERBOSE, "############# -> Returned in fc_send_all_you_can ############## ");
            return 1;
        }
    }

    /* check, if the destination path has been idle for more than one RTO */
    /* if so, reset CWND to 2*MTU                                         */
    fc_reset_cwnd(fc->myAssociation, destination);

    while (fc_send_okay(asok, fc, fcpp, dat, total_size, obpa, pathMTU)) {

        total_size += dat->chunk_len;

        /* ---------------------------------- DEBUGGING ----------------------------- */
        event_logiii(VVERBOSE, "Chunk: len=%u, tsn=%u, gap_reports=%u",
                     dat->chunk_len, dat->chunk_tsn, dat->gap_reports);
        event_logii(VVERBOSE, "Chunk: ack_time=%d, num_of_transmissions=%u",
                    dat->ack_time, dat->num_of_transmissions);
        /* ---------------------------------- DEBUGGING ----------------------------- */

        result = bu_put_Data_Chunk(fc->myAssociation, (SCTP_simple_chunk *) dat->data, destination);
        data_is_submitted = TRUE;
        adl_gettime(&(fcpp->last_send_time));

        event_logi(VERBOSE, "sent chunk (tsn=%u) to bundling -- now updating chunk...", dat->chunk_tsn);

        fc_update_chunk_data(fc, dat, destination);

        if (dat->num_of_transmissions == 1) {
            adl_gettime(&(dat->transmission_time));
            result = rtx_save_retrans_chunks(asok, dat);
        } else {
            if (lowest_tsn_is_retransmitted == FALSE) {
                /* must not be reset to FALSE here */
                lowest_tsn_is_retransmitted = rtx_is_lowest_tsn(fc->myAssociation, dat->chunk_tsn);
            }
        }
        fc->chunk_list = g_list_remove(fc->chunk_list, (gpointer) dat);
        fc->list_length--;
        fc->one_packet_inflight = TRUE;
        dat = NULL;

        dat = fc_get_chunk(asok);

        if (dat != NULL) {
            if (dat->num_of_transmissions >= 1)  data_is_retransmitted = TRUE;
            else if (dat->num_of_transmissions == 0) data_is_retransmitted = FALSE;

            oldDestination = destination;
            destination = fc_select_destination(fc, dat, data_is_retransmitted, &destination);
            if (destination != oldDestination) {
                obpa = rtx_get_obpa(fc->myAssociation, destination, &fc->outstanding_bytes);
                if (obpa < 0) {
                    error_log(ERROR_MAJOR, "rtx_get_obpa_error in fc_send_all_you_can !");
                    break;
                }
                total_size = 0;
            }
            addressIndex = mdi_getIndexForKey(fc->myAssociation, destination);
            fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
            assert(fcpp);

            if ((rtx_read_remote_receiver_window(fc->myAssociation) < dat->chunk_len && data_is_retransmitted == FALSE))  {
                break;
            }
            event_logii(VERBOSE, "After fc_select_destination...next destination == %u for CHUNK %u", destination, dat->chunk_tsn);
        }     /* if (dat != NULL) */

    }  /* while ((dat != NULL) && */


    if ((fc->waiting_for_sack == TRUE) && (fc->t3_retransmission_sent == FALSE)) {
        if (data_is_submitted == TRUE && data_is_retransmitted == TRUE) {
            event_log(VERBOSE, "Retransmission Condition in fc_send_all_you_can !!!!!!!! ");
            /* Keep me from retransmitting more than once */
            fc->t3_retransmission_sent = TRUE;
        }
    }

    /* ------------------------------ DEBUGGING ---------------------------------------- */
    event_log(VVERBOSE, "Printing Chunk List / Congestion Params in fc_send_all_you_can");
    chunk_list_debug(VVERBOSE, fc->chunk_list);
    /* fc_debug_cparams(asok, VVERBOSE);*/
    /* ------------------------------ DEBUGGING ---------------------------------------- */

    rto_time = pm_readRTO(fc->myAssociation, destination);

    if (fcpp->T3_timer == 0) { /* see section 5.1 */
        fcpp->T3_timer = adl_startTimer(rto_time,
                                        &fc_timer_cb_t3_timeout,
                                        TIMER_TYPE_RTXM,
                                        fc->myAssociation,
                                        fcpp->pathKeyPtr);
        event_logiii(INTERNAL_EVENT_0,
                     "fc_send_all_you_can started T3 Timer with RTO(%u)==%u msecs on address %u",
                     destination, rto_time, fcpp->pathKey);
    } else {
        /* restart only if lowest TSN is being retransmitted, else leave running */
        /* see section 6.1 */
        if (lowest_tsn_is_retransmitted) {
            event_logiii(INTERNAL_EVENT_0,
                         "RTX of lowest TSN: Restarted T3 Timer with RTO(%u)==%u msecs on address %u",
                         destination, rto_time, fcpp->pathKey);

            fcpp->T3_timer = adl_restartTimer(fcpp->T3_timer, rto_time);
        }
    }

    queueLen = fc->list_length  + se_getQueuedChunksSendQueue(asok);

    if (data_is_submitted == TRUE) {
        fc->one_packet_inflight = TRUE;
        bu_sendAllChunks(fc->myAssociation, destination);

        if (fc->maxQueueLen != 0) {
            if ((queueLen < fc->maxQueueLen && oldListLen >= fc->maxQueueLen) ||
                (queueLen > fc->maxQueueLen && oldListLen <= fc->maxQueueLen))   {
                 mdi_queueStatusChangeNotif(fc->myAssociation, SCTP_SEND_QUEUE, 0, queueLen);
            }
        }
        return 0;
    }

    if (fc->maxQueueLen != 0) {
        if ((queueLen < fc->maxQueueLen && oldListLen >= fc->maxQueueLen) ||
            (queueLen > fc->maxQueueLen && oldListLen <= fc->maxQueueLen))   {
             mdi_queueStatusChangeNotif(fc->myAssociation, SCTP_SEND_QUEUE, 0, queueLen);
        }
    }
    return 1;
}


/*
  this function checks whether T3 may be stopped, restarted or left running
  @param ad_idx  index of the destination address concerned (which may have a T3 timer running)
  @param all_acked   has all data been acked ?
  @param new_acked   have new chunks been acked ? CHECKME : has the ctsna advanced ?
*/
void fc_check_t3(fc_data *fc, fcPathParam *fcpp, boolean all_acked, boolean new_acked)
{
    int result, obpa;
    unsigned int count;
    fcPathParam *tmp_pp = NULL;
    /* int addressIndex; */

    assert(fc);

    event_logiii(INTERNAL_EVENT_0, "fc_check_t3(%u,all_acked=%s, new_acked=%s)... ", fcpp->pathKey,
                 (all_acked == TRUE) ? "true" : "false", (new_acked == TRUE) ? "true" : "false");

    obpa = rtx_get_obpa(fc->myAssociation, fcpp->pathKey, &fc->outstanding_bytes);
    if(obpa < 0) return;

    /* if all is acked, stop all timers, after update of fc->outstanding_bytes */
    if (all_acked == TRUE) {
        for (count = 0; count < fc->number_of_addresses; count++) {
            tmp_pp = &g_array_index(fc->pathParams ,fcPathParam, count);
            assert(tmp_pp);
            if (tmp_pp->T3_timer != 0) {
                result = sctp_stopTimer(tmp_pp->T3_timer);
                event_logii(INTERNAL_EVENT_0, "Stopped T3 Timer(%d), Result was %d ", count, result);
                tmp_pp->T3_timer = 0;
            }
        }
        return;
    }

    /* here, if all is acked for one path, stop only the path timer */
    if (obpa == 0) {
        if (fcpp->T3_timer != 0) {
           result = sctp_stopTimer(fcpp->T3_timer);
           event_logii(INTERNAL_EVENT_0, "Stopped T3 Timer(%d), Result was %d ", fcpp->T3_timer, result);
           fcpp->T3_timer = 0;
        }
        return;
    }

    /*
     * 6.3.2 R3) Whenever a SACK is received that acknowledges new data chunks
     * including the one with the earliest outstanding TSN on that address,
     * restart T3-rxt timer of that address with its current RTO.
     */
    if (new_acked == TRUE) {
        /* 6.2.4.4) Restart T3, if SACK acked lowest outstanding tsn, OR
         *                      we are retransmitting the first outstanding data chunk
         */
        if (fcpp->T3_timer != 0) {
            fcpp->T3_timer = adl_restartTimer(fcpp->T3_timer, pm_readRTO(fc->myAssociation, fcpp->pathKey));
            event_logii(INTERNAL_EVENT_0,
                        "Restarted T3 Timer with RTO==%u msecs on address %u",
                        pm_readRTO(fc->myAssociation, fcpp->pathKey), fcpp->pathKey);
        } else {
            fcpp->T3_timer = adl_startTimer(pm_readRTO(fc->myAssociation, fcpp->pathKey),
                                            &fc_timer_cb_t3_timeout, TIMER_TYPE_RTXM,
                                            fc->myAssociation , fcpp->pathKeyPtr);
            event_logii(INTERNAL_EVENT_0,
                        "Started T3 Timer with RTO==%u msecs on address %u",
                        pm_readRTO(fc->myAssociation, fcpp->pathKey), fcpp->pathKey);

        }
        return;
    }
    event_log(INTERNAL_EVENT_0, "Left all T3 Timers running...");
    /* else leave T3 running  */
    return;
}


int fc_dequeue_acked_chunks(Association* asok, unsigned int ctsna)
{
    fc_data* fc=NULL;
    chunk_data *dat = NULL;
    GList* tmp = NULL;
    guint oldQueueLen, newQueueLen;

    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);

    oldQueueLen = fc->list_length + se_getQueuedChunksSendQueue(asok);
    tmp = g_list_first(fc->chunk_list);

    while (tmp != NULL) {
        dat = (chunk_data*)tmp->data;
         if (before(dat->chunk_tsn, ctsna) || (dat->chunk_tsn == ctsna)) {
            tmp = g_list_next(tmp);
            fc->chunk_list = g_list_remove(fc->chunk_list, (gpointer) dat);
            fc->list_length--;
            event_logii(INTERNAL_EVENT_0, "Removed chunk %u from Flowcontrol-List, Listlength now %u",
                dat->chunk_tsn, fc->list_length);
        } else
            break;
    }

    if (fc->maxQueueLen != 0) {
        newQueueLen = fc->list_length + se_getQueuedChunksSendQueue(asok);
        if (oldQueueLen >= fc->maxQueueLen && newQueueLen < fc->maxQueueLen) {
            mdi_queueStatusChangeNotif(fc->myAssociation, SCTP_SEND_QUEUE, 0, newQueueLen);
        }
    }
    return 0;
}

int fc_adjustCounters(fc_data *fc, fcPathParam *fcpp,
                        unsigned int num_acked,
                        gboolean all_data_acked,
                        gboolean new_data_acked)

{
    int obpa;
    unsigned int rtt_time, diff;
    struct timeval last_update, now;

    obpa = rtx_get_obpa(fc->myAssociation, fcpp->pathKey, &fc->outstanding_bytes);

    /* see section 6.2.1, section 6.2.2 */
    if (fcpp->cwnd <= fcpp->ssthresh) { /* SLOW START */
        fcpp->partial_bytes_acked = 0;
        if (new_data_acked == TRUE) {
            fcpp->cwnd += min(MAX_MTU_SIZE, num_acked);
            adl_gettime(&(fcpp->time_of_cwnd_adjustment));
        }
    } else {                    /* CONGESTION AVOIDANCE, as per section 6.2.2 */
        if (new_data_acked == TRUE) {
            fcpp->partial_bytes_acked += num_acked;
            event_logii(VVERBOSE, "CONG. AVOIDANCE : new data acked: increase PBA(key %u) to %u",
                fcpp->pathKey,fcpp->partial_bytes_acked);
        }

        rtt_time = pm_readSRTT(fc->myAssociation, fcpp->pathKey);
        last_update = fcpp->time_of_cwnd_adjustment;
        adl_add_msecs_totime(&last_update, rtt_time);
        adl_gettime(&now);
        diff = adl_timediff_to_msecs(&now, &last_update); /* a-b */
        event_logii(VVERBOSE, "CONG. AVOIDANCE : rtt_time=%u diff=%d", rtt_time, diff);

        if (diff >= 0) {
            if ((fcpp->partial_bytes_acked >= fcpp->cwnd) && (obpa >= fcpp->cwnd)) {
                fcpp->cwnd += MAX_MTU_SIZE;
                fcpp->partial_bytes_acked -= fcpp->cwnd;
                /* update time of window adjustment (i.e. now) */
                event_log(VVERBOSE, "CONG. AVOIDANCE : updating time of adjustment !! NOW ! ");
                adl_gettime(&(fcpp->time_of_cwnd_adjustment));
            }
            event_logii(VERBOSE, "CONG. AVOIDANCE : updated counters: %u bytes outstanding, cwnd=%u",
                        fc->outstanding_bytes, fcpp->cwnd);
        }

        /* see section 7.2.2 */
        if (all_data_acked == TRUE) fcpp->partial_bytes_acked = 0;

    }

    return SCTP_SUCCESS;
}


/**
 * function called by Reliable Transfer, when it requests retransmission
 * in SDL diagram this signal is called (Req_RTX, RetransChunks)
 * @param  all_data_acked indicates whether or not all data chunks have been acked
 * @param   new_data_acked indicates whether or not new data has been acked
 * @param   num_acked number of bytes that have been newly acked, else 0
 * @param   number_of_addresses so many addresses may have outstanding bytes
 *          actually that value may also be retrieved from the association struct (?)
 * @param   number_of_rtx_chunks number indicating, how many chunks are to be retransmitted in on datagram
 * @param   chunks  array of pointers to data_chunk structures. These are to be retransmitted
 * @return   -1 on error, 0 on success, (1 if problems occurred ?)
 */
int fc_fast_retransmission(Association* asok,
                     unsigned int addressKey,
                     unsigned int arwnd, unsigned int ctsna,
                     unsigned int rtx_bytes, boolean all_data_acked,
                     boolean new_data_acked, unsigned int num_acked,
                     int number_of_rtx_chunks, chunk_data ** chunks)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    int addressIndex;
    int count, result;

    unsigned int peer_rwnd, oldListLen;

    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);

    event_logii(INTERNAL_EVENT_0, "fc_fast_retransmission...bytes acked=%u on address key %u ",
                    num_acked, addressKey);

    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(asok, VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    oldListLen = fc->list_length;
    fc->t3_retransmission_sent = FALSE; /* we have received a SACK, so reset this */
    /* the other guy is still alive */
    fc->waiting_for_sack = FALSE;

    for (count = 0; count < number_of_rtx_chunks; count++) {
        event_logi(VERBOSE, "fc_fast_retransmission: Got TSN==%u for RTXmit\n", chunks[count]->chunk_tsn);
    }

    addressIndex = mdi_getIndexForKey(asok, addressKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    result = fc_adjustCounters(fc, fcpp, num_acked, all_data_acked, new_data_acked);

    result = -2;

    /*
     * We HAVE retransmission, so DO UPDATE OF WINDOW PARAMETERS , see section 7.2.3 and 7.2.4 and
     * the implementors guide
     */
    if (rtx_is_in_fast_recovery(asok) == FALSE) {
        fcpp->ssthresh =  max(fcpp->cwnd / 2, 2 * pm_readPathSCTPMTU(asok, fcpp->pathKey));
        fcpp->cwnd = fcpp->ssthresh;
        /* as per implementors guide */
        fcpp->partial_bytes_acked = 0;
        rtx_enter_fast_recovery(asok);
    }
    event_logiiii(VERBOSE, "fc_fast_retransmission: now %u bytes outstanding, cwnd[%u]=%u, ssthresh=%u",
                  fc->outstanding_bytes, fcpp->pathKey, fcpp->cwnd, fcpp->ssthresh);

    /* This is to be an ordered list containing no duplicate entries ! */
    for (count = number_of_rtx_chunks - 1; count >= 0; count--) {

        if (g_list_find(fc->chunk_list, chunks[count]) != NULL){
            event_logii(VERBOSE, "chunk_tsn==%u, count==%u already in the list -- skip to next",
                        chunks[count]->chunk_tsn, count);
            continue;
        }
        event_logii(INTERNAL_EVENT_0, "inserting chunk_tsn==%u, count==%u in the list",
                    chunks[count]->chunk_tsn, count);

        fc->chunk_list = g_list_insert_sorted(fc->chunk_list, chunks[count], (GCompareFunc) sort_tsn);
        fc->list_length++;
    }

    event_log(VVERBOSE, "fc_fast_retransmission: FlowControl Chunklist after Re-Insertion");
    chunk_list_debug(VVERBOSE, fc->chunk_list);

    /*
     * FIXME: optimization:
     * both  fc_adjustCounters() and  fc_check_t3() call  rtx_get_obpa().
     * should not be necessary, really...
     */
    fc_check_t3(fc, fcpp, all_data_acked, new_data_acked);

    /* section 6.2.1.D ?? */
    if (arwnd >= fc->outstanding_bytes) {
        peer_rwnd = arwnd - fc->outstanding_bytes;
    } else {
        peer_rwnd = 0;
    }
    /* section 6.2.1.C */
    rtx_set_remote_receiver_window(asok, peer_rwnd);

    if (fc->outstanding_bytes==0) {
        fc->one_packet_inflight = FALSE;
    } 

    /* send as many to bundling as allowed, requesting new destination address */
    result = fc_send_all_you_can(fc->myAssociation, TRUE);
    /* make sure that SACK chunk is actually sent ! */
    if (result != 0) bu_sendAllChunks(asok, 0);

    return SCTP_SUCCESS;
}     /* end: fc_fast_retransmission */


/**
 * function called by Reliable Transfer, after it has got a SACK chunk
 * in SDL diagram this signal is called SACK_Info
 * @param   all_data_acked indicates whether or not all data chunks have been acked
 * @param   new_data_acked indicates whether or not new data has been acked
 * @param   num_acked number of bytes that have been newly acked, else 0
 * @param   number_of_addresses so many addresses may have outstanding bytes
 *          actually that value may also be retrieved from the association struct (?)
 * @param   num_acked_per_address array of integers, that hold number of bytes acked for each address
 */
void fc_sack_info(Association* asok,
                  unsigned int addressKey,
                  unsigned int arwnd, unsigned int ctsna,
                  boolean all_data_acked, boolean new_data_acked,
                  unsigned int num_acked)
{
    fc_data *fc = NULL;
    fcPathParam *fcpp = NULL;
    int addressIndex;
    int result = 0;

    fc = (fc_data *) mdi_readFlowControl(asok);
    assert(fc);
    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(asok, VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    event_logii(INTERNAL_EVENT_0, "fc_sack_info...bytes acked=%u on address key %u ",
                    num_acked, addressKey);

    fc->t3_retransmission_sent = FALSE; /* we have received a SACK, so reset this */

    /* the other guy is still alive */
    fc->waiting_for_sack = FALSE;

    addressIndex = mdi_getIndexForKey(asok, addressKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    result = fc_adjustCounters(fc, fcpp, num_acked, all_data_acked, new_data_acked);

    fc_check_t3(fc, fcpp, all_data_acked, new_data_acked);

    if (fc->outstanding_bytes == 0) {
        fc->one_packet_inflight = FALSE;
    } 

    /* section 6.2.1.C */
    if (arwnd > fc->outstanding_bytes)
        rtx_set_remote_receiver_window(asok, arwnd - fc->outstanding_bytes);
    else
        rtx_set_remote_receiver_window(asok, 0);

    fc_send_all_you_can(asok, FALSE);
}    /* end: fc_sack_info  */



int fc_dequeueUnackedChunk(Association* asok, unsigned int tsn)
{
    fc_data *fc = NULL;
    chunk_data *dat = NULL;
    GList *tmp = NULL;
    gboolean found = FALSE;

    fc = (fc_data *) mdi_readFlowControl(asok);
    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    dat = g_list_nth_data(fc->chunk_list, 0);
    tmp = fc->chunk_list;
    while (dat != NULL && tmp != NULL) {
        event_logii(VVERBOSE, "fc_dequeueOldestUnsentChunks(): checking chunk tsn=%u, num_rtx=%u ", dat->chunk_tsn, dat->num_of_transmissions);
        if (dat->chunk_tsn == tsn) {
            found = TRUE;
            break;
        } else {
            tmp = g_list_next(tmp);
            if (tmp != NULL) {
                dat = (chunk_data*)tmp->data;
            } else {
                dat = NULL;
            }
        }
    }
    if (found) { /* delete */
        fc->chunk_list = g_list_remove(fc->chunk_list, (gpointer) dat);
        fc->list_length--;
        event_log(VVERBOSE, "fc_dequeueUnackedChunk(): checking list");
        chunk_list_debug(VVERBOSE, fc->chunk_list);
        return 1;
    }
    /* else */
    return 0;
}

int fc_dequeueOldestUnsentChunk(Association* asok,
                                unsigned char *buf, unsigned int *len, unsigned int *tsn,
                                unsigned short *sID, unsigned short *sSN,unsigned int* pID,
                                unsigned char* flags, gpointer* ctx)
{
    fc_data *fc = NULL;
    chunk_data *dat = NULL;
    GList *tmp = NULL;
    SCTP_data_chunk* dchunk = NULL;
    int listlen;

    fc = (fc_data *) mdi_readFlowControl(asok);
    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    listlen =  fc_readNumberOfUnsentChunks(asok);

    if (listlen <= 0)           return SCTP_UNSPECIFIED_ERROR;
    if (fc->chunk_list == NULL) return SCTP_UNSPECIFIED_ERROR;

    dat = g_list_nth_data(fc->chunk_list, 0);
    tmp = fc->chunk_list;
    while (dat != NULL && tmp != NULL) {
        event_logii(VVERBOSE, "fc_dequeueOldestUnsentChunks(): checking chunk tsn=%u, num_rtx=%u ", dat->chunk_tsn, dat->num_of_transmissions);
        if (dat->num_of_transmissions != 0) {
            tmp = g_list_next(tmp);
            dat = (chunk_data*)tmp->data;
        /* should be a sorted list, and not happen here */
        } else break;
    }
    if ((*len) <  (dat->chunk_len - FIXED_DATA_CHUNK_SIZE)) return SCTP_BUFFER_TOO_SMALL;

    event_logii(VVERBOSE, "fc_dequeueOldestUnsentChunks(): returning chunk tsn=%u, num_rtx=%u ", dat->chunk_tsn, dat->num_of_transmissions);

    dchunk = (SCTP_data_chunk*) dat->data;
    *len = dat->chunk_len - FIXED_DATA_CHUNK_SIZE;
    memcpy(buf, dchunk->data, dat->chunk_len - FIXED_DATA_CHUNK_SIZE);
    *tsn = dat->chunk_tsn;
    *sID = ntohs(dchunk->stream_id);
    *sSN = ntohs(dchunk->stream_sn);
    *pID = ntohl(dchunk->protocolId);
    *flags = dchunk->chunk_flags;
    *ctx = dat->context;
    fc->chunk_list = g_list_remove(fc->chunk_list, (gpointer) dat);
    fc->list_length--;
    /* be careful ! data may only be freed once: this module ONLY takes care of _untransmitted_ chunks */
    free(dat);
    event_log(VVERBOSE, "fc_dequeueOldestUnsentChunks(): checking list");
    chunk_list_debug(VVERBOSE, fc->chunk_list);
    return (listlen-1);
}


int fc_readNumberOfUnsentChunks(Association* asok)
{
    int queue_len = 0;
    fc_data *fc = NULL;
    GList* tmp = NULL;
    chunk_data *cdat = NULL;

    fc = (fc_data *) mdi_readFlowControl(asok);
    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (fc->chunk_list == NULL) return 0;
    tmp = g_list_first(fc->chunk_list);
    while (tmp) {
        cdat = (chunk_data*)tmp->data; /* deref list data */
        event_logii(VERBOSE, "fc_readNumberOfUnsentChunks(): checking chunk tsn=%u, num_rtx=%u ", cdat->chunk_tsn, cdat->num_of_transmissions);
        if (cdat->num_of_transmissions == 0) queue_len++;
        tmp = g_list_next(tmp);
    }
    event_logi(VERBOSE, "fc_readNumberOfUnsentChunks() returns %u", queue_len);
    return queue_len;
}


/**
 * function returns number of chunks, that are waiting in the transmission queue
 * @return size of the send queue of the current flowcontrol module, or SCTP_MODULE_NOT_FOUND
 */
int fc_readNumberOfQueuedChunks(Association* asok)
{
    unsigned int queue_len;
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (fc->chunk_list != NULL){
        queue_len = fc->list_length;
    }
    else
        queue_len=0;

    event_logi(VERBOSE, "fc_readNumberOfQueuedChunks() returns %u", queue_len);
    return queue_len;
}




/**
 * Function returns cwnd value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current cwnd value, else SCTP_MODULE_NOT_FOUND, SCTP_PARAMETER_PROBLEM
 */
int fc_readCWND(Association* asok, unsigned int pathKey)
{
    fc_data *fc;
    fcPathParam *fcpp = NULL;
    int addressIndex;

    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "Wrong Address Parameter %u!", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
    addressIndex = mdi_getIndexForKey(asok, pathKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    return (int)fcpp->cwnd;
}
/**
 * Function returns cwnd2 value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current cwnd value, else SCTP_MODULE_NOT_FOUND, SCTP_PARAMETER_PROBLEM
 */
int fc_readCWND2(Association* asok, unsigned int pathKey)
{
    fc_data *fc;
    fcPathParam *fcpp = NULL;
    int addressIndex;

    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "Wrong Address Parameter %u!", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
    addressIndex = mdi_getIndexForKey(asok, pathKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    return (int)fcpp->cwnd2;
}

/**
 * Function returns ssthresh value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current cwnd value, else SCTP_MODULE_NOT_FOUND, SCTP_PARAMETER_PROBLEM
 */
int fc_readSsthresh(Association* asok, unsigned int pathKey)
{
    fc_data *fc;
    fcPathParam *fcpp = NULL;
    int addressIndex;

    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "Wrong Address Parameter %u!", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
    addressIndex = mdi_getIndexForKey(asok, pathKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    return (int)fcpp->ssthresh;
}


/**
 * Function returns mtu value of a certain path.
 * @param pathKey    path key of which we want to know the mtu
 * @return current MTU value, else SCTP_MODULE_NOT_FOUND, SCTP_PARAMETER_PROBLEM
 */
 /*
int fc_readMTU(Association* asok, unsigned int pathKey)
{
    fc_data *fc;
    fcPathParam *fcpp = NULL;
    int addressIndex;

    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "Wrong Address Parameter %u!", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
    addressIndex = mdi_getIndexForKey(asok, pathKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    return (int)fcpp->mtu;
}
*/

/**
 * Function returns the partial bytes acked value of a certain path.
 * @param path_id    path index of which we want to know the PBA
 * @return current PBA value, else -1
 */
int fc_readPBA(Association* asok, unsigned int pathKey)
{
    fc_data *fc;
    fcPathParam *fcpp = NULL;
    int addressIndex;

    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "Wrong Address Parameter %u!", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
    addressIndex = mdi_getIndexForKey(asok, pathKey);
    fcpp = &g_array_index(fc->pathParams ,fcPathParam, addressIndex);
    assert(fcpp);

    return (int)fcpp->partial_bytes_acked;
}

/**
 * Function returns the outstanding byte count value of this association.
 * @return current outstanding_bytes value, else -1
 */
int fc_readOutstandingBytes(Association* asok)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    return (int)fc->outstanding_bytes;
}

int fc_get_maxSendQueue(Association* asok, unsigned int * queueLen)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    *queueLen = fc->maxQueueLen;
    return SCTP_SUCCESS;

}

int fc_set_maxSendQueue(Association* asok, unsigned int maxQueueLen)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    fc->maxQueueLen = maxQueueLen;
    return SCTP_SUCCESS;
}

int fc_addDestinationAddress(Association* asok,
                             unsigned int newKey,
                             int newIndex)
{
    fc_data *fc;
    fcPathParam fcpp;

    fc = (fc_data *) mdi_readFlowControl(asok);
    
    if (!fc) {
        error_log(ERROR_MAJOR, "fc_addDestinationAddress: flow_control not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    fcpp.T3_timer   = 0;          /* i.e. timer not running */
    fcpp.cwnd       = 2 * MAX_MTU_SIZE;
    fcpp.cwnd2      = 0L;
    fcpp.partial_bytes_acked = 0L;
    fcpp.ssthresh   = fc->announced_rwnd;
    fcpp.pathKey    = newKey;
    fcpp.pathKeyPtr = malloc(sizeof(unsigned int));
    *(fcpp.pathKeyPtr) = fcpp.pathKey;

    adl_gettime( &(fcpp.time_of_cwnd_adjustment) );

    fc->pathParams = g_array_append_val(fc->pathParams, fcpp);

    fc->number_of_addresses++;
    
    return SCTP_SUCCESS;

}

int fc_delDestinationAddress(Association* asok,
                             unsigned int delKey,
                             int delIndex,
                             unsigned int altKey,
                             int altIndex)
{
    fc_data *fc;
    fcPathParam *fcpp = NULL;
	int result;
    unsigned int rto_time = 0;

    fc = (fc_data *) mdi_readFlowControl(asok);

    if (!fc) {
        error_log(ERROR_MAJOR, "fc_delDestinationAddress: flow control instance not set !");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (fc->number_of_addresses == 1) {
		error_log(ERROR_FATAL, "fc_delDestinationAddress: Only ONE address left...");
        return SCTP_UNSPECIFIED_ERROR;
    }		

    /* if RTX/T3 timer is running, start this timer for replacement path, if that path
       doesn't already have a timer running */
    fcpp = &g_array_index(fc->pathParams, fcPathParam, delIndex);
    assert(fcpp);

    free(fcpp->pathKeyPtr);

    if (fcpp->T3_timer != 0) {
        result = sctp_stopTimer(fcpp->T3_timer);
        fcpp->T3_timer = 0;
        if (result == 1) error_log(ERROR_MINOR, "Timer not correctly reset to 0 !");
        event_logii(VVERBOSE, "Stopping T3-Timer(%d) = %d ", delIndex, result);

        fcpp = &g_array_index(fc->pathParams, fcPathParam, altIndex);
        assert(fcpp);
        if (fcpp->T3_timer == 0) {
            rto_time = pm_readRTO(fc->myAssociation, altKey);
            fcpp->T3_timer = adl_startTimer(rto_time,
                                           &fc_timer_cb_t3_timeout,
                                            TIMER_TYPE_RTXM,
                                            fc->myAssociation,
                                            fcpp->pathKeyPtr);
        }
    }
    fc->pathParams = g_array_remove_index(fc->pathParams, delIndex);
    fc->number_of_addresses--;
       
    return SCTP_SUCCESS;

}


