/*
 *  $Id: errorhandler.c,v 1.1.1.1 2002/11/20 14:51:16 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This module decodes error chunks and handles them accordingly.
 *          They are sent to the module that can treat these chunks...
 *          For chunks that are not treated in any way, we log their occurrence
 */

 
#include "globals.h"            /* for chunk struct definition */
#include "chunkHandler.h"
#include "bundling.h"
#include "sctp-control.h"


void eh_init_errorhandler(void)
{
}

/*
 * eh_new: Create a new instance and returns a pointer to its data.
 */
void *eh_new(void)
{
   return NULL;
}

/* 
 * eh_delete: Deletes a bundling instance
 * 
 * Params: Pointer/handle which was returned by eh_new()
 */
void eh_delete(void *instancePtr)
{

}

/*
 *  eh_recv_chunk gets a pointer to an error chunk and decodes it
 *  accordingly....
 *  @return  error code, 0 for success, less than one for error
 */
int eh_recv_chunk(Association* asok, SCTP_simple_chunk * errchunk)
{
    SCTP_error_chunk *chunk;
    SCTP_vlparam_header *header;
    SCTP_error_cause *cause;
    unsigned char* data;

    unsigned short err_cause;
    unsigned short cause_len;
    int result = (-1);

    chunk = (SCTP_error_chunk *) errchunk;
    cause = (SCTP_error_cause *) chunk->data;
    data =  cause->cause_information;

    err_cause = ntohs(cause->cause_code);
    cause_len = ntohs(cause->cause_length);
    switch (err_cause) {
    case ECC_INVALID_STREAM_ID:
        event_logi(EXTERNAL_EVENT, "Invalid Stream Id Error with Len %u ", cause_len);
        break;
    case ECC_MISSING_MANDATORY_PARAM:
        event_logi(EXTERNAL_EVENT, "Missing Mandatory Parameter Error, Len %u ", cause_len);
        break;
    case ECC_STALE_COOKIE_ERROR:
        event_logi(EXTERNAL_EVENT, "Stale Cookie Error, Len %u ", cause_len);
        scr_staleCookie(asok, (SCTP_simple_chunk *) errchunk);
        result = 0;
        break;
    case ECC_OUT_OF_RESOURCE_ERROR:
        event_logi(EXTERNAL_EVENT, "Out Of Resource Error with Len %u ", cause_len);
        break;
    case ECC_UNRESOLVABLE_ADDRESS:
        event_logi(EXTERNAL_EVENT, "Unresovable Address Error with Len %u ", cause_len);
        break;
    case ECC_UNRECOGNIZED_CHUNKTYPE:
        event_logi(EXTERNAL_EVENT, "Unrecognized Chunktype Len %u ", cause_len);
        break;
    case ECC_INVALID_MANDATORY_PARAM:
        event_logi(EXTERNAL_EVENT, "Invalid Mandatory Parameter : Len %u ", cause_len);
        break;
    case ECC_UNRECOGNIZED_PARAMS:
        event_logi(EXTERNAL_EVENT, "Unrecognized Params Error with Len %u ", cause_len);
        header = (SCTP_vlparam_header*)data;
        if (ntohs(header->param_type) == VLPARAM_PRSCTP) {
            /* set peer does not understand PRSCTP - do not use it in this ASSOC ! */
            event_log(EXTERNAL_EVENT, "Unrecognized Parameter: PRSCTP ");
        }
        break;
    case ECC_NO_USER_DATA:
        event_logi(EXTERNAL_EVENT, "No User Data Error with Len %u ", cause_len);
        break;
    case ECC_COOKIE_RECEIVED_DURING_SHUTDWN:
        event_logi(EXTERNAL_EVENT, "Error : Cookie Received During Shutdown, Len: %u ", cause_len);
        break;
    default:
        error_logii(ERROR_MINOR, "Unrecognized Error Cause %u with Len %u ", err_cause, cause_len);
    }
    return result;
}

/**
 * function to trigger sending of error chunk, after receiving an invalid stream id
 * @return error value, 0 on success, -1 on error
 */
int eh_send_invalid_streamid(Association* asok, unsigned short streamid)
{
    return -1;
}


int eh_send_invalid_chunk(Association* asok)
{
    return -1;
}


/**
 * function to put an error_chunk with type UNKNOWN PARAMETER
 * @return error value, 0 on success, -1 on error
 */
int eh_send_unrecognized_chunktype(Association* asok, unsigned char* faulty_chunk, unsigned short length)
{
    ChunkID errorCID;

    /* build chunk */
    errorCID = ch_makeErrorChunk();
    /* add parameters */
    ch_enterErrorCauseData(errorCID, ECC_UNRECOGNIZED_CHUNKTYPE, length, faulty_chunk);

    bu_put_Ctrl_Chunk(asok, ch_chunkString(errorCID), 0);
    return bu_sendAllChunks(asok, 0);
}

/**
 * function to trigger sending of error chunk, after mandatory parameter(s) was(were) missing
 * @return error value, 0 on success, -1 on error
 */
int eh_send_missing_mandatory_param(unsigned int number, unsigned short *param_types)
{
    return -1;
}

/**
 * function to trigger sending of error chunk, after receiving an invalid stream id
 * @param number number of pointers passed as second argument
 * @param addresses pointers (or array of pointers) to unrecognized addresses
 * @return error value, 0 on success, -1 on error
 */
int eh_send_unresolvable_address(unsigned int number, unsigned char *addresses)
{
    return -1;
}
