/*
 *  $Id: echo_server.c,v 1.7 2004/01/13 16:13:14 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * echo_server.c  - main program module
 *
 */

#include "sctp_wrapper.h"


#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#define min(x,y)            (x)<(y)?(x):(y)

#define ECHO_PORT                             7
#define MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES    10
#define MAXIMUM_PAYLOAD_LENGTH             8192
#define MAXIMUM_NUMBER_OF_IN_STREAMS         17
#define MAXIMUM_NUMBER_OF_OUT_STREAMS        17

struct ulp_data {
    int maximumStreamID;
};

static unsigned char  localAddressList[MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES][SCTP_MAX_IP_LEN];
static unsigned short noOfLocalAddresses = 0;

static int verbose          = 0;
static int vverbose         = 0;
static int unknownCommand   = 0;
static int sendOOTBAborts   = 1;
static int timeToLive       = SCTP_INFINITE_LIFETIME;

void printUsage(void)
{
    printf("Usage:   echo_server [options]\n");
    printf("options:\n");
    printf("-i       ignore OOTB packets\n");
    printf("-s       source address\n");
    printf("-t       time to live in ms\n");
    printf("-v       verbose mode\n");
    printf("-V       very verbose mode\n");   
}

void getArgs(int argc, char **argv)
{
    int c;
    extern char *optarg;
    extern int optind;

    while ((c = getopt(argc, argv, "hs:t:ivV")) != -1)
    {
        switch (c) {
        case 'h':
            printUsage();
            exit(0);
        case 's':
            if ((noOfLocalAddresses < MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES) &&
                (strlen(optarg) < SCTP_MAX_IP_LEN  )) {
                strcpy(localAddressList[noOfLocalAddresses], optarg);
                noOfLocalAddresses++;
            };
            break;
        case 'i':
            sendOOTBAborts = 0;
            break;
        case 't':
            timeToLive = atoi(optarg);
            break;
        case 'v':
            verbose = 1;
            break;
        case 'V':
            verbose = 1;
            vverbose = 1;
            break;
        default:
            unknownCommand = 1;
            break;
        }
    }
}
void checkArgs(void)
{
    int abortProgram;
    int printUsageInfo;
    
    abortProgram = 0;
    printUsageInfo = 0;
    
    if (noOfLocalAddresses == 0) {
#ifdef HAVE_IPV6
        strcpy(localAddressList[noOfLocalAddresses], "::0");
#else
        strcpy(localAddressList[noOfLocalAddresses], "0.0.0.0");
#endif
        noOfLocalAddresses++;
    }
    if (unknownCommand ==1) {
         printf("Error:   Unkown options in command.\n");
         printUsageInfo = 1;
    }
    
    if (printUsageInfo == 1)
        printUsage();
    if (abortProgram == 1)
        exit(-1);
}

void dataArriveNotif(unsigned int assocID, unsigned int streamID, unsigned int len,
                     unsigned short streamSN,unsigned int TSN, unsigned int protoID,
                     unsigned int unordered, void* ulpDataPtr)
{
    unsigned char chunk[MAXIMUM_PAYLOAD_LENGTH];
    int length, numMessages, result;
    unsigned short ssn;
    unsigned int tsn;

    if (vverbose) {  
      fprintf(stdout, "%-8x: Data arrived (%u bytes on stream %u, %s)\n",
                      assocID, len, streamID, (unordered==SCTP_ORDERED_DELIVERY)?"ordered":"unordered");
      fflush(stdout);
    }
    /* read it */
    numMessages = (len / MAXIMUM_PAYLOAD_LENGTH) + 1;
    length = MAXIMUM_PAYLOAD_LENGTH;
    while (numMessages > 0) {
        result = SCTP_receive(assocID, streamID, chunk, &length,&ssn, &tsn, SCTP_MSG_DEFAULT);
        /* and send it */
        if (vverbose) {
            fprintf(stdout, "%-8x: Send Data (%u bytes on stream %u, %s), old result was %d\n",
                      assocID, length, streamID, (unordered==SCTP_ORDERED_DELIVERY)?"ordered":"unordered", result);
            fflush(stdout);
        }
        if (result >= 0) {
            SCTP_send(assocID,
                      min(streamID, ((struct ulp_data *) ulpDataPtr)->maximumStreamID),
                      chunk, length,
                      protoID,
                      SCTP_USE_PRIMARY, SCTP_NO_CONTEXT, timeToLive, unordered, SCTP_BUNDLING_DISABLED);
        }
        numMessages--;
    }
}

void sendFailureNotif(unsigned int assocID,
                      unsigned char *unsent_data, unsigned int dataLength, unsigned int *context, void* ulpDataPtr)
{
  if (verbose) {  
    fprintf(stdout, "%-8u: Send failure\n", assocID);
    fflush(stdout);
  }
}

void networkStatusChangeNotif(unsigned int assocID, unsigned int destAddrIndex, int newState, void* ulpDataPtr)
{
    SCTP_AssociationStatus assocStatus;
    SCTP_PathStatus pathStatus;
    unsigned int pathIndex;
    int pathKey, i;
    unsigned char status[40];

    if (newState == SCTP_PATH_OK) strcpy(status, "ACTIVE STATE");
    if (newState == SCTP_PATH_UNREACHABLE) strcpy(status,"INACTIVE");
    if (newState == SCTP_PATH_ADDED) strcpy(status, "PATH ADDED");
    if (newState == SCTP_PATH_REMOVED) strcpy(status, "PATH DELETED");
    if (newState == SCTP_ASCONF_SUCCEEDED) strcpy(status, "ASCONF SUCCEEDED");
    if (newState == SCTP_ASCONF_FAILED) strcpy(status, "ASCONF FAILED");
    if (newState == SCTP_PATH_CONFIRMED) strcpy(status, "PATH CONFIRMED");
    if (newState == SCTP_PATH_UNCONFIRMED) strcpy(status, "PATH UNCONFIRMED");

    if (verbose) {  
        fprintf(stdout, "%-8x: Network status change notification: path %u status is  %s (%d)\n", 
                        assocID, destAddrIndex, status, newState);
        fflush(stdout);
    }
    
    /* if the primary path has become inactive */
    if ((newState == SCTP_PATH_UNREACHABLE) && (destAddrIndex == SCTP_getPrimary(assocID))) {
        
        /* select a new one */ /* should we have a sctp_get_primary()? */
        SCTP_getAssocStatus(assocID, &assocStatus);
        for (pathIndex=0; pathIndex < assocStatus.numberOfDestinationPaths ; pathIndex++){
            pathKey =  assocStatus.destinationPathIDs[pathIndex];
            SCTP_getPathStatus(assocID, pathKey, &pathStatus);
            if (pathStatus.state == SCTP_PATH_OK)
                break;
        }
        
        /* and use it */
        if (pathIndex < assocStatus.numberOfDestinationPaths) {
            SCTP_setPrimary(assocID, pathKey);
        }
    }
    if (newState == SCTP_PATH_ADDED || newState == SCTP_PATH_REMOVED) {
        if (vverbose) {
            SCTP_getAssocStatus(assocID, &assocStatus);
            fprintf(stdout, "%-8x: We now have %d paths to our peer, primary is %s)\n", assocID, assocStatus.numberOfDestinationPaths, assocStatus.primaryDestinationAddress);
            fflush(stdout);
            for (i = 0; i < assocStatus.numberOfDestinationPaths; i++) {
                pathKey =  assocStatus.destinationPathIDs[i];
                memset(&pathStatus, 0, sizeof(SCTP_PathStatus));
                SCTP_getPathStatus(assocID, pathKey, &pathStatus);
                fprintf(stdout, "%-8x: Path %d is to address %s (key=%u)\n", assocID, i, pathStatus.destinationAddress, pathKey);
                fflush(stdout);
            }
        }
    }
}


void* communicationUpNotif(unsigned int assocID, int status,
                           unsigned int noOfDestinations,
                           unsigned short noOfInStreams, unsigned short noOfOutStreams,
                           int associationSupportsPRSCTP, int adLayerIndLen, void* adLayerInd, void* dummy)
{	
    struct ulp_data *ulpDataPtr;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Communication up (%u In-Streams, %u Out-Streams, %u Paths)\n",
                assocID, noOfInStreams, noOfOutStreams, noOfDestinations);
        if (adLayerInd != NULL) {
            fprintf(stdout, "%-8x: Got Adaptation Layer Indication, Len=%d", assocID, adLayerIndLen);
        }

        fflush(stdout);
    }
  
    ulpDataPtr                  = malloc(sizeof(struct ulp_data));
    ulpDataPtr->maximumStreamID = noOfOutStreams - 1;
    return((void *) ulpDataPtr);
}

void communicationLostNotif(unsigned int assocID, int status, void* ulpDataPtr)
{	
    unsigned char buffer[MAXIMUM_PAYLOAD_LENGTH];
    unsigned int bufferLength;
    unsigned short streamID, streamSN;
    unsigned int protoID;
    unsigned int tsn;

    if (verbose) {
        fprintf(stdout, "%-8x: Communication lost (status %u)\n", assocID, status);
        fflush(stdout);
    }
  
    /* retrieve data */
    bufferLength = sizeof(buffer);
    while (SCTP_receiveUnsent(assocID, buffer, &bufferLength,  &tsn,
                               &streamID, &streamSN, &protoID) >= 0){
        /* do something with the retrieved data */
        /* after that, reset bufferLength */
        bufferLength = sizeof(buffer);
    }
    
    bufferLength = sizeof(buffer);
    while (SCTP_receiveUnacked(assocID, buffer, &bufferLength, &tsn,
			       &streamID, &streamSN, &protoID) >= 0){
        /* do something with the retrieved data */
        /* after that, reset bufferLength */
        bufferLength = sizeof(buffer);
    }
                      
    /* free ULP data */
    free((struct ulp_data *) ulpDataPtr);
    
    /* delete the association */
    SCTP_deleteAssociation(assocID);
    exit(0);
}

void communicationErrorNotif(unsigned int assocID, int status, void* dummy)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Communication error (status %u)\n", assocID, status);
    fflush(stdout);
  }
}

void restartNotif(unsigned int assocID, void* ulpDataPtr)
{
    SCTP_AssociationStatus assocStatus;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Restart\n", assocID);
        fflush(stdout);
    }
    SCTP_getAssocStatus(assocID, &assocStatus);
    /* update ULP data */
    ((struct ulp_data *) ulpDataPtr)->maximumStreamID = assocStatus.outStreams - 1;
}

void shutdownCompleteNotif(unsigned int assocID, void* ulpDataPtr)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Shutdown complete\n", assocID);
    fflush(stdout);
  }
  /* free ULP data */
  free((struct ulp_data *) ulpDataPtr);
  SCTP_deleteAssociation(assocID);

}

void shutdownReceivedNotif(unsigned int assocID, void* ulpDataPtr)
{
    if (verbose) {  
        fprintf(stdout, "%-8x: Shutdown received\n", assocID);
        fflush(stdout);
    }
}


int main(int argc, char **argv)
{
    SCTP_ulpCallbacks echoUlp;
    SCTP_LibraryParameters params;
    
    /* initialize the echo_ulp variable */
    echoUlp.dataArriveNotif           = &dataArriveNotif;
    echoUlp.sendFailureNotif          = &sendFailureNotif;
    echoUlp.networkStatusChangeNotif  = &networkStatusChangeNotif;
    echoUlp.communicationUpNotif      = &communicationUpNotif;
    echoUlp.communicationLostNotif    = &communicationLostNotif;
    echoUlp.communicationErrorNotif   = &communicationErrorNotif;
    echoUlp.restartNotif              = &restartNotif;
    echoUlp.shutdownCompleteNotif     = &shutdownCompleteNotif;
    echoUlp.peerShutdownReceivedNotif = &shutdownReceivedNotif;


    /* handle all command line options */
    getArgs(argc, argv);
    checkArgs();

    SCTP_initLibrary();
    SCTP_getLibraryParameters(&params);
    params.sendOotbAborts = sendOOTBAborts;
    /* params.checksumAlgorithm = SCTP_CHECKSUM_ALGORITHM_ADLER32; */
    params.checksumAlgorithm = SCTP_CHECKSUM_ALGORITHM_CRC32C;
    SCTP_setLibraryParameters(&params);

    /* set up the "server" */
    SCTP_registerInstance(ECHO_PORT,
                          MAXIMUM_NUMBER_OF_IN_STREAMS, MAXIMUM_NUMBER_OF_OUT_STREAMS,
                          noOfLocalAddresses, localAddressList,
                          echoUlp);
 
    /* run the event handler forever */
    while (1) {
        SCTP_eventLoop();
    }

    /* this will never be reached */
    exit(0);
}




