/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMSPDU_H
#define SMSPDU_H

#include "charsets.h"
#include "timeincl.h"
#include "intincl.h"

#define SMS_NUMBER_TYPEMASK  0x70
#define SMS_NUMTYPE_INTERNAT 0x10
#define SMS_NUMTYPE_TEXT     0x50  
struct sms_number {
  unsigned int type; //see ETSI 23.040, Ch. 9.1.2.5
  unsigned char digits[21];
  ucs4char_t* text;
};
void struct_sms_number_init (struct sms_number* s);
void struct_sms_number_delete (struct sms_number* s);
int struct_sms_number_compare (const struct sms_number* s1, 
			       const struct sms_number* s2);

struct sms_pdu_time {
  enum {
    SMS_PDU_TIME_NONE = 0, //ignore value
    SMS_PDU_TIME_RELATIVE, //value is relative
    SMS_PDU_TIME_ABSOLUTE  //value is absolute(GMT)
  } format;
  time_t value;
};
void struct_sms_pdu_time_init (struct sms_pdu_time* s);

enum sms_direction {
  SMS_INCOMING,
  SMS_OUTGOING
};

/* different than the defined values above
 * does not need sms_direction (implied by each value)
 */
enum sms_pdu_type {
  SMS_TYPE_DELIVER,
  SMS_TYPE_SUBMIT_REPORT,
  SMS_TYPE_STATUS_REPORT,
  SMS_TYPE_DELIVER_REPORT,
  SMS_TYPE_SUBMIT,
  SMS_TYPE_COMMAND
};
/* type is the full PDU type field
 * but only the lowest two bits are really needed
 */
enum sms_pdu_type get_sms_pdu_type (enum sms_direction d, uint8_t type);

struct sms_pdu_options {
  enum sms_pdu_type type;
  uint8_t udh_present; //needed by sms_udh_fill()
  uint8_t mms; //more messages to send indication
  uint8_t rd; //reject duplicate request
  uint8_t rp; //reply path request/indication
  uint8_t sr; //status report request/indication/qualifier
};
void struct_sms_pdu_options_init (struct sms_pdu_options* s);

enum sms_encoding {
  SMS_CHARSET_GSM,
  SMS_CHARSET_UCS2,
  SMS_CHARSET_8BIT
};
struct sms_pdu_dcs {
  //needed for SMS-STATUS-REPORT (DCS is optional)
  unsigned char dcs_present;

  /* note, that not all combination are possible,
   * see 3GPP TS 23.038, ETSI TS 129 038
   */
  enum sms_encoding encoding;
  uint8_t compressed; // =0 not compressed, !=0 compressed
  uint8_t autodel; // =0 normal mode, !=0 marked as auto-delete
  enum {
    SMS_DCS_OPT_NONE,
    SMS_DCS_OPT_CLASS, //use class
    SMS_DCS_OPT_IND    //use indication
  } options;
  uint8_t class; // 0-3, other values match SMS_DCS_OPT_NONE
  enum {
    SMS_DCS_IND_STORE,
    SMS_DCS_IND_DISCARD
  } indgroup;
  uint8_t indsense; // =0 inactive, !=0 active
  enum {
    SMS_DCS_IND_VOICE,
    SMS_DCS_IND_FAX,
    SMS_DCS_IND_EMAIL,
    SMS_DCS_IND_OTHER
  } indtype;
};
void struct_sms_pdu_dcs_init (struct sms_pdu_dcs* s);

struct sms_pdu_message_status {
  uint8_t s;
  struct sms_pdu_time t;
  struct sms_pdu_ud* message_parent;
  struct sms_pdu_data* status_parent;
};
void struct_sms_pdu_message_status_init(struct sms_pdu_message_status* s);

struct sms_pdu_ud_header {
  uint8_t type;
  uint8_t len;
  char* data;
};
void struct_sms_pdu_ud_header_init (struct sms_pdu_ud_header* s);
void struct_sms_pdu_ud_header_delete (struct sms_pdu_ud_header* s);

struct sms_pdu_ud {
  // a NULL terminated list
  struct sms_pdu_ud_header** header;

  //the text in the message
  ucs4char_t* text;

  /* status of the message if a SMS-STATUS-REPORT can be matched
   * If the message type is SMS_STATUS_REPORT or no
   * SMS-STATUS-REPORT matches, this shall be NULL.
   * Never free this pointer from here, except that
   * ack->status_parent is NULL
   */
  struct sms_pdu_message_status* ack; //never free this from here
};
void struct_sms_pdu_ud_init (struct sms_pdu_ud* s);
void struct_sms_pdu_ud_delete (struct sms_pdu_ud* s);

struct sms_pdu_data {
  struct sms_pdu_options options;

  /* message reference
   * This is not valid for all message types
   * (currently only SMS-SUBMIT and SMS-STATUS-REPORT)
   * and meaning depends on type, too.
   */
  uint8_t mref; //message reference

  struct sms_number address;

  uint8_t pid; //protocol ID
  struct sms_pdu_dcs scheme;

  struct sms_pdu_time timedata;

  uint16_t multipart_id; //multipart short message id
  uint8_t partnum; //number of this part, count from 0

  /* ud is a list of size (parts*sizeof(*ud))
   * if parts is 0, ud should be NULL
   */  
  uint8_t parts; //amount of parts, count from 1
  struct sms_pdu_ud* ud; //user data,

  /* service center status of a previously sent message
   * only valid for SMS-STATUS-REPORT
   */
  struct sms_pdu_message_status* status;
};
void struct_sms_pdu_data_init (struct sms_pdu_data* s);
void struct_sms_pdu_data_delete (struct sms_pdu_data* s);

struct sms_tpdu_data {
  struct sms_number sca; //service center address
  struct sms_pdu_data* pdu;
};
void struct_sms_tpdu_data_init (struct sms_tpdu_data* s);
void struct_sms_tpdu_data_delete (struct sms_tpdu_data* s);

struct sms_slot_data {
  int slot; // >0 for true values
  char tpdu[363]; //as hexstring
  enum sms_direction type;
};
void struct_sms_slot_data_init (struct sms_slot_data* s);

struct sms {
  struct sms_slot_data** encoded;
  struct sms_tpdu_data* decoded;
};
void struct_sms_init (struct sms* s,
		      struct sms_slot_data* encoded,
		      struct sms_tpdu_data* decoded);
void struct_sms_delete (struct sms* s);


#include <stdio.h>


/***************************
 * encoding short messages *
 ***************************/
/* create a new short message of type SMS-SUBMIT
 * return value is a TPDU allocated with malloc
 */
#include "options.h"
char* sms_pdu_create_submit (char* text, char* number, struct smsopts* opts);

/* both return length of pdu (pdu is part of tpdu)
 */
unsigned int sms_tpdu_len (char* tpdu); //for TPDU (means: with SMSC field)
unsigned int sms_pdu_len (char* pdu); //for PDU

/***************************
 * decoding short messages *
 ***************************/
/* STEP 1:
 * decode sms which is of type type
 * return value allocated with malloc
 */
struct sms_tpdu_data* sms_pdu_decode (enum sms_direction type,
				      struct sms_slot_data* sms);

/* STEP 2:
 * see struct_sms_init()
 */

/* STEP 3.1:
 * to match the status report messages to sent messages
 * The number of elements in list does not change with this call.
 * This MUST be done BEFORE merging concatenated message,
 * otherwise not all can be matched.
 *
 * If you don't need or want this, just omit it.
 */
void sms_statusreport_match (struct sms** list);

/* STEP 3.2:
 * to merge concatenated short messages
 * list may have less elements after this call
 *
 * If you don't need or want this, just omit it.
 */
int sms_multipart_merge (struct sms** list);

/* STEP 3.3:
 * order can currently be "type", "slot" or "type,slot"
 * The number of elements in list does not change with this call.
 * This can be done at any time in STEP 3.
 */
void sms_pdu_sort (struct sms** list, const char* order);

/* STEP 4:
 * print sms as text to fp
 */
void sms_pdu_print (FILE* fp, struct sms* sms);

#endif
