
/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "common.h"
#include "helper.h"
#include "options.h"
#include "atcommand.h"
#include "ttyaccess.h"

#include <locale.h>
#include <string.h>
#include <stdlib.h>

int main (int argc, char** argv) {
  char* supported_phones[] = {
    "S25",
    "S35i", "M35i", "MC35i", "C35i",
    "SL42", "SLIN", "SL45", "SLIK",
    "C45",  "ME45", "S45",  "S45i",
    "M50",  "MT50", "M50I",
    //there are even some phones from north america
    "S46",
    NULL
  };

  unsigned int i = 0;
  char* vendor;
  char* model;
        
  struct args_t args;
  char** files;

  //start setup
  //init global vars	
  VERBOSE_LEVEL = 0;

  //init local vars    
  vendor = NULL;
  model = NULL;

  //setting program's locale
  setlocale(LC_ALL,"");

  if (argc==1) {
    help(argv[0]);
    exit(EXIT_SUCCESS);
  }

  args_init(&args);
  files = args_parse(argc,argv,&args);
  use_envvars(&args);

  VERBOSE_LEVEL=args.verbose;

  if (args.help || args.version) {
    if (args.help) {
      help(argv[0]);
    }
    if (args.version) {
      version();
    }
    exit(EXIT_SUCCESS);
  }

  //now open the port
  tty_open(&args.port);

  at_init(args.pin);

  if (args.reset) {
    /* sending a sync frame
     * the phone may not respond without it because it thinks that all
     * send commands are still data from a previous run
     */
    myprintf(0,"%s","Setting reset frame...");
    tty_write("\x1a\r\n",3);
    myprintf(0,"%s","done\n");
    exit(EXIT_SUCCESS);
  }

  /* Now perform initial phone setup
   */
  //testing the device
  if (phone_init()) {
    myprintf(0,"%s","OK, a modem device is present.\n");
  } else {
    errexit("%s","ERROR, cannot communicate with device.\n");
  }
  //disabling command echo
  command_echo(0);
  
  //enabling advanced error response
  //not needed everywhere but this reduces code duplication
  verbose_errors(1);
  
  //changing to GSM charset
  set_charset("GSM");

  /* Check for registered phones and print a warning on unknown ones
   */
  if (!args.info) {
    vendor = get_vendor();
    model = get_model();
    if (vendor!=NULL) {
      if (!strcasecmp(vendor,"SIEMENS")) {
	myprintf(0,"Detected %s %s\n",vendor,model);
	for (i=0; supported_phones[i] != NULL; ++i) {
	  myprintf(3,"Comparing to registered model %s\n",supported_phones[i]);
	  if (!strcmp(model,supported_phones[i])) {
	    break;
	  }
	}
	if (supported_phones[i] == NULL) {
	  myprintf(1,"%s","WARNING: this phone was not confirmed to be working with this software.\n");
	}
      } else {
	myprintf(0,"%s","WARNING: Phones from this vendor were not confirmed to be working with this software!\n");
      }
    }
    mem_realloc(vendor,0);
    mem_realloc(model,0);
  }

  if (args.time || args.smsc ||
      args.info || args.meminfo) {
    if (args.time) {
      set_time();
    }
    if (args.smsc) {
      if (str_len(args.myparams.number)) {
	set_smsc(args.myparams.number);
      } else {
	errexit("%s","You must define a number with the --number option.\n");
      }
    }
    if (args.info || args.meminfo) {
      info(args.myFILE,args.info,args.meminfo);
    }
    exit(EXIT_SUCCESS);
  }

  switch(args.scmxx_ftype) {
  default:
    errexit("%s","You must specifiy a valid memory type.\n");
    break;
  case SCMXX_FTYPE_BINARY:
    file_transfer_manage(args.scmxx_action,files,
			 args.myFILE,args.myPIPE,
			 args.myparams.mem,
			 args.myparams.slot);
    break;
  case SCMXX_FTYPE_PBOOK:
    pbook_manage(args.scmxx_action,args.myFILE,
		 files,&args.myparams);
    break;
  case SCMXX_FTYPE_SMS:
    sms_manage(args.scmxx_action,files,
	       args.myFILE,args.myPIPE,
	       &args.myparams,&args.mysmsopts);
    break;
  }
  tty_close();
  exit(EXIT_SUCCESS);
}
