/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "common.h"
#include "charsets.h"
#include "helpers.h"
#include "smscoding.h"
#include "smspdu.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

unsigned int sms_pdu_len (char* pdu) {
  //short but does not look for invalid PDUs
  return str_len(pdu)/2;
}

unsigned int sms_tpdu_len (char* tpdu) {
  return sms_pdu_len(tpdu+2+hexstr2int(tpdu,2));
}

char* sms_pdu_create_submit (char* text, char* number,
			     struct smsopts* opts)
{
  sms_number_t sca; //short message service center address
  uint8_t pdutype; //PDU type
  uint8_t mr; //message reference
  sms_number_t da; //destination address
  uint8_t pid = 0; //protocol identifier (normally 0)
  uint8_t dcs; //data coding scheme
  uint8_t vp_rel = 0xff; //validity period (0xff = max = 63 weeks)
  //char vp_abs[15]; //validity period (YYMMDDhhmmssTZ)
  int udl; //user data length
  unsigned char* ud; //user data

  ucs4char_t* wide_str;
  char buffer;
  int i;

  char* pdu;

  /* not yet supported
   * zero length means that the phone will use its default
   */
  memset(sca.number,0,sizeof(sca.number));
  sca.type = numtype(sca.number);
  sca.length = strlen(sca.number)/2;
  if (sca.length) {
    ++sca.length;
  }
  
  /* set PDU type to standard values
   */
  pdutype = SMS_SUBMIT; //message type
  pdutype |= (1<<4); //validity period format: relative
  if (opts->srr) {
    pdutype |= (1<<5); //status report request;
  }

  /* message reference must be zero for Siemens phones!
   */
  mr = 0;

  /* destination number
   */
  memset(da.number,0,sizeof(da.number));
  da.type = numtype(number);
  if (da.type == 0x91) {
    i = 1;
  } else {
    i = 0;
  }
  strncpy(da.number,number+i,sizeof(da.number)-1);
  for (i=0;i<strlen(da.number);++i) {
    if (!isdigit(da.number[i])) {
      errexit("Error: only digits are allowed in the destination number of a SMS.\n");
    }
  }
  da.length = strlen(da.number);
  if (strlen(da.number)%2) {
    da.number[strlen(da.number)] = 'F';
  }
  for (i=0;i<strlen(da.number);i+=2) {
    buffer = da.number[i];
    da.number[i] = da.number[i+1];
    da.number[i+1] = buffer;
  }
  
  /* support "flash-SMS" and unicode here
   */
  dcs = 0;
  if (opts->flash) {
    dcs &= 0xec; //clear all affected bits
    dcs |= 0x10; //set class 0 message (immediate display)
  }
  if (opts->unicode) {
    dcs &= 0xf3; //clear all affected bits
    dcs |= 0x08; //set unicode charset
  }

  //leave pid and vp_rel as is

  /* proces user data
   */
  wide_str = convert_from_system(text);
  if (opts->unicode) {
    ud = convert_to_ucs2_hexstring(wide_str);
    udl = strlen(ud)/2;
    //check user data content length
    if (udl/2 > MAXSMSSIZEUCS) {
      errexit ("SMS text is too long (max. %d characters).\n", MAXSMSSIZEUCS);
    }
  } else {
    ud = sms_data_7bit_encode(wide_str,&udl);
  }
  mem_realloc(wide_str,0);

  //create PDU
  pdu = mem_alloc((176*2)+1,1);
  sprintf(pdu,"%02X",sca.length);
  if (sca.length) {
    sprintf(pdu+2,"%02X%s",sca.type,sca.number);
  }
  sprintf(pdu+sca.length+strlen(pdu),"%02X%02X%02X%02X%s%02X%02X%02X%02X%s",
	  pdutype,mr,da.length,da.type,da.number,
	  pid,dcs,vp_rel,udl,ud);
  mem_realloc(ud,0);
  return pdu;
}

void sms_pdu_decode (FILE* fp, int slot, int type, char* tpdu) {
  struct sms_tpdu_data* decoded;
  uint8_t mti; //message type indicator
  char* pdu;

  int i;
  int k = 0;

  decoded = mem_alloc(sizeof(*decoded),0);
  pdu = NULL;

  //extracting SMSC
  memset(decoded->sca.number,0,sizeof(decoded->sca.number));
  decoded->sca.length = hexstr2int(tpdu,2);
  if (decoded->sca.length > 0) {
    decoded->sca.length *= 2;
    decoded->sca.type = hexstr2int(tpdu+2,2);
    if ((decoded->sca.type&0x70) == 0x10) {
      decoded->sca.number[0] ='+';
      k = 1;
    }
    for (i=0;i<decoded->sca.length-2;i+=2) {
      decoded->sca.number[i+k] = tpdu[4+i+1];
      decoded->sca.number[i+k+1] = tpdu[4+i];
    }
    if (decoded->sca.number[i+k-1] == 'F') {
      memset(decoded->sca.number+i+k-1,0,1);
    }
  }
  pdu = str_dup(tpdu+decoded->sca.length+2);

  //evaluating PDU type (only message type)
  mti = hexstr2int(pdu,2)&3;
  //checking if PDUtype is supported and can thus be decoded
  switch (type) {
  case SMS_INCOMING:
    switch(mti) {
    case SMS_DELIVER:
      decoded->pdu = sms_pdu_decode_deliver(pdu);
      mem_realloc(pdu,0);
      sms_pdu_print_deliver(fp,slot,SMS_INCOMING,decoded);
      break;
    case SMS_SUBMIT_REPORT:
      myprintf(0,"Unsupported pdu type: %s\n","SMS-SUBMIT-REPORT");
      mem_realloc(pdu,0);
      return;
      break;
    case SMS_STATUS_REPORT:
      myprintf(0,"Unsupported pdu type: %s\n","SMS-STATUS-REPORT");
      mem_realloc(pdu,0);
      return;
      break;
    default:
      myprintf(0,"Unknown pdu type.\n");
      mem_realloc(pdu,0);
      return;
      break;
    }
    break;
  case SMS_OUTGOING:
    switch(mti) {
    case SMS_DELIVER_REPORT:
      myprintf(0,"Unsupported pdu type: %s\n","SMS-DELIVER-REPORT");
      mem_realloc(pdu,0);
      return;
      break;
    case SMS_SUBMIT:
      /* SMS-SUBMIT almost equals SMS-DELIVER but
       * the message reference octet must be removed
       * We don't need it anyway as it's always 0.
       */
      memmove(pdu+2,pdu+4,strlen(pdu+4)+1);
      decoded->pdu = sms_pdu_decode_deliver(pdu);
      mem_realloc(pdu,0);
      sms_pdu_print_deliver(fp,slot,SMS_OUTGOING,decoded);
      break;
    case SMS_COMMAND:
      myprintf(0,"Unsupported pdu type: %s\n","SMS-COMMAND");
      mem_realloc(pdu,0);
      return;
      break;
    default:
      myprintf(0,"Unknown pdu type.\n");
      mem_realloc(pdu,0);
      return;
      break;
    }
    break;
  default:
    myprintf(0,"\nUnknown sms status %d\n", type);
    mem_realloc(pdu,0);
    return;
  }  
}

struct sms_pdu_data*  sms_pdu_decode_deliver (char* pdu) {
  struct sms_pdu_data* sms;
  int i, k;
  int offset = 0;
  int udhsize = 0;
  ucs4char_t* wide_str;
  char* temp;
  struct tm sct_tm;

  temp = NULL;

  sms = mem_alloc(sizeof(struct sms_pdu_data),0);

  sms->type = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  sms->address.length = hexstr2int(pdu,2);
  sms->address.length += sms->address.length%2;
  sms->address.type = hexstr2int(pdu+2,2);
  memset(sms->address.number,0,sizeof(sms->address.number));
  if (sms->address.length) {
    switch (sms->address.type&0x70) {
    case 0x50:
      for (i=0;i<sms->address.length;i+=2) {
	sms->address.number[i] = pdu[4+i+1];
	sms->address.number[i+1] = pdu[4+i];
      }
      wide_str = sms_data_7bit_decode(sms->address.number,
				      (sms->address.length*4)/7);
      temp = convert_to_system(wide_str,REPMODE_QUESTIONMARK);
      mem_realloc(wide_str,0);
      strncpy(sms->address.number,temp,sizeof(sms->address.number));
      mem_realloc(temp,0);
      break;
    case 0x10:
      sms->address.number[0] = '+';
      offset=1;
      //no break
    default:
      for (i=0;i<sms->address.length;i+=2) {
	sms->address.number[i+offset] = pdu[4+i+1];
	sms->address.number[i+offset+1] = pdu[4+i];
      }
      if (sms->address.number[i+offset-1] == 'F') {
	memset(sms->address.number+i+offset-1,0,1);
      }
    }
  }
  pdu += 2+2+sms->address.length;

  sms->pid = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  sms->dcs = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  switch (sms->type&0x1b) {
  case 0x00+SMS_SUBMIT: //VP not present
    sms->timedata.format = SMS_PDU_TIME_NONE;
    break;
  case 0x08+SMS_SUBMIT: //VP enhanced
    i = hexstr2int(pdu,2)&0xFF; //reading octet 1
    if (i&0x80) { //extension bit
      //not supported
      sms->timedata.format = SMS_PDU_TIME_NONE;
    }
    //ignore bit 6 (single shot SM)
    //bits 5,4 and 3 are reserved
    switch (i&0x07) { //bits 2, 1 and 0
    default: //reserved values
    case 0x00:
      sms->timedata.format = SMS_PDU_TIME_NONE;
      break;
    case 0x01: //value in seconds (1 octet)
      sms->timedata.format = SMS_PDU_TIME_RELATIVE;
      sms->timedata.value = hexstr2int(pdu+2,2)&0xFF;
      if (sms->timedata.value == 0) {
	sms->timedata.format = SMS_PDU_TIME_NONE;
      }
      break;
    case 0x02: //real relative value
      sms->timedata.format = SMS_PDU_TIME_RELATIVE;
      i = hexstr2int(pdu,2)&0xFF;
      if (0 <= i && i <= 143) { //5 minutes
	sms->timedata.value = (i+1)*5*60;
      } else if (144 <= i && i <= 167) { //30 minutes
	sms->timedata.value = (i-143+24)*30*60;
      } else if (168 <= i && i <= 196) {//1 day
	sms->timedata.value = (i-166)*(3600*24);
      } else if (197 <= i && i <= 255) {//1 week
	sms->timedata.value = (i-192)*(3600*24*7);
      }
      break;
    case 0x03: // value as HH:MM:SS
      sms->timedata.format = SMS_PDU_TIME_RELATIVE;
      temp = mem_alloc(9,1);
      sprintf(temp,"%c%c:%c%c:%c%c",
	      pdu[1],pdu[0],pdu[3],pdu[2],pdu[5],pdu[4]);
      sms->timedata.value = atoi(temp)*3600 + atoi(temp+3)*60 + atoi(temp+6);
      break;
    }
    pdu += 14;
    break;
  case 0x10+SMS_SUBMIT: //VP relative
    sms->timedata.format = SMS_PDU_TIME_RELATIVE;
    i = hexstr2int(pdu,2)&0xFF;
    if (0 <= i && i <= 143) { //5 minutes
      sms->timedata.value = (i+1)*5*60;
    } else if (144 <= i && i <= 167) { //30 minutes
      sms->timedata.value = (i-143+24)*30*60;
    } else if (168 <= i && i <= 196) {//1 day
      sms->timedata.value = (i-166)*(3600*24);
    } else if (197 <= i && i <= 255) {//1 week
      sms->timedata.value = (i-192)*(3600*24*7);
    }
    pdu += 2;
    break;
  case 0x18+SMS_SUBMIT: //VP absolute
  case 0x00+SMS_DELIVER: //Service Center Time Stamp
    sms->timedata.format = SMS_PDU_TIME_ABSOLUTE;
    temp = mem_alloc(18,1);
    sprintf(temp,"%c%c/%c%c/%c%c,%c%c:%c%c:%c%c",
	    pdu[1],pdu[0],pdu[3],pdu[2],pdu[5],pdu[4],
	    pdu[7],pdu[6],pdu[9],pdu[8],pdu[11],pdu[10]);
    memset(&sct_tm,0,sizeof(sct_tm));
    if (strptime(temp,"%y/%m/%d,%H:%M:%S",&sct_tm) != NULL) {
      /* Not set by strptime(); tells mktime() to determine whether
       * daylight saving time is in effect
       */
      sct_tm.tm_isdst = -1;
      /* work-around for lazy *BSD implementation of strptime()
       * that does not set tm_yday in struct tm
       */
      tzset();
      sms->timedata.value = mktime(&sct_tm);
      localtime_r(&sms->timedata.value,&sct_tm);

      /* mktime() does not work correctly for this, so we use the following from
       * http://www.opengroup.org/onlinepubs/007904975/basedefs/xbd_chap04.html#tag_04_14
       * (slightly modified)
       */
      sms->timedata.value = sct_tm.tm_sec + sct_tm.tm_min*60 + sct_tm.tm_hour*3600;
      sms->timedata.value += 86400 * (sct_tm.tm_yday + (sct_tm.tm_year-70)*365
				      + (sct_tm.tm_year-69)/4
				      - (sct_tm.tm_year-1)/100
				      + (sct_tm.tm_year+299)/400);

      //read real timezone offset
      temp = mem_realloc(temp,3);
      memset(temp,0,3);
      temp[0] = pdu[13];
      temp[1] = pdu[12];
      if ((atoi(temp)%80)/4 <= 12) {
	if (atoi(temp) >= 80) { //apply timezone offset
	  sms->timedata.value += (atoi(temp)%80)*15*60;
	} else {
	  sms->timedata.value -= (atoi(temp)%80)*15*60;
	}
      }
    } else { //fallback for strptime failure
      sms->timedata.format = SMS_PDU_TIME_NONE;
    }
    mem_realloc(temp,0);
    pdu += 14;
    break;
  }

  sms->ud.len = hexstr2int(pdu,2);
  pdu += 2;
  
  if (sms->ud.len) {
    if (sms->type&0x40) {
      //extract the user data header data
      udhsize = hexstr2int(pdu,2);
      //split the udh into its parts
      sms->ud.header = mem_alloc(((udhsize/2)+1)*sizeof(*sms->ud.header),0);
      offset = 0;
      i = 0;
      sms->ud.header[i] = NULL;
      while (offset+4 <= udhsize*2) {
	sms->ud.header[i] = mem_alloc(sizeof(**sms->ud.header),0);
	//read fields that are mandatory for each user data header
	sms->ud.header[i]->type = hexstr2int(pdu+2+offset,2)&0xFF;
	sms->ud.header[i]->len = hexstr2int(pdu+4+offset,2)&0xFF;
	offset += 4;
	//find the udh type in the list of known headers
	k = 0;
	while (header_defs[k].type != -1 &&
	       header_defs[k].type != sms->ud.header[i]->type) {
	  ++k;
	}
	//check that everything is correct with the udh data
	if (offset+(sms->ud.header[i]->len*2) <= udhsize*2 &&
	    header_defs[k].type != -1 &&
	    header_defs[k].minlen <= sms->ud.header[i]->len &&
	    sms->ud.header[i]->len <= header_defs[k].maxlen) {
	  sms->ud.header[i]->data = strn_dup(pdu+2+offset, sms->ud.header[i]->len*2);
	  offset += sms->ud.header[i]->len*2;
	  sms->ud.header[i+1] = NULL;
	} else {
	  //bogus or unknown header, delete it and stop processing the headers
	  mem_realloc(sms->ud.header[i],0);
	  sms->ud.header[i] = NULL;
	  break;
	}
	++i;
      }
    } else {
      udhsize = 0;
      sms->ud.header = NULL;
    }
    //decode text
    if ((sms->dcs&0x8c) == 0x00 || //0xxx 00xx
        (sms->dcs&0xe0) == 0xb0 || //110x xxxx
        (sms->dcs&0xf4) == 0xf0    //1111 x0xx
        ) {
      //all encodings of uncompressed 7bit
      sms->ud.text = sms_data_7bit_decode(pdu,sms->ud.len);
      if (sms->type&0x40) {
	//offset = 1 + ((udhsize*8)+((udhsize*8)%7))/7;
	offset = ((udhsize+1)*8)/7; //size of udh (including the length field) in 7bits
	if ((udhsize+1)%7 != 0) ++offset; //7bit header padding
	memmove(sms->ud.text,&sms->ud.text[offset],(ucs4len(&sms->ud.text[offset])+1)*sizeof(ucs4char_t));
      }
      sms->ud.len = ucs4len(sms->ud.text);
    } else if ((sms->dcs&0x8c) == 0x08 || //0xxx 10xx
               (sms->dcs&0xf0) == 0xe0    //1110 xxxx
               ) {
      //all encodings of uncompressed 16bit unicode
      if (sms->type&0x40) {
	sms->ud.text = convert_from_ucs2_hexstring(pdu+((udhsize+1)*2));
      } else {
	sms->ud.text = convert_from_ucs2_hexstring(pdu);
      }
      sms->ud.len = ucs4len(sms->ud.text);
    } else {
      //all encodings of 8bit data: ASCII assumed
      if (sms->type&0x40) {
	offset = (udhsize+1)*2;
      } else {
	offset = 0;
      }
      sms->ud.len -= offset;
      sms->ud.text = mem_alloc(sms->ud.len,1);
      for (i=0;i<sms->ud.len;++i) {
	sms->ud.text[i] = hexstr2int(pdu+(2*i),2)&0xFF;
      }
    }
  }
  return sms;
}

void sms_pdu_print_deliver (FILE* fp, int slot, enum sms_direction type, struct sms_tpdu_data* sms) {
  char* temp;
  unsigned short weeks;
  unsigned short days;
  unsigned short hours;
  unsigned short minutes;
  unsigned short seconds;
  struct tm sct_tm;

  if (fp == NULL || sms == NULL) {
    return;
  }

  if (slot >= SCMXX_SLOT_SMS_MIN) {
    fprintf(fp,"Slot: %d\n", slot);
  }
  if (str_len(sms->pdu->address.number) > 0) {
    fprintf(fp,"%s: %s\n", ((type == SMS_OUTGOING) ? "To" : "From"),
	    sms->pdu->address.number);
  }

  switch (sms->pdu->timedata.format) {
  case SMS_PDU_TIME_NONE:
    break;
  case SMS_PDU_TIME_RELATIVE:
/*     fprintf(fp,"Valid for: %ld seconds\n",sms->pdu->timedata.value); */
    weeks = sms->pdu->timedata.value / (7*24*3600);
    days = (sms->pdu->timedata.value / (24*3600))%7;
    hours = (sms->pdu->timedata.value / 3600)%24;
    minutes = (sms->pdu->timedata.value / 60)%60;
    seconds = sms->pdu->timedata.value%60;
    fprintf(fp,"Valid for:");
    if (weeks) fprintf(fp," %d %s", weeks, (weeks > 1) ? "weeks" : "week");
    if (days) fprintf(fp," %d %s", days, (days > 1) ? "days" : "day");
    if (hours) fprintf(fp," %d %s", hours, (hours > 1) ? "hours" : "hour");
    if (minutes) fprintf(fp," %d %s", minutes, (minutes > 1) ? "minutes" : "minute");
    if (seconds) fprintf(fp," %d %s", seconds, (seconds > 1) ? "seconds" : "second");
    fprintf(fp,"\n");
    break;
  case SMS_PDU_TIME_ABSOLUTE:
    temp = mem_alloc(80+1,1);
    memset(&sct_tm,0,sizeof(sct_tm));
    if (localtime_r(&sms->pdu->timedata.value,&sct_tm) != NULL) {
      if (strftime(temp,80,"%c",&sct_tm) > 0) {
	fprintf(fp,"%s: %s\n",
		(type == SMS_OUTGOING) ? "Valid until" : "Date",
		temp);
      }
    }
    break;
  }

  if (sms->sca.length) {
    fprintf(fp,"SMSC number: %s\n",sms->sca.number);
  }

  fprintf(fp,"PDU type: %s\n",(type == SMS_INCOMING) ? "SMS-DELIVER" : "SMS-SUBMIT");
  if (sms->pdu->type&0xA4) {
    fprintf(fp,"PDU flags:");
    if (sms->pdu->type&0x80) fprintf(fp," ReplyPath");
    if (sms->pdu->type&0x20) fprintf(fp," StatusRequest");
    if (sms->pdu->type&0x04) fprintf(fp," %s",(type == SMS_INCOMING) ? "MoreMessagesToSend" : "RejectDuplicate");
    fprintf(fp,"\n");
  }

  fprintf(fp,"Data coding scheme:");
  if ((sms->pdu->dcs&0x80) == 0) {
    if (sms->pdu->dcs&0x20) fprintf(fp," compressed");
    switch ((sms->pdu->dcs>>2)&3) {
    case 0: fprintf(fp," 7bit-GSM"); break;
    case 1: fprintf(fp," 8bit"); break;
    case 2: fprintf(fp," UCS-2"); break;
    case 3: break;
    }
    if (sms->pdu->dcs&0x08) fprintf(fp," (class %d)",sms->pdu->dcs&3);
    if ((sms->pdu->dcs&0x40) == 1) fprintf(fp," marked as auto-delete");
  } else {
    if (sms->pdu->dcs >= 0xF0) {
      fprintf(fp," %s", (sms->pdu->dcs&0x40) ? "8bit" : "7bit-GSM");
      fprintf(fp," (class %d)",sms->pdu->dcs&3);
    } else if (sms->pdu->dcs >= 0xC0) {
      fprintf(fp," %s", (sms->pdu->dcs&0x20) ? "UCS-2" : "7bit-GSM");
      fprintf(fp,"\nIndication: %s", (sms->pdu->dcs&0x08) ? "new" : "no more");
      switch(sms->pdu->dcs&3) {
      case 0: fprintf(fp," voicemail"); break;
      case 1: fprintf(fp," fax"); break;
      case 2: fprintf(fp," e-mail"); break;
      case 3: fprintf(fp," misc."); break;
      }
      fprintf(fp," message(s) waiting");
    }
  }
  fprintf(fp,"\n");
  
  sms_udh_print(fp,sms->pdu->ud.header);

  fprintf(fp,"Message length: %d\n",ucs4len(sms->pdu->ud.text));
  temp = convert_to_system(sms->pdu->ud.text,REPMODE_QUESTIONMARK);
  fprintf(fp,"\n%s\n",temp);
  mem_realloc(temp,0);
  fprintf(fp,"\n");
}
