/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "common.h"
#include "charsets.h"
#include "helpers.h"
#include "pbmaskgsm.h"
#include "options.h"
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <ctype.h>

void pbook_manage (int action, char* outfile,
		   char** files, char* phonebook)
{
  int i = 0;

  if (phonebook == NULL) {
    //SIM phonebook seems to be a good default
    phonebook = "SM";
  }
  if (strcmp(phonebook,"?") == 0) {
    pbook_print_memlist(NULL,1);
    exit(0);
  }

  if (action == 0) {
    errexit ("You must specify a valid action.\n");
  }

  if (!pbook_select_mem(phonebook)) {
    errexit("Cannot select phonebook %s\n",phonebook);
  }

  switch (action) {
  default:
    errexit("You must specify exactly one operation.\n");
    break;
  case SCMXX_ACTION_REMOVE: //deleting
    pbook_delete(phonebook);
    break;
  case SCMXX_ACTION_SEND: //sending
    while (files[i] != NULL) {
      pbook_send_file(files[i++],phonebook);
    }
    break;
  case SCMXX_ACTION_GET: //getting
    pbook_get(outfile,phonebook);
    break;
  }
}

void pbook_delete (char* phonebook) {
  char at_command[BUFSIZ];
  char* ausgabe;
  char* ack;
  int i,first_entry,last_entry;

  memset(at_command,0,sizeof(at_command));

  if ((!strcasecmp(phonebook,"DC"))
      || (!strcasecmp(phonebook,"MD"))
      || (!strcasecmp(phonebook,"LD"))) {
    new_at_command(at_command,"^SDLD");
    ack=tty_write_read(at_command);
    if (!strcmp(ack,"OK")) {
      myprintf(0,"%s deleted.\n",phonebook);
    } else {
      errexit("\n%s\nAn ERROR occured. Possible data corruption!\n",ack);
    }
    mem_realloc(ack,0);
  } else {
    new_at_command(at_command,"+CPBW=?");
    ausgabe=tty_write_read(at_command);
    if (!strncmp(ausgabe,"+CME ERROR: ",12)) {
      errexit("Error deleting phonebook %s: %s\n",phonebook,ausgabe+12);
    } else {
      mem_realloc(tty_read(at_command),0);
    }
    first_entry=atoi((char *)strtok(ausgabe+7,"(-),"));
    last_entry=atoi((char *)strtok(NULL,"(-),"));
    mem_realloc(ausgabe,0);
    myprintf(0,"Preparing to delete %d entries\n",last_entry-first_entry+1);
    myprintf(0,"Deleting: ");
    for (i=first_entry;i<=last_entry;i++) {
      new_at_command(at_command,"+CPBW=");
      add_at_command(at_command,"%d",i);
      ack=tty_write_read(at_command);
      if (!strcmp(ack,"OK")) {
	myprintf(0,"%d ",i);
	mem_realloc(ack,0);
      } else {
	errexit("\n%s\nAn ERROR occured. Possible data corruption!\n",ack);
      }				
    }
    myprintf(0,"\n");
  }
}

enum charset_type {
  CHARSET_UCS2 = 0,
  CHARSET_GSM
};
struct pbook_entry {
  unsigned int slot;
  char* number;
  ucs4char_t* text;
};

int pbook_file_read_line (FILE* fp,struct pbook_entry* entry) {
  char buffer[BUFSIZ];
  unsigned int slot;
  char* number;
  char* text;
  char* temp;
  
  if (fp == NULL || entry == NULL) {
    return -1;
  }

  if (fgets(buffer,BUFSIZ,fp) == NULL) {
    return 0;
  } else {
    //token 1: slot
    if (isdigit((int)*buffer)) {
      slot = atoi(buffer);
    } else {
      return -1;
    }

    //token 2 and 3: number and numbertype
    number = buffer+numlen(slot);
    if (*number != ',' || *(++number) != '"') {
      return -1;
    }
    ++number;
    text = index(number,'"');
    if (text == NULL) {
      return -1;
    }
    memset(text++,0,1);

    //token 4: text
    if (*text != ',' || *(++text) != '"' ||
	strlen(++text) == 0) {
      return -1;
    }
    temp = rindex(text,'"');
    if (temp != NULL &&
	(strcmp(temp,"\"\n") ||
	 strcmp(temp,"\"\r\n") ||
	 strcmp(temp,"\""))) {
      memset(temp,0,1);;
    } else {
      return -1;
    }

    entry->slot = slot;
    entry->number = str_dup(number);
    entry->text = convert_from_system(text);
  }
  return 1;
}

struct pbook_entry** pbook_file_read (char* file) {
  FILE* fp;
  size_t i = 0;
  struct pbook_entry** plist;
  size_t psize = 100;
  int status;

  if (file == NULL) {
    return NULL;
  }
  if (str_len(file) && strcmp(file,"-") != 0) {
    if ((fp = fdopen(open_myFile_ro(file),"r")) == NULL) {
      return NULL;
    }
  } else {
    fp = stdin;
  }
  plist = mem_alloc(psize*sizeof(*plist),0);
  do {
    if (i+1 >= psize) {
      psize += 100;
      plist = mem_realloc(plist,psize*sizeof(*plist));
    }
    plist[i] = mem_alloc(sizeof(**plist),0);
    status = pbook_file_read_line(fp,plist[i]);
    if (status > 0) {
      ++i;
    }
  } while (status > 0);
  if (status == -1) { //error in file
    plist[i] = mem_realloc(plist[i],0);
    plist[i] = NULL;
    i=0;
    while (plist[i] != NULL) {
      mem_realloc(plist[i]->number,0);
      mem_realloc(plist[i]->text,0);
      mem_realloc(plist[i],0);
      ++i;
    }
    mem_realloc(plist,0);
    return NULL;
  }
  plist[i] = mem_realloc(plist[i],0);
  plist[i] = NULL;
  plist = mem_realloc(plist,(i+1)*sizeof(*plist));
  if (fp != stdin) {
    fclose(fp);
  }
  return plist;
}

int pbook_send_entry (struct pbook_entry* entry,
		      enum charset_type charset)
{
  char* command;
  char* temp;
  char* translated;
  int retval = 0;

  if (entry == NULL) {
    return 0;
  }
  translated = NULL;
  switch(charset) {
  case CHARSET_UCS2:
    translated = convert_to_ucs2_hexstring(entry->text);
    break;
  case CHARSET_GSM:
    temp = convert_to_gsm(entry->text);
    translated = pb_mask_chars(temp);
    mem_realloc(temp,0);
    break;
  default:
    return 0;
    break;
  }

  command = mem_alloc(sizeof(AT_PREFIX)-1
		      + sizeof(AT_WRITE_PB_ENTRY)-1 + 1
		      + numlen(entry->slot) + 1
		      + 1 + strlen(entry->number) + 2
		      + numlen(numtype(entry->number)) + 1
		      + 1 + strlen(translated) + 2 ,1);
  new_at_command(command,AT_WRITE_PB_ENTRY);
  add_at_command(command,"=%d,\"%s\",%d,\"%s\"",entry->slot,
		 entry->number,numtype(entry->number),translated);
  mem_realloc(translated,0);
  temp = tty_write_read(command);
  mem_realloc(command,0);
  if (check_return_code(temp,NULL,0) == 1) {
    retval = 1;
  }
  return retval;
}

int pbook_get_ranges (char* mem, int write, int* min, int* max, int* nrlen, int* txtlen) {
  char* command;
  char* prefix;
  size_t prefix_len;
  char* ausgabe;
  char* temp;
  
  if (mem == NULL) {
    return 0;
  }
  if (write) {
    command = AT_TEST(AT_WRITE_PB_ENTRY);
    prefix = AT_REPLY_PREFIX(AT_WRITE_PB_ENTRY);
  } else {
    if (strcasecmp(mem,"RD") == 0 ||
	strcasecmp(mem,"CS") == 0) {    
      //get sorted pb index
      command = AT_TEST(AT_SORTED_PB_ENTRY);
      prefix = AT_REPLY_PREFIX(AT_SORTED_PB_ENTRY);
    } else {
      //get pb index
      command = AT_TEST(AT_READ_PB_ENTRY);
      prefix = AT_REPLY_PREFIX(AT_READ_PB_ENTRY);
    }
  }
  prefix_len = strlen(prefix);

  ausgabe = tty_write_read(command);
  if (check_return_code(ausgabe,NULL,0) != -1) {
    return 0;
  }
  mem_realloc(tty_read(command),0);
  /* examples:
   * +CPBW: (1-100),20,(128-255),17
   * +CPBR: (1-100),20,17
   * ^SPBG: (1-75),20,17
   */
  //minimum of range
  temp = ausgabe + prefix_len;
  if (min != NULL) {
    if (*temp != '(') {
      return 0;
    }
    *min = atoi(temp+1);
  }
  //maximum of range
  if (max != NULL) {
    while (isdigit((int)*temp) || *temp == '(') {
      ++temp;
    }
    switch(*temp) {
    case '-':
      *max = atoi(temp+1);
      break;
    case ')':
      *max = *min;
      break;
    default:
      return 0;
    }
  }
  //maximum number field length
  if (nrlen != NULL) {
    temp = index(temp,',');
    if (temp == NULL) {
      return 0;
    }
    *nrlen = atoi(temp+1);
  }
  //maximum text field length
  if (txtlen != NULL) {
    temp = rindex(temp,',');
    if (temp == NULL) {
      return 0;
    }
    *txtlen = atoi(temp+1);
  }  
  return 1;
}

void pbook_send_file (char* file, char* phonebook) {
  int first_entry;
  int last_entry;
  int nr_length;
  int text_length;
  enum charset_type ct = CHARSET_GSM;
  struct pbook_entry** plist;
  int i = 0;

  if (set_charset("UCS2")) {
    ct = CHARSET_UCS2;
  }

  if (!pbook_get_ranges(phonebook,1,&first_entry,&last_entry,&nr_length,&text_length)) {
    errexit("Error on getting limits of phonebook %s.\n",phonebook);
  }

  plist = pbook_file_read(file);
  if (plist == NULL) {
    errexit("Error: invalid phonebook file %s.\n",file);
  }
  if (plist[0] == NULL) {
    errexit("Error: empty phonebook file %s.\n",file);
  }

  while(plist[i] != NULL) {
    if (first_entry > plist[i]->slot ||
	plist[i]->slot > last_entry) {
      errexit ("Error: slot %d is out of range %d-%d.\n",plist[i]->slot,first_entry,last_entry);
    }
    if (strlen(plist[i]->number) > nr_length) {
      errexit("Error: number of slot %d is too long (max. %d).\n",plist[i]->slot,nr_length);
    }
    if (ucs4len(plist[i]->text) > text_length) {
      errexit("Error: text of entry %d is too long (max. %d)\n",plist[i]->slot,text_length);
    }
    if (i > last_entry-first_entry+1) {
      errexit("Error: too many entries in file %s (max. %d).\n",file,i);
    }
    ++i;
  }

  if (i > 1) {
    myprintf(0,"Sending %d entries...\n",i);
  } else {
    myprintf(0,"Sending one entry to slot %d...\n",plist[0]->slot);    
  }
  i=0;
  while(plist[i] != NULL) {
    //send the entry
    if (pbook_send_entry(plist[i],ct)) {
      if (i > 0) {
	myprintf(0," %d",plist[i]->slot);
      } else if (plist[i+1] != NULL) {
	myprintf(0,"%d",plist[i]->slot);
      }
    } else {
      if (i > 0) {
	myprintf(0,"\n");
      }      
      errexit("Error on sending entry to slot %d.\nPossible data corruption!\n",plist[i]->slot);
    }
    mem_realloc(plist[i]->number,0);
    mem_realloc(plist[i]->text,0);
    mem_realloc(plist[i],0);
    ++i;
  }
  mem_realloc(plist,0);
  if (i > 0) {
    myprintf(0,"\n");
  }      
  myprintf(0,"done\n");
  set_charset("GSM");
}

#include <ctype.h>

int pbook_parse_entry (char* entry_line, int* index,
		       char** number, char** text)
{
  char* start;
  char* temp;

  if (entry_line == NULL ||
      index == NULL ||
      number == NULL ||
      text == NULL) {
    return -1;
  }

  *index = 0;
  *number = NULL;
  *text = NULL;

  /* 1st token: index */
  /*   checking field */
  temp = entry_line;
  while(isdigit((int)*temp) && *temp != 0) {
    ++temp;
  }
  if (*temp != ',') {
    return 0;
  }
  /*   reading field */
  *index=atoi(entry_line);


  /* 2nd token: number enclosed in "" */
  /*   checking field */
  if (*(++temp) != '"' || *(++temp) == 0) {
    return 0;
  }
  start = temp;
  while (*temp != '"' && *temp != 0) {
    ++temp;
  }
  if (*temp != '"' || *(++temp) != ',') {
    return 0;
  }
  /*   reading field */
  *number = strn_dup(start,temp-1-start);


  /* 3rd token: number format integer (ignored) */
  ++temp;
  while (*temp != ',' && *temp != 0) {
    ++temp;
  }
  if (*temp != ',') {
    return 0;
  }
  
  /* 4th token: text enclosed in "" */
  /*   checking field */
  if (*(++temp) != '"' || *(++temp) == 0) {
    return 0;
  }
  start = temp;
  temp += strlen(temp)-1;
  if (*temp != '"') {
    return 0;
  }
  /*   reading field */
  *text = strn_dup(start,temp-start);

  return 1;
}

int pbook_write_entry_to_file (int fd, unsigned int index,
			       char* number, char* text)
{
  char* num_ptr;
  char* txt_ptr;
  char* empty = "";
  int retval = 1;
  char* buffer;

  if (fd < 0) {
    return 0;
  }

  buffer = mem_alloc(numlen(index)+1,1);
  sprintf(buffer,"%u",index);

  if (number == NULL) {
    num_ptr = empty;
  } else {
    num_ptr = number;
  }
  if (text == NULL) {
    txt_ptr = empty;
  } else {
    txt_ptr = text;
  }

  if (write(fd,buffer,strlen(buffer)) == -1 ||
      write(fd,",\"",2) == -1 ||
      write(fd,num_ptr,strlen(num_ptr)) == -1 ||
      write(fd,"\",\"",3) == -1 ||
      write(fd,txt_ptr,strlen(txt_ptr)) == -1 ||
      write(fd,"\"\n",2) == -1){
    retval = -1;
  }
  mem_realloc(buffer,0);
  return retval;
}

void pbook_get (char* file, char* phonebook) {
  char at_command[BUFSIZ];
  char* ausgabe;
  char temp[BUFSIZ];
  int myfd=-1,
    i,
    first_entry=-1,
    current_entry=-1,
    last_entry=-1,
    next_entry=-1;
  ucs4char_t* wide_str;
  int unicode;
  char *command;
  char *reply_prefix;
  int  reply_prefix_len;

  int status;
  int index;
  char* number;
  char* text;

  ausgabe = NULL;

  unicode = set_charset("UCS2");

  memset(at_command,0,sizeof(at_command));
  memset(temp,0,sizeof(temp));

  //reading available indexes
  if (!pbook_get_ranges(phonebook,0,&first_entry, &last_entry,NULL,NULL)) {
    errexit("Error on getting limits of phonebook %s.\n",phonebook);
  }
  if (first_entry==0 && last_entry==0) {
    //emty phonebook
    myprintf(0,"Nothing to get.\n");
    return;
  }
					
  //accessing file
  if (str_len(file)) {
    myfd=open_myFile_rw(file);
  }
  //getting the data
  myprintf(0,"Receiving:");
  if (str_len(file) && !strcmp(file,"-")) {
    myprintf(0,"\n");
  }

// number of requesting numbers
// higher means a little bit more speed
// lower helps slow phones/SIM-cards to not time out
#define delta	5

  for (next_entry=first_entry; next_entry<=last_entry; next_entry+=delta) {
    ausgabe=NULL;

    if ((!strcasecmp(phonebook,"RD"))||(!strcasecmp(phonebook,"CS"))) {
    /* Get sorted pb */
      command   = AT_WRITE(AT_SORTED_PB_ENTRY);
      reply_prefix = AT_REPLY_PREFIX(AT_SORTED_PB_ENTRY);
    } else {
    /* Get pb */
      command = AT_WRITE(AT_READ_PB_ENTRY);
      reply_prefix = AT_REPLY_PREFIX(AT_READ_PB_ENTRY);
    }
    reply_prefix_len=strlen(reply_prefix);

    strcpy(at_command, command);		// new_at_command
    sprintf(temp,"%d,%d",next_entry,min(next_entry+delta-1, last_entry));
    add_at_command(at_command,temp);
    ausgabe=tty_write_read(at_command);
    if (strstr(ausgabe,"ERROR")!=NULL) {
      errexit("Error on selecting phonebook\n");
    }
    current_entry=next_entry;
    //read until OK or ERROR or Invalid Index
    while (strcmp(ausgabe,"OK")
	   && strcmp(ausgabe,"ERROR")
	   && strncmp(ausgabe,"+CME ERROR",10)) {
      status = pbook_parse_entry(ausgabe+reply_prefix_len,&index,&number,&text);
      if (status <= 0) {
	close(myfd);
	errexit("\nERROR on parsing entry: %s\n",ausgabe+reply_prefix_len);
      } else {
	for (i=current_entry;i<index;i++) {
	  if (pbook_write_entry_to_file(myfd,i,"","") < 0) {
	    close(myfd);
	    errexit("\nERROR on writing to %s: %s\n",file, strerror(errno));
	  }
	  if (str_len(file) == 0 || strcmp(file,"-")) {
	    myprintf(0," (%d)",i);
	  }
	}
	current_entry=index;
	if (unicode) {
	  wide_str = convert_from_ucs2_hexstring(text);
	} else {
	  wide_str = convert_from_gsm(text);
	}
	text = mem_realloc(text,0);
	text = convert_to_system(wide_str,REPMODE_ESCAPE_CHARS);
	if (pbook_write_entry_to_file(myfd,index,number,text) < 0) {
	  close(myfd);
	  errexit("\nERROR on writing to %s: %s\n",file, strerror(errno));
	}
	if (str_len(file) == 0 || strcmp(file,"-")) {
	  myprintf(0," %d",index);
	}
	wide_str = mem_realloc(wide_str,0);
      }
      number = mem_realloc(number,0);
      text = mem_realloc(text,0);

      //reading next entry
      mem_realloc(ausgabe,0);
      ausgabe=tty_read(at_command);
      current_entry++;
    }
    
    for (i=current_entry;i<=min(next_entry+delta-1, last_entry);i++) {
      if (pbook_write_entry_to_file(myfd,i,"","") < 0) {
	close(myfd);
	errexit("\nERROR on writing to %s: %s\n",file, strerror(errno));
      }
      if (str_len(file) == 0 || strcmp(file,"-")) {
	myprintf(0," (%d)",i);
      }
    }
  }
  myprintf(0,"\n");

  //be nice to other programs
  set_charset("GSM");

  if (ausgabe==NULL || !strcmp(ausgabe,"OK")
      /*||!strcasecmp(ausgabe,"+CME ERROR: INVALID INDEX")*/){
    myprintf(0,"Received all gettable entries\n");
  } else {
    errexit("An ERROR occured. Possible data corruption!\n");
  }
  mem_realloc(ausgabe,0);
}
