/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
	 
#include "common.h"
#include "helpers.h"
#include "memtypes.h"

#include <stdlib.h>
#include <string.h>
#include <strings.h>

inline char* get_vendor (void) {
  return get_value("AT+CGMI");
}

inline char* get_model (void) {
  return get_value("AT+CGMM");
}

inline char* get_revision (void) {
  return get_value("AT+CGMR");
}

inline char* get_phoneserial (void) {
  return get_value("AT+CGSN");
}

inline char* get_simserial (void) {
  return get_value("AT+CIMI");
}

int pbook_select_mem (char* mem) {
  char* command;
  char* temp;
  int retval;

  command = mem_alloc(sizeof(AT_PREFIX)-1 + 5
		      + 2 + str_len(mem) + 2, 1);
  temp=get_vendor();
  if (temp!=NULL && !strcmp(temp,"SIEMENS")) {
    new_at_command(command,"^SPBS=");
  } else {
    new_at_command(command,"+CPBS=");
  }
  add_at_command(command,mem);
  retval = verify_command(command,NULL,0);
  mem_realloc(command,0);
  return retval;
}

int sms_select_mem (char* mem, struct slot_range* r,
		    int* current_fill)
{
  char* command;
  char* temp;
  char* ausgabe;
  int retval = 0;

  if (str_len(mem) == 0) {
    return 0;
  }
  command = mem_alloc(sizeof(AT_PREFIX)-1 + 5 + 1
		      + 2*(str_len(mem)+2) + 2, 1);
  new_at_command(command,"+CPMS");
  add_at_command(command,"=\"%s\",\"%s\"",mem,mem);
  ausgabe = get_value(command);
  //example: +CPMS: 10,112,10,112,8,12
  mem_realloc(command,0);
  if (str_len(ausgabe) != 0 &&
      strncmp(ausgabe,"+CPMS: ",7) == 0) {
    if (r != NULL) {
      strncpy(r->name,mem,sizeof(r->name));
      temp = index(ausgabe,',');
      if (str_len(temp)) {
	r->min = 1;
	r->max = atoi(temp+1);
      }
    }
    if (current_fill != NULL && str_len(ausgabe+7)) {
      *current_fill = atoi(ausgabe+7);
    }
    retval = 1;
  }
  mem_realloc(ausgabe,0);
  return retval;
}

int phone_init (void) {
  int retval;
  char* command = "ATZ";
  char* ack;

  tty_write_command("");
  retval = verify_command(command,NULL,0);
  if (retval < 0) { //timeout
    ack = tty_read(command);
    retval = check_return_code(ack,NULL,0);
    mem_realloc(ack,0);
    if (retval < 0) {
      retval = 0;
    }
  }
  return retval;;
}

int command_echo (int enable) {
  char* command;
  if (enable) {
    command = "ATE1";
  } else {
    command = "ATE0";
  }
  return verify_command(command,NULL,0);
}

int verbose_errors (int enable) {
  char* command;
  if (enable) {
    command = "AT+CMEE=2";
  } else {
    command = "AT+CMEE=0";
  }
  return verify_command(command,NULL,0);
}

char* get_charset (void) {
  char* ausgabe;
  char* temp;
    
  ausgabe=get_value("AT+CSCS?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CSCS: \"",8)) {
    if ((temp=index(ausgabe+8,'"'))!=NULL) {
      temp=strn_dup(ausgabe+8,temp-ausgabe-8);
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

char** get_charsets (void) {
  char* ausgabe;
  char** retval;
  int p;

  ausgabe=get_value("AT+CSCS=?");
  p = strlen(AT_REPLY_PREFIX(AT_CHARSET));
  //example: +CSCS: ("GSM","UCS2")
  if (str_len(ausgabe) &&
      !strncmp(ausgabe,AT_REPLY_PREFIX(AT_CHARSET),p)) {
    retval = parse_at_stringlist(ausgabe+p);
  } else {
    retval = NULL;
  }
  return retval;
}

int set_charset(char *charset) {
  char *command;
  int  retval;

  if (!str_len(charset)) {
    errexit ("No charset specified.\n");
  }
  command = mem_alloc(sizeof(AT_PREFIX)-1
		      + sizeof(AT_CHARSET)-1
		      + 2 + str_len(charset) + 2, 1);
  new_at_command(command, AT_CHARSET);
  add_at_command(command, "=\"%s\"",charset);
  retval = verify_command(command,NULL,0);
  mem_realloc(command,0);
  return retval;
}

void set_smsc (char* number) {
  char* command;
  char error[128];

  if (!str_len(number)) {
    errexit("No number specified.\n");
  } else if (!is_pnumber(number,1)) {
    errexit("No valid number specified.\n");
  }
  command = mem_alloc(sizeof(AT_PREFIX)-1
		      + sizeof(AT_SMSC)-1
		      + 2 + str_len(number) + 2, 1);
  new_at_command(command,AT_SMSC);
  add_at_command(command,"=\"%s\",%d",number,numtype(number));

  if (verify_command(command,error,sizeof(error))) {
    myprintf(0,"SMS server number was set to %s\n",number);
  } else {
    errexit("Error: %s\n",error);
  }
  mem_realloc(command,0);
}

char* get_smsc (void) {
  char* ausgabe;
  char* temp;
    
  ausgabe=get_value("AT+CSCA?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CSCA: \"",8)) {
    if ((temp=strstr(ausgabe+8,"\","))!=NULL) {
      temp=strn_dup(ausgabe+8,temp-ausgabe-8);
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

#include "timeincl.h"

void set_time (void) {
  char at_command[128];
  char* ack;
  time_t seconds;
  char timestr[128];
	
  memset(at_command,0,sizeof(at_command));
  memset(timestr,0,sizeof(timestr));
	
  new_at_command(at_command,"+CCLK");
  time(&seconds); //seconds since Unix
  //string formatting with conversion to local time
  strftime(timestr,sizeof(timestr),"%y/%m/%d,%H:%M:%S",localtime(&seconds));
  add_at_command(at_command,"=\"%s\"",timestr);
  ack=tty_write_read(at_command);
  if (!strcmp(ack,"OK")) {
    myprintf(0,"Time was synchronized\n");
  } else {
    errexit("Time could not be synchronized\n");
  }
  mem_realloc(ack,0);
}

struct tm* get_time (void) {
  char* ausgabe;
  struct tm* phonetime;
  
  ausgabe=get_value("AT+CCLK?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CCLK: \"",8)) {
    phonetime=mem_alloc(sizeof(struct tm),1);
    if (strptime(ausgabe+8,"%y/%m/%d,%H:%M:%S",phonetime)!=NULL) {
      mem_realloc(ausgabe,0);
      return phonetime;
    }
    mem_realloc(ausgabe,0);
    mem_realloc(phonetime,0);
  }
  return NULL;
}

char* get_simid (void) {
  char* ausgabe;

  ausgabe=get_vendor();
  if(ausgabe!=NULL && !strcasecmp(ausgabe,"SIEMENS")) {
    ausgabe=get_value("AT^SCID");
    if (ausgabe!=NULL && !strncmp(ausgabe,"^SCID: ",7)) {
      memmove(ausgabe,ausgabe+7,strlen(ausgabe+7)+1);
      return mem_realloc(ausgabe,strlen(ausgabe)+1);
    }
  }
  return NULL;
}

char* get_operator (void) {
  char* ausgabe;
  char* temp;

  ausgabe=get_value("AT+COPS?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+COPS: ",7)) {
    if ((temp=index(ausgabe+7,'"'))!=NULL) {
      ++temp;
      if (temp[strlen(temp)-1]=='"') {
	temp[strlen(temp)-1]=0;
      }
      temp=strn_dup(temp,strlen(temp));
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

char* get_battery (void) {
  char* ausgabe;
  char* temp;
    
  ausgabe=get_value("AT+CBC");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CBC: ",6)) {
    if ((temp=index(ausgabe+6,','))!=NULL) {
      temp=strn_dup(temp+1,strlen(temp+1));
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

int get_signal (int* ber_p) {
  char* ausgabe;
  char* temp;
  int signal = -1;

  if (ber_p != NULL) {
    *ber_p = -1;
  }    
  ausgabe=get_value("AT+CSQ");
  //example: +CSQ: 28,99
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CSQ: ",6)) {
    signal=atoi(ausgabe+6);
    if (signal==99) {
      signal=-1;
    } else {
      if ((signal=111-((signal-1)*2)) < 0) {
	signal = -1;
      }
    }
    if (ber_p != NULL) {
      if ((temp=index(ausgabe,','))!=NULL) {
	*ber_p=atoi(temp+1);
	if (*ber_p==7) {
	  *ber_p=255;
	} else if (*ber_p>=0 && *ber_p <7) {
	  *ber_p=2<<(*ber_p);
	} else {
	  // unknown value (the standard 99 means the same)
	  *ber_p=-1;
	}
      }
    }
    mem_realloc(ausgabe,0);
  }
  return signal;
}

char* get_netstatus (unsigned int* areacode_p, unsigned int* cellid_p) {
  char* ausgabe;
  char* index1;
  char* index2;

  ausgabe=get_value("AT+CREG=?");
  if (index(ausgabe,'2')==NULL) {
    mem_realloc(ausgabe,0);
    ausgabe=get_value("AT+CREG?");
  } else {
    mem_realloc(ausgabe,0);
    /* Fix: firmware may forget to send LAC/IC (ME45 v23)
     * Provided by Dominik Neubauer
     */
    ausgabe=get_value("AT+CREG=2;+CREG?;+CREG=0");
  }
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CREG: ",7)) {
    if (areacode_p!=NULL) {
      *areacode_p=0;
    }
    if (areacode_p!=NULL) {
      *cellid_p=0;
    }
    index1=index(ausgabe+7,'"');
    if (index1!=NULL) {
      index2=index(++index1,'"');
      if (index2!=NULL && areacode_p!=NULL) {
	*areacode_p=hexstr2int(index1,index2-index1);
	index1=index(index2+1,'"');
	if (index1!=NULL) {
	  index2=index(++index1,'"');
	  if (index2!=NULL && cellid_p!=NULL) {
	    *cellid_p=hexstr2int(index1,index2-index1);
	  }
	}
      }
    }
    if ((index1=index(ausgabe+7,','))!=NULL) {
      switch (*(index1+1)) {
      case '0':
	mem_realloc(ausgabe,0);
	return "not checked in, not seeking";
      case '1':
	mem_realloc(ausgabe,0);
	return "checked in";
      case '2':
	mem_realloc(ausgabe,0);
	return "not checked in, but seeking a network";
      case '3':
	mem_realloc(ausgabe,0);
	return "check-in denied by network";
      case '5':
	mem_realloc(ausgabe,0);
	return "registered, roaming";
      default:
	mem_realloc(ausgabe,0);
	return "unknown status";
      }
    }
  }
  return NULL;
}

char* get_gprs_status (unsigned int* areacode_p, unsigned int* cellid_p) {
  char* ausgabe;
  char* index1;
  char* index2;

  ausgabe=get_value("AT+CGREG?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CGREG: ",8)) {
    if (areacode_p!=NULL) {
      *areacode_p=0;
    }
    if (areacode_p!=NULL) {
      *cellid_p=0;
    }
    index1=index(ausgabe+8,'"');
    if (index1!=NULL) {
      index2=index(++index1,'"');
      if (index2!=NULL && areacode_p!=NULL) {
	*areacode_p=hexstr2int(index1,index2-index1);
	index1=index(index2+1,'"');
	if (index1!=NULL) {
	  index2=index(++index1,'"');
	  if (index2!=NULL && cellid_p!=NULL) {
	    *cellid_p=hexstr2int(index1,index2-index1);
	  }
	}
      }
    }
    if ((index1=index(ausgabe+8,','))!=NULL) {
      switch (*(index1+1)) {
      case '0':
	mem_realloc(ausgabe,0);
	return "not registered, not searching";
      case '1':
	mem_realloc(ausgabe,0);
	return "registered, home network";
      case '2':
	mem_realloc(ausgabe,0);
	return "not registered, searching";
      case '3':
	mem_realloc(ausgabe,0);
	return "registration denied by network";
      case '5':
	mem_realloc(ausgabe,0);
	return "registered, roaming";
      default:
	mem_realloc(ausgabe,0);
	return "unknown status";
      }
    }
  }
  return NULL;
}

int get_gprs_attach (void) {
  char* ausgabe;

  ausgabe=get_value("AT+CGATT?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CGATT: ",8)) {
    switch (ausgabe[8]) {
    case '0':
      mem_realloc(ausgabe,0);
      return 0;
    case '1':
      mem_realloc(ausgabe,0);
      return 1;
    }
  }
  return -1;
}

char* get_gprs_class (void) {
  char* ausgabe;

  ausgabe=get_value("AT+CGCLASS?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CGCLASS: ",10)) {
    memmove(ausgabe,ausgabe+10,strlen(ausgabe+10)+1);
    return mem_realloc(ausgabe,strlen(ausgabe)+1);
  }
  return NULL;
}

void info_misc (FILE* fd) {
  char *temp;
  int i,j;
  struct tm* ltime;
  char** memlist;

  fprintf(fd,"Phone related information:\n");
  fprintf(fd,"Vendor:\t\t%s\n",temp=get_vendor());
  mem_realloc(temp,0);
  fprintf(fd,"Model:\t\t%s\n",temp=get_model());
  mem_realloc(temp,0);
  fprintf(fd,"Revision:\t%s\n",temp=get_revision());
  mem_realloc(temp,0);
  fprintf(fd,"IMEI:\t\t%s\n",temp=get_phoneserial());
  mem_realloc(temp,0);
  fprintf(fd,"Battery:\t%s%%\n",temp=get_battery());
  mem_realloc(temp,0);
  memlist=get_charsets();
  fprintf(fd,"Charsets:\t");
  if (memlist == NULL) {
    fprintf(fd,"none");
  } else {
    for (i=0;memlist[i]!=NULL;++i) {
      if (i != 0) {
	fprintf(fd,", ");
      }
      fprintf(fd,"%s",memlist[i]);
      mem_realloc(memlist[i],0);
    }
  }
  fprintf(fd,"\n");
  mem_realloc(memlist,0);
  temp=mem_alloc(81,1);
  ltime=get_time();
  if (ltime!=NULL && strftime(temp,80,"%c",ltime)) {
    fprintf(fd,"Time:\t\t%s\n",temp);
    mem_realloc(ltime,0);
  }
  mem_realloc(temp,0);

  fprintf(fd,"\nSIM card related information:\n");
  fprintf(fd,"IMSI:\t\t%s\n",temp=get_simserial());
  mem_realloc(temp,0);
  fprintf(fd,"card ID:\t%s\n",temp=get_simid());
  mem_realloc(temp,0);

  fprintf(fd,"\nNetwork related information:\n");
  temp=get_netstatus(&i,&j);
  if (temp!=NULL) {
    fprintf(fd,"Status:\t\t%s\n",temp);
    if (i) {
      fprintf(fd,"Area code:\t%04X\n",i);
      if (j) {
	fprintf(fd,"Cell ID:\t%04X\n",j);
      }
    }
  }
  temp=get_operator();
  if (temp!=NULL) {
    fprintf(fd,"Operator:\t%s\n",temp);
  }
  mem_realloc(temp,0);
  temp=get_smsc();
  if (temp!=NULL) {
    fprintf(fd,"SMS Server:\t%s\n",temp);
  }
  mem_realloc(temp,0);
  if ((i=get_signal(&j)) >= 0) {
    fprintf(fd,"Signal:\t\t%d dBm\n",-i);
    if (j >= 0) {
      if (j <= 128) {
	fprintf(fd,"BER:\t\t<=%d,%d%%\n",j/10,j%10);
      } else {
	fprintf(fd,"BER:\t\t>12,8%%\n");
      }
    }
  }
  temp=get_gprs_class();
  if (temp!=NULL) {
    fprintf(fd,"GPRS class:\t");
    temp=(char *)strtok(temp,"()\",");
    while (temp!=0) {
      fprintf(fd,"%s",temp);
      temp=(char *)strtok(NULL,"()\",");
      if (temp!=0) {
	fprintf(fd,", ");
      }
    }
    fprintf(fd,"\n");
  }
  mem_realloc(temp,0);
  temp=get_gprs_status(NULL,NULL);
  if (temp!=NULL) {
    fprintf(fd,"GRPS status:\t%s",temp);
    i=get_gprs_attach();
    if (i==0) {
      fprintf(fd,", detached\n");
    } else if (i==1) {
      fprintf(fd,", attached\n");
    }
  }

  fprintf(fd,"\nAvailable memories:\n");
}

void info_mem (FILE* fd, int full) {
  fprintf(fd,"Binary files:\t");
  file_print_memlist(fd,full);
  fprintf(fd,"Phonebooks:\t");
  pbook_print_memlist(fd,full);
  fprintf(fd,"SMS storages:\t");
  sms_print_memlist(fd,full);
}

void info (char* file, int misc, int mem){
  if (misc || mem) {
    FILE* fd;
    if (!str_len(file) || !strcmp(file,"-")) {
      fd=stdout;
    } else {
      fd=fdopen(open_myFile_rw(file),"w+");
    }
    if (misc) {
      info_misc(fd);
    }
    info_mem(fd,mem);
    if (str_len(file) && strcmp(file,"-")) {
      fclose(fd);
    }
  }
}

