/** @file scim_socket_transaction.cpp
 *  @brief Implementation of SocketTransaction related classes.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_transaction.cpp,v 1.26.2.1 2004/10/03 13:16:12 suzhe Exp $
 *
 */

#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_STL_ALGORITHM
#define Uses_C_STDLIB

#include "scim_private.h"
#include "scim.h"

namespace scim {

#define SCIM_TRANS_MIN_BUFSIZE 128
#define SCIM_TRANS_MAX_BUFSIZE (1048576*16)
#define SCIM_TRANS_MAGIC       0x4d494353
#define SCIM_TRANS_HEADER_SIZE (sizeof (uint32) * 4)

struct SocketTransaction::SocketTransactionImpl
{
    size_t         m_buffer_size;
    size_t         m_read_pos;
    size_t         m_write_pos;

    unsigned char *m_buffer;

    SocketTransactionImpl (size_t bufsize)
        : m_buffer_size (std::max ((size_t)SCIM_TRANS_MIN_BUFSIZE, bufsize)),
          m_read_pos (SCIM_TRANS_HEADER_SIZE),
          m_write_pos (SCIM_TRANS_HEADER_SIZE),
          m_buffer (new unsigned char [std::max ((size_t)SCIM_TRANS_MIN_BUFSIZE, bufsize)]) {
    }

    ~SocketTransactionImpl () {
        delete [] m_buffer;
    }

    void request_buffer_size (size_t request) {
        if (m_write_pos + request >= m_buffer_size) {
            size_t bufsize = std::max ((size_t) SCIM_TRANS_MIN_BUFSIZE, request + 1) + m_buffer_size;
            unsigned char *tmp = new unsigned char [bufsize];

            memcpy (tmp, m_buffer, m_buffer_size);
            std::swap (tmp, m_buffer);
            delete [] tmp;

            m_buffer_size = bufsize;
        }
    }

    uint32 calc_checksum () const {
        uint32 sum = 0;

        unsigned char *ptr = m_buffer + SCIM_TRANS_HEADER_SIZE;
        unsigned char *ptr_end = m_buffer + m_write_pos;

        while (ptr < ptr_end) {
            sum += (uint32) (*ptr);
            sum = (sum << 1) | (sum >> 31);
            ++ ptr;
        }

        return sum;
    }
};

SocketTransaction::SocketTransaction (size_t bufsize)
    : m_impl (new SocketTransactionImpl (bufsize))
{
}

SocketTransaction::~SocketTransaction ()
{
    delete m_impl;
}

bool
SocketTransaction::valid () const
{
    return m_impl->m_buffer && m_impl->m_buffer_size;
}

bool
SocketTransaction::write_to_socket (const Socket &socket, uint32 signature) const
{
    if (socket.valid ()) {
        scim_uint32tobytes (m_impl->m_buffer, signature);
        scim_uint32tobytes (m_impl->m_buffer + sizeof (uint32), SCIM_TRANS_MAGIC);
        scim_uint32tobytes (m_impl->m_buffer + sizeof (uint32) * 2, m_impl->m_write_pos - SCIM_TRANS_HEADER_SIZE);
        scim_uint32tobytes (m_impl->m_buffer + sizeof (uint32) * 3, m_impl->calc_checksum ());
        return socket.write (m_impl->m_buffer, m_impl->m_write_pos) == (int) m_impl->m_write_pos;
    }
    return false;
}

bool
SocketTransaction::read_from_socket (const Socket &socket, int timeout)
{
    if (socket.valid ()) {
        unsigned char buf [sizeof (uint32) * 2];
        uint32 sign1, sign2;
        uint32 checksum;
        int size;
        int nbytes;

        nbytes = socket.read_with_timeout (buf, sizeof (uint32) * 2, timeout);
        if (nbytes < sizeof (uint32) * 2)
            return false;

        sign1 = scim_bytestouint32 (buf);
        sign2 = scim_bytestouint32 (buf + sizeof (uint32));

        if (sign1 != SCIM_TRANS_MAGIC && sign2 != SCIM_TRANS_MAGIC)
            return false;

        if (sign2 == SCIM_TRANS_MAGIC) {
            nbytes = socket.read_with_timeout (buf, sizeof (uint32), timeout);
            if (nbytes < sizeof (uint32))
                return false;
            size = scim_bytestouint32 (buf);
        } else {
            size = (int) sign2;
        }

        nbytes = socket.read_with_timeout (buf, sizeof (uint32), timeout);
        if (nbytes < sizeof (uint32))
            return false;

        checksum = scim_bytestouint32 (buf);

        if (size <= 0 || size > SCIM_TRANS_MAX_BUFSIZE)
            return false;

        clear ();

        m_impl->request_buffer_size (size);

        while (size != 0) {
            nbytes = socket.read_with_timeout (m_impl->m_buffer + m_impl->m_write_pos, size, timeout);
            if (nbytes <= 0) {
                m_impl->m_write_pos = SCIM_TRANS_HEADER_SIZE;
                return false;
            }

            size -= nbytes;
            m_impl->m_write_pos += nbytes;
        }

        if (checksum != m_impl->calc_checksum ()) {
            m_impl->m_write_pos = SCIM_TRANS_HEADER_SIZE;
            return false;
        }

        return true;
    }
    return false;
}

void
SocketTransaction::put_command (int type)
{
    m_impl->request_buffer_size (1);

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_COMMAND;
    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, (uint32) type);
    m_impl->m_write_pos += sizeof (uint32);
}

void
SocketTransaction::put_data (uint32 val)
{
    m_impl->request_buffer_size (sizeof (val));

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_UINT32;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, val);

    m_impl->m_write_pos += sizeof (val);
}

void
SocketTransaction::put_data (const String &str)
{
    m_impl->request_buffer_size (str.length () + sizeof (uint32));

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_STRING;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, str.length ());

    m_impl->m_write_pos += sizeof (uint32);

    if (str.length ())
        memcpy (m_impl->m_buffer + m_impl->m_write_pos, str.c_str (), str.length ());

    m_impl->m_write_pos += str.length ();
}

void
SocketTransaction::put_data (const WideString &str)
{
    String mbs = utf8_wcstombs (str);

    m_impl->request_buffer_size (mbs.length () + sizeof (uint32));

    m_impl->m_buffer [m_impl->m_write_pos] = (unsigned char) SCIM_TRANS_DATA_WSTRING;

    m_impl->m_write_pos ++;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, mbs.length ());

    m_impl->m_write_pos += sizeof (uint32);

    if (mbs.length ())
        memcpy (m_impl->m_buffer + m_impl->m_write_pos, mbs.c_str (), mbs.length ());

    m_impl->m_write_pos += mbs.length ();
}

void
SocketTransaction::put_data (const KeyEvent &key)
{
    m_impl->request_buffer_size (sizeof (uint32) * 2);

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_KEYEVENT;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, key.code);

    m_impl->m_write_pos += sizeof (uint32);

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, key.mask);

    m_impl->m_write_pos += sizeof (uint32);
}

void
SocketTransaction::put_data (const AttributeList &attrs)
{
    size_t size = attrs.size () * (sizeof (unsigned int) * 3 + 1) + sizeof (uint32);

    m_impl->request_buffer_size (size);

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_ATTRIBUTE_LIST;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, attrs.size ());

    m_impl->m_write_pos += sizeof (uint32);

    for (size_t i=0; i<attrs.size (); i++) {
        m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) attrs[i].get_type ();
        scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, attrs[i].get_value ());
        m_impl->m_write_pos += sizeof (uint32);
        scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, attrs[i].get_start ());
        m_impl->m_write_pos += sizeof (uint32);
        scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, attrs[i].get_length ());
        m_impl->m_write_pos += sizeof (uint32);
    }
}

void
SocketTransaction::put_data (const Property &property)
{
    size_t request = property.get_key ().length () +
                     property.get_label ().length () +
                     property.get_icon ().length () +
                     property.get_tip ().length () +
                     sizeof (uint32) * 4 + 2;

    m_impl->request_buffer_size (request);

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_PROPERTY;

    put_data (property.get_key ());
    put_data (property.get_label ());
    put_data (property.get_icon ());
    put_data (property.get_tip ());

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) property.visible ();
    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) property.active ();
}

void
SocketTransaction::put_data (const PropertyList &properties)
{
    m_impl->request_buffer_size (sizeof(uint32));

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_PROPERTY_LIST;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, properties.size ());

    m_impl->m_write_pos += sizeof (uint32);

    for (PropertyList::const_iterator it = properties.begin (); it != properties.end (); ++ it)
        put_data (*it);
}

void
SocketTransaction::put_data (const LookupTable &table)
{
    unsigned char stat = 0;
    size_t i;

    m_impl->request_buffer_size (3);

    //Can be page up.
    if (table.get_current_page_start ())
        stat |= 1;

    //Can be page down.
    if (table.get_current_page_start () + table.get_current_page_size () <
        table.number_of_candidates ())
        stat |= 2;

    //Cursor is visible.
    if (table.is_cursor_visible ())
        stat |= 4;

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_LOOKUP_TABLE;
    m_impl->m_buffer [m_impl->m_write_pos ++] = stat;
    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) table.get_current_page_size ();
    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) table.get_cursor_pos_in_current_page ();

    // Store page labels.
    for (i = 0; i < table.get_current_page_size (); ++i)
        put_data (table.get_candidate_label (i));

    // Store page candidates and attributes.
    for (i = 0; i < table.get_current_page_size (); ++i) {
        put_data (table.get_candidate_in_current_page (i));
        put_data (table.get_attributes_in_current_page (i));
    }
}

void
SocketTransaction::put_data (const std::vector<uint32> &vec)
{
    size_t size = vec.size () * sizeof (uint32) + sizeof (uint32);

    m_impl->request_buffer_size (size);

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_VECTOR_UINT32;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, vec.size ());

    m_impl->m_write_pos += sizeof (uint32);

    for (size_t i=0; i<vec.size ();i++) {
        scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, vec[i]);
        m_impl->m_write_pos += sizeof (uint32);
    }
}

void
SocketTransaction::put_data (const std::vector<String> &vec)
{
    m_impl->request_buffer_size (sizeof(uint32));

    m_impl->m_buffer [m_impl->m_write_pos ++] = (unsigned char) SCIM_TRANS_DATA_VECTOR_STRING;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, vec.size ());

    m_impl->m_write_pos += sizeof (uint32);

    for (size_t i=0; i<vec.size ();i++) {
        put_data (vec[i]);
    }
}

void
SocketTransaction::put_data (const std::vector<WideString> &vec)
{
    m_impl->request_buffer_size (sizeof(uint32));

    m_impl->m_buffer [m_impl->m_write_pos] = (unsigned char) SCIM_TRANS_DATA_VECTOR_WSTRING;

    m_impl->m_write_pos ++;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, vec.size ());

    m_impl->m_write_pos += sizeof (uint32);

    for (size_t i=0; i<vec.size ();i++) {
        put_data (vec[i]);
    }
}

void
SocketTransaction::put_data (const char *raw, size_t bufsize)
{
    if (!raw || !bufsize)
        return;

    m_impl->request_buffer_size (bufsize);

    m_impl->m_buffer [m_impl->m_write_pos] = (unsigned char) SCIM_TRANS_DATA_RAW;

    m_impl->m_write_pos ++;

    scim_uint32tobytes (m_impl->m_buffer + m_impl->m_write_pos, (uint32) bufsize);

    m_impl->m_write_pos += sizeof (uint32);

    memcpy (m_impl->m_buffer + m_impl->m_write_pos, raw, bufsize);

    m_impl->m_write_pos += bufsize;
}

SocketTransactionDataType
SocketTransaction::get_data_type () const
{
    if (m_impl->m_write_pos <= m_impl->m_read_pos)
        return SCIM_TRANS_DATA_UNKNOWN;

    return (SocketTransactionDataType) m_impl->m_buffer [m_impl->m_read_pos];
}

bool
SocketTransaction::get_command (int &type)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_COMMAND) {

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        type = (int) scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);

        m_impl->m_read_pos += sizeof (uint32);

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (uint32 &val)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_UINT32) {

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        val = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);

        m_impl->m_read_pos += sizeof (uint32);

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (String &str)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_STRING) {

        size_t len;
        size_t old_read_pos = m_impl->m_read_pos;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        len = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        if (m_impl->m_write_pos < (m_impl->m_read_pos + len)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }

        if (len)
            str = String (m_impl->m_buffer + m_impl->m_read_pos, m_impl->m_buffer + m_impl->m_read_pos + len);
        else
            str = String ("");

        m_impl->m_read_pos += len;

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (WideString &str)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_WSTRING) {

        String mbs;
        size_t len;
        size_t old_read_pos = m_impl->m_read_pos;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        len = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        if (m_impl->m_write_pos < (m_impl->m_read_pos + len)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }

        if (len)
            mbs = String (m_impl->m_buffer + m_impl->m_read_pos, m_impl->m_buffer + m_impl->m_read_pos + len);
        else
            mbs = String ("");

        m_impl->m_read_pos += len;

        str = utf8_mbstowcs (mbs);
        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (KeyEvent &key)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_KEYEVENT) {

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) * 2 + 1))
            return false;

        m_impl->m_read_pos ++;

        key.code = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        key.mask = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (AttributeList &attrs)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_ATTRIBUTE_LIST) {

        AttributeType type;
        size_t num;
        uint32 value;
        uint32 start;
        uint32 length;

        attrs.clear ();

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        num = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        if (m_impl->m_write_pos < (m_impl->m_read_pos + (sizeof (uint32) * 3 + 1) * num)) {
            m_impl->m_read_pos -= (sizeof (uint32) + 1);
            return false;
        }

        for (size_t i=0; i<num; i++) {
            type = (AttributeType) m_impl->m_buffer [m_impl->m_read_pos];
            m_impl->m_read_pos ++;
            value = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
            m_impl->m_read_pos += sizeof (uint32);
            start = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
            m_impl->m_read_pos += sizeof (uint32);
            length = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
            m_impl->m_read_pos += sizeof (uint32);

            attrs.push_back (Attribute (start, length, type, value));
        }
        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (Property &property)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_PROPERTY) {

        size_t old_read_pos = m_impl->m_read_pos;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) * 4 + 3))
            return false;

        m_impl->m_read_pos ++;

        String str;

        if (! get_data (str)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }
        property.set_key (str);

        if (! get_data (str)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }
        property.set_label (str);

        if (! get_data (str)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }
        property.set_icon (str);

        if (! get_data (str)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }
        property.set_tip (str);

        if (m_impl->m_write_pos < (m_impl->m_read_pos + 2)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }

        property.show ((bool) m_impl->m_buffer [m_impl->m_read_pos]);
        m_impl->m_read_pos ++;

        property.set_active ((bool) m_impl->m_buffer [m_impl->m_read_pos]);
        m_impl->m_read_pos ++;

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (PropertyList &properties)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_PROPERTY_LIST) {

        size_t old_read_pos = m_impl->m_read_pos;
        size_t num;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        num = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        properties.clear ();
        Property prop;

        for (size_t i = 0; i < num; ++ i) {
            if (!get_data (prop)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }
            properties.push_back (prop);
        }

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (CommonLookupTable &table)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_LOOKUP_TABLE) {

        size_t i;

        size_t old_read_pos = m_impl->m_read_pos;

        unsigned char stat;
        uint32 page_size;
        uint32 cursor_pos;

        WideString    wstr;
        AttributeList attrs;

        std::vector<WideString> labels;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + 4))
            return false;

        table.clear ();

        m_impl->m_read_pos ++;

        stat = m_impl->m_buffer [m_impl->m_read_pos];
        m_impl->m_read_pos ++;

        page_size = (uint32) m_impl->m_buffer [m_impl->m_read_pos];
        m_impl->m_read_pos ++;

        cursor_pos = (uint32) m_impl->m_buffer [m_impl->m_read_pos];
        m_impl->m_read_pos ++;

        if (page_size > SCIM_LOOKUP_TABLE_MAX_PAGESIZE ||
            cursor_pos >= page_size) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }

        table.set_page_size (page_size);

        for (i = 0; i < page_size; ++i) {
            if (!get_data (wstr)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }
            labels.push_back (wstr);
        }

        table.set_candidate_labels (labels);

        //Can be paged up.
        if (stat & 1)
            table.append_candidate (0x3400);

        for (i = 0; i < page_size; ++i) {
            if (!get_data (wstr)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }
            if (!get_data (attrs)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }
            table.append_candidate (wstr, attrs);
        }

        // Can be paged down.
        if (stat & 2)
            table.append_candidate (0x3400);

        if (stat & 1) {
            table.set_page_size (1);
            table.page_down ();
            table.set_page_size (page_size);
        }

        table.set_cursor_pos_in_current_page (cursor_pos);

        if (stat & 4)
            table.show_cursor (true);
        else
            table.show_cursor (false);

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (std::vector<uint32> &vec)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_VECTOR_UINT32) {

        size_t old_read_pos = m_impl->m_read_pos;
        size_t num;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        num = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) * num)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }

        vec.clear ();

        for (size_t i=0; i<num; i++) {
            vec.push_back (scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos));
            m_impl->m_read_pos += sizeof (uint32);
        }

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (std::vector<String> &vec)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_VECTOR_STRING) {

        size_t old_read_pos = m_impl->m_read_pos;
        size_t num;
        String str;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        num = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        vec.clear ();

        for (size_t i=0; i<num; i++) {
            if (!get_data (str)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }
            vec.push_back (str);
        }

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (std::vector<WideString> &vec)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_VECTOR_WSTRING) {

        size_t old_read_pos = m_impl->m_read_pos;
        size_t num;
        WideString str;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        num = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        vec.clear ();

        for (size_t i=0; i<num; i++) {
            if (!get_data (str)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }
            vec.push_back (str);
        }

        return true;
    }
    return false;
}

bool
SocketTransaction::get_data (char **raw, size_t &bufsize)
{
    if (m_impl->m_write_pos > m_impl->m_read_pos &&
        m_impl->m_buffer [m_impl->m_read_pos] == SCIM_TRANS_DATA_RAW) {

        size_t old_read_pos = m_impl->m_read_pos;

        if (m_impl->m_write_pos < (m_impl->m_read_pos + sizeof (uint32) + 1))
            return false;

        m_impl->m_read_pos ++;

        bufsize = scim_bytestouint32 (m_impl->m_buffer + m_impl->m_read_pos);
        m_impl->m_read_pos += sizeof (uint32);

        if (!bufsize || m_impl->m_write_pos < (m_impl->m_read_pos + bufsize)) {
            m_impl->m_read_pos = old_read_pos;
            return false;
        }

        if (raw) {
            *raw = new char [bufsize];
            if (! (*raw)) {
                m_impl->m_read_pos = old_read_pos;
                return false;
            }

            memcpy (*raw, m_impl->m_buffer + m_impl->m_read_pos, bufsize);
        }

        m_impl->m_read_pos += bufsize;
        return true;
    }
    return false;
}

bool
SocketTransaction::skip_data ()
{
    if (m_impl->m_write_pos > m_impl->m_read_pos) {
        switch (m_impl->m_buffer [m_impl->m_read_pos]) {
            case SCIM_TRANS_DATA_COMMAND:
            {
                int cmd;
                return get_command (cmd);
            }
            case SCIM_TRANS_DATA_UINT32:
            {
                uint32 val;
                return get_data (val);
            }
            case SCIM_TRANS_DATA_STRING:
            {
                String str;
                return get_data (str);
            }
            case SCIM_TRANS_DATA_WSTRING:
            {
                WideString wstr;
                return get_data (wstr);
            }
            case SCIM_TRANS_DATA_KEYEVENT:
            {
                KeyEvent key;
                return get_data (key);
            }
            case SCIM_TRANS_DATA_ATTRIBUTE_LIST:
            {
                AttributeList attrs;
                return get_data (attrs);
            }
            case SCIM_TRANS_DATA_PROPERTY:
            {
                Property prop;
                return get_data (prop);
            }
            case SCIM_TRANS_DATA_PROPERTY_LIST:
            {
                PropertyList proplist;
                return get_data (proplist);
            }
            case SCIM_TRANS_DATA_LOOKUP_TABLE:
            {
                CommonLookupTable table;
                return get_data (table);
            }
            case SCIM_TRANS_DATA_VECTOR_UINT32:
            {
                std::vector <uint32> vec;
                return get_data (vec);
            }
            case SCIM_TRANS_DATA_VECTOR_STRING:
            {
                std::vector <String> vec;
                return get_data (vec);
            }
            case SCIM_TRANS_DATA_VECTOR_WSTRING:
            {
                std::vector <WideString> vec;
                return get_data (vec);
            }
            case SCIM_TRANS_DATA_RAW:
            {
                size_t bufsize;
                return get_data (NULL, bufsize);
            }
        }
    }
    return false;
}

void
SocketTransaction::rewind_read_pos ()
{
    m_impl->m_read_pos = SCIM_TRANS_HEADER_SIZE;
}

void
SocketTransaction::clear ()
{
    m_impl->m_write_pos = SCIM_TRANS_HEADER_SIZE;
    m_impl->m_read_pos = SCIM_TRANS_HEADER_SIZE;
}

static bool
scim_socket_trans_check_type (const String &types,
                              const String &atype)
{
    std::vector <String> type_list;
    scim_split_string_list (type_list, types, ',');

    return std::find (type_list.begin (), type_list.end (), atype) != type_list.end ();
}

bool
scim_socket_trans_open_connection   (uint32       &key,
                                     const String &client_type,
                                     const String &server_type,
                                     const Socket &socket,
                                     int           timeout)
{
    if (!socket.valid () || !client_type.length () || !server_type.length ())
        return false;

    SocketTransaction trans;

    trans.put_command (SCIM_TRANS_CMD_REQUEST);
    trans.put_command (SCIM_TRANS_CMD_OPEN_CONNECTION);
    trans.put_data (String (SCIM_BINARY_VERSION));
    trans.put_data (client_type);

    if (trans.write_to_socket (socket)) {
        int cmd;
        String server_types;
        if (trans.read_from_socket (socket, timeout) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (server_types) && scim_socket_trans_check_type (server_types, server_type) &&
            trans.get_data (key)) {
            trans.clear ();
            trans.put_command (SCIM_TRANS_CMD_REPLY);
            trans.put_command (SCIM_TRANS_CMD_OK);
            if (trans.write_to_socket (socket))
                return true;
        } else {
            trans.clear ();
            trans.put_command (SCIM_TRANS_CMD_REPLY);
            trans.put_command (SCIM_TRANS_CMD_FAIL);
            trans.write_to_socket (socket);
        }
    }

    return false;
}

String
scim_socket_trans_accept_connection (uint32       &key,
                                     const String &server_types,
                                     const String &client_types,
                                     const Socket &socket,
                                     int           timeout)
{
    if (!socket.valid () || !client_types.length () || !server_types.length ())
        return String ("");

    SocketTransaction trans;

    if (trans.read_from_socket (socket, timeout)) {
        int cmd;
        String version;
        String client_type;
        if (trans.get_command (cmd)  && cmd == SCIM_TRANS_CMD_REQUEST &&
            trans.get_command (cmd)  && cmd == SCIM_TRANS_CMD_OPEN_CONNECTION &&
            trans.get_data (version) && version == String (SCIM_BINARY_VERSION) &&
            trans.get_data (client_type) && 
            (scim_socket_trans_check_type (client_types, client_type) || client_type == "ConnectionTester")) {
            key = (uint32) rand ();
            trans.clear ();
            trans.put_command (SCIM_TRANS_CMD_REPLY);
            trans.put_data (server_types);
            trans.put_data (key);

            if (trans.write_to_socket (socket) &&
                trans.read_from_socket (socket, timeout) &&
                trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
                trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {

                // Client is ok, return the client type.
                return (client_type == "ConnectionTester") ? String ("") : client_type;
            }
        }
    }
    return String ("");
}

} // namespace scim

/*
vi:ts=4:nowrap:ai:expandtab
*/

