/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@tsinghua.org.cn>
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_launcher.cpp,v 1.1.2.1 2004/10/03 13:54:53 suzhe Exp $
 *
 */

#define Uses_SCIM_FRONTEND_MODULE
#define Uses_SCIM_IMENGINE_MODULE
#define Uses_SCIM_BACKEND
#define Uses_SCIM_CONFIG_PATH
#define Uses_C_LOCALE
#include "scim_private.h"
#include <scim.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>

using namespace scim;
using std::cout;
using std::cerr;
using std::endl;

FrontEndModule *frontend_module = 0;
ConfigModule   *config_module = 0;
ConfigPointer   config;

void signalhandler(int sig)
{
    config->flush ();

    cerr << "SCIM successfully exited.\n";

    exit (0);
}

void make_daemon ()
{
#if HAVE_DAEMON
    if (daemon (0, 0) == -1)
        cerr << "Error to make SCIM into a daemon!\n";

    return;
#else        
    pid_t id;
 
    id = fork ();
    if (id == -1) {
        cerr << "Error to make SCIM into a daemon!\n";
        return;
    } else if (id > 0) {
        exit (0);
    }

    id = fork ();
    if (id == -1) {
        cerr << "Error to make SCIM into a daemon!\n";
        return;
    } else if (id > 0) {
        exit (0);
    }

    return;
#endif
}
/**
 * argv [1]  config module name
 * argv [2]  imengine module list
 * argv [3]  frontend module name
 * argv [4]  debug verbose level
 * argv [5]  debug mask
 * argv [6]  debug output target
 */
int main (int argc, char *argv [])
{
    BackEndPointer      backend;

    std::vector<String> engine_list;
    std::vector<String> debug_mask_list;

    int new_argc;
    char * new_argv [40];

    bool print_msg = false;
    int verbose_level = 0;

    if (argc < 7) {
        cerr << "Missing arguments.\n";
        return 1;
    }

    DebugOutput::set_output (String (argv [6]));

    if (String (argv [5]) != "none") {
        scim_split_string_list (debug_mask_list, argv [5], ',');
        DebugOutput::disable_debug (SCIM_DEBUG_AllMask);
        for (size_t i=0; i<debug_mask_list.size (); i++)
            DebugOutput::enable_debug_by_name (debug_mask_list [i]);
    }

    verbose_level = atoi (argv [4]);

    DebugOutput::set_verbose_level (verbose_level);

    if (verbose_level > 0 && argv [6] != "none")
        print_msg = true;

    if (String (argv [2]) == "all") {
        scim_get_imengine_module_list (engine_list);
        for (size_t i = 0; i < engine_list.size (); ++i) {
            if (engine_list [i] == "socket") {
                engine_list.erase (engine_list.begin () + i);
                break;
            }
        }
    } else if (String (argv [2]) != "none") {
        scim_split_string_list (engine_list, argv [2], ',');
    }

    // put config module option
    new_argv [0] = argv [0];
    new_argv [1] = "-c";
    new_argv [2] = argv [1];

    // put additional option
    for (int i = 7; i < argc && i < 43; ++i)
        new_argv [i - 4] = argv [i];

    new_argc = std::min (argc - 4, 39);
    new_argv [new_argc] = 0;

#if ENABLE_DEBUG
    cerr << "Running: ";
    for (int i = 0; i < 40 && new_argv [i]; ++i)
        cerr << new_argv [i] << " ";
    cerr << "\n";
#endif

    try {
        // Try to load config module
        if (print_msg)
            cout << "Loading " << argv [1] << " Config module ...\n";

        if (String (argv [1]) != "dummy") {
            //load config module
            config_module = new ConfigModule (argv[1]);

            if (!config_module->valid ()) {

                if (print_msg)
                    cerr << "Can not load " << argv [1] << " Config module. Using dummy module instead.\n";

                delete config_module;
                config_module = 0;
            }

        }

        if (config_module) {
            config = config_module->create_config ("scim");
        } else {
            config = new DummyConfig ("scim");
        }

        if (config.null ()) {
            cerr << "Can not create Config Object!\n";
            return 1;
        }

        // create backend
        if (print_msg)
            cout << "Creating backend ...\n";

        backend = new CommonBackEnd (config, engine_list);

        //load FrontEnd module
        if (print_msg)
            cout << "Loading " << argv [3] << " FrontEnd module ...\n";

        frontend_module = new FrontEndModule (argv [3], backend, config, new_argc, new_argv);

        if (!frontend_module || !frontend_module->valid ()) {

            if (print_msg)
                cerr << "Failed to load " << argv [3] << " FrontEnd module.\n";

            return 1;
        }

        //reset backend pointer, in order to destroy backend automatically.
        backend.reset ();

        signal(SIGQUIT, signalhandler);
        signal(SIGTERM, signalhandler);
        signal(SIGINT,  signalhandler);
        signal(SIGHUP,  signalhandler);

        if (String (argv [0]).find ("daemon") != String::npos) {

            if (print_msg)
                cout << "Starting SCIM as daemon ..." << endl;

            make_daemon ();
        } else {
            if (print_msg)
                cout << "Starting SCIM ..." << endl;
        }

        frontend_module->run ();
    } catch (const std::exception & err) {

        if (print_msg)
            cerr << err.what () << "\n";

        return 1;
    }

    return 0;
}

/*
vi:ts=4:ai:nowrap:expandtab
*/
