#include <assert.h>
#include <malloc.h>
#include <pthread.h>
#include <string.h>

#include "scim-bridge-imcontext-manager.h"

typedef struct _IMContextContainer
{
    ScimBridgeIMContextID id;

    ScimBridgeIMContextID prev;
    ScimBridgeIMContextID next;

    ScimBridgeIMContext *imcontext;
} IMContextContainer;

static IMContextContainer *imcontext_containers = NULL;

static ScimBridgeIMContextID free_imcontext_container_root;
static ScimBridgeIMContextID used_imcontext_container_root;

static size_t imcontext_capacity;

static pthread_mutex_t mutex;

static int initialized = 0;

static size_t imcontext_count = 0;

/* Implementations */
void scim_bridge_initialize_imcontext_manager ()
{
    assert (!initialized);

    pthread_mutex_init (&mutex, NULL);
    pthread_mutex_lock (&mutex);

    imcontext_count = 0;
    imcontext_capacity = 0;

    free_imcontext_container_root = -1;
    used_imcontext_container_root = -1;

    initialized = 1;

    pthread_mutex_unlock (&mutex);

    return;
}


void scim_bridge_finalize_imcontext_manager ()
{
    assert (initialized);

    pthread_mutex_lock (&mutex);

    free (imcontext_containers);
    imcontext_containers = NULL;

    imcontext_count = 0;
    initialized = 0;

    pthread_mutex_unlock (&mutex);
}


void scim_bridge_add_imcontext (ScimBridgeIMContext *imcontext)
{
    assert (initialized);

    pthread_mutex_lock (&mutex);

    if (free_imcontext_container_root == -1) {
        const size_t new_imcontext_capacity = imcontext_capacity + 5;
        imcontext_containers = realloc (imcontext_containers, sizeof (IMContextContainer) * new_imcontext_capacity);

        int i;
        for (i = imcontext_capacity; i < new_imcontext_capacity; ++i) {
            IMContextContainer *container = &imcontext_containers[i];

            container->id = i;

            if (i > imcontext_capacity) container->prev = i - 1;
            else container->prev = -1;
            if (i < new_imcontext_capacity - 1) container->next = i + 1;
            else container->next = -1;

            container->imcontext = NULL;
        }

        free_imcontext_container_root = imcontext_capacity;
        imcontext_capacity = new_imcontext_capacity;
    }

    IMContextContainer *container = &imcontext_containers[free_imcontext_container_root];

    free_imcontext_container_root = container->next;
    if (free_imcontext_container_root != -1) imcontext_containers[free_imcontext_container_root].prev = -1;

    container->prev = -1;
    container->next = used_imcontext_container_root;
    if (used_imcontext_container_root != -1) imcontext_containers[used_imcontext_container_root].prev = container->id;
    used_imcontext_container_root = container->id;

    container->imcontext = imcontext;
    imcontext->parent.id = container->id;

    ++imcontext_count;

    pthread_mutex_unlock (&mutex);
}


void scim_bridge_remove_imcontext (ScimBridgeIMContext *imcontext)
{
    assert (initialized);
    assert (imcontext->parent.id >= 0 && imcontext->parent.id < imcontext_capacity);

    pthread_mutex_lock (&mutex);

    IMContextContainer *container = &imcontext_containers[imcontext->parent.id];

    assert (container->imcontext == imcontext);

    if (container->prev != -1) imcontext_containers[container->prev].next = container->next;
    if (container->next != -1) imcontext_containers[container->next].prev = container->prev;

    if (container->id == used_imcontext_container_root) used_imcontext_container_root = container->next;

    container->prev = -1;
    container->next = free_imcontext_container_root;
    if (free_imcontext_container_root != -1) imcontext_containers[free_imcontext_container_root].prev = container->id;
    free_imcontext_container_root = container->id;

    container->imcontext = NULL;

    imcontext->parent.id = -1;

    --imcontext_count;

    pthread_mutex_unlock (&mutex);
}


ScimBridgeIMContext *scim_bridge_find_imcontext (ScimBridgeIMContextID id)
{
    assert (initialized);
    assert (id >= 0 && id < imcontext_capacity);

    pthread_mutex_lock (&mutex);
    ScimBridgeIMContext *imcontext;

    if (id < 0 || id >= imcontext_capacity) {
        imcontext = NULL;
    } else {
        imcontext = imcontext_containers[id].imcontext;
    }

    pthread_mutex_unlock (&mutex);
    return imcontext;
}


size_t scim_bridge_get_imcontext_count ()
{
    return imcontext_count;
}


ScimBridgeIMContext *scim_bridge_get_first_imcontext ()
{
    pthread_mutex_lock (&mutex);

    ScimBridgeIMContext *imcontext;

    if (used_imcontext_container_root == -1) {
        imcontext = NULL;
    } else {
        imcontext = imcontext_containers[used_imcontext_container_root].imcontext;
    }

    pthread_mutex_unlock (&mutex);
    return imcontext;
}


ScimBridgeIMContext *scim_bridge_get_next_imcontext (ScimBridgeIMContext *imcontext)
{
    pthread_mutex_lock (&mutex);
    const ScimBridgeIMContextID id = imcontext->parent.id;

    ScimBridgeIMContext *next_imcontext;

    if (id < 0 || id >= imcontext_capacity) {
        next_imcontext = NULL;
    } else {
        IMContextContainer *container = &imcontext_containers[id];
        if (container->next != -1) {
            next_imcontext = imcontext_containers[container->next].imcontext;
        } else {
            next_imcontext = NULL;
        }
    }

    pthread_mutex_unlock (&mutex);
    return next_imcontext;
}
