#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <malloc.h>

#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>

#include "scim-bridge-client-kernel-protected.h"
#include "scim-bridge-client-messenger.h"
#include "scim-bridge-exception.h"
#include "scim-bridge-environment.h"
#include "scim-bridge-keyevent.h"
#include "scim-bridge-message.h"
#include "scim-bridge-messenger.h"
#include "scim-bridge-output.h"
#include "scim-bridge-string.h"

/* Private static variables */
static const int CONNECTION_TRY_MAX_COUNT = 3;

/* Private variables */
static int input_fd;
static int output_fd;

static ScimBridgeMessengerID messenger_id = -1;

static pthread_t messenger_thread;

/* Private functions */
static int launch_agent ();
static int open_connection (ScimBridgeException *except);
static int start_thread (ScimBridgeException *except);
static void *run_thread ();

static int received_message_unknown (ScimBridgeException *except, ScimBridgeMessageCode code);
static int received_message_commit (ScimBridgeException *except);
static int received_message_update_preedit (ScimBridgeException *except);
static int received_message_set_preedit_string (ScimBridgeException *except);
static int received_message_set_preedit_attributes (ScimBridgeException *except);
static int received_message_set_preedit_cursor_position (ScimBridgeException *except);
static int received_message_set_preedit_shown (ScimBridgeException *except);
static int received_message_forward_keyevent (ScimBridgeException *except);
static int received_message_beep (ScimBridgeException *except);
static int received_message_get_surrounding_string (ScimBridgeException *except);
static int received_message_delete_surrounding_string (ScimBridgeException *except);

/* Implementations */
int scim_bridge_client_initialize_messenger (ScimBridgeException *except)
{
    input_fd = -1;
    output_fd = -1;

    int retry_count;
    for (retry_count = 0; retry_count < CONNECTION_TRY_MAX_COUNT; ++retry_count) {
        if (open_connection (except)) {
            if (retry_count == 0) {
                launch_agent ();
            } else {
                usleep (100);
            }
        } else {
            break;
        }
    }
    if (input_fd < 0 || output_fd < 0) return -1;

    if (scim_bridge_initialize_messenger (except, &messenger_id, input_fd, output_fd))
        return -1;

    return start_thread (except);
}


int scim_bridge_client_finalize_messenger (ScimBridgeException *except)
{
    if (input_fd > 0 && output_fd > 0) {
        input_fd = -1;
        output_fd = -1;
    }

    if (messenger_id >= 0) {
        scim_bridge_finalize_messenger (except, messenger_id);
        messenger_id = -1;
    }

    return 0;
}


/* Helper functions */
int launch_agent ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 1, "Invoking the agent...");
    if (system ("scim-bridge-agent")) {
        scim_bridge_perrorln ("Failed to invoking the agent: %s", strerror (errno));

        return -1;
    }
    return 0;
}


int open_connection (ScimBridgeException *except)
{
    int output_socket_fd = socket (PF_UNIX, SOCK_STREAM, 0);
    if (output_socket_fd < 0) {
        scim_bridge_exception_set_errno (except, errno);
        scim_bridge_exception_set_message (except, "Failed to create the output socket");
        return -1;
    }

    struct sockaddr_un output_socket_addr;
    memset (&output_socket_addr, 0, sizeof (struct sockaddr_un));
    output_socket_addr.sun_family = AF_UNIX;
    strcpy (output_socket_addr.sun_path, scim_bridge_environment_get_client_to_agent_socket_path ());

    if (connect (output_socket_fd, (struct sockaddr*)&output_socket_addr, sizeof (struct sockaddr_un)) != 0) {
        scim_bridge_exception_set_errno (except, errno);
        scim_bridge_exception_set_message (except, "Failed to open the output socket");
        return -1;
    }

    int input_socket_fd = socket (PF_UNIX, SOCK_STREAM, 0);
    if (input_socket_fd < 0) {
        scim_bridge_exception_set_errno (except, errno);
        scim_bridge_exception_set_message (except, "Failed to create the input socket");
        return -1;
    }

    struct sockaddr_un input_socket_addr;
    memset (&input_socket_addr, 0, sizeof (struct sockaddr_un));
    input_socket_addr.sun_family = AF_UNIX;
    strcpy (input_socket_addr.sun_path, scim_bridge_environment_get_agent_to_client_socket_path ());

    if (connect (input_socket_fd, (struct sockaddr*)&input_socket_addr, sizeof (struct sockaddr_un)) != 0) {
        scim_bridge_exception_set_errno (except, errno);
        scim_bridge_exception_set_message (except, "Failed to open the input socket");
        return -1;
    }

    input_fd = input_socket_fd;
    output_fd = output_socket_fd;

    return 0;
}


int start_thread (ScimBridgeException *except)
{
    if (pthread_create (&messenger_thread, NULL, &run_thread, NULL) || pthread_detach (messenger_thread)) {
        scim_bridge_exception_set_errno (except, errno);
        scim_bridge_exception_set_message (except, "Failed to create new thread");
        return -1;
    }

    return 0;
}


void *run_thread ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 1, "run_messenger");

    int connection_broken = 0;
    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    while (!connection_broken && input_fd > 0 && output_fd > 0) {
        if (scim_bridge_messenger_open_input (&except, messenger_id)) {
            connection_broken = 1;
            break;
        }

        ScimBridgeMessageCode code;
        if (scim_bridge_messenger_read_input (&except, messenger_id, &code, sizeof (code))) {
            connection_broken = 1;
            break;
        }
        switch (code) {
            case SCIM_BRIDGE_MESSAGE_COMMIT:
                if (received_message_commit (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_SET_PREEDIT_STRING:
                if (received_message_set_preedit_string (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_SET_PREEDIT_ATTRIBUTES:
                if (received_message_set_preedit_attributes (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_SET_PREEDIT_CURSOR_POSITION:
                if (received_message_set_preedit_cursor_position (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_SET_PREEDIT_SHOWN:
                if (received_message_set_preedit_shown (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_UPDATE_PREEDIT:
                if (received_message_update_preedit (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_FORWARD_KEYEVENT:
                if (received_message_forward_keyevent (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_BEEP:
                if (received_message_beep (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_GET_SURROUNDING_STRING:
                if (received_message_get_surrounding_string (&except)) connection_broken = 1;
                break;
            case SCIM_BRIDGE_MESSAGE_DELETE_SURROUNDING_STRING:
                if (received_message_delete_surrounding_string (&except)) connection_broken = 1;
                break;
            default:
                if (received_message_unknown (&except, code)) {
                    connection_broken = 1;
                }
        }

    }

    if (input_fd > 0 || output_fd > 0) {
        scim_bridge_client_kernel_exception_occured (&except);
    } else {
        scim_bridge_client_kernel_cleanup (&except);
    }
    scim_bridge_exception_finalize (&except);

    return NULL;
}


/* Received message handler */

int received_message_unknown (ScimBridgeException *except, ScimBridgeMessageCode code)
{
    scim_bridge_perrorln ("Unknown message received: %d", code);
    scim_bridge_exception_set_errno (except, EBADRQC);
    scim_bridge_exception_set_message (except, "Unknwon message received");

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return -1;
}


int received_message_commit (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Commit' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    scim_bridge_client_kernel_commit (id);

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return 0;
}


int received_message_set_preedit_shown (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Set preedit shown' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    int shown;
    if (scim_bridge_messenger_read_input (except, messenger_id, &shown, sizeof (shown)))
        return -1;

    scim_bridge_client_kernel_set_preedit_shown (id, shown);

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return 0;
}


int received_message_set_preedit_string (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Set preedit string' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    size_t wstr_len;
    if (scim_bridge_messenger_read_input (except, messenger_id, &wstr_len, sizeof (wstr_len)))
        return -1;

    ucs4_t *wstr;
    wstr = malloc (sizeof (ucs4_t) * (wstr_len + 1));
    if (scim_bridge_messenger_read_input (except, messenger_id, wstr, sizeof (ucs4_t) * (wstr_len + 1)))
        return -1;

    scim_bridge_client_kernel_set_preedit_string (id, wstr, wstr_len);

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;
    return 0;
}


int received_message_set_preedit_attributes (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Set preedit attributes' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    size_t attr_count;
    if (scim_bridge_messenger_read_input (except, messenger_id, &attr_count, sizeof (attr_count)))
        return -1;

    ScimBridgeAttribute *attrs;
    if (attr_count != 0) attrs = malloc (sizeof (ScimBridgeAttribute) * attr_count);
    else attrs = NULL;
    if (scim_bridge_messenger_read_input (except, messenger_id, attrs, sizeof (ScimBridgeAttribute) * (attr_count)))
        return -1;

    scim_bridge_client_kernel_set_preedit_attributes (id, attrs, attr_count);

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;
    return 0;
}


int received_message_set_preedit_cursor_position (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Set preedit cursor position' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    int cur_pos;
    if (scim_bridge_messenger_read_input (except, messenger_id, &cur_pos, sizeof (cur_pos)))
        return -1;

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    scim_bridge_client_kernel_set_preedit_cursor_position (id, cur_pos);
    return 0;
}


int received_message_update_preedit (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Update preedit' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    scim_bridge_client_kernel_update_preedit (id);

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return 0;
}


int received_message_forward_keyevent (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'forward keyevent' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    ScimBridgeKeyEvent keyevent;
    if (scim_bridge_messenger_read_input (except, messenger_id, &keyevent, sizeof (ScimBridgeKeyEvent)))
        return -1;

    scim_bridge_client_kernel_forward_keyevent (id, &keyevent);

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return 0;
}


int received_message_beep (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'beep' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    scim_bridge_client_kernel_beep (id);

    return 0;
}


int received_message_get_surrounding_string (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Get surrounding text' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    size_t buffer_length;
    if (scim_bridge_messenger_read_input (except, messenger_id, &buffer_length, sizeof (buffer_length)))
        return -1;

    ucs4_t wstr[buffer_length + 1];
    wstr[0] = L'\0';

    size_t wstr_len;
    int cursor_position;
    scim_bridge_client_kernel_get_surrounding_string (id, wstr, buffer_length, &wstr_len, &cursor_position);

    if (scim_bridge_messenger_write_input (except, messenger_id, &wstr_len, sizeof (wstr_len)))
        return -1;

    if (scim_bridge_messenger_write_input (except, messenger_id, wstr, sizeof (ucs4_t) * (wstr_len + 1)))
        return -1;

    if (scim_bridge_messenger_write_input (except, messenger_id, &cursor_position, sizeof (cursor_position)))
        return -1;

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return 0;
}


int received_message_delete_surrounding_string (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'Delete surrounding text' message");

    ScimBridgeIMContextID id;
    if (scim_bridge_messenger_read_input (except, messenger_id, &id, sizeof (id)))
        return -1;

    size_t offset;
    if (scim_bridge_messenger_read_input (except, messenger_id, &offset, sizeof (offset)))
        return -1;

    size_t length;
    if (scim_bridge_messenger_read_input (except, messenger_id, &length, sizeof (length)))
        return -1;

    int retval;
    scim_bridge_client_kernel_delete_surrounding_string (id, offset, length, &retval);

    if (scim_bridge_messenger_write_input (except, messenger_id, &retval, sizeof (retval)))
        return -1;

    if (scim_bridge_messenger_close_input (except, messenger_id))
        return -1;

    return 0;
}


/* Remote functions */
int scim_bridge_client_call_alloc_imcontext (ScimBridgeException *except, ScimBridgeIMContextID id, ScimBridgeIMContextID *opponent_id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Sending 'alloc imcontext' message...");

    if (scim_bridge_messenger_open_output (except, messenger_id))
        return -1;

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_ALLOC_IMCONTEXT;
    if (scim_bridge_messenger_write_output (except, messenger_id, &code, sizeof (code)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &id, sizeof (id)))
        return -1;

    if (scim_bridge_messenger_read_output (except, messenger_id, opponent_id, sizeof (*opponent_id)))
        return -1;

    if (scim_bridge_messenger_close_output (except, messenger_id))
        return -1;

    return 0;
}


int scim_bridge_client_call_free_imcontext (ScimBridgeException *except, ScimBridgeIMContextID opponent_id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Sending 'free imcontext' message...");

    if (scim_bridge_messenger_open_output (except, messenger_id))
        return -1;

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_FREE_IMCONTEXT;
    if (scim_bridge_messenger_write_output (except, messenger_id, &code, sizeof (code)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &opponent_id, sizeof (opponent_id)))
        return -1;

    if (scim_bridge_messenger_close_output (except, messenger_id))
        return -1;

    return 0;
}


int scim_bridge_client_call_reset_imcontext (ScimBridgeException *except, ScimBridgeIMContextID opponent_id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Sending 'reset imcontext' message...");

    if (scim_bridge_messenger_open_output (except, messenger_id))
        return -1;

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_RESET_IMCONTEXT;
    if (scim_bridge_messenger_write_output (except, messenger_id, &code, sizeof (code)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &opponent_id, sizeof (opponent_id)))
        return -1;

    if (scim_bridge_messenger_close_output (except, messenger_id))
        return -1;

    return 0;
}


int scim_bridge_client_call_cursor_location_changed (ScimBridgeException *except, ScimBridgeIMContextID opponent_id, int cursor_x, int cursor_y)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Sending 'cursor location changed' message");

    if (scim_bridge_messenger_open_output (except, messenger_id))
        return -1;

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_CURSOR_LOCATION_CHANGED;
    if (scim_bridge_messenger_write_output (except, messenger_id, &code, sizeof (code)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &opponent_id, sizeof (opponent_id)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &cursor_x, sizeof (cursor_x)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &cursor_y, sizeof (cursor_y)))
        return -1;

    if (scim_bridge_messenger_close_output (except, messenger_id))
        return -1;

    return 0;
}


int scim_bridge_client_call_focus_changed (ScimBridgeException *except, ScimBridgeIMContextID opponent_id, int focus_in)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Sending 'focus changed' message...");

    if (scim_bridge_messenger_open_output (except, messenger_id))
        return -1;

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_FOCUS_CHANGED;
    if (scim_bridge_messenger_write_output (except, messenger_id, &code, sizeof (code)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &opponent_id, sizeof (opponent_id)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &focus_in, sizeof (focus_in)))
        return -1;

    if (scim_bridge_messenger_close_output (except, messenger_id))
        return -1;

    return 0;
}


int scim_bridge_client_call_keyevent_occured (ScimBridgeException *except, ScimBridgeIMContextID opponent_id, const ScimBridgeKeyEvent *keyevent, int *consumed)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT | SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Sending 'keyevent occured' message...");

    if (scim_bridge_messenger_open_output (except, messenger_id))
        return -1;

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_KEYEVENT_OCCURED;
    if (scim_bridge_messenger_write_output (except, messenger_id, &code, sizeof (code)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, &opponent_id, sizeof (opponent_id)))
        return -1;

    if (scim_bridge_messenger_write_output (except, messenger_id, keyevent, sizeof (ScimBridgeKeyEvent)))
        return -1;

    if (scim_bridge_messenger_read_output (except, messenger_id, consumed, sizeof (*consumed)))
        return -1;

    if (scim_bridge_messenger_close_output (except, messenger_id))
        return -1;

    return 0;
}
