#include <assert.h>
#include <errno.h>
#include <poll.h>
#include <pthread.h>
#include <signal.h>
#include <string.h>

#include <sys/socket.h>
#include <sys/time.h>

#include <list>
#include <vector>
#include <sstream>

#include "scim-bridge-agent.h"
#include "scim-bridge-agent-client-listener.h"
#include "scim-bridge-agent-exception.h"
#include "scim-bridge-agent-imcontext.h"
#include "scim-bridge-agent-kernel.h"
#include "scim-bridge-environment.h"
#include "scim-bridge-imcontext-common.h"
#include "scim-bridge-keyevent.h"
#include "scim-bridge-output.h"

#include "scim-bridge-agent-client-peer.h"

using std::find;
using std::vector;

using namespace scim;

using std::list;
using std::stringstream;

/* Macros */
#define _(String) (String)
#define N_(String) (String)

#ifndef SCIM_VERSION
#define SCIM_VERSION "<Unknown>"
#endif

#ifndef SCIM_KEYBOARD_ICON_FILE
#define SCIM_KEYBOARD_ICON_FILE (SCIM_ICONDIR "/keyboard.png")
#endif

static const bool OFF = false;
static const bool ON = true;

static const unsigned int LOCKFILE_UPDATE_INTERVAL = 60 * 60;

static const int SIGEXIT = SIGRTMIN + 5;

/* Class definition */
enum PanelListenerStatus
{
    STATUS_RUNNING,
    STATUS_SHUTDOWN_TRIGGERED,
    STATUS_SHUTDOWNED
};

class ScimBridgeAgentKernelImpl: public ScimBridgeAgentKernel, public ScimBridgeAgentClientListener, public ScimBridgeAgentPanelListener
{

    friend ScimBridgeAgentKernel *ScimBridgeAgentKernel::create (bool exit_with_no_client_value) throw (ScimBridgeAgentException);

    public:

        ~ScimBridgeAgentKernelImpl ();

        void connect (const int input_fd, const int output_fd) throw (ScimBridgeAgentException);

        const scim::KeyboardLayout &get_keyboard_layout ();

        ScimBridgeAgentIMContext *alloc_imcontext (ScimBridgeAgentClientPeer &client_peer);
        void free_imcontext (ScimBridgeAgentIMContext &ic);
        void reset_imcontext (ScimBridgeAgentIMContext &ic);
        void focus_changed (ScimBridgeAgentIMContext &ic, bool focus_in);
        bool keyevent_occured (ScimBridgeAgentIMContext &ic, const scim::KeyEvent &keyevent);
        void cursor_location_changed (ScimBridgeAgentIMContext &ic, int cursor_x, int cursor_y);

        void connection_closed (ScimBridgeAgentClientPeer &client_peer);

        int get_scim_panel_fd ();
        void scim_panel_handler (int condition);
        void scim_panel_shutdowned ();
        bool is_scim_panel_shutdown_triggered ();

        void agent_update_lockfile ();

    private:

        pthread_t panel_listener_thread;

        int scim_panel_fd;
        PanelListenerStatus panel_listener_status;

        const bool exit_with_no_client;

        pthread_mutex_t mutex;

        std::list<ScimBridgeAgentClientPeer*> client_peer_list;

        /* Variables for SCIM */
        bool scim_initialized;

        scim::KeyboardLayout scim_keyboard_layout;
        int scim_valid_key_mask;

        scim::FrontEndHotkeyMatcher scim_frontend_hotkey_matcher;
        scim::IMEngineHotkeyMatcher scim_imengine_hotkey_matcher;

        scim::String scim_language;
        int scim_imengine_count;

        scim::ConfigModule *scim_config_module;
        scim::ConfigPointer scim_config;
        scim::BackEndPointer scim_backend;

        scim::IMEngineFactoryPointer scim_fallback_factory;
        scim::IMEngineInstancePointer scim_fallback_instance;

        scim::PanelClient scim_panel_client;

        /* Pure private functions */
        ScimBridgeAgentKernelImpl (bool exit_with_no_client_value) throw (ScimBridgeAgentException);

        void initialize_scim ();
        void finalize_scim ();
        void open_imengine_by_factory (ScimBridgeAgentIMContext *ic, scim::IMEngineFactoryPointer factory);
        void open_previous_imengine (ScimBridgeAgentIMContext *ic);
        void open_next_imengine (ScimBridgeAgentIMContext *ic);
        void open_imengine_by_factory_uuid (ScimBridgeAgentIMContext *ic, const scim::String &uuid);
        void attach_imengine (scim::IMEngineInstancePointer new_imengine);

        /* Protected functions */
        void add_imcontext (ScimBridgeIMContext &imcontext);
        void remove_imcontext (ScimBridgeIMContext &imcontext);
        ScimBridgeAgentIMContext *find_imcontext (ScimBridgeIMContextID id);

        /* Slot functions */
        void slot_show_preedit_string (scim::IMEngineInstanceBase *imengine);
        void slot_show_aux_string (scim::IMEngineInstanceBase *imengine);
        void slot_show_lookup_table (scim::IMEngineInstanceBase *imengine);

        void slot_hide_preedit_string (scim::IMEngineInstanceBase *imengine);
        void slot_hide_aux_string (scim::IMEngineInstanceBase *imengine);
        void slot_hide_lookup_table (scim::IMEngineInstanceBase *imengine);

        void slot_update_preedit_caret (scim::IMEngineInstanceBase *imengine, int caret);
        void slot_update_preedit_string (scim::IMEngineInstanceBase *imengine, const scim::WideString &str, const scim::AttributeList &attrs);
        void slot_update_aux_string (scim::IMEngineInstanceBase *imengine, const scim::WideString &str, const scim::AttributeList &attrs);
        void slot_commit_string (scim::IMEngineInstanceBase *imengine, const scim::WideString &str);
        void slot_forward_key_event (scim::IMEngineInstanceBase *imengine, const scim::KeyEvent &key);
        void slot_update_lookup_table (scim::IMEngineInstanceBase *imengine, const scim::LookupTable &table);

        void slot_register_properties (scim::IMEngineInstanceBase *imengine, const scim::PropertyList &properties);
        void slot_update_property (scim::IMEngineInstanceBase *imengine, const scim::Property &property);
        void slot_beep (scim::IMEngineInstanceBase *imengine);
        void slot_start_helper (scim::IMEngineInstanceBase *imengine, const scim::String &helper_uuid);
        void slot_stop_helper (scim::IMEngineInstanceBase *imengine, const scim::String &helper_uuid);
        void slot_send_helper_event (scim::IMEngineInstanceBase *imengine, const scim::String &helper_uuid, const scim::Transaction &trans);
        bool slot_get_surrounding_text (scim::IMEngineInstanceBase *imengine, scim::WideString &text, int &cursor, int maxlen_before, int maxlen_after);
        bool slot_delete_surrounding_text (scim::IMEngineInstanceBase *imengine, int offset, int len);

        void reload_config_callback (const scim::ConfigPointer &config);
        void fallback_commit_string_cb (scim::IMEngineInstanceBase *si, const scim::WideString &wstr);

        /* Panel related callbacks */
        void panel_slot_reload_config (ScimBridgeIMContextID id);
        void panel_slot_exit (ScimBridgeIMContextID id);
        void panel_slot_update_lookup_table_page_size (ScimBridgeIMContextID id, int page_size);
        void panel_slot_lookup_table_page_up (ScimBridgeIMContextID id);
        void panel_slot_lookup_table_page_down (ScimBridgeIMContextID id);
        void panel_slot_trigger_property (ScimBridgeIMContextID id, const scim::String &property);
        void panel_slot_process_helper_event (ScimBridgeIMContextID id, const scim::String &target_uuid, const scim::String &helper_uuid, const scim::Transaction &trans);
        void panel_slot_move_preedit_caret (ScimBridgeIMContextID id, int caret_pos);
        void panel_slot_select_candidate (ScimBridgeIMContextID id, int cand_index);
        void panel_slot_process_key_event (ScimBridgeIMContextID id, const scim::KeyEvent &keyevent);
        void panel_slot_commit_string (ScimBridgeIMContextID id, const scim::WideString &wstr);
        void panel_slot_forward_key_event (ScimBridgeIMContextID id, const scim::KeyEvent &keyevent);
        void panel_slot_request_help (ScimBridgeIMContextID id);
        void panel_slot_request_factory_menu (ScimBridgeIMContextID id);
        void panel_slot_change_factory (ScimBridgeIMContextID id, const scim::String &uuid);

        /* Panel related requestions */
        void panel_req_focus_in (ScimBridgeAgentIMContext *ic);
        void panel_req_update_screen (ScimBridgeAgentIMContext *ic);
        void panel_req_update_factory_info (ScimBridgeAgentIMContext *ic);
        void panel_req_update_spot_location (ScimBridgeAgentIMContext *ic);
        void panel_req_show_help (ScimBridgeAgentIMContext *ic);
        void panel_req_show_factory_menu (ScimBridgeAgentIMContext *ic);

        /* Utility functions */
        bool filter_hotkeys (ScimBridgeAgentIMContext *ic, const scim::KeyEvent &keyevent);
        void set_imcontext_status (ScimBridgeAgentIMContext *ic, bool on);

        /* Panel iochannel handler */
        bool initialize_panel ();
        void finalize_panel ();
};

/* Heleper functions */
static bool check_socket_frontend ()
{
    SocketAddress address;
    SocketClient client;

    uint32 magic;

    address.set_address (scim_get_default_socket_frontend_address ());

    if (!client.connect (address))
        return false;

    if (!scim_socket_open_connection (magic, "ConnectionTester", "SocketFrontEnd", client, 1000)) {
        return false;
    } else {
        return true;
    }
}


static void sig_exit (int signum)
{
}


static void *run_panel_listener (void *arg)
{
    ScimBridgeAgentPanelListener *kernel_ptr = static_cast<ScimBridgeAgentPanelListener*> (arg);

    struct sigaction sigact;
    sigact.sa_handler = sig_exit;
    sigact.sa_flags = 0;
    if (sigaction (SIGEXIT, &sigact, NULL)) {
        scim_bridge_perrorln ("Cannot setup panel listener exit hook: %s", strerror (errno));
        kernel_ptr->scim_panel_shutdowned ();
        return NULL;
    }

    struct timeval lockfile_last_update_time;
    gettimeofday (&lockfile_last_update_time, NULL);

    while (!kernel_ptr->is_scim_panel_shutdown_triggered ()) {
        pollfd poll_fd;
        poll_fd.fd = kernel_ptr->get_scim_panel_fd ();
        poll_fd.events = POLLIN | POLLERR | POLLHUP | POLLNVAL;
        poll_fd.revents = 0;

        const int poll_retval = poll (&poll_fd, 1, LOCKFILE_UPDATE_INTERVAL * 1000);
        if (kernel_ptr->is_scim_panel_shutdown_triggered ()) {
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "Panel listener exiting...");
            break;
        } else {
            struct timeval current_time;
            gettimeofday (&current_time, NULL);
            if (current_time.tv_sec - lockfile_last_update_time.tv_sec > LOCKFILE_UPDATE_INTERVAL) {
                kernel_ptr->agent_update_lockfile ();
                lockfile_last_update_time = current_time;
            }

            if (poll_retval > 0) {
                kernel_ptr->scim_panel_handler (poll_fd.revents);
            } else if (poll_retval < 0 && errno != EINTR) {
                scim_bridge_perrorln ("Panel listener failed: %s", strerror (errno));
                kernel_ptr->scim_panel_handler (POLLNVAL);
                usleep (100);
            }
        }
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM | SCIM_BRIDGE_DEBUG_AGENT, 5, "Panel listener thread exited");
    kernel_ptr->scim_panel_shutdowned ();
    return NULL;
}


ScimBridgeAgentKernel *ScimBridgeAgentKernel::create (bool exit_with_no_client_value) throw (ScimBridgeAgentException)
{
    return new ScimBridgeAgentKernelImpl (exit_with_no_client_value);
}


/* Implementations */
ScimBridgeAgentKernelImpl::ScimBridgeAgentKernelImpl (bool exit_with_no_client_value) throw (ScimBridgeAgentException):
panel_listener_status (STATUS_SHUTDOWNED), exit_with_no_client (exit_with_no_client_value),
scim_keyboard_layout (SCIM_KEYBOARD_Default), scim_valid_key_mask (SCIM_KEY_AllMasks), scim_imengine_count (0),
scim_config_module (NULL)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Initializing the agent kernel...");

    pthread_mutex_init (&mutex, 0);

    initialize_scim ();

    panel_listener_status = STATUS_RUNNING;
    if (pthread_create (&panel_listener_thread, NULL, run_panel_listener, static_cast<ScimBridgeAgentPanelListener*> (this))) {
        panel_listener_status = STATUS_SHUTDOWNED;
        throw ScimBridgeAgentException (String ("Cannot invoke panel listener: ") + String (strerror (errno)));
    }
}


ScimBridgeAgentKernelImpl::~ScimBridgeAgentKernelImpl ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Finalizing the agent kernel...");

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 1, "The agent kernel is waiting for panel listener thread finishing...");

    pthread_mutex_lock (&mutex);
    panel_listener_status = STATUS_SHUTDOWN_TRIGGERED;
    pthread_mutex_unlock (&mutex);

    for (list<ScimBridgeAgentClientPeer*>::iterator i = client_peer_list.begin (); i != client_peer_list.end (); ++i) {
        ScimBridgeAgentClientPeer *client_peer = *i;

        delete client_peer;
    }

    while (panel_listener_status != STATUS_SHUTDOWNED) {
        pthread_kill (panel_listener_thread, SIGEXIT);
        usleep (50);
    }
    pthread_join (panel_listener_thread, NULL);

    finalize_scim ();

    pthread_mutex_destroy (&mutex);

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "The agent kernel is now down");
}


/* Panel watcher related codes */
void ScimBridgeAgentKernelImpl::scim_panel_shutdowned ()
{
    panel_listener_status = STATUS_SHUTDOWNED;
    pthread_exit (NULL);
}


bool ScimBridgeAgentKernelImpl::is_scim_panel_shutdown_triggered ()
{
    return panel_listener_status == STATUS_SHUTDOWN_TRIGGERED;
}


int ScimBridgeAgentKernelImpl::get_scim_panel_fd ()
{
    return scim_panel_fd;
}


void ScimBridgeAgentKernelImpl::agent_update_lockfile ()
{
    update_lockfile ();
}


void ScimBridgeAgentKernelImpl::scim_panel_handler (int condition)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM | SCIM_BRIDGE_DEBUG_AGENT, 4, "scim_panel_handler...");
    if (condition & (POLLERR | POLLHUP | POLLNVAL)) {
        finalize_panel ();
        initialize_panel ();
    } else if (condition & POLLIN) {
        if (!scim_panel_client.filter_event ()) {
            finalize_panel ();
            initialize_panel ();
        }
    }
}


/* Scim related codes */
void ScimBridgeAgentKernelImpl::connect (const int input_fd, const int output_fd) throw (ScimBridgeAgentException)
{
    ScimBridgeAgentClientPeer *client_peer = ScimBridgeAgentClientPeer::create (input_fd, output_fd, this);
    client_peer_list.push_back (client_peer);
}


void ScimBridgeAgentKernelImpl::attach_imengine (IMEngineInstancePointer new_imengine)
{
    new_imengine->signal_connect_show_preedit_string (slot (this, &ScimBridgeAgentKernelImpl::slot_show_preedit_string));
    new_imengine->signal_connect_show_aux_string (slot (this, &ScimBridgeAgentKernelImpl::slot_show_aux_string));
    new_imengine->signal_connect_show_lookup_table (slot (this, &ScimBridgeAgentKernelImpl::slot_show_lookup_table));

    new_imengine->signal_connect_hide_preedit_string (slot (this, &ScimBridgeAgentKernelImpl::slot_hide_preedit_string));
    new_imengine->signal_connect_hide_aux_string (slot (this, &ScimBridgeAgentKernelImpl::slot_hide_aux_string));
    new_imengine->signal_connect_hide_lookup_table (slot (this, &ScimBridgeAgentKernelImpl::slot_hide_lookup_table));

    new_imengine->signal_connect_update_preedit_caret (slot (this, &ScimBridgeAgentKernelImpl::slot_update_preedit_caret));
    new_imengine->signal_connect_update_preedit_string (slot (this, &ScimBridgeAgentKernelImpl::slot_update_preedit_string));
    new_imengine->signal_connect_update_aux_string (slot (this, &ScimBridgeAgentKernelImpl::slot_update_aux_string));
    new_imengine->signal_connect_update_lookup_table (slot (this, &ScimBridgeAgentKernelImpl::slot_update_lookup_table));

    new_imengine->signal_connect_commit_string (slot (this, &ScimBridgeAgentKernelImpl::slot_commit_string));

    new_imengine->signal_connect_forward_key_event (slot (this, &ScimBridgeAgentKernelImpl::slot_forward_key_event));

    new_imengine->signal_connect_register_properties (slot (this, &ScimBridgeAgentKernelImpl::slot_register_properties));

    new_imengine->signal_connect_update_property (slot (this, &ScimBridgeAgentKernelImpl::slot_update_property));

    new_imengine->signal_connect_beep (slot (this, &ScimBridgeAgentKernelImpl::slot_beep));

    new_imengine->signal_connect_start_helper (slot (this, &ScimBridgeAgentKernelImpl::slot_start_helper));

    new_imengine->signal_connect_stop_helper (slot (this, &ScimBridgeAgentKernelImpl::slot_stop_helper));

    new_imengine->signal_connect_send_helper_event (slot (this, &ScimBridgeAgentKernelImpl::slot_send_helper_event));

    new_imengine->signal_connect_get_surrounding_text (slot (this, &ScimBridgeAgentKernelImpl::slot_get_surrounding_text));

    new_imengine->signal_connect_delete_surrounding_text (slot (this, &ScimBridgeAgentKernelImpl::slot_delete_surrounding_text));
}


void ScimBridgeAgentKernelImpl::initialize_scim ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "Initializing scim...");

    // Get system language.
    scim_language = scim_get_locale_language (scim_get_current_locale ());

    // Get modules list
    vector<String> imengine_vector;
    vector<String> config_vector;
    scim_get_imengine_module_list (imengine_vector);
    scim_get_config_module_list (config_vector);

    // Use socket if avaiable.
    bool use_socket = true;
    if (find (imengine_vector.begin (), imengine_vector.end (), "socket") == imengine_vector.end () ||
        find (config_vector.begin (), config_vector.end (), "socket") == config_vector.end ())
        use_socket = false;

    //Use simple config module as default if available.
    String config_module_name;
    if (config_vector.size () > 0) {
        config_module_name = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));
        if (find (config_vector.begin (), config_vector.end (), config_module_name) == config_vector.end ())
            config_module_name = config_vector [0];
    } else {
        config_module_name = "dummy";
    }

    // Use user defined immodules if available
    bool use_user_imegine = false;
    vector <String> user_imengine_vector;
    const char *imengine_list_str = getenv ("BRIDGE_IM_SCIM_IMENGINE_MODULES");
    if (imengine_list_str != NULL) {
        vector <String> spec_imengine_vector;
        scim_split_string_list (spec_imengine_vector, imengine_list_str, ',');

        user_imengine_vector.clear ();

        for (size_t i = 0; i < spec_imengine_vector.size (); ++i) {
            if (find (imengine_vector.begin (), imengine_vector.end (), imengine_vector [i]) != imengine_vector.end ()) {
                user_imengine_vector.push_back (spec_imengine_vector [i]);
                use_user_imegine = true;
            }
        }
    }

    // If you try to use the socket feature manually,
    // then let you do it by yourself.
    if (config_module_name == "socket" ||
        find (user_imengine_vector.begin (), user_imengine_vector.end (), "socket") != user_imengine_vector.end ())
        use_socket = false;

    // Try to start a SCIM SocketFrontEnd process if necessary.
    if (scim_get_default_socket_frontend_address () != scim_get_default_socket_imengine_address () &&
        scim_get_default_socket_frontend_address () != scim_get_default_socket_config_address ())
        use_socket = false;

    if (use_socket) {
        // If no Socket FrontEnd is running, then launch one.
        // And set manual to false.
        if (!check_socket_frontend ()) {
            scim_bridge_perrorln ("Launching a SCIM daemon with Socket FrontEnd...");
            char *new_argv [] = { "--no-stay", 0 };
            scim_launch (true,
                config_module_name,
                (!user_imengine_vector.empty () ? scim_combine_string_list (user_imengine_vector, ',') : "all"),
                "socket",
                new_argv);
            use_user_imegine = false;
        }

        // If there is one Socket FrontEnd running and it's not manual mode,
        // then just use this Socket Frontend.
        if (!use_user_imegine) {
            for (int i = 0; i < 100; ++i) {
                if (check_socket_frontend ()) {
                    config_module_name = "socket";
                    user_imengine_vector.clear ();
                    user_imengine_vector.push_back ("socket");
                    break;
                }
                scim_usleep (100000);
            }
        }
    }

    if (config_module_name != "dummy") {
        //load config module
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 2, "Loading Config module...: %s", config_module_name.c_str ());
        scim_config_module = new ConfigModule (config_module_name);

        //create config instance
        if (scim_config_module != NULL && scim_config_module->valid ())
            scim_config = scim_config_module->create_config ();
    }

    if (scim_config.null ()) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 2, "Config module cannot be loaded, using dummy Config.");

        if (scim_config_module) delete scim_config_module;
        scim_config_module = NULL;

        scim_config = new DummyConfig ();
        config_module_name = "dummy";
    }

    reload_config_callback (scim_config);
    scim_config->signal_connect_reload (slot (this, &ScimBridgeAgentKernelImpl::reload_config_callback));

    // create backend
    scim_backend = new CommonBackEnd (scim_config, user_imengine_vector.empty () ? user_imengine_vector : imengine_vector);

    if (scim_backend.null ()) {
        scim_bridge_perrorln ("Cannot create BackEnd Object!");
    } else {
        scim_fallback_factory = scim_backend->get_factory (SCIM_COMPOSE_KEY_FACTORY_UUID);
    }

    if (scim_fallback_factory.null ())
        scim_fallback_factory = new DummyIMEngineFactory ();

    scim_fallback_instance = scim_fallback_factory->create_instance (String ("UTF-8"), 0);
    scim_fallback_instance->signal_connect_commit_string (slot (this, &ScimBridgeAgentKernelImpl::fallback_commit_string_cb));

    // Attach Panel Client signal.
    scim_panel_client.signal_connect_reload_config (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_reload_config));
    scim_panel_client.signal_connect_exit (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_exit));
    scim_panel_client.signal_connect_update_lookup_table_page_size (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_update_lookup_table_page_size));
    scim_panel_client.signal_connect_lookup_table_page_up (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_lookup_table_page_up));
    scim_panel_client.signal_connect_lookup_table_page_down (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_lookup_table_page_down));
    scim_panel_client.signal_connect_trigger_property (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_trigger_property));
    scim_panel_client.signal_connect_process_helper_event (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_process_helper_event));
    scim_panel_client.signal_connect_move_preedit_caret (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_move_preedit_caret));
    scim_panel_client.signal_connect_select_candidate (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_select_candidate));
    scim_panel_client.signal_connect_process_key_event (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_process_key_event));
    scim_panel_client.signal_connect_commit_string (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_commit_string));
    scim_panel_client.signal_connect_forward_key_event (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_forward_key_event));
    scim_panel_client.signal_connect_request_help (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_request_help));
    scim_panel_client.signal_connect_request_factory_menu (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_request_factory_menu));
    scim_panel_client.signal_connect_change_factory (slot (this, &ScimBridgeAgentKernelImpl::panel_slot_change_factory));

    if (!initialize_panel ()) {
        scim_bridge_perrorln ("Cannot connect to Panel!");
    }

    scim_initialized = true;

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Initialize scim, done!");
}


void ScimBridgeAgentKernelImpl::finalize_scim ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "Finalizing scim...");

    ScimBridgeAgentIMContext::release_shared_imengine ();

    scim_fallback_instance.reset ();
    scim_fallback_factory.reset ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, " Releasing BackEnd...");
    scim_backend.reset ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, " Releasing Config...");
    scim_config.reset ();

    if (scim_config_module) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, " Deleting config module...");
        delete scim_config_module;
        scim_config_module = NULL;
    }

    scim_initialized = false;
    finalize_panel ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Finalize scim, done!");
}


void ScimBridgeAgentKernelImpl::open_imengine_by_factory (ScimBridgeAgentIMContext *ic, IMEngineFactoryPointer factory)
{
    if (!factory.null ()) {
        set_imcontext_status (ic, OFF);
        ic->reset_preedit ();
        IMEngineInstancePointer new_imengine = factory->create_instance ("UTF-8", ic->get_imengine ()->get_id ());
        ic->set_imengine (new_imengine);
        attach_imengine (new_imengine);
        scim_backend->set_default_factory (scim_language, factory->get_uuid ());
        scim_panel_client.register_input_context (ic->parent.id, factory->get_uuid ());
        set_imcontext_status (ic, ON);
    }
}


void ScimBridgeAgentKernelImpl::open_next_imengine (ScimBridgeAgentIMContext *ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "open_next_factory context=%d", ic->parent.id);
    open_imengine_by_factory (ic, scim_backend->get_next_factory ("", "UTF-8", ic->get_factory_uuid ()));
}


void ScimBridgeAgentKernelImpl::open_previous_imengine (ScimBridgeAgentIMContext *ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "open_previous_factory context=%d", ic->parent.id);
    open_imengine_by_factory (ic, scim_backend->get_previous_factory ("", "UTF-8", ic->get_imengine ()->get_factory_uuid ()));
}


void ScimBridgeAgentKernelImpl::open_imengine_by_factory_uuid (ScimBridgeAgentIMContext *ic, const String &uuid)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "open_imengine_by_factory_uuid context=%d factory-uuid=%s", ic->parent.id, uuid.c_str ());

    // The same input method is selected, just turn on the IC.
    if (ic->get_imengine ()->get_factory_uuid () == uuid) {
        set_imcontext_status (ic, ON);
        return;
    }

    IMEngineFactoryPointer factory = scim_backend->get_factory (uuid);
    if (!uuid.empty () && !factory.null ()) {
        open_imengine_by_factory (ic, factory);
    } else {
        set_imcontext_status (ic, OFF);
    }
}


/* Slot functions */
void ScimBridgeAgentKernelImpl::slot_show_preedit_string (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_show_preedit_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        if (ic->is_on_the_spot_enabled ()) {
            if (!ic->get_preedit_string ().empty ()) {
                ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
                client_peer->set_preedit_shown (*ic, true);
                client_peer->set_preedit_cursor_position (*ic, ic->get_preedit_caret ());
                client_peer->set_preedit_string (*ic, ic->get_preedit_string ());
                client_peer->set_preedit_attributes (*ic, ic->get_preedit_attribute_list ());
                client_peer->update_preedit (*ic);
            }
        } else {
            scim_panel_client.show_preedit_string (ic->parent.id);
        }
    }
}


void ScimBridgeAgentKernelImpl::slot_show_aux_string (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_show_aux_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.show_aux_string (ic->parent.id);
}


void ScimBridgeAgentKernelImpl::slot_show_lookup_table (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_show_lookup_table...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.show_lookup_table (ic->parent.id);
}


void ScimBridgeAgentKernelImpl::slot_hide_preedit_string (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_hide_preedit_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        bool emit = false;
        if (!ic->get_preedit_string ().empty ()) {
            ic->reset_preedit ();
            emit = true;
        }
        if (ic->is_on_the_spot_enabled ()) {
            if (emit) {
                ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
                client_peer->set_preedit_shown (*ic, false);
                client_peer->update_preedit (*ic);
            }
        } else {
            scim_panel_client.hide_preedit_string (ic->parent.id);
        }
    }
}


void ScimBridgeAgentKernelImpl::slot_hide_aux_string (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_hide_aux_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.hide_aux_string (ic->parent.id);
}


void ScimBridgeAgentKernelImpl::slot_hide_lookup_table (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_hide_lookup_table...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.hide_lookup_table (ic->parent.id);
}


void ScimBridgeAgentKernelImpl::slot_update_preedit_caret (IMEngineInstanceBase *imengine, int caret)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_update_preedit_caret...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic && ic->get_preedit_caret () != caret) {
        ic->set_preedit_caret (caret);
        if (ic->is_on_the_spot_enabled ()) {
            ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
            client_peer->set_preedit_shown (*ic, true);
            client_peer->set_preedit_cursor_position (*ic, ic->get_preedit_caret ());
            client_peer->update_preedit (*ic);
        } else {
            scim_panel_client.update_preedit_caret (ic->parent.id, caret);
        }
    }
}


void ScimBridgeAgentKernelImpl::slot_update_preedit_string (IMEngineInstanceBase *imengine, const WideString &str, const AttributeList & attrs)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_update_preedit_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic && (ic->get_preedit_string () != str || !str.empty ())) {
        ic->set_preedit_string (str);
        ic->set_preedit_attribute_list (attrs);
        if (ic->is_on_the_spot_enabled ()) {
            ic->set_preedit_caret (str.length ());

            ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
            client_peer->set_preedit_shown (*ic, true);
            client_peer->set_preedit_cursor_position (*ic, ic->get_preedit_caret ());
            client_peer->set_preedit_string (*ic, ic->get_preedit_string ());
            client_peer->set_preedit_attributes (*ic, ic->get_preedit_attribute_list ());
            client_peer->update_preedit (*ic);
        } else {
            scim_panel_client.update_preedit_string (ic->parent.id, str, attrs);
        }
    }
}


void ScimBridgeAgentKernelImpl::slot_update_aux_string (IMEngineInstanceBase *imengine, const WideString &str, const AttributeList &attrs)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_update_aux_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.update_aux_string (ic->get_id (), str, attrs);
}


void ScimBridgeAgentKernelImpl::slot_commit_string (IMEngineInstanceBase *imengine, const WideString &str)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_commit_string...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        client_peer->set_preedit_string (*ic, str);
        client_peer->commit (*ic);
    }
}


void ScimBridgeAgentKernelImpl::slot_forward_key_event (IMEngineInstanceBase *imengine, const KeyEvent &keyevent)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_forward_key_event...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        if (!scim_fallback_instance->process_key_event (keyevent)) {
            ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
            client_peer->forward_keyevent (*ic, keyevent);
        }
    }
}


void ScimBridgeAgentKernelImpl::slot_update_lookup_table (IMEngineInstanceBase *imengine, const LookupTable & table)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_update_lookup_table...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.update_lookup_table (ic->get_id (), table);
}


void ScimBridgeAgentKernelImpl::slot_register_properties (IMEngineInstanceBase *imengine, const PropertyList & properties)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_register_properties...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.register_properties (ic->get_id (), properties);
}


void ScimBridgeAgentKernelImpl::slot_update_property (IMEngineInstanceBase *imengine, const Property & property)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_update_property...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) scim_panel_client.update_property (ic->get_id (), property);
}


void ScimBridgeAgentKernelImpl::slot_beep (IMEngineInstanceBase *imengine)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_beep...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        client_peer->beep (*ic);
    }
}


void ScimBridgeAgentKernelImpl::slot_start_helper (IMEngineInstanceBase *imengine, const String &helper_uuid)
{
    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_start_helper helper=%s context=%d ic=%s ic-uuid=%s...",
        helper_uuid.c_str (), ic ? ic->get_id () : -1, ic ? "valid" : "NULL", ic ? ic->get_imengine ()->get_factory_uuid ().c_str () : "");

    if (ic)
        scim_panel_client.start_helper (ic->get_id (), helper_uuid);
}


void ScimBridgeAgentKernelImpl::slot_stop_helper (IMEngineInstanceBase *imengine, const String &helper_uuid)
{
    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_stop_helper helper=%s context=%d ic=%s...",
        helper_uuid.c_str (), ic ? ic->get_id () : -1, ic ? "valid" : "NULL");

    if (ic)
        scim_panel_client.stop_helper (ic->get_id (), helper_uuid);
}


void ScimBridgeAgentKernelImpl::slot_send_helper_event (IMEngineInstanceBase *imengine, const String &helper_uuid, const Transaction &trans)
{
    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_send_helper_event helper=%s context=%d ic=%s ic-uuid=%s...",
        helper_uuid.c_str (), ic ? ic->get_id () : -1, ic ? "valid" : "NULL",  ic ? ic->get_imengine ()->get_factory_uuid ().c_str () : "");

    if (ic)
        scim_panel_client.send_helper_event (ic->get_id (), helper_uuid, trans);
}


bool ScimBridgeAgentKernelImpl::slot_get_surrounding_text (IMEngineInstanceBase *imengine, WideString &text, int &cursor, int maxlen_before, int maxlen_after)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_get_surrounding_text...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        WideString surrounding;
        int cursor_pos;

        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        if (client_peer->get_surrounding_string (*ic, surrounding, maxlen_before + maxlen_after, cursor_pos)) {
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 4, "Surrounding text: %s", surrounding.c_str ());
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 4, "Cursor position: %d", cursor_pos);

            int begin_pos;
            if (maxlen_before < 0) {
                begin_pos = 0;
            } else {
                begin_pos = cursor_pos - maxlen_before;
                if (begin_pos < 0)
                    begin_pos = 0;
            }

            unsigned int end_pos;
            if (maxlen_after < 0) {
                end_pos = surrounding.length ();
            } else {
                end_pos = surrounding.length ();
                if (end_pos > surrounding.length ())
                    end_pos = surrounding.length ();
            }

            text = surrounding.substr (begin_pos, end_pos - begin_pos);
            cursor = cursor_pos - begin_pos;
            return true;
        }
    }

    return false;
}


bool ScimBridgeAgentKernelImpl::slot_delete_surrounding_text (IMEngineInstanceBase *imengine, int offset, int len)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "slot_delete_surrounding_text...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        return client_peer->delete_surrounding_string (*ic, offset, len);
    } else {
        return false;
    }
}


void ScimBridgeAgentKernelImpl::reload_config_callback (const ConfigPointer &config)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "reload_config_callback...");

    scim_frontend_hotkey_matcher.load_hotkeys (config);
    scim_imengine_hotkey_matcher.load_hotkeys (config);

    KeyEvent key;

    scim_string_to_key (key, config->read (String (SCIM_CONFIG_HOTKEYS_FRONTEND_VALID_KEY_MASK), String ("Shift+Control+Alt+Lock")));

    scim_valid_key_mask = (key.mask > 0)? (key.mask):0xFFFF;
    scim_valid_key_mask |= SCIM_KEY_ReleaseMask;

    const bool on_the_spot_default = ScimBridgeAgentIMContext::is_on_the_spot_enabled ();
    ScimBridgeAgentIMContext::set_on_the_spot_enabled (config->read (String (SCIM_CONFIG_FRONTEND_ON_THE_SPOT), on_the_spot_default));

    const bool imengine_shared_default = ScimBridgeAgentIMContext::is_imengine_shared ();
    ScimBridgeAgentIMContext::set_imengine_shared (config->read (String (SCIM_CONFIG_FRONTEND_SHARED_INPUT_METHOD), imengine_shared_default));

    // Get keyboard layout setting
    // Flush the global config first, in order to load the new configs from disk.
    scim_global_config_flush ();

    scim_keyboard_layout = scim_get_default_keyboard_layout ();
}


void ScimBridgeAgentKernelImpl::fallback_commit_string_cb (IMEngineInstanceBase *imengine, const WideString &wstr)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "fallback_commit_string_cb...");

    ScimBridgeAgentIMContext *ic = static_cast<ScimBridgeAgentIMContext*> (imengine->get_frontend_data ());

    if (ic) {
        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        client_peer->set_preedit_string (*ic, wstr);
        client_peer->commit (*ic);
    }
}


/* Panel Slot functions */
void ScimBridgeAgentKernelImpl::panel_slot_reload_config (ScimBridgeIMContextID id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_reload_config...");
    scim_config->reload ();
}


void ScimBridgeAgentKernelImpl::panel_slot_exit (ScimBridgeIMContextID id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_exit...");
    shutdown ();
}


void ScimBridgeAgentKernelImpl::panel_slot_update_lookup_table_page_size (ScimBridgeIMContextID id, int page_size)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_update_lookup_table_page_size context=%d page_size=%d ic=%s", id, page_size, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        ic->update_lookup_table_page_size (page_size);
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_lookup_table_page_up (ScimBridgeIMContextID id)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_lookup_table_page_up context=%d ic=%s", id, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        ic->lookup_table_page_up ();
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_lookup_table_page_down (ScimBridgeIMContextID id)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_lookup_table_page_down context=%d ic=%s", id, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        ic->lookup_table_page_down ();
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_trigger_property (ScimBridgeIMContextID id, const String &property)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_trigger_property context=%d property=%s ic=%s", id, property.c_str (), ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        ic->trigger_property (property);
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_process_helper_event (ScimBridgeIMContextID id, const String &target_uuid, const String &helper_uuid, const Transaction &trans)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    if (!ic) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_process_helper_event context=%d target=%s helper=%s ic=%s ic-uuid=%s",
            id, target_uuid.c_str (), helper_uuid.c_str (), "NULL", "NULL");
    } else {
        const String factory_uuid = ic->get_factory_uuid ();
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "panel_slot_process_helper_event context=%d target=%s helper=%s ic=%s ic-uuid=%s",
            id, target_uuid.c_str (), helper_uuid.c_str (), "valid", factory_uuid.c_str ());
        if (factory_uuid == target_uuid) {
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "call process_helper_event");
            scim_panel_client.prepare (id);
            ic->process_helper_event (helper_uuid, trans);
            scim_panel_client.send ();
        }
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_move_preedit_caret (ScimBridgeIMContextID id, int caret_pos)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_move_preedit_caret context=%s caret=%s ic=%s", id, caret_pos, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        ic->move_preedit_caret (caret_pos);
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_select_candidate (ScimBridgeIMContextID id, int cand_index)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_select_candidate context=%s candidate=%s ic=%s",
        id, cand_index, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        ic->select_candidate (cand_index);
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_process_key_event (ScimBridgeIMContextID id, const KeyEvent &keyevent)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_process_key_event context=%d key=%s ic=%s", id, keyevent.get_key_string ().c_str (), ic ? "valid": "NULL");

    if (ic) {
        scim_panel_client.prepare (id);

        if (!filter_hotkeys (ic, keyevent)) {
            ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
            if (!ic->is_on () || !ic->process_keyevent (keyevent)) {
                if (!scim_fallback_instance->process_key_event (keyevent)) {
                    // Just send it to key_snooper and bypass to client directly (because send_event is set to TRUE).
                    ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
                    client_peer->forward_keyevent (*ic, keyevent);
                }
            }
        }

        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_commit_string (ScimBridgeIMContextID id, const WideString &wstr)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_commit_string context=%d str=%s ic=%s", id, wstr.c_str (), ic ? "valid": "NULL");

    if (ic) {
        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        client_peer->set_preedit_string (*ic, wstr);
        client_peer->commit (*ic);
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_forward_key_event (ScimBridgeIMContextID id, const KeyEvent &keyevent)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_forward_key_event context=%d key=%s ic=%s", id, keyevent.get_key_string ().c_str (), ic ? "valid": "NULL");
    if (ic) {
        // Just send it to key_snooper and bypass to client directly (because send_event is set to TRUE).
        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        client_peer->forward_keyevent (*ic, keyevent);
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_request_help (ScimBridgeIMContextID id)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_request_help context=%d ic=%s", id, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        panel_req_show_help (ic);
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_request_factory_menu (ScimBridgeIMContextID id)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_request_factory_menu context=%d ic=%s", id, ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        panel_req_show_factory_menu (ic);
        scim_panel_client.send ();
    }
}


void ScimBridgeAgentKernelImpl::panel_slot_change_factory (ScimBridgeIMContextID id, const String &uuid)
{
    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (id);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_slot_change_factory context=%d factory=%s ic=%s", id, uuid.c_str (), ic ? "valid": "NULL");
    if (ic) {
        scim_panel_client.prepare (id);
        open_imengine_by_factory_uuid (ic, uuid);
        scim_panel_client.send ();
    }
}


/* Panel Requestion functions */
void ScimBridgeAgentKernelImpl::panel_req_update_screen (ScimBridgeAgentIMContext *ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_req_update_screen");

    scim_panel_client.update_screen (ic->get_id (), scim_bridge_environment_get_screen_number ());
}


void ScimBridgeAgentKernelImpl::panel_req_show_help (ScimBridgeAgentIMContext *ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "panel_req_show_help: ic = %s", ic ? "valid" : "NULL");

    String help = String (_ ("Smart Common Input Method platform")) +
        String (SCIM_VERSION) +
        String (_ ("\n (C) 2002-2005 James Su <suzhe@tsinghua.org.cn>\n\n"));

    if (ic) {
        IMEngineFactoryPointer sf = scim_backend->get_factory (ic->get_factory_uuid ());
        help += utf8_wcstombs (sf->get_name ());
        help += String (_ (":\n\n"));

        help += utf8_wcstombs (sf->get_authors ());
        help += String (_ ("\n\n"));

        help += utf8_wcstombs (sf->get_help ());
        help += String (_ ("\n\n"));

        help += utf8_wcstombs (sf->get_credits ());
    }

    scim_panel_client.show_help (ic->get_id (), help);
}


void ScimBridgeAgentKernelImpl::panel_req_show_factory_menu (ScimBridgeAgentIMContext *ic)
{
    std::vector<IMEngineFactoryPointer> factories;
    std::vector <PanelFactoryInfo> menu;

    scim_backend->get_factories_for_encoding (factories, "UTF-8");

    for (size_t i = 0; i < factories.size (); ++i) {
        menu.push_back (PanelFactoryInfo (
            factories [i]->get_uuid (),
            utf8_wcstombs (factories [i]->get_name ()),
            factories [i]->get_language (),
            factories [i]->get_icon_file ()));
    }

    if (menu.size ())
        scim_panel_client.show_factory_menu (ic->get_id (), menu);
}


void ScimBridgeAgentKernelImpl::panel_req_update_factory_info (ScimBridgeAgentIMContext *ic)
{
    if (ic) {
        PanelFactoryInfo info;
        if (ic->is_on ()) {
            IMEngineFactoryPointer sf = scim_backend->get_factory (ic->get_factory_uuid ());
            info = PanelFactoryInfo (sf->get_uuid (), utf8_wcstombs (sf->get_name ()), sf->get_language (), sf->get_icon_file ());
        } else {
            info = PanelFactoryInfo (String (""), String (_ ("English/Keyboard")), String ("C"), String (SCIM_KEYBOARD_ICON_FILE));
        }
        scim_panel_client.update_factory_info (ic->get_id (), info);
    }
}


void ScimBridgeAgentKernelImpl::panel_req_focus_in (ScimBridgeAgentIMContext *ic)
{
    scim_panel_client.focus_in (ic->get_id (), ic->get_factory_uuid ());
}


void ScimBridgeAgentKernelImpl::panel_req_update_spot_location (ScimBridgeAgentIMContext *ic)
{
    scim_panel_client.update_spot_location (ic->get_id (), ic->get_cursor_x (), ic->get_cursor_y ());
}


/* Miscs */
bool ScimBridgeAgentKernelImpl::filter_hotkeys (ScimBridgeAgentIMContext *ic, const KeyEvent &key)
{
    scim_frontend_hotkey_matcher.push_key_event (key);
    scim_imengine_hotkey_matcher.push_key_event (key);

    FrontEndHotkeyAction hotkey_action = scim_frontend_hotkey_matcher.get_match_result ();

    if (hotkey_action == SCIM_FRONTEND_HOTKEY_TRIGGER) {
        set_imcontext_status (ic, !ic->is_on ());
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_ON) {
        if (!ic->is_on ())
            set_imcontext_status (ic, ON);
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_OFF) {
        if (ic->is_on ())
            set_imcontext_status (ic, OFF);
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_NEXT_FACTORY) {
        open_next_imengine (ic);
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_PREVIOUS_FACTORY) {
        open_previous_imengine (ic);
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_SHOW_FACTORY_MENU) {
        panel_req_show_factory_menu (ic);
        return true;
    } else if (scim_imengine_hotkey_matcher.is_matched ()) {
        String sfid = scim_imengine_hotkey_matcher.get_match_result ();
        open_imengine_by_factory_uuid (ic, sfid);
        return true;
    }

    return false;
}


bool ScimBridgeAgentKernelImpl::initialize_panel ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "panel_initialize..");
    const String display_name = String (scim_bridge_environment_get_display_name ());

    if (scim_panel_client.open_connection (scim_config->get_name (), display_name) >= 0) {
        scim_panel_fd = scim_panel_client.get_connection_number ();

        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, " Panel FD = %d", scim_panel_fd);
        return true;
    } else {
        scim_panel_fd = -1;
        return false;
    }
}


void ScimBridgeAgentKernelImpl::finalize_panel ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Finalizing panel...");

    shutdown (scim_panel_fd, SHUT_RDWR);
    scim_panel_client.close_connection ();
    scim_panel_fd = -1;
}


void ScimBridgeAgentKernelImpl::set_imcontext_status (ScimBridgeAgentIMContext *ic, bool on)
{
    if (ic) {
        if (on) {
            ic->turn_on ();
            panel_req_focus_in (ic);
            panel_req_update_screen (ic);
            panel_req_update_spot_location (ic);
            panel_req_update_factory_info (ic);
            scim_panel_client.turn_on (ic->get_id ());
            scim_panel_client.hide_preedit_string (ic->get_id ());
            scim_panel_client.hide_aux_string (ic->get_id ());
            scim_panel_client.hide_lookup_table (ic->get_id ());
            ic->focus_in ();
        } else {
            ic->turn_off ();
            ic->focus_out ();

            panel_req_update_factory_info (ic);
            scim_panel_client.turn_off (ic->get_id ());
        }

        // Record the IC on/off panel_listener_status
        if (ScimBridgeAgentIMContext::is_imengine_shared ())
            scim_config->write (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), on);

        ScimBridgeAgentClientPeer *client_peer = ic->get_client_peer ();
        if (ic->is_on_the_spot_enabled ()) {
            if (on && !ic->get_preedit_string ().empty ()) client_peer->set_preedit_shown (*ic, true);
            else client_peer->set_preedit_shown (*ic, false);
            client_peer->update_preedit (*ic);
        }
    }
}


/* Message handler */
const KeyboardLayout &ScimBridgeAgentKernelImpl::get_keyboard_layout ()
{
    return scim_keyboard_layout;
}


ScimBridgeAgentIMContext *ScimBridgeAgentKernelImpl::alloc_imcontext (ScimBridgeAgentClientPeer &client_peer)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "alloc_imcontext...");

    if (scim_backend.null ()) return NULL;

    IMEngineInstancePointer imengine = NULL;
    if (ScimBridgeAgentIMContext::is_imengine_shared ()) {
        ScimBridgeAgentIMContext::get_shared_imengine ();
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "Use default instance");
    }

    if (imengine.null ()) {
        IMEngineFactoryPointer factory = scim_backend->get_default_factory (scim_language, "UTF-8");
        if (factory.null ()) return NULL;
        imengine = factory->create_instance ("UTF-8", scim_imengine_count++);
        if (imengine.null ()) return NULL;

        attach_imengine (imengine);

        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "create new default instance: %d %s", imengine->get_id (), imengine->get_factory_uuid ().c_str ());
    }

    ScimBridgeAgentIMContext *imcontext = new ScimBridgeAgentIMContext ();

    imcontext->set_imengine (imengine);
    imcontext->set_client_peer (&client_peer);
    imcontext->reset ();

    if (ScimBridgeAgentIMContext::is_imengine_shared ()) {
        const bool new_imcontext_panel_listener_status = scim_config->read (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), imcontext->is_on ());
        if (new_imcontext_panel_listener_status) imcontext->turn_on ();
        else imcontext->turn_off ();
    }

    scim_panel_client.prepare (imcontext->get_id ());

    scim_panel_client.register_input_context (imcontext->get_id (), imengine->get_factory_uuid ());

    scim_panel_client.send ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "Imcontext created: id = %d", imcontext->get_id ());
    return imcontext;
}


void ScimBridgeAgentKernelImpl::free_imcontext (ScimBridgeAgentIMContext &ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "free_imcontext...");

    scim_panel_client.prepare (ic.get_id ());

    ic.focus_out ();

    scim_panel_client.turn_off (ic.get_id ());
    scim_panel_client.focus_out (ic.get_id ());

    scim_panel_client.remove_input_context (ic.get_id ());
    scim_panel_client.send ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "imcontext freed: id = %d", ic.get_id ());
}


void ScimBridgeAgentKernelImpl::reset_imcontext (ScimBridgeAgentIMContext &ic)
{
    scim_panel_client.prepare (ic.get_id ());
    ic.reset ();
    scim_panel_client.send ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "IMContext reseted: id = %d", ic.get_id ());
}


void ScimBridgeAgentKernelImpl::focus_changed (ScimBridgeAgentIMContext &ic, bool focus_in)
{
    if (focus_in) scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 2, "focus in: ic = %d", ic.get_id ());
    else scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 2, "focus out: ic = %d", ic.get_id ());

    const ScimBridgeIMContextID id = ic.get_id ();

    scim_panel_client.prepare (id);

    if (focus_in) {
        panel_req_focus_in (&ic);
        panel_req_update_screen (&ic);
        panel_req_update_spot_location (&ic);
        panel_req_update_factory_info (&ic);

        if (ic.is_on ()) {
            scim_panel_client.turn_on (id);
            scim_panel_client.hide_preedit_string (id);
            scim_panel_client.hide_aux_string (id);
            scim_panel_client.hide_lookup_table (id);
            ic.focus_in ();
        } else {
            scim_panel_client.turn_off (id);
        }

    } else {

        scim_panel_client.turn_on (id);
        ic.focus_out ();
        if (ScimBridgeAgentIMContext::is_imengine_shared ()) ic.reset ();
        scim_panel_client.turn_off (id);
        scim_panel_client.focus_out (id);
    }

    scim_panel_client.send ();
}


bool ScimBridgeAgentKernelImpl::keyevent_occured (ScimBridgeAgentIMContext &ic, const KeyEvent &keyevent)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "Keyevent occured....");

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 1, "key: %s", keyevent.get_key_string ().c_str ());

    scim_panel_client.prepare (ic.get_id ());
    if (filter_hotkeys (&ic, keyevent)) {
        scim_panel_client.send ();
        return true;
    } else if (ic.is_on () && ic.process_key_event (keyevent)) {
        scim_panel_client.send ();
        return true;
    } else if (scim_fallback_instance->process_key_event (keyevent)) {
        scim_panel_client.send ();
        return true;
    } else {
        scim_panel_client.send ();
        return false;
    }
}


void ScimBridgeAgentKernelImpl::cursor_location_changed (ScimBridgeAgentIMContext &ic, int cursor_x, int cursor_y)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 3, "Cursor location changed: x = %d y = %d", cursor_x, cursor_y);

    ic.set_cursor_location (cursor_x, cursor_y);
    scim_panel_client.prepare (ic.get_id ());
    panel_req_update_spot_location (&ic);
    scim_panel_client.send ();
}


void ScimBridgeAgentKernelImpl::connection_closed (ScimBridgeAgentClientPeer &client_peer)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 4, "The connection with a client is closed");

    pthread_mutex_lock (&mutex);
    if (panel_listener_status == STATUS_RUNNING) {
        client_peer_list.remove (&client_peer);
        delete &client_peer;

        if (client_peer_list.empty () && exit_with_no_client) shutdown ();
    }
    pthread_mutex_unlock (&mutex);

    pthread_exit (NULL);
}
