#include <list>

#define Uses_SCIM_ATTRIBUTE
#define Uses_SCIM_HOTKEY

#include <scim.h>

#include "scim-bridge-agent-client-listener.h"
#include "scim-bridge-agent-client-peer.h"
#include "scim-bridge-agent-client-peer-protected.h"
#include "scim-bridge-agent-exception.h"
#include "scim-bridge-agent-kernel.h"
#include "scim-bridge-agent-messenger.h"
#include "scim-bridge-imcontext-manager.h"

#include "scim-bridge-output.h"

using namespace scim;

using std::list;

/* Class definition */
class ScimBridgeAgentClientPeerImpl: public ScimBridgeAgentClientPeer, public ScimBridgeAgentClientPeerProtected
{

    friend ScimBridgeAgentClientPeer *ScimBridgeAgentClientPeer::create (int input_fd, int output_fd, ScimBridgeAgentClientListener *kernel);

    public:

        void connection_closed ();

        void commit (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException);
        void set_preedit_string (const ScimBridgeAgentIMContext &ic, const scim::WideString &string) throw (ScimBridgeAgentException);
        void set_preedit_attributes (const ScimBridgeAgentIMContext &ic, const scim::AttributeList &attributes) throw (ScimBridgeAgentException);
        void set_preedit_cursor_position (const ScimBridgeAgentIMContext &ic, int cursor_position) throw (ScimBridgeAgentException);
        void set_preedit_shown (const ScimBridgeAgentIMContext &ic, bool shown) throw (ScimBridgeAgentException);
        void update_preedit (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException);
        void forward_keyevent (const ScimBridgeAgentIMContext &ic, const scim::KeyEvent &keyevent) throw (ScimBridgeAgentException);
        void beep (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException);
        bool get_surrounding_string (const ScimBridgeAgentIMContext &ic, scim::WideString &surrounding, size_t max_length, int &cursor_pos) throw (ScimBridgeAgentException);
        bool delete_surrounding_string (const ScimBridgeAgentIMContext &ic, size_t offset, size_t len) throw (ScimBridgeAgentException);

        const scim::KeyboardLayout &get_keyboard_layout ();

        ScimBridgeAgentIMContext *alloc_imcontext ();
        void free_imcontext (ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException);
        void reset_imcontext (ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException);
        void focus_changed (ScimBridgeAgentIMContext &ic, bool focus_in) throw (ScimBridgeAgentException);
        bool keyevent_occured (ScimBridgeAgentIMContext &ic, const scim::KeyEvent &keyevent) throw (ScimBridgeAgentException);
        void cursor_location_changed (ScimBridgeAgentIMContext &ic, int cursor_x, int cursor_y) throw (ScimBridgeAgentException);

    protected:

        ~ScimBridgeAgentClientPeerImpl ();

    private:

        ScimBridgeAgentClientPeerImpl (int input_fd, int output_fd, ScimBridgeAgentClientListener *kernel);

        ScimBridgeAgentMessenger *messenger;

        ScimBridgeAgentClientListener *kernel;

        std::list<ScimBridgeIMContextID> imcontext_list;
};

/* Helpers */
ScimBridgeAgentClientPeer *ScimBridgeAgentClientPeer::create (int input_fd, int output_fd, ScimBridgeAgentClientListener *new_kernel)
{
    return new ScimBridgeAgentClientPeerImpl (input_fd, output_fd, new_kernel);
}


/* Implementation */
ScimBridgeAgentClientPeerImpl::ScimBridgeAgentClientPeerImpl (int input_fd, int output_fd, ScimBridgeAgentClientListener *new_kernel):
kernel (new_kernel)
{
    messenger = new ScimBridgeAgentMessenger (*this);
    messenger->open_connection (input_fd, output_fd);
}


ScimBridgeAgentClientPeerImpl::~ScimBridgeAgentClientPeerImpl ()
{
    for (list<ScimBridgeIMContextID>::iterator i = imcontext_list.begin (); i != imcontext_list.end (); ++i) {
        ScimBridgeIMContextID ic_id = *i;
        ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (ic_id);

        if (ic != NULL) delete ic;
    }
    imcontext_list.clear ();

    delete messenger;
}


void ScimBridgeAgentClientPeerImpl::connection_closed ()
{
    kernel->connection_closed (*this);
}


const KeyboardLayout &ScimBridgeAgentClientPeerImpl::get_keyboard_layout ()
{
    return kernel->get_keyboard_layout ();
}


/* Remote calls */
void ScimBridgeAgentClientPeerImpl::commit (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at commit");

    if (messenger) messenger->call_commit (ic);
}


void ScimBridgeAgentClientPeerImpl::set_preedit_string (const ScimBridgeAgentIMContext &ic, const WideString &string) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at set_preedit_attributes");

    if (messenger) messenger->call_set_preedit_string (ic, string);
}


void ScimBridgeAgentClientPeerImpl::set_preedit_attributes (const ScimBridgeAgentIMContext &ic, const AttributeList &attributes) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at set_preedit_attributes");

    if (messenger) messenger->call_set_preedit_attributes (ic, attributes);
}


void ScimBridgeAgentClientPeerImpl::set_preedit_cursor_position (const ScimBridgeAgentIMContext &ic, int cursor_position) throw (ScimBridgeAgentException)
{
    if (messenger) messenger->call_set_preedit_cursor_position (ic, cursor_position);
}


void ScimBridgeAgentClientPeerImpl::set_preedit_shown (const ScimBridgeAgentIMContext &ic, bool shown) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at set_preedit_shown");

    if (messenger) messenger->call_set_preedit_shown (ic, shown);
}


void ScimBridgeAgentClientPeerImpl::update_preedit (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at update_preedit");

    if (messenger) messenger->call_update_preedit (ic);
}


void ScimBridgeAgentClientPeerImpl::forward_keyevent (const ScimBridgeAgentIMContext &ic, const KeyEvent &keyevent) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at forward_keyevent");

    if (messenger) messenger->call_forward_keyevent (ic, keyevent);
}


void ScimBridgeAgentClientPeerImpl::beep (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at beep");

    if (messenger) messenger->call_beep (ic);
}


bool ScimBridgeAgentClientPeerImpl::get_surrounding_string (const ScimBridgeAgentIMContext &ic, scim::WideString &surrounding, size_t max_length, int &cursor_pos) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at get_surrounding_string");

    if (messenger) {
        return messenger->call_get_surrounding_string (ic, surrounding, max_length, cursor_pos);
    } else {
        return false;
    }
}


bool ScimBridgeAgentClientPeerImpl::delete_surrounding_string (const ScimBridgeAgentIMContext &ic, size_t offset, size_t len) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at delete_surrounding_string");

    if (messenger) {
        return messenger->call_delete_surrounding_string (ic, offset, len);
    } else {
        return false;
    }
}


/* Calls for kernel */
ScimBridgeAgentIMContext *ScimBridgeAgentClientPeerImpl::alloc_imcontext ()
{
    ScimBridgeAgentIMContext *imcontext = kernel->alloc_imcontext (*this);
    if (imcontext != NULL) imcontext_list.push_back (imcontext->get_id ());

    return imcontext;
}


void ScimBridgeAgentClientPeerImpl::free_imcontext (ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at free_imcontext");

    kernel->free_imcontext (ic);
    imcontext_list.remove (ic.get_id ());
    delete &ic;
}


void ScimBridgeAgentClientPeerImpl::reset_imcontext (ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at reset_imcontext");

    kernel->reset_imcontext (ic);
}


void ScimBridgeAgentClientPeerImpl::focus_changed (ScimBridgeAgentIMContext &ic, bool focus_in) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at focus_changed");

    kernel->focus_changed (ic, focus_in);
}


bool ScimBridgeAgentClientPeerImpl::keyevent_occured (ScimBridgeAgentIMContext &ic, const scim::KeyEvent &keyevent) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at keyevent_occured");

    return kernel->keyevent_occured (ic, keyevent);
}


void ScimBridgeAgentClientPeerImpl::cursor_location_changed (ScimBridgeAgentIMContext &ic, int cursor_x, int cursor_y) throw (ScimBridgeAgentException)
{
    if (ic.get_client_peer () != this) throw ScimBridgeAgentException ("Invalid imcontext at cursor_location_changed");

    return kernel->cursor_location_changed (ic, cursor_x, cursor_y);
}
