//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [sma,ecc] = CL_op_rpvp2ae(rp,vp,mu)
// Periapsis radius and velocity at periapsis to semi major axis and eccentricity
//
// Calling Sequence
// [sma,ecc] = CL_op_rpvp2ae(rp,vp[,mu])
//
// Description
// <itemizedlist><listitem>
// Computes keplerian parameters semi major axis and eccentricity from
// periapsis radius and velocity at periapsis (elliptic and hyperbolic orbits).
// </listitem>
// </itemizedlist>
//
// Parameters
// rp: periapsis radius [m] (1xN)
// vp: velocity at periapsis [m/s] (1xN)
// mu : (optional) geocentric gravitational constant [m^3/s^2] (default value is %CL_mu)
// sma: semi major axis [m] (1xN)
// ecc: eccentricity (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Elliptic orbit
// rp = 7078.e3;
// vp = 8.e3;
// [sma,ecc] = CL_op_rpvp2ae(rp,vp)
//
// // Hyperbolic orbit
// rp = 7078.e3;
// vp = 12.e3;
// [sma,ecc] = CL_op_rpvp2ae(rp,vp)

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

Nrp = size(rp,2)
Nvp = size(vp,2)
N = max(Nrp,Nvp)
coherence = (Nrp==N|Nrp==1) & (Nvp==N|Nvp==1)
if ~coherence then CL__error('bad size of input arguments'); end
if N~=1
  if Nrp==1 then rp=rp*ones(1,N); end
  if Nvp==1 then vp=vp*ones(1,N); end
end

if ~exists('mu','local') then mu=%CL_mu; end

//

orb_type = zeros(1,N)
sma = zeros(1,N)
ecc = zeros(1,N)

energy = vp.^2/2-mu./rp  //calculation of energy to decide whether it is hyperb or ellipse

if find(energy==0)~=[] then CL__error('invalid input values (energy=0)'); end

i_ellip = find(energy<0)  //ellipse
  orb_type(i_ellip) = 1
  sma(i_ellip) = (mu.*rp(i_ellip))./(2*mu-(vp(i_ellip).^2).*rp(i_ellip))
  ecc(i_ellip) = (sma.*vp(i_ellip).^2-mu)/(mu+sma(i_ellip).*vp(i_ellip).^2)

i_hyperb = find(energy>0) //hyperbole
  orb_type(i_hyperb) = 2
  sma(i_hyperb) = (mu.*rp(i_hyperb))./((vp(i_hyperb).^2).*rp(i_hyperb)-2*mu)
  ecc(i_hyperb) = (rp(i_hyperb).*vp(i_hyperb).^2-mu)./mu

endfunction
