//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [q, Inok] = CL_rot_defRotVec(u1, v1, u2, v2)
// Rotation transforming 2 vectors into 2 vectors
//
// Calling Sequence
// [q, Inok] = CL_rot_defRotVec(u1, v1, u2, v2)
//
// Description
// <itemizedlist><listitem>
// <para>Computes the rotation that transforms 2 vectors into 2 vectors. </para>
// <para>Except for possible scale factors and vector adjustments, the rotation R is such that 
// R(u1)=u2 and R(v1)=v2. The rotation R is returned as a quaternion (<emphasis role="bold">q</emphasis>). </para>
// <para>If the angular separation between u1 and v1 is not the same as the angular separation between u2 and
// v2, then v2 is adjusted. The adjusted vector v2' is in the (u2,v2) plane, and the signed angle (u1,v1) 
// is equal to (u2,v2').</para></listitem>
// <listitem>
// <para>Notes : </para>
// <para>If (u1 and v1) or (u2 and v2) are colinear, then the rotation is not uniquely defined or doesn't exist, 
// and the corresponding indices are returned in <emphasis role="bold">Inok</emphasis>. In that case, the returned 
// rotation is the rotation that transforms u1 into u2 about an axis perpendicular to the plane (u1,u2) which is 
// the one with the smallest associated angle. If additionnaly u1 and u2 are colinear, 
// then the rotation is either the identity or a 180 degree rotation about a random axis perpendicular to u1.</para>
// </listitem>
// </itemizedlist>
//
// Parameters
// u1: First vector (3xN)
// v1: Second vector (3xN)
// u2: Desired image of u1 by the rotation (3xN)
// v2: Desired image of v1 by the rotation (3xN)
// q: Quaternion that defines the rotation (dim N)
// Inok: Indices for which u1,v1,u2,v2 do not define a unique rotation. 
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_defQuat
// CL_rot_defFrameVec
//
// Examples
// // Case: (u1,v1) and (u2,v2) have the same angular separation:
// u1 = [ 1 ; 2 ; 3 ];
// v1 = [ 3 ; 1 ; 2 ];
// q = CL_rot_axAng2quat([0;0;1], 0.1)
// u2 = CL_rot_rotVect(q,u1); 
// v2 = CL_rot_rotVect(q,v1);
// q2 = CL_rot_defRotVec(u1,v1,u2,v2) // q == q2
//
// // Case: (u1,v1) and (u2,v2) have different angular separations:
// u1 = [ 1 ; 2 ; 3 ];
// v1 = [ 3 ; 1 ; 2 ];
// q = CL_rot_axAng2quat([0;0;1], 0.1)
// u2 = CL_rot_rotVect(q,u1); 
// v2 = CL_rot_rotVect(q,v1) + 10 * u2;
// q2 = CL_rot_defRotVec(u1,v1,u2,v2) //q == q2
//
// // Case: One pair of vectors
// u1 = [ 1 ; 2 ; 3 ];
// u2 = [ 1 ; 4 ; 5 ];
// [q,ind] = CL_rot_defRotVec(u1,u1,u2,u2);
// ind // rotation is not uniquely defined.
// [axis,ang] = CL_rot_quat2axAng(q);
// axis // colinear to CL_cross(u1,u2)
// 


// Declarations:

// Code:
tol = 1.e-16; // tolerance to control alignment (square of angle) 

s = [size(u1); size(v1); size(u2); size(v2)];
 
smin = min(s, "r"); // min nb of rows, min nb of columns
smax = max(s, "r"); // max nb of rows, max nb of columns
N = smax(2); // max number of columns

// check rows (must be 3)
if (smin(1) <> 3 | smax(1) <> 3) 
  CL__error("Invalid argument sizes (number of rows)");
end

// check columns (must be 1 or N)
I = find(s(:,2) <> 1 & s(:,2) <> N);
if (I <> [])
  CL__error("Invalid argument sizes (number of columns)");
end

// initialization
axis = %nan * ones(3,N);
Imaj = zeros(1,N); // are set to 1 as indices are handled


// unitvectors before resizing (for efficiency)
[u1, nu1] = CL_unitVector(u1); 
[v1, nv1] = CL_unitVector(v1); 
[u2, nu2] = CL_unitVector(u2); 
[v2, nv2] = CL_unitVector(v2); 

if (s(1,2) < N); u1 = u1 * ones(1,N); end
if (s(2,2) < N); v1 = v1 * ones(1,N); end
if (s(3,2) < N); u2 = u2 * ones(1,N); end
if (s(4,2) < N); v2 = v2 * ones(1,N); end

// arguments checks (vectors should not be 0)
// NB: norms compared to 0 volontarily
if (find (nu1 .* nu2 .* nv1 .* nv2 == 0))
  CL__error("Vectors should not be zero");
end

// function: find axis perpendicular to u (any direction)
function [v] = axis_perp(u)
  x = CL_cross(u, [1;0;0]);
  y = CL_cross(u, [0;1;0]);
  v = x; 
  I = find(CL_dot(y) > CL_dot(x));
  v(:,I) = y(:,I);
endfunction



// --- case u1 == u2 
// => axis = u1 
// NB: not a special case, but avoids dealing with this 
// case several times in the rest

I = find(CL_dot(u2-u1) < tol);
if (I <> [])
  axis(:,I) = u1(:,I); 
  Imaj(I) = 1; 
end


// --- case u2 // v2 (=> v2 cannot be adjusted)
// => axis = u1^u2 
// but if u2 == -u1 => arbitrary axis perpendicular to u1

// --- cases considered as "not nominal" : 
// not nominal if : 
// - v2 cannot be adjusted or 
// - u1==v1 (information missing)
Inok = find(CL_dot(CL_cross(u2,v2)) < tol | CL_dot(CL_cross(u1,v1)) < tol);


I = Inok(find(Imaj(Inok) == 0)); // indices / Imaj(indices) == 0 and indices in Inok 

if (I <> [])
  w = CL_cross(u1(:,I), u2(:,I)); 
  axis(:,I) = w; // u1^u2 by default

  J = I(find(CL_dot(w) < tol)); // w = 0
  if (J <> [])
    axis(:,J) = axis_perp(u1(:,J)); 
  end

  Imaj(I) = 1; 
end


// --- adjust v2 (-> v2') so that u1.v1 = u2.v2' (||v2'|| == 1)
// v2' is in the plane (u2,v2) and u1^v1 == u2^v2'  

// w = unit vector perpendicular to v2 (OK as u2 <> v2)
w = CL_unitVector( v2 - CL_dMult(CL_dot(u2,v2), u2) ); 
v2 = CL_dMult(CL_dot(u1,v1), u2)  + CL_dMult(CL_norm(CL_cross(u1,v1)), w);


// --- "general case" 
// w = (u2-u1) ^ (v2-v1) 
// axis == w, except if w == 0 
// (=> axis in the plane (u1,v1) instead)

w = CL_cross(u2-u1,v2-v1);

I = find(Imaj == 0 & CL_dot(w) < tol);
if (I <> [])
  
  // case : rotation axis in (u1,v1) plane
  J = I(find(Imaj(I) == 0));
  if (J <> [])
    coef = -CL_dot(v1(:,J), u2(:,J)-u1(:,J)) ./ CL_dot(u1(:,J), u2(:,J)-u1(:,J)); 
    axis(:,J) = v1(:,J) + CL_dMult(coef, u1(:,J));
    Imaj(J) = 1; 
  end
end 

I = find(Imaj == 0); 
axis(:,I) = w(:,I); 


// quaternion (all cases)

// angle obtained using (u1 and u2) except if
// axis is u1 (in this case: use (v1,v2))
I = find(CL_dot(u1-u2) < tol); 
u1(:,I) = v1(:,I);
u2(:,I) = v2(:,I);

u1 = CL_cross(axis, u1); 
u2 = CL_cross(axis, u2);

ud = CL_dot(u1); // = norm(u1) * norm(u2) 
w = CL_cross(u1,u2); 
 
cosang = %nan * ones(1,N);
I = find(ud >= tol);
cosang(I) = CL_dot(u1(:,I),u2(:,I)) ./ ud(I);
I = find(ud < tol);
cosang(I) = 1; 

ca = real(sqrt((1+cosang)/2)); // cos(ang/2)
sa = real(sqrt((1-cosang)/2)); // sin(ang/2)

I = find(CL_dot(w, axis) < 0);
sa(I) = -sa(I);

// quaternion from real and imaginary parts
axis = CL_unitVector(axis);
q = CL_rot_defQuat(ca, sa.*axis(1,:), sa.*axis(2,:), sa.*axis(3,:)); 


endfunction








