module Schleuder
  def log
    # setup a logger
    if Schleuder.list
      Schleuder.list.log
    else
      unless @log
        @log = SchleuderLogger.new('Schleuder', config.logfile, config.loglevel)
      end
      @log
    end
  end
  module_function :log

  def config
    # read the base config (class overloads itself with conf/schleuder.conf)
    @config = SchleuderConfig.new unless @config
    @config
  end
  module_function :config

  # This is quite ugly, I don't like module functions. But we need the
  # list-object in Schleuder.log and in Schleuder::Mail, so until somebody has
  # got a better idea this has to stay.
  def list
    @list || nil
  end
  module_function :list

  def list=(list)
    @list = list
  end
  module_function :list=

  class Processor
    def self.run(listname, message)
      begin
        Schleuder.log.debug "Creating list, logging cont'd in the list-specific logfile"
        Schleuder.list = List.new(listname)

        Schleuder.log.debug "Parsing incoming message"

        mail = Mail.parse(message)
        Schleuder.log.info "New mail incoming from #{mail.from}"

        #  if mail is a bounce we forward it directly to adminaddr, as
        #  dealing with those probably fails (encrypted attachments etc.)
        if not mail.to.nil? and mail.to.include? Schleuder.list.bounce_addr
          Schleuder.log.info "This is a bounce, forwarding to adminaddr and exiting"
          Schleuder.log.notify_admin "Hello,\n\nI catched a bounce. Please take care of it:\n\n\n#{message}", "Problem"
          Schleuder.log.info "Exiting cleanly"
          exit(0)
        end

        # if mail is a send-key-request we answer directly
        if (not mail.to.nil? and mail.to.include? Schleuder.list.sendkey_addr) or (mail.subject.strip.downcase == 'send key!' and mail.body.strip.empty?)
          Schleuder.log.info "Found send-key-request"
          # TODO: refactor with Plugin::reply
          out = Mail.new
          out.subject = "Re: #{mail.subject}"
          Schleuder.log.info "Building reply"
          iout = out.individualize(Member.new({'email' => mail.from.first}))
          iout.in_reply_to = mail.message_id
          Schleuder.log.debug "Filling body with key-id and key"
          me = Schleuder.list.config.myaddr
          iout.body = "#{mail.crypt.get_key(me).to_s}\n#{mail.crypt.export(me).to_s}"
          Schleuder.log.debug "Signing outgoing email"
          rawmail = iout.sign
          Schleuder.log.info "Handing over to Mailer"
          Mailer.send(rawmail, iout.to)
          Schleuder.log.info "Exiting"
          exit 0
        end

        # Analyse data (hopefully an incoming mail).
        # Is it multi-part? encrypted? pgp/mime? signed?
        Schleuder.log.debug "Analysing incoming mail"
        begin
          mail.decrypt!
        rescue GPGME::Error::DecryptFailed => e
          Schleuder.log.error "#{e}\n\nOriginal message:\n#{message}"
          STDOUT.puts "Schleuder speaking. Cannot decrypt. Please check your setup.\nMessages to this list need to be encrypted with this key:\n#{mail.crypt.get_key(Schleuder.list.config.myaddr)}"
          exit 100
        end

        if Schleuder.list.config.receive_signed_only and not mail.in_signed
          self.bounce_or_drop "not validly signed", "This address accepts only messages validly signed in an OpenPGP-compatible way", mail, message
        end

        if Schleuder.list.config.receive_encrypted_only and not mail.in_encrypted
          self.bounce_or_drop 'not encrypted', "This address accepts only messages encrypted with this key:\n#{mail.crypt.get_key(Schleuder.list.config.myaddr)}", mail, message
        end

        if Schleuder.list.config.receive_authenticated_only and not mail.from_member
          self.bounce_or_drop "not authenticated", "This address accepts only messages validly signed by a list members key", mail, message
        end

        if mail.from_member or mail.from_admin
          if mail.in_encrypted
            Schleuder.log.info 'Email is encrypted and authorized, processing plugins'
            # collect keywords from message body
            Schleuder.log.debug 'Collecting keywords from message body...'
            mail.collect_keywords!
            Schleuder.log.debug "Inspecting found keywords: #{mail.keywords.inspect}"
            # process plugins
            Schleuder.log.debug 'Processing plugins'
            mail.process_plugins!
          else
            Schleuder.log.debug "Incoming mail is not encrypted or authorized, skipping plugins"
          end
        else
          Schleuder.log.debug "Mail is not from a list-member, adding prefix_in"
          mail.add_prefix_in!
        end

        # first encrypt and send message to external recipients and collect
        # information about the process
        mail.resend_to.each do |receiver|
          imail = mail.individualize(receiver)
          imail.add_public_footer!

          if Schleuder.list.config.send_encrypted_only
            enc_only = true
          else
            enc_only = receiver.encrypted_only
          end

          begin
            if ret = self.send(imail, receiver, enc_only)
              # store meta-data about the sent message
              crypt = " (#{ret})"
              if mail.metadata['resent_to']
                mail.metadata['resent_to'] << ", #{receiver.email + crypt}"
              else
                mail.metadata['resent_to'] = receiver.email + crypt
              end
            else
              mail.metadata['note'] = "Not resent to #{receiver.email}"
            end
          rescue => e
            Schleuder.log.error "#{e}\n#{e.backtrace.join("\n")}\n\nOriginal message:\n#{mail.to_s}"
          end
        end

        if mail.metadata['resent_to'] and not mail.metadata['resent_to'].empty?
          mail.add_prefix_out!
        end
        mail.add_prefix!

        mail.add_metadata!

        # encrypt message and send mail for each list-member
        Schleuder.list.members.each do |receiver|
          imail = mail.individualize_member(receiver)

          if Schleuder.list.config.send_encrypted_only
            enc_only = true
          else
            enc_only = receiver.encrypted_only
          end

          Schleuder.log.info "enc: #{enc_only}"
          # encrypt for each receiver
          begin
            unless self.send(imail, receiver, enc_only)
              Schleuder.log.error "Sending mail to list-member #{receiver} failed!"
            end
          rescue => e
            Schleuder.log.error "#{e}\n#{e.backtrace.join("\n")}\n\nOriginal message:\n#{mail.to_s}"
          end
        end
      rescue SystemExit => e
        exit e.status
      rescue Exception => e
        Schleuder.log.error "#{e}\n#{e.backtrace.join("\n")}\n\nOriginal message:\n#{message}"
      end
    end

    def self.send(mail, receiver, encrypted_only = true)
      if mail.encrypt!(receiver)
        Mailer.send(mail)
        return 'encrypted'
      else
        Schleuder.log.warn 'Encryption failed'
        if encrypted_only
          Schleuder.log.warn "Sending plaintext not allowed, message not sent to #{receiver.inspect}!"
        else
          Schleuder.log.info "Sending plaintext allowed, doing so"
          # sign msg
          rawmail = mail.sign
          if rawmail
            Mailer.send(rawmail, mail.to)
            return 'unencrypted'
          end
        end
        return false
      end
    end

    def self.test(listname=nil)
      # Doing things that trigger exceptions if they fail, which we rescue and
      # put to stderr
      # TODO: test #{smtp_host}:25
      begin
        # test listdir
        Dir.entries Schleuder.config.lists_dir

        # test superadminaddr
        Utils::verify_addr('superadminaddr', Schleuder.config.superadminaddr)

        if listname
          # test for listdir
          Dir.entries File.join(Schleuder.config.lists_dir, listname)
          # testwise create a list-object (reads list-config etc.)
          list = List.new listname
          # test adminaddr
          unless list.config.adminaddr.kind_of? Array
            raise 'adminaddr must be an array!'
          else
            list.config.adminaddr.each do |a|
              Utils::verify_addr('adminaddr', a)
            end
          end

          crypt = Crypt.new(list.config.gpg_password)

          list.members.each do |m|
            Utils::verify_addr('member address', m.email)
            unless crypt.get_key(m.email || m.name)
              raise "#{m.inspect} matches more than one key in the pubring!"
            end
          end
        end
      rescue => e
        STDERR.puts e.message
        exit 1
      end
    end

    def self.newlist(listname, interactive='true', args=nil)
      created_list = Schleuder::ListCreator.create(listname,interactive,args)
    end

    def self.bounce_or_drop status, bounce_msg, mail, message
      Schleuder.log.warn "Mail is #{status}, not passing it along"

      if Schleuder.list.config.bounces_drop_all
        Schleuder.log.warn "Found bounces_drop_all being true: dropping!"
        self.bounce_notify_admin message, status, "bounces_drop_all is true"
        exit 0
      end

      Schleuder.log.debug "Testing bounces_drop_on_headers"
      Schleuder.list.config.bounces_drop_on_headers.each do |header,value|
        Schleuder.log.debug "Testing #{header} => #{value}"
        if mail.header[header].to_s.downcase == value.to_s.downcase
          Schleuder.log.warn "Found matching drop-header: #{header} => #{value} -- dropping!"
          self.bounce_notify_admin message, status, "Forbidden header found: '#{header.capitalize}: #{value.capitalize}'"
          exit 0
        end
      end

      # if we're still alive: bounce message
      Schleuder.log.info "bouncing mail to sender"
      self.bounce_notify_admin message, status
      #STDOUT.puts bounce_msg
      exit 100
    end

    def self.bounce_notify_admin message, reason, drop_reason=''
      msg = "Hello list-admin,\n\nthe following incoming email has not been passed to the list.\nReason: #{reason}.\n"
      if drop_reason.empty?
        msg += "\nIt has been bounced to the sender.\n"
      else
        msg += "\nIt has *not* been bounced but dropped.\nReason: #{drop_reason}.\n"
      end
      msg += "\n\nOriginal message:\n\n#{message}\n"
      Schleuder.log.notify_admin(msg, "Notice") if Schleuder.list.config.bounces_notify_admin
    end
  end
end
