/*
 * scamper_addresslist.h
 *
 * $Id: scamper_addresslist.h,v 1.37 2006/12/21 03:05:56 mjl Exp $
 *
 * Copyright (C) 2004 The University of Waikato
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __SCAMPER_ADDRESSLIST_H
#define __SCAMPER_ADDRESSLIST_H

typedef struct scamper_source_params
{
  /*
   * these parameters end up in a 'scamper_list' structure with the
   * scamper_trace itself:
   *  list_id:  the list id number assigned by a human.
   *  cycle_id: the initial cycle id to use.
   *  name:     the name of the list being probed.
   *  descr:    a description of the addresses that are stored in this source
   *  command:  a default command to use with the source
   */
  uint32_t           list_id;
  uint32_t           cycle_id;
  char              *name;
  char              *descr;
  char              *command;

  /* how many addresses per cycle of all sources to read */
  int                priority;

  /* if set, this source is adhoc. */
  int                adhoc;

  /* the output file, if somewhere other than the default output file */
  scamper_outfile_t *sof;

  /*
   * if the source is not adhoc, then we're managing a file.
   *  filename:   the name of the file to read
   *  cycles:     how many cycles to make over the list.
   *  autoreload: if set, the file should be re-loaded at each cycle
   */
  char              *filename;
  int                cycles;
  int                autoreload;

} scamper_source_params_t;

typedef struct scamper_source scamper_source_t;

/* add a new source */
scamper_source_t *scamper_addresslist_addsource(scamper_source_params_t *ssp);

/* find the source given a name */
scamper_source_t *scamper_addresslist_getsource(char *name);

/* remove the source from rotation */
int scamper_addresslist_delsource(scamper_source_t *source);

/* return the next address to probe */
scamper_task_t *scamper_addresslist_get(void);

/* functions for getting various source properties */
const char *scamper_source_getname(const scamper_source_t *source);
const char *scamper_source_getdescr(const scamper_source_t *source);
const char *scamper_source_getfilename(const scamper_source_t *source);
scamper_outfile_t *scamper_source_getoutfile(const scamper_source_t *s);
uint32_t scamper_source_getlistid(const scamper_source_t *source);
uint32_t scamper_source_getcycleid(const scamper_source_t *source);
int scamper_source_getpriority(const scamper_source_t *source);
int scamper_source_getadhoc(const scamper_source_t *source);
int scamper_source_getcycles(const scamper_source_t *source);
int scamper_source_getautoreload(const scamper_source_t *source);

/* functions to give a source something to do */
int scamper_source_do(scamper_source_t *ss, char *command);
int scamper_source_do_file(scamper_source_t *ss, char *file);
int scamper_source_do_array(scamper_source_t *ss, char *command,
			    char **iparray, int ipcount);

/* functions to reference / dereference a source */
scamper_source_t *scamper_source_use(scamper_source_t *source);
void scamper_source_unuse(scamper_source_t *source);

/* update various parameters */
int scamper_source_update(scamper_source_t *source,
			  const int *reload,const int *cycles,const int *pri);

/* function for iterating through all sources */
void scamper_addresslist_foreach(void *p,
				 int (*func)(void *p, scamper_source_t *src));

/* mark the cycle point */
int scamper_addresslist_cyclesource(scamper_source_t *source);

/* if the address list is empty, this function will tell us */
int scamper_addresslist_isempty(void);

/* tell the caller if the address list appears to have an address ready */
int scamper_addresslist_isready(void);

/* remove all addresses and sources in the address list */
void scamper_addresslist_empty(void);

/* functions to free up the core address list data structures */
int scamper_addresslist_init(void);
void scamper_addresslist_cleanup(void);

/*
 * interface to observe source events.
 *
 *
 */
typedef struct scamper_source_event
{
  scamper_source_t *source;
  time_t            sec;
  int               event;

#define SCAMPER_SOURCE_EVENT_ADD     0x01
#define SCAMPER_SOURCE_EVENT_UPDATE  0x02
#define SCAMPER_SOURCE_EVENT_CYCLE   0x03
#define SCAMPER_SOURCE_EVENT_DELETE  0x04
#define SCAMPER_SOURCE_EVENT_FINISH  0x05

  union
  {

    struct sse_update
    {
      uint8_t flags;  /* 0x01 == autoreload, 0x02 == cycles, 0x03 = priority */
      int     autoreload;
      int     cycles;
      int     priority;
    } sseu_update;

#define sse_update_flags       sse_un.sseu_update.flags
#define sse_update_autoreload  sse_un.sseu_update.autoreload
#define sse_update_cycles      sse_un.sseu_update.cycles
#define sse_update_priority    sse_un.sseu_update.priority

    struct sse_cycle
    {
      int     cycle_id;
    } sseu_cycle;

#define sse_cycle_cycle_id     sse_un.sseu_cycle.cycle_id

  } sse_un;

} scamper_source_event_t;

typedef void (*scamper_source_event_func_t)(const scamper_source_event_t *sse,
					    void *param);

void *scamper_addresslist_observe(scamper_source_event_func_t func, void *p);
void  scamper_addresslist_unobserve(void *handle);

#endif /* SCAMPER_ADDRESSLIST_H */
