
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Umesh Kumar V. Rajasekaran 
//          Timothy J. McBrayer 
//	    Magnus Danielson	

#include "IIR_FunctionCall.hh"
#include "IIR_AssociationElement.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "StandardPackage.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

using std::cerr;
using std::ends;
using std::string;

IIRScram_FunctionCall::~IIRScram_FunctionCall() {}

set<IIR_Declaration> *
IIRScram_FunctionCall::_symbol_lookup(){
  ASSERT( get_implementation() != NULL );
  return new set<IIR_Declaration>( get_implementation() );
}

ostream &
IIRScram_FunctionCall::_print( ostream &os ){
  ASSERT(get_implementation() != NULL);

  os << *get_implementation()->get_declarator();
  if (parameter_association_list.num_elements() != 0) {
    os << "(";
    os << parameter_association_list;
    os << ") ";
  }

  return os;
}

void 
IIRScram_FunctionCall::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT(get_implementation() != NULL);

  get_implementation()->get_declarator()->_publish_vhdl(_vhdl_out);
  if (parameter_association_list.num_elements() != 0) {
    _vhdl_out << "(";
    parameter_association_list._publish_vhdl_without_formals(_vhdl_out);
    _vhdl_out << ") ";
  }
}

void
IIRScram_FunctionCall::_publish_cc_ams_function(published_file &_cc_out) {
  IIR_TextLiteral *declarator = get_implementation()->get_declarator();
  
  if ((IIRScram_Identifier::_cmp(declarator, "now") == 0)) {
    if(_get_currently_publishing_unit() == IIRScram::SIMULTANEOUS_STATEMENT) {
      _cc_out << "  new equationNode('T',0,node"
              << (int)(_stmt_node_index/2) << ");" << NL();
    }
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "exp") == 0)) {
    if(_get_currently_publishing_unit() == IIRScram::SIMULTANEOUS_STATEMENT) {
      _cc_out << "  equationNode *node"<<_stmt_node_index<<" = new equationNode('E',0,node"
              << (int)(_stmt_node_index/2) << ");" << NL();
    }
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "log") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "sqrt") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "sin") == 0)) {
    if(_get_currently_publishing_unit() == IIRScram::SIMULTANEOUS_STATEMENT) {
      _cc_out << "  equationNode *node"<<_stmt_node_index<<" = new equationNode('S',0,node"
              << (int)(_stmt_node_index/2) << ");" << NL();
    }
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "cos") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "tan") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }  
  else if ((IIRScram_Identifier::_cmp(declarator, "asin") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "acos") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "atan") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "atan2") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "pow") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "log10") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "sinh") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "cosh") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "tanh") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "ceil") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "floor") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "asinh") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "acosh") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "atanh") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "fabs") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "fmax") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "fmin") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "ldexp") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "frexp") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "erf") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "rand") == 0)) {
    _publish_cc_ams_math_functions(declarator,_cc_out);
  }
  else {
    _cc_out << "((("; 
    get_subtype()->_publish_cc_universal_type(_cc_out);
    _cc_out << "*)(&(";
    get_implementation()->_publish_cc_lvalue(_cc_out);
    _cc_out << "_" << this << ".getVHDLData())))->val) ";
  }
}

void
IIRScram_FunctionCall::_publish_cc_ams_math_functions(IIR_TextLiteral *math_function_name,
                                                      published_file &_cc_out) {
  IIR_AssociationElement *elem = parameter_association_list.first();
  while (elem != NULL) {   
    elem->_get_actual()->_publish_cc_ams_function(_cc_out);
    elem = parameter_association_list.successor(elem);
    if (elem != NULL) {
      elem->_get_actual()->_publish_cc_ams_function(_cc_out);
    }
  }   
}

void
IIRScram_FunctionCall::_publish_cc_ams_function_call_in_simult_stmt(published_file &_cc_out) {
  ostringstream retval_buf;
  string func_retval;
  
  IIR_TextLiteral *declarator = get_implementation()->get_declarator();
  
  if ((IIRScram_Identifier::_cmp(declarator, "exp") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "log") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "sqrt") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "sin") == 0)) { 
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "cos") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "tan") == 0)) {
    // No need to do anything
  } 
  else if ((IIRScram_Identifier::_cmp(declarator, "asin") == 0)) {  
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "acos") == 0)) {   
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "atan") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "atan2") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "pow") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "log10") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "sinh") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "cosh") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "tanh") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "ceil") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "floor") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "asinh") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "acosh") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "atanh") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "fabs") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "fmax") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "fmin") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "ldexp") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "frexp") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "erf") == 0)) {
    // No need to do anything
  }
  else if ((IIRScram_Identifier::_cmp(declarator, "rand") == 0)) {
    // No need to do anything
  }
  else
  {
    ASSERT(_get_currently_publishing_unit() == IIRScram::ARCHITECTURE_DECL);
    ASSERT( get_implementation() != NULL);
    
    get_implementation()->_get_declarator()->_print(retval_buf);
    retval_buf << "_" << this << ends;
    func_retval = retval_buf.str();
    _cc_out << get_subtype()->_get_cc_type_name( );
    _cc_out << " " << func_retval << "(ObjectBase::VARIABLE, ";
    get_implementation()->_publish_cc_lvalue(_cc_out);
    _cc_out << "(";
    if (_get_currently_publishing_unit() == SIMULTANEOUS_STATEMENT) {
      _cc_out << "(VHDLKernel *)currentEquation->ckt";
    }
    if ( parameter_association_list.num_elements() != 0) {
      _cc_out << ", ";
      parameter_association_list._publish_cc_lvalue( _cc_out );
    }
    _cc_out << "));" << NL();
  }
}

void
IIRScram_FunctionCall::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  parameter_association_list._build_sensitivity_list(sensitivity_list);
}

void
IIRScram_FunctionCall::_add_decl_into_cgen_symbol_table() {
  IIR_AssociationElement *elem = parameter_association_list.first();

  while (elem != NULL) {
    if (elem->_get_actual() != NULL) {
      elem->_get_actual()->_add_decl_into_cgen_symbol_table();
    }
    elem = parameter_association_list.successor(elem);
  }
}

IIR_FunctionCall *
IIRScram_FunctionCall::_build_function_call( IIR_SubprogramDeclaration *func_decl,
					     IIR_IndexedName * ){
  IIR_FunctionCall *retval = new IIR_FunctionCall;
  copy_location( func_decl, retval );
  retval->set_implementation( func_decl );
  return retval;
}

void
IIRScram_FunctionCall::_publish_cc_as_operator( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc_as_operator" );

  ASSERT ( get_implementation() != NULL );
  get_implementation()->_publish_cc_lvalue( _cc_out );
  _cc_out << OS("(");
  parameter_association_list._publish_cc_lvalue( _cc_out );
  if ((IIR_TextLiteral::_cmp(get_implementation()->get_declarator(), "\"&\"") == 0) && (parameter_association_list._are_all_scalars() == TRUE)) {
    ASSERT ( get_subtype() != NULL );
    _cc_out << "," << NL() 
	    << "(" << get_subtype()->_get_cc_type_name() << " *) NULL";
  }
  _cc_out << CS(")");
}
    
void 
IIRScram_FunctionCall::_publish_cc_lvalue( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc" );
  ASSERT(get_implementation() != NULL);
  
  if ( get_implementation()->_is_operator() &&
      get_implementation()->_is_implicit_declaration() ){
    _publish_cc_as_operator( _cc_out );
    return;
  }
  
  IIR_TextLiteral *declarator = get_implementation()->get_declarator();
  if ( IIRScram_Identifier::_cmp(declarator, "now") == 0 ){
    get_implementation()->_publish_cc_lvalue( _cc_out );
    return;
  }
  else {
    // Special handling is required for "now"call within a subprogram.
    if ( _is_currently_publishing_subprogram() == TRUE ){
      if ( IIRScram_Identifier::_cmp(declarator, "now") == 0 ){
	_cc_out << "processPtr->";
	get_implementation()->_publish_cc_lvalue( _cc_out );
	_cc_out << OS("(");
      } 
      else {
	// Pass on the process ptr to any function that is being called from
	// this function.
	get_implementation()->_publish_cc_lvalue( _cc_out );
	_cc_out << OS("(processPtr");
      }
    } 
    else {
      get_implementation()->_publish_cc_lvalue( _cc_out );
      switch(_get_currently_publishing_unit()) {
      case IIRScram::ENTITY_DECL:
      case IIRScram::ARCHITECTURE_DECL:
      case IIRScram::PACKAGE_PUB:
      case IIRScram::PACKAGE_BODY:
      case IIRScram::PROCESS_STATE:
      case IIRScram::CONFIGURATION_DECLARATION:	
      case IIRScram::GENERIC_MAP:
	_cc_out << OS("(NULL");
	break;
      case IIRScram::PROCESS:
      case IIRScram::BLOCK:
	_cc_out << OS("(this");
	break;
      default:
	cerr << "Yet to Consider Function call invocation in the Publishing Unit: "  
	     << (int)_get_currently_publishing_unit() << "\n";
	abort();
	break;
      }
    }
    
    if (parameter_association_list.num_elements() != 0) {
      //     parameter_association_list._publish_cc_subprogram_arguments( _cc_out );
      _cc_out << "," << NL();
      parameter_association_list._publish_cc_lvalue( _cc_out );
    }
    _cc_out << CS(")");
  }
}

void 
IIRScram_FunctionCall::_publish_cc_elaborate( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc_elaborate" );
  _publish_cc_lvalue( _cc_out );
}

// This method is called from
// IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_Add.
// This happens ONLY when this is used as a type conversion function.
// Then, this function call MUST have ONLY one argument.
void 
IIRScram_FunctionCall::_publish_cc_elaborate_arg( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc_elaborate_arg" );

  ASSERT(parameter_association_list.num_elements() == 1);
  parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
}

void 
IIRScram_FunctionCall::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc_bounds" );

  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION );

  ((IIR_FunctionDeclaration *)get_implementation())->get_return_type()->_publish_cc_bounds( _cc_out );
}
  
void 
IIRScram_FunctionCall::_type_check( set<IIR_TypeDefinition> *possible_types ){
  if (has_been_type_checked  == FALSE) {
    has_been_type_checked = TRUE;
    ASSERT(get_implementation() != NULL);
    IIR_SubprogramDeclaration *my_decl = get_implementation();
  
    if (parameter_association_list._is_resolved() == FALSE) {
      parameter_association_list._resolve_and_order( &my_decl->interface_declarations,
						     NULL,
						     this );
    }
  
    parameter_association_list._fill_in_defaults( this, &my_decl->interface_declarations );

    ASSERT( parameter_association_list._is_resolved() == TRUE );
  }
}

void
IIRScram_FunctionCall::_get_list_of_input_signals( set<IIR> *list ){
  parameter_association_list._get_list_of_input_signals(list);

  // There isn't any real type checking - it was all done during the
  // semantic_transform of the indexed_name.
}

IIR *
IIRScram_FunctionCall::_rval_to_decl( IIR_TypeDefinition *my_type ){
  // These are some simple sanity checks...
  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->get_subtype() != NULL );
  ASSERT( my_type != NULL );
  ASSERT(  get_implementation()->get_subtype()->_is_compatible( my_type ) != NULL );

  return (IIR *)this;
}

IIR_TypeDefinition *
IIRScram_FunctionCall::get_subtype(){
  ASSERT( get_implementation() != NULL );
  ASSERT(get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION);
  IIR_FunctionDeclaration* func_decl = (IIR_FunctionDeclaration*)get_implementation();
  return func_decl->get_return_type();  
}

set<IIR_TypeDefinition> *
IIRScram_FunctionCall::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  ASSERT( get_implementation() != NULL );

  return get_implementation()->_get_rval_set(constraint_function);
}

void
IIRScram_FunctionCall::_publish_cc_first_objectParameter( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc_first_objectParameter" );

  // This function cannot be called on function call nodes with no
  // parameters. This function is for TCs that usually have a parameter
  ASSERT (parameter_association_list.first() != NULL);

  parameter_association_list.first()->_publish_cc_first_objectParameter( _cc_out );
}

const string
IIRScram_FunctionCall::_get_cc_kernel_type(){
  //  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionCall::_publish_cc_kernel_type" );
  ASSERT( get_implementation() != NULL );
  return get_implementation()->get_subtype()->_get_cc_kernel_type();
}

IIR*
IIRScram_FunctionCall::_clone() {
  IIR_FunctionCall *funccall = new IIR_FunctionCall();

  IIR_Expression::_clone(funccall);
  funccall->set_implementation(get_implementation());
  
  IIR_AssociationElement *newassoc, *assoc = parameter_association_list.first();
  while (assoc != NULL) {
    newassoc = (IIR_AssociationElement*)assoc->_clone();
    funccall->parameter_association_list.append(newassoc);
    assoc = parameter_association_list.successor(assoc);
  }

  return funccall;
}

IIR_Mode
IIRScram_FunctionCall::_get_mode() {
  ASSERT(parameter_association_list.num_elements() == 1);
  ASSERT(parameter_association_list.first()->_is_signal() == TRUE);

  return parameter_association_list.first()->_get_actual()->_get_mode();
}

IIR_Declaration *
IIRScram_FunctionCall::_find_formal_declaration(){
  ASSERT( _is_resolved() == TRUE );
  ASSERT( parameter_association_list.num_elements() == 1 );
  ASSERT( parameter_association_list.first() != NULL );
  return parameter_association_list.first()->_find_formal_declaration();
}

IIR_Boolean 
IIRScram_FunctionCall::_is_globally_static_primary(){
  IIR_Boolean retval = TRUE;

  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION );
  if ( ((IIR_FunctionDeclaration *)get_implementation())->get_pure() == IIR_IMPURE_FUNCTION ){
    retval = FALSE;
  }
  
  if ( parameter_association_list._is_globally_static_for_function() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean
IIRScram_FunctionCall::_is_static_expression() {
  if (_is_globally_static_primary() == FALSE)  {
     return FALSE;
  }

  IIR_AssociationElement *current = parameter_association_list.first();
  while (current != NULL)  {
    if (current->_is_static_expression() == FALSE)  {
      return FALSE;
    }
    current = parameter_association_list.successor(current);
  }

  return TRUE;
}

set<IIR_Declaration> *
IIRScram_FunctionCall::_get_implicit_declarations( ){ 
  ASSERT( get_implementation() != 0 );
  return get_implementation()->_get_implicit_declarations();
}

void 
IIRScram_FunctionCall::_set_implicit_declarations( set<IIR_Declaration> *id ){
  ASSERT( get_implementation() != 0 );
  get_implementation()->_set_implicit_declarations( id );
}

void
IIRScram_FunctionCall::_set_stmt_node_index(IIR_Int32 *index, bool _is_right_child, bool &reducibleFlag) {
  if (_is_right_child == true) {
   _stmt_node_index = 2 * (*index)+ 1;
  }
  else {
   _stmt_node_index = 2 * (*index);
  }
  IIR_AssociationElement *elem = parameter_association_list.first();
  while(elem != NULL) {
    elem->_get_actual()->_set_stmt_node_index(&_stmt_node_index, false, reducibleFlag);
    elem = parameter_association_list.successor(elem);
    if (elem!=NULL) {
     elem->_get_actual()->_set_stmt_node_index(&_stmt_node_index, true, reducibleFlag);
    }
  }
}

void
IIRScram_FunctionCall::_set_stmt_qty_index(IIR_Int32 *index,
                                           set<IIR_Declaration> *quantity_set) {
  IIR_AssociationElement *elem = parameter_association_list.first();
  while(elem != NULL) {
    elem->_get_actual()->_set_stmt_qty_index(index, quantity_set);
    elem = parameter_association_list.successor(elem);
  }
} 

void
IIRScram_FunctionCall::_set_stmt_signal_index(IIR_Int32 *index,
                                              set<IIR_Declaration> *signal_set) {
  
  IIR_AssociationElement *elem = parameter_association_list.first();
  while(elem != NULL) {
    elem->_get_actual()->_set_stmt_signal_index(index, signal_set);
    elem = parameter_association_list.successor(elem);
  }
}

void
IIRScram_FunctionCall::_flush_stmt_index() {
  IIR_AssociationElement *elem = parameter_association_list.first();
  while(elem != NULL) {
    elem->_get_actual()->_flush_stmt_index();
    elem = parameter_association_list.successor(elem);
  }
}

void
IIRScram_FunctionCall::_build_generic_parameter_set(set<IIR_Declaration> *to_build) {

  IIR_AssociationElement *elem = parameter_association_list.first();
  while(elem != NULL) {
    elem->_get_actual()->_build_generic_parameter_set(to_build);
    elem = parameter_association_list.successor(elem);
  }
} 

visitor_return_type *IIRScram_FunctionCall::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_FunctionCall(this, arg);
};
