
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur    
//          Krishnan Subramani 
//          Timothy J. McBrayer
//	    Magnus Danielson	cfmd@swipnet.se	

#include "IIR_DyadicOperator.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_ReferenceAttribute.hh"
#include "IIR_ContributionAttribute.hh"
#include "IIR_TerminalDeclaration.hh"
#include "IIR_AboveAttribute.hh"
#include "published_file.hh"
using std::cerr;

IIRScram_DyadicOperator::~IIRScram_DyadicOperator() {}

const string
IIRScram_DyadicOperator::_get_function_name() const {
  string retval;
  switch( get_kind() ){
  case IIR_AND_OPERATOR:{
    retval = "\"and\"";
    break;
  }
  case IIR_OR_OPERATOR:{
    retval = "\"or\"";
    break;
  }
  case IIR_NAND_OPERATOR:{
    retval = "\"nand\"";
    break;
  }
  case IIR_NOR_OPERATOR:{
    retval = "\"nor\"";
    break;
  }
  case IIR_XOR_OPERATOR:{
    retval = "\"xor\"";
    break;
  }
  case IIR_XNOR_OPERATOR:{
    retval = "\"xnor\"";
    break;
  }
  case IIR_EQUALITY_OPERATOR:{
    retval = "\"=\"";
    break;
  }
  case IIR_INEQUALITY_OPERATOR:{
    retval = "\"/=\"";
    break;
  }
  case IIR_LESS_THAN_OPERATOR:{
    retval = "\"<\"";
    break;
  }
  case IIR_LESS_THAN_OR_EQUAL_OPERATOR:{
    retval = "\"<=\"";
    break;
  }
  case IIR_GREATER_THAN_OPERATOR:{
    retval = "\">\"";
    break;
  }
  case IIR_GREATER_THAN_OR_EQUAL_OPERATOR:{
    retval = "\">=\"";
    break;
  }
  case IIR_SLL_OPERATOR:{
    retval = "\"sll\"";
    break;
  }
  case IIR_SLA_OPERATOR:{
    retval = "\"sla\"";
    break;
  }
  case IIR_SRL_OPERATOR:{
    retval = "\"srl\"";
    break;
  }
  case IIR_SRA_OPERATOR:{
    retval = "\"sra\"";
    break;
  }
  case IIR_ROL_OPERATOR:{
    retval = "\"rol\"";
    break;
  }
  case IIR_ROR_OPERATOR:{
    retval = "\"ror\"";
    break;
  }
  case IIR_ADDITION_OPERATOR:{
    retval = "\"+\"";
    break;
  }
  case IIR_SUBTRACTION_OPERATOR:{
    retval = "\"-\"";
    break;
  }
  case IIR_CONCATENATION_OPERATOR:{
    retval = "\"&\"";
    break;
  }
  case IIR_MULTIPLICATION_OPERATOR:{
    retval = "\"*\"";
    break;
  }
  case IIR_DIVISION_OPERATOR:{
    retval = "\"/\"";
    break;
  }
  case IIR_MODULUS_OPERATOR:{
    retval = "\"mod\"";
    break;
  }  
  case IIR_REMAINDER_OPERATOR:{
    retval = "\"rem\"";
    break;
  }  
  case IIR_EXPONENTIATION_OPERATOR:{
    retval = "\"**\"";
    break;
  }  
  default:{
    cerr << "Unknown operator type in IIRScram_DyadicOperator::_get_function_name()";
    abort();
  }
  }
  return retval;
}

IIR * 
IIRScram_DyadicOperator::_transmute() {
  IIR_FunctionCall* functioncall = new IIR_FunctionCall;
  IIR_AssociationElementByExpression* associationelement;

  copy_location( this, functioncall );
  functioncall->set_implementation(get_implementation());

  associationelement = new IIR_AssociationElementByExpression;
  copy_location( this, associationelement );
  associationelement->set_actual(get_left_operand());
  functioncall->parameter_association_list.append(associationelement);

  associationelement = new IIR_AssociationElementByExpression;
  copy_location( this, associationelement );
  associationelement->set_actual(get_right_operand());
  functioncall->parameter_association_list.append(associationelement);

  return functioncall;
}

IIR_AssociationList *
IIRScram_DyadicOperator::_build_argument_list(){
  IIR_AssociationList *retval = new IIR_AssociationList;
  //  copy_location( this, retval );
  
  IIR_AssociationElementByExpression *new_assoc = new IIR_AssociationElementByExpression();
  copy_location( get_left_operand(), new_assoc );
  new_assoc->set_actual( get_left_operand() );
  retval->append( new_assoc );

  new_assoc = new IIR_AssociationElementByExpression();
  copy_location( get_right_operand(), new_assoc );
  new_assoc->set_actual( get_right_operand() );
  retval->append( new_assoc );

  return retval;
}

void
IIRScram_DyadicOperator::_publish_cc_headers( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_DyadicOperator::_publish_cc_headers" );

  ASSERT(get_left_operand() != NULL);
  ASSERT(get_right_operand() != NULL);
  get_left_operand()->_publish_cc_headers( _cc_out );
  get_right_operand()->_publish_cc_headers( _cc_out );
}



void
IIRScram_DyadicOperator::_publish_cc_short_circuit_positive( published_file &_cc_out,
							     void (IIR::*_publish_function)( published_file & )){
    if(get_subtype()->_is_bit_type()) {
    _cc_out << "((";
    (get_left_operand()->*_publish_function)( _cc_out );
    _cc_out << " == EnumerationType::getEnumerationOne())"
	    << " ? EnumerationType::getEnumerationOne() : "
	    << "EnumerationType::getEnumerationOne()."
	    << _get_cc_operator_name() << "(";
    (get_right_operand()->*_publish_function)( _cc_out );
    _cc_out << "))";
  }
  else {
    _cc_out << "((";
    (get_left_operand()->*_publish_function)( _cc_out );
    _cc_out << " == EnumerationType::getEnumerationTrue())"
	    << " ? EnumerationType::getEnumerationTrue() : "
	    << "EnumerationType::getEnumerationTrue()." 
	    << _get_cc_operator_name() << "(";
    (get_right_operand()->*_publish_function)( _cc_out );
    _cc_out << "))";
  }
}

void
IIRScram_DyadicOperator::_publish_cc_short_circuit_negative( published_file &_cc_out,
							     void (IIR::*_publish_function)( published_file & )){
    if(get_subtype()->_is_bit_type()) {
    _cc_out << "((";
    (get_left_operand()->*_publish_function)( _cc_out );
    _cc_out << " == EnumerationType::getEnumerationOne())"
	    << " ? EnumerationType::getEnumerationZero() : "
	    << "EnumerationType::getEnumerationZero()."
	    << _get_cc_operator_name() << "(";
    (get_right_operand()->*_publish_function)( _cc_out );
    _cc_out << "))";
  }
  else {
    _cc_out << "((";
    (get_left_operand()->*_publish_function)( _cc_out );
    _cc_out << " == EnumerationType::getEnumerationTrue())"
	    << " ? EnumerationType::getEnumerationFalse() : "
	    << "EnumerationType::getEnumerationFalse()." 
	    << _get_cc_operator_name() << "(";
    (get_right_operand()->*_publish_function)( _cc_out );
    _cc_out << "))";
  }
}

// void
// IIRScram_DyadicOperator::_publish_cc_short_circuit( published_file &_cc_out ){
//   _publish_cc_short_circuit( _cc_out, &IIRScram::_publish_cc_rvalue );
// }

// void
// IIRScram_DyadicOperator::_publish_cc_wait_data_short_circuit( published_file &_cc_out ){
//   _publish_cc_short_circuit( _cc_out, &IIRScram::_publish_cc_wait_data );
// }

void
IIRScram_DyadicOperator::_publish_cc_lvalue( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_DyadicOperator::_publish_cc_lvalue" );
  _publish_cc_xvalue( _cc_out, &IIRScram::_publish_cc_lvalue );
}

void
IIRScram_DyadicOperator::_publish_cc_rvalue( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_DyadicOperator::_publish_cc_rvalue" );
  _publish_cc_xvalue( _cc_out, &IIRScram::_publish_cc_rvalue );
}

void
IIRScram_DyadicOperator::_publish_cc_xvalue( published_file &_cc_out,
					     void (IIR::*_publish_function)( published_file & )) {
  SCRAM_CC_REF( _cc_out, "IIRScram_DyadicOperator::_publish_cc_xvalue" );

  (get_left_operand()->*_publish_function)( _cc_out );
  _cc_out << "." << _get_cc_operator_name() << OS("(");
  (get_right_operand()->*_publish_function)( _cc_out );
  _cc_out << CS( ")" );
}

void
IIRScram_DyadicOperator::_publish_cc_wait_data( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_DyadicOperator::_publish_cc_wait_data" );
  _publish_cc_xvalue( _cc_out, &IIRScram::_publish_cc_wait_data );
}

void
IIRScram_DyadicOperator::_publish_cc_ams_function(published_file &_cc_out) {
  _cc_out << "  equationNode *node" <<_stmt_node_index;
  _cc_out << " = new equationNode('O',";
  switch(get_kind()) {
  case IIR_ADDITION_OPERATOR:{
    _cc_out<<1;
    break;
  }
  case IIR_SUBTRACTION_OPERATOR:{
    _cc_out<<2;
    break;
  }
  case IIR_MULTIPLICATION_OPERATOR:{
    _cc_out<<3;
    break;
  }
  case IIR_DIVISION_OPERATOR:{
    _cc_out<<4;
    break;
  }
  default:{
    cerr<<"Unknown operator type in IIRScram_DyadicOperator::_get_function_name()";
    abort();
  }
  }
  _cc_out << ",";
  _cc_out << "node" <<(int)(_stmt_node_index/2);
  _cc_out << ");" << NL();
  get_left_operand()->_publish_cc_ams_function(_cc_out);
  get_right_operand()->_publish_cc_ams_function(_cc_out);
}

void
IIRScram_DyadicOperator::_publish_cc_ams_operator_name( published_file & ) {
  _report_undefined_scram_fn("_publish_cc_ams_operator_name()");
}

void
IIRScram_DyadicOperator::_publish_cc_ams_function_call_in_simult_stmt(published_file &_cc_out) {
  get_left_operand()->_publish_cc_ams_function_call_in_simult_stmt(_cc_out);
  get_right_operand()->_publish_cc_ams_function_call_in_simult_stmt(_cc_out);
}

void
IIRScram_DyadicOperator::_get_list_of_input_signals( set<IIR> *list ){
  get_left_operand()->_get_list_of_input_signals(list);
  get_right_operand()->_get_list_of_input_signals(list);
}

void
IIRScram_DyadicOperator::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  get_left_operand()->_build_sensitivity_list(sensitivity_list);
  get_right_operand()->_build_sensitivity_list(sensitivity_list);
}

IIR_Boolean
IIRScram_DyadicOperator::_is_readable(){
  if( get_left_operand() && get_left_operand()->_is_readable() == TRUE &&
      get_right_operand() && get_right_operand()->_is_readable() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_DyadicOperator::_is_resolved(){
  if( get_implementation() != NULL && get_implementation()->_is_resolved() == FALSE ){
    return FALSE;
  }
  ASSERT( get_left_operand() != NULL );
  ASSERT( get_right_operand() != NULL );

  if( get_left_operand()->_is_resolved() == TRUE 
      && get_right_operand()->_is_resolved() == TRUE
      && get_subtype() != NULL ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void 
IIRScram_DyadicOperator::_add_decl_into_cgen_symbol_table() {
  get_left_operand()->_add_decl_into_cgen_symbol_table();
  get_right_operand()->_add_decl_into_cgen_symbol_table();
}

void
IIRScram_DyadicOperator::_clone( IIR *copy_into ){
  ASSERT( copy_into->_is_iir_dyadic_operator() == TRUE );
  
  IIR_DyadicOperator *as_dyadic = (IIR_DyadicOperator *)copy_into;

  IIR *cloneop;

  // Copy the base info
  IIR_Expression::_clone( copy_into );

  // Implementation
  as_dyadic->set_implementation( get_implementation() );
  
  // Left
  cloneop = get_left_operand()->_clone();
  as_dyadic->set_left_operand( cloneop );

  // Right
  cloneop = get_right_operand()->_clone();
  as_dyadic->set_right_operand( cloneop );
}

IIR_Boolean
IIRScram_DyadicOperator::_is_static_expression() {
  if ((get_left_operand()->_is_static_expression() == TRUE) &&
      (get_right_operand()->_is_static_expression() == TRUE)) {
    return TRUE;
  }
  return FALSE;
}

IIR_Boolean
IIRScram_DyadicOperator::_is_operator() { return TRUE; }

IIR_Boolean
IIRScram_DyadicOperator::_is_locally_static_primary() {
  ASSERT( _is_resolved() == TRUE );

  if ( get_left_operand()->_is_locally_static_primary() == TRUE &&
       get_right_operand()->_is_locally_static_primary() == TRUE ) {
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void
IIRScram_DyadicOperator::_publish_vhdl(ostream &_vhdl_out){
  bool left_bracket = FALSE;
  bool right_bracket = FALSE ;
  OperatorPrecedenceLevel left_precedence = OTHERS;
  OperatorPrecedenceLevel right_precedence = OTHERS;
  OperatorPrecedenceLevel self_precedence = OTHERS;
  
  left_precedence = get_left_operand()->_get_operator_precedence();
  right_precedence = get_right_operand()->_get_operator_precedence();
  self_precedence = _get_operator_precedence();

  if( (get_left_operand()->get_subtype() != NULL) && 
      (get_right_operand()->get_subtype() != NULL) ){
    if( ((get_left_operand()->get_subtype())->_is_physical_type() == TRUE )||
	((get_right_operand()->get_subtype())->_is_physical_type() == TRUE ) ){
      _vhdl_out << "(";
    }
  }
  if( ( left_precedence < self_precedence ) || (get_left_operand()->_is_relational_operator() == TRUE) ){
    left_bracket = TRUE ;
  }
  if( ( get_left_operand()->get_kind() == get_kind())  && (_is_left_associative() == FALSE ) ){
    left_bracket = TRUE ;
  }
  if( ( left_precedence == self_precedence ) && ( _is_logical_operator() == TRUE ) && ( get_kind() != get_left_operand()->get_kind() ) ){
    left_bracket = TRUE ;
  }
  if (( right_precedence < self_precedence ) || (get_right_operand()->_is_relational_operator() == TRUE)){
    right_bracket = TRUE ;
  } 
  if ((right_precedence == self_precedence) && (_is_associative(get_right_operand()->get_kind()) == FALSE)) {
    right_bracket = TRUE;
  }
  if ( left_bracket == TRUE ){
    _vhdl_out << "(";
  }
  get_left_operand()->_publish_vhdl(_vhdl_out);
  if ( left_bracket == TRUE ){
    _vhdl_out << ")";
  }
  _publish_vhdl_operator(_vhdl_out);
  if ( right_bracket == TRUE ){
    _vhdl_out << "(";
  }
  get_right_operand()->_publish_vhdl(_vhdl_out);
  if ( right_bracket == TRUE ){
    _vhdl_out << ")";
  }
  if ((get_left_operand()->get_subtype() != NULL) && 
      (get_right_operand()->get_subtype() != NULL)){
    if (((get_left_operand()->get_subtype())->_is_physical_type() == TRUE )||
	((get_right_operand()->get_subtype())->_is_physical_type() == TRUE )) {
      _vhdl_out << ")";
    }
  }
}

void 
IIRScram_DyadicOperator::_type_check_operands(){
  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->interface_declarations.num_elements() == 2 );

  IIR_InterfaceDeclaration *first_argument_decl = 
    get_implementation()->interface_declarations.first();
  IIR_InterfaceDeclaration *second_argument_decl = 
    get_implementation()->interface_declarations.successor( first_argument_decl );
  
  IIR_TypeDefinition *left_type = first_argument_decl->get_subtype();
  IIR_TypeDefinition *right_type = second_argument_decl->get_subtype();

  set_left_operand( get_left_operand()->_semantic_transform( left_type ) );
  get_left_operand()->_type_check( left_type );
  set_left_operand( get_left_operand()->_rval_to_decl( left_type ) );

  set_right_operand( get_right_operand()->_semantic_transform( right_type ) );
  get_right_operand()->_type_check( right_type );
  set_right_operand( get_right_operand()->_rval_to_decl( right_type ) );

}

IIR *
IIRScram_DyadicOperator::_rval_to_decl( IIR_TypeDefinition *my_rval ){
  if( get_implementation() == NULL ){
    set_left_operand( get_left_operand()->_semantic_transform( my_rval ) );
    get_left_operand()->_type_check( my_rval );
    set_left_operand( get_left_operand()->_rval_to_decl( my_rval ) );
    
    set_right_operand( get_right_operand()->_semantic_transform( my_rval ) );
    get_right_operand()->_type_check( my_rval );
    set_right_operand( get_right_operand()->_rval_to_decl( my_rval ) );
  }
  set_subtype( my_rval );
  ASSERT( _is_resolved() == TRUE );
  return this;
}

void
IIRScram_DyadicOperator::_set_stmt_node_index(IIR_Int32 *index, bool _is_right_child, bool &reducibleFlag) {
  ASSERT(index!=NULL);
  if (_is_right_child == true) {
   _stmt_node_index = 2 * (*index)+ 1;
  }
  else {
   _stmt_node_index = 2 * (*index);
  }
  get_left_operand()->_set_stmt_node_index(&_stmt_node_index, false, reducibleFlag);
  get_right_operand()->_set_stmt_node_index(&_stmt_node_index, true, reducibleFlag);
}

void
IIRScram_DyadicOperator::_set_stmt_qty_index(IIR_Int32 *index,
                                             set<IIR_Declaration> *quantity_set) {
  get_left_operand()->_set_stmt_qty_index(index, quantity_set);
  get_right_operand()->_set_stmt_qty_index(index, quantity_set);
}
  
void
IIRScram_DyadicOperator::_set_stmt_signal_index(IIR_Int32 *index,
                                                set<IIR_Declaration> *signal_set) {
  get_left_operand()->_set_stmt_signal_index(index, signal_set);
  get_right_operand()->_set_stmt_signal_index(index, signal_set);
}

void
IIRScram_DyadicOperator::_flush_stmt_index() {
  get_left_operand()->_flush_stmt_index();
  get_right_operand()->_flush_stmt_index();
}

IIR_Boolean   
IIRScram_DyadicOperator::_contribution_quantity_found() {
  IIR_Boolean retval = FALSE;
  retval = get_left_operand()->_contribution_quantity_found() ||
    get_right_operand()->_contribution_quantity_found();
  return retval;
}
  
IIR_Boolean
IIRScram_DyadicOperator::_reference_quantity_found() {
  IIR_Boolean retval = FALSE;
  retval = get_left_operand()->_reference_quantity_found() ||
    get_right_operand()->_reference_quantity_found();
  return retval;
}

IIR_Boolean
IIRScram_DyadicOperator::_differential_quantity_found() {
  IIR_Boolean retval = FALSE;
  retval = get_left_operand()->_differential_quantity_found() ||
    get_right_operand()->_differential_quantity_found();
  return retval;
}

void
IIRScram_DyadicOperator::_build_contribution_quantity_list(dl_list<IIR_ContributionAttribute>
                                                           *contribution_quantity_list) {
  get_left_operand()->_build_contribution_quantity_list(contribution_quantity_list);
  get_right_operand()->_build_contribution_quantity_list(contribution_quantity_list);
}
 
void
IIRScram_DyadicOperator::_build_reference_quantity_list(dl_list<IIR_ReferenceAttribute>
                                                           *reference_quantity_list) {
  get_left_operand()->_build_reference_quantity_list(reference_quantity_list);
  get_right_operand()->_build_reference_quantity_list(reference_quantity_list);
}

void
IIRScram_DyadicOperator::_build_differential_quantity_list(dl_list<IIR_DotAttribute>
                                                           *differential_quantity_list) {
  get_left_operand()->_build_differential_quantity_list(differential_quantity_list);
  get_right_operand()->_build_differential_quantity_list(differential_quantity_list);
}

void
IIRScram_DyadicOperator::_build_generic_parameter_set(set<IIR_Declaration> *to_build) {
  get_left_operand()->_build_generic_parameter_set(to_build);
  get_right_operand()->_build_generic_parameter_set(to_build);
}

IIR_Boolean
IIRScram_DyadicOperator::_is_above_attribute_found() {
  IIR_Boolean retval = FALSE;
  retval = get_left_operand()->_is_above_attribute_found() ||
    get_right_operand()->_is_above_attribute_found();
  return retval;
}

void
IIRScram_DyadicOperator::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  get_left_operand()->_build_above_attribute_set(to_build);
 get_right_operand()->_build_above_attribute_set(to_build);
}
