
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Radharamanan Radhakrishnan
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam 
//          Swaminathan Subramanian
//	    Magnus Danielson    cfmd@swipnet.se

#include "IIR_DesignFile.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_Identifier.hh"
#include "scram.hh"
#include "set.hh"
#include "published_makefile.hh"
#include "published_cc_file.hh"
#include "sstream-wrap.hh"
#include "savant_config.hh"
#include "language_processing_control.hh"

extern char *entity_name, *arch_name, *config_name;
extern string design_library_name;
extern language_processing_control *lang_proc;
#ifdef PROCESS_GRAPH
extern bool signal_graph;
#endif

IIRScram_DesignFile::~IIRScram_DesignFile() {}

void 
IIRScram_DesignFile::_publish_vhdl(ostream &_vhdl_out) {
  comments._publish_vhdl(_vhdl_out);
  library_units._publish_vhdl(_vhdl_out);
}

void 
IIRScram_DesignFile::_publish_cc( bool last_file_this_run ) {
  _set_currently_publishing_unit(IIRScram::NONE);

  library_units._publish_cc();

  // We're assuming that the last file this run is the top level.
  _publish_cc_makefile( last_file_this_run );

  library_units._publish_cc_elaborate();
  if ( design_library_name == "" ){
    _publish_cc_main();
  }
}

void
IIRScram_DesignFile::_publish_cc_makefile( bool top_level ) {
  if( top_level == true ){
    published_makefile makefile( _get_work_library()->_get_path_to_directory(),
				 "Makefile",
				 this );

    SCRAM_MAKE_REF( makefile, "IIRScram_DesignFile::_publish_cc_makefile" );
    makefile << "# Begin publishing " << *get_name() << "\n";

    string objectname;
    if( _get_current_architecture_name() != "" ){
      objectname =  _get_current_architecture_name();
    }
    else {
      objectname = "_savant_entity_elab";
    }
  
    makefile << "SIMSRCS = $(wildcard *.cc)\n"
	     << "ifdef SAVANTROOT\n" 
	     << "  include $(SAVANTROOT)/lib/Makefile.common\n" 
	     << "else\n  include " << BUILD_SAVANTROOT << "/savant/lib/Makefile.common\n"
	     << "endif\n"
	     << "CPPFLAGS=$(SIMCPPFLAGS)\n"
	     << "CXXFLAGS=$(SIMCXXFLAGS)\n"
	     << "ALLSRCS=$(wildcard *.cc)\n"
	     << "ALLDOTOS=$(patsubst %.cc,%.o, $(SIMSRCS))\n"
	     << "ALLDOTLOS=$(patsubst %.cc,%.lo, $(SIMSRCS))\n";
    // include design libraries here...
    _publish_cc_library_data( makefile );

    makefile << "LIBS += $(LINK_THESE_LIBRARIES)\n\n";
    // Then we'll generate a magical "all" target.
    makefile << "all: ";
      
    if( design_library_name == "" ){
      makefile << objectname << "\n\n";
    }
    else{
      makefile << design_library_name 
	      << library_manager::get_library_suffix() << ".la\n\n";
    }
      
    makefile << "depend: cleandep\n"
	     << "\t makedepend -p$(VHDLDIR)/ -f- -- $(CPPFLAGS) $(CXXFLAGS)"
	     << " -- $(DEPENDINC) $(SIMSRCS) > .depend\n"
	
	     << "cleandep:\n"
	     << "\t-rm -f .depend\n\n"

	     << "clean:\n"
	     << "\trm -rf -f *~ \\#*\\# .libs;\n"
	     << "\trm -f *.a *.la *.lo *.so *.o\n"
      
	     << "-include .depend\n";
      
    // Now we'll generate how to build whatever this is we're talking about.
    string targetName = objectname;
    if( design_library_name == "" ){
      // Needed for running benchmarks...
      makefile << targetName << ": $(ALLDOTOS)\n"
	       << "\tlibtool $(CXX) $(CXXFLAGS) -static -rdynamic $^\\\n"
	       << "\t$(LIBVHDL) $(LIBTW) $(LIBCOMMON) $(LDFLAGS)\\\n"
	       << "\t-o $@ $(LIBS) \n\n";

    }
    else{
      targetName = design_library_name + library_manager::get_library_suffix();
    }

    makefile << targetName << ".la: $(ALLDOTLOS)\n"
	     << "\tlibtool $(CXX) $(CXXFLAGS) -module -rpath $(PWD) $^\\\n"
	     << "\t$(LIBVHDL) $(LIBTW) $(LIBCOMMON) $(LDFLAGS)\\\n"
	     << "\t-o $@ $(LIBS) \n\n";
    
    makefile << targetName << ".so: " << targetName << ".la" << "\n"
	     << "\t libtool install $^ $(PWD)\n\n";

    makefile << "# End publishing " << *get_name() << "\n";
  }
}

const string
IIRScram_DesignFile::_get_top_level_design_unit_name() {
  string objectname;
  if( _get_current_configuration_name() != "" ){
    objectname = "SCFG" 
      + _get_current_configuration_name() + "_"
      + _get_current_entity_name() + "_" 
      + _get_current_architecture_name() + "_elab";
  }
  else if( _get_current_entity_name() != "" && _get_current_architecture_name() != "" ){
    objectname = _get_current_architecture_name() + "_elab";
  }
  else {
    objectname = "_savant_entity_elab";
  }
  return objectname;
}

void
IIRScram_DesignFile::_publish_cc_main(){
  
  //Lot of work needed in this function
  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(),
			     "main",
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_DesignFile::_publish_cc_main" );

  _publish_cc_include( cc_file, _get_top_level_design_unit_name() + ".hh" );
  _publish_cc_include( cc_file, "warped/WarpedMain.h", true );
  _publish_cc_include( cc_file, "tyvis/VHDLApplication.hh", true );
  if (lang_proc->processing_vhdl_ams()) {
    _publish_cc_include( cc_file, "tyvis/AMSApplication.hh", true);
  }
  
  cc_file << "int" << NL()
	  << OS("main( int argc, char **argv ){")
	  << OS("WarpedMain wm(");
  if ( lang_proc->processing_vhdl_ams() ){
    cc_file << "new AMSApplication( new " << _get_top_level_design_unit_name() << " )";
  }
  else {
    cc_file << "new VHDLApplication( new " << _get_top_level_design_unit_name() << " )";
  }
  cc_file << CS(");")
	  << "wm.main( argc, argv );" << NL()
	  << CS("}");
}

void 
IIRScram_DesignFile::_set_file_name( const string &new_file_name ){
  set_name( IIR_Identifier::get( new_file_name ) );
}

#ifdef PROCESS_COMBINATION
IIR_ArchitectureDeclaration *
IIRScram_DesignFile::_static_elaborate_design() {
  IIR_LibraryUnit *lib_unit = library_units.first();
  IIR_EntityDeclaration *ent;
  // I sure hope component hierarchies aren't any longer than this!
  char hier_location[1024]; 

#ifdef DEBUG_ELAB
  cout << "Elaborating " << entity_name << " " << arch_name << " "
       << config_name << "\n";
#endif
  
  if (config_name != NULL) {
    cout << "Sorry, top-level configurations not supported yet\n";
    exit(1);
  }

  while (lib_unit != NULL) {
    if (lib_unit->get_kind() == IIR_ARCHITECTURE_DECLARATION) {
      if (!IIR_TextLiteral::_cmp(lib_unit->get_declarator(), arch_name)) {
	IIR_ArchitectureDeclaration *arch, *newarch;
	IIR_DeclarationList *cfglist = NULL;
	arch = (IIR_ArchitectureDeclaration*)lib_unit;

	ent = arch->get_entity();
	if (!IIR_TextLiteral::_cmp(ent->get_declarator(), entity_name)) {
	  // we've found the correct entity/arch pair now
	  hier_location[0] = 'v';
	  hier_location[1] = '_';
	  hier_location[2] = '\0';
	  newarch = new IIR_ArchitectureDeclaration;
	  arch->_clone(newarch);
	  newarch->set_entity(ent);
	  arch->_static_elaborate( newarch, cfglist, hier_location );
	  library_units._replace( arch, newarch );
	  return newarch;
	}
      }
    }
    lib_unit = library_units.successor(lib_unit);
  }
  cout << "Error: couldn't find specified design entity.\n";
  return NULL;
}
#endif

symbol_table *
IIRScram_DesignFile::_get_symbol_table(){
  ASSERT( _my_parser != NULL );
  ASSERT( _my_parser->get_symbol_table() != NULL );
  return _my_parser->get_symbol_table();
}

IIR_LibraryDeclaration *
IIRScram_DesignFile::_get_work_library(){
  ASSERT( _my_parser != NULL );
  return _my_parser->get_work_library();
}

void
IIRScram_DesignFile::_publish_cc_library_data( published_file &_cc_out ){
  set<IIR_LibraryDeclaration> *libraries = library_manager::instance()->get_libraries();
  
  ostringstream include_stream;
  ostringstream library_stream;
  
  include_stream << "LIBRARY_INCLUDES=";
  library_stream << "LINK_THESE_LIBRARIES=";
  IIR_LibraryDeclaration *current_library = libraries->get_element();
  while( current_library != NULL ){
    include_stream << " -I";
    library_stream << " ";
    current_library->_publish_cc_include_path( include_stream );
    if(	current_library != _get_work_library() ){
      current_library->_publish_cc_library_file_name( library_stream );
    }
    
    current_library = libraries->get_next_element();
  }
  include_stream << "\n";
  library_stream << " -lVHDL -lltdl ";
  if ( lang_proc->processing_vhdl_ams() ){
    library_stream << "-lAMS -lsparse -lad";
  }
  _cc_out << include_stream.str() << "\n";
  _cc_out << library_stream.str() << "\n";

  _cc_out << "\n\n";
  _cc_out << "CPPFLAGS += $(LIBRARY_INCLUDES)\n";

  delete libraries;
}

visitor_return_type *IIRScram_DesignFile::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_DesignFile(this, arg);
};
