
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur
//          Krishnan Subramani
//          Umesh Kumar V. Rajasekaran
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIR_Attribute.hh"
#include "IIR_AttributeDeclaration.hh"
#include "IIR_BranchQuantityDeclaration.hh"
#include "IIR_ConstantDeclaration.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_DesignatorList.hh"
#include "IIR_FreeQuantityDeclaration.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_Identifier.hh"
#include "IIR_IndexedName.hh"
#include "IIR_LeftAttribute.hh"
#include "IIR_RightAttribute.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_VariableInterfaceDeclaration.hh"
#include "IIR_PhysicalLiteral.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_EnumerationSubtypeDefinition.hh"
#include "IIR_PhysicalSubtypeDefinition.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_IntegerLiteral.hh"
#include "IIR_EnumerationLiteral.hh"
#include "StandardPackage.hh"
#include "error_func.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "resolution_func.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"
#include "IIR_ScalarNatureDefinition.hh"

extern symbol_table *cgen_sym_tab_ptr;

IIRScram_Attribute::~IIRScram_Attribute() {}

IIR *
IIRScram_Attribute::_clone(){
  _report_undefined_scram_fn("_clone()");  
  return NULL;
}

void 
IIRScram_Attribute::set_suffix( IIR * ){
  ostringstream err;
  err << "Internal error - IIRScram_Attribute::set_suffix called."
      << "The instantiated node"
      << " was a " << get_kind_text() << ", which apparently has no suffix.";
	
  report_error( this, err.str() );
  abort();
}

IIR *
IIRScram_Attribute::get_suffix( ){
  return NULL;
}

void 
IIRScram_Attribute::_resolve_suffix(){
  if( get_suffix() == NULL || get_suffix()->_is_resolved() == TRUE ) {
    return;
  }
  else{
    // Can't do much more here.  Some of the classes have limitations on
    // the types you can put in the suffix.
		
    // This will call the correct virtual method in the derived class.
    _resolve_suffix_special();
  }
}

void 
IIRScram_Attribute::_resolve_suffix_special(){
  _report_undefined_scram_fn("_resolve_suffix_special()");
}

IIR_TextLiteral *
IIRScram_Attribute::_build_attribute_name(){
  _report_undefined_scram_fn("_build_attribute_name()");
  return NULL;
}

IIR_RangeTypeDefinition *
IIRScram_Attribute::_build_range_type(){
  // Common code between range and reverse range.
  IIR_RangeTypeDefinition *retval = new IIR_RangeTypeDefinition();
  copy_location( this, retval );
  IIR_LeftAttribute *left = new IIR_LeftAttribute();
  copy_location( this, left );
  IIR_RightAttribute *right = new IIR_RightAttribute();
  copy_location( this, right );
	
  left->set_prefix( ((IIR *)get_prefix())->_clone() );
  right->set_prefix( ((IIR *)get_prefix())->_clone() );
  IIR_TypeDefinition *subtype = get_subtype();
  ASSERT( subtype != NULL );
  if( get_kind() == IIR_RANGE_ATTRIBUTE ){
    retval->set_left( left );
    retval->set_right( right );
    if( subtype->_get_direction() == NULL || subtype->_is_ascending_range() == TRUE ){
      // This is the "true" literal
      retval->set_direction( StandardPackage::get_boolean_type()->get_right() );
    }
    else{
      // This is the "false" literal
      retval->set_direction( StandardPackage::get_boolean_type()->get_left() );
    }
  }
  else{
    ASSERT( get_kind() == IIR_REVERSE_RANGE_ATTRIBUTE );
    retval->set_left( right );
    retval->set_right( left );
    if( subtype->_get_direction() == NULL || subtype->_is_ascending_range() == TRUE ){
      // This is the "false" literal
      retval->set_direction( (IIR *)StandardPackage::get_false_literal() );
    }
    else{
      // This is the "true" literal
      retval->set_direction( (IIR *)StandardPackage::get_true_literal() );
    }
  }
	
  return retval;
}

IIR_TextLiteral *
IIRScram_Attribute::_get_attribute_name(){
  if( my_attribute_name == NULL ){
    my_attribute_name = _build_attribute_name();
    copy_location( this, my_attribute_name );
  }
  
  return my_attribute_name;
}

set<IIR_Declaration> *
IIRScram_Attribute::_symbol_lookup(){
  // We have a virtual function, "_get_attribute_name", that returns
  // to us the name of attribute that this instance created...    
  IIR_TypeDefinition *my_subtype = NULL;
	
  set<IIR_Declaration> *my_decls;
  my_decls = _get_attribute_name()->_symbol_lookup( &IIR::_is_iir_attribute_declaration );
  if( my_decls == NULL ){
    my_subtype = get_subtype();
    // Then this attribute hasn't been declared.  Now we need to create
    // a declaration for it...
    IIR_AttributeDeclaration *attribute_decl = new IIR_AttributeDeclaration();
    copy_location( this, attribute_decl );
		
    attribute_decl->set_declarator( _get_attribute_name() );
    attribute_decl->set_subtype( my_subtype );
    attribute_decl->_set_attribute( (IIR_Attribute *)this );
    attribute_decl->_add_declaration();
		
    my_decls = _get_attribute_name()->_symbol_lookup();
  }
  else{
    // OK, there was at least one declaration of an attribute with our 
    // name.  We need to find the one with OUR subtype...
		
    // Now, if there is no attribute declaration for whatever
    // attribute this is, we'll generate one.
    if( get_kind() != IIR_USER_ATTRIBUTE ){
      IIR_AttributeDeclaration *current_attribute_decl;
      current_attribute_decl = (IIR_AttributeDeclaration *)my_decls->get_element();
      while( current_attribute_decl != NULL ){
	if( get_subtype()->_is_compatible( current_attribute_decl->get_subtype() ) == 0 ){
	  my_decls->remove( current_attribute_decl );
	}
	current_attribute_decl = (IIR_AttributeDeclaration *)my_decls->get_next_element();
      }    
      
      if(my_decls->num_elements() == 0) {
				// Then this attribute hasn't been declared.  Now we need to create
				// a declaration for it...
	IIR_AttributeDeclaration *attribute_decl = new IIR_AttributeDeclaration();
	copy_location( this, attribute_decl );
	attribute_decl->set_declarator( _get_attribute_name() );
	attribute_decl->set_subtype( get_subtype() );
	attribute_decl->_set_attribute( (IIR_Attribute *)this );
	_get_symbol_table()->add_declaration( attribute_decl );
				//      my_decls = sym_tab->find_set( _get_attribute_name() );
	my_decls->add(attribute_decl);
      }
      //$$$ - end of my code 
    }
  }
	
  ASSERT( my_decls->num_elements() == 1 );
  IIR_Declaration *my_decl = my_decls->get_element();
  
  // So, we have a declaration for this attribute now.  What this attribute
  // represents is a signal, or a function or something.  We want
  // "_symbol_lookup" to return a reference to that "thing".  We'll take
  // the prefix "foo" in foo'quiet and generate something like foo_quiet,
  // and then use a virtual method to generate a declaration...
  ostringstream decl_name;
  
  if( get_prefix()->_is_iir_declaration() == true ){
    decl_name << *((IIR_Declaration *)get_prefix())->get_declarator() << "_" 
	      << *_get_attribute_name();
  }
  else{
    decl_name << *get_prefix() << "_" << *_get_attribute_name();
  }
	
  string decl_name_string = decl_name.str();
	
  set<IIR_Declaration> *attached_to = get_prefix()->_symbol_lookup();
	
  if( attached_to == NULL ){
    ostringstream err;
    err << *get_prefix() << " undefined";
    report_error( this, err.str() );
    return NULL;
  }
	
  IIR_Declaration *signal_decl = NULL;
  switch ( attached_to->num_elements() ) {
  case 1:{
    signal_decl = attached_to->get_element();
    break;
  }
  case 0:
  default:
    cerr << "Internal error in overload resolution in "
	 << "IIRScram_Attribute::_symbol_lookup() - got "
	 << attached_to->num_elements() << " possibilities." << endl;
    abort();
  }
	
  IIR_Declaration *implicit_decl = signal_decl->_find_in_implicit_list( decl_name_string );
  
  if( implicit_decl == NULL ){
    IIR_TypeDefinition *my_type = my_decl->get_subtype();
		
    implicit_decl = _get_implicit_declaration( decl_name_string, my_type );
    implicit_decl->_set_attribute_name( (IIR_Attribute *)this );
    if( signal_decl->_get_implicit_declarations() == NULL ){
      signal_decl->_set_implicit_declarations( new set<IIR_Declaration> );
    }
    signal_decl->_get_implicit_declarations()->add( implicit_decl );
  }
  
  delete my_decls;
  delete attached_to;
	
  _set_implicit_declaration( implicit_decl );
 
  set<IIR_Declaration> *retval = new set<IIR_Declaration>( implicit_decl );
	
  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_Attribute::_get_rval_set( IIR_Boolean (IIR::*)() ){
  IIR_TypeDefinition *rval = get_subtype();
  set<IIR_TypeDefinition> *my_rvals = NULL;
	
  if( rval != NULL ){
    my_rvals = new set<IIR_TypeDefinition>( rval );
  }
  return my_rvals;
}

IIR_TypeDefinition *
IIRScram_Attribute::_get_prefix_subtype( IIR_Boolean (IIR::*constraint_function)() ){
  IIR_TypeDefinition *retval = NULL;
  if( get_prefix()->_is_resolved() == true ){
    retval = get_prefix()->get_subtype();
  }
  else{
    set<IIR_TypeDefinition> *prefix_types = get_prefix()->_get_rval_set( constraint_function );
    if( prefix_types == NULL ){
      report_undefined_symbol( get_prefix() );
      return NULL;
    }
    switch( prefix_types->num_elements() ){
    case 0:{
      ostringstream err;
      err << "|" << *get_prefix()
	  << "| has no definition appropriate for use as a prefix "
	  << "to attribute " << *_get_attribute_name();
      report_error( this, err.str() );
      break;
    }
    case 1:{
      retval = prefix_types->get_element();
      break;
    }
    default: {
      ostringstream err;
      err << "Cannot disambiguate |" << *get_prefix() << "| in it's usage.";
      report_error( this, err.str() );
    }
    }
    delete prefix_types;
  }
  return retval;
}

void 
IIRScram_Attribute::_type_check( set<IIR_TypeDefinition> * ){
  // We need to resolve our prefix. 
  _resolve_prefix();
  if( _has_suffix() == TRUE ){
    _resolve_suffix();
  }
  ASSERT( _is_resolved() == TRUE );
}

void 
IIRScram_Attribute::_resolve_prefix(){
  bool done = false;
  
  // The prefix can be a library unit and therefore not have a return type.
  // It can also be an indexed name and not have a declaration.  Therefore,
  // we have to check for both...
  set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup();
  if( prefix_decls != NULL ){
    switch( prefix_decls->num_elements() ){
    case 1:{
      set_prefix( get_prefix()->_decl_to_decl( prefix_decls->get_element() ) );
      done = true;
      break;
    }
    default:{
      // We're not done and we'll fall through.
    }
    }
  }
  set<IIR_TypeDefinition> *prefix_rvals = NULL;
  if( done == false ){
    prefix_rvals = get_prefix()->_get_rval_set();
    
    if( prefix_rvals == NULL ){
      report_undefined_symbol( get_prefix() );
      goto finish;
    }
    set_prefix( get_prefix()->_semantic_transform( prefix_rvals ) );
    
    delete prefix_rvals;
    prefix_rvals = get_prefix()->_get_rval_set();
    
    if( prefix_rvals == NULL || prefix_rvals->num_elements() == 0 ){
      report_undefined_symbol( get_prefix() );
      goto finish;
    }
    
    if( prefix_rvals->num_elements() > 1 ){
      report_ambiguous_error( get_prefix(), prefix_rvals );
      goto finish;
    }
    get_prefix()->_type_check( prefix_rvals );  
    set_prefix( get_prefix()->_rval_to_decl( prefix_rvals->get_element() ) );
    ASSERT( get_prefix()->_is_resolved() == TRUE );
    
    if( get_prefix()->get_kind() == IIR_FUNCTION_CALL ){
      IIR_FunctionCall *as_func_call = (IIR_FunctionCall *)get_prefix();
      ASSERT( as_func_call->get_implementation() != NULL );
      IIR_Declaration *new_prefix = as_func_call->get_implementation();
      delete get_prefix();
      set_prefix( new_prefix );
    }
  }
 finish:
  delete prefix_rvals;
  delete prefix_decls;
}

IIR *
IIRScram_Attribute::_decl_to_decl( IIR_Declaration * ){
  // So, we're being told our declaration is "my_decl".  We don't really
  // care, but we DO need to make that our prefix is resolved.  We probably
  // need to do the exact same stuff in rval_to_decl.  This should be put
  // into a function "resolve_prefix".
  _resolve_prefix();
  if( _has_suffix() == TRUE ){
    _resolve_suffix();
  }
  return this;
}

IIR_Declaration *
IIRScram_Attribute::_get_implicit_declaration( const string &, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_get_implicit_declaration( const string &, IIR_TypeDefinition * )");
  return NULL;
}

IIR *
IIRScram_Attribute::_rval_to_decl( IIR_TypeDefinition *resolved_rval ){
  IIRScram_Attribute *retval = this;
  
  if( get_prefix()->_is_resolved() == TRUE ){
    ostringstream imp_name;
    imp_name << *get_prefix();
    imp_name << "_";
    imp_name << *_get_attribute_name();
    
    ASSERT( get_suffix() == NULL || get_suffix()->_is_resolved() == TRUE );
    
    string decl_name = imp_name.str();
    IIR_Declaration *implicit_decl = get_prefix()->_find_in_implicit_list( decl_name );
    
    if( implicit_decl == NULL ){
      // Then we need to add it!
      implicit_decl = _get_implicit_declaration( decl_name, resolved_rval );
      
      if( get_prefix()->_get_implicit_declarations() == NULL ){
	get_prefix()->_set_implicit_declarations( new set<IIR_Declaration> );
      }
      get_prefix()->_get_implicit_declarations()->add( implicit_decl );
    }
    implicit_decl->_set_attribute_name( (IIR_Attribute *)this );
    _set_implicit_declaration( implicit_decl );
    
    ASSERT( retval != NULL );
    
    if( _need_to_process_prefix_prefix() == TRUE ){
      _process_prefix_prefix( resolved_rval );
    }
  }
  return retval;
}

IIR_Boolean 
IIRScram_Attribute::_is_resolved(){
  IIR_Boolean retval = TRUE;
  if( get_prefix()->_is_resolved() == FALSE || 
      ( get_suffix() != NULL && get_suffix()->_is_resolved() == FALSE ) ){
    retval = FALSE;
  }
  return retval;
}

void
IIRScram_Attribute::_publish_cc_necessary_decl_in_state_last_event( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_decl_in_state_last_event" );

  ASSERT( get_prefix() != NULL );
  ASSERT( get_prefix()->_is_resolved() == TRUE );
  ASSERT( get_prefix()->_is_signal() == TRUE || get_prefix()->_is_attribute() == TRUE );
  
  if( !get_prefix()->_is_published_attribute_in_state( IIR_Declaration::LAST_EVENT )){
    if( get_prefix()->_is_scalar_type() == TRUE) {
      _cc_out << "PhysicalType ";
    }
    else {
      _cc_out << get_prefix()->get_subtype()->_get_cc_type_name();
      _cc_out << "_lastevent ";
    }
    get_prefix()->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << "_last_event;" << NL();
    get_prefix()->_add_published_attribute_in_state(IIR_Declaration::LAST_EVENT);
  }
}

void
IIRScram_Attribute::_publish_cc_necessary_decl_in_state( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_decl_in_state" );
  _report_undefined_scram_fn("_publish_cc_necessary_decl_in_state( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_name_in_caps( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_name_in_caps" );
  _report_undefined_scram_fn("_publish_cc_name_in_caps( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_attrib_type( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_attrib_type" );
  _report_undefined_scram_fn("_publish_cc_attrib_type( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_necessary_decl_init( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_decl_init" );
  _report_undefined_scram_fn("_publish_cc_necessary_decl_init( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_init_val( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_init_val" );
  _report_undefined_scram_fn("_publish_cc_init_val( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_headers( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_headers" );
  if( get_prefix()->_is_type() == TRUE || get_prefix()->_is_object() == TRUE ) {
    get_prefix()->_publish_cc_include( _cc_out );
  }
}

void
IIRScram_Attribute::_publish_cc_necessary_copying( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_copying" );
  _report_undefined_scram_fn("_publish_cc_necessary_copying( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_necessary_attribute_copying( published_file &_cc_out,
							     SignalAttribute,
							     const char *object_suffix ) {
  IIR_Declaration *decl   = _get_actual_signal_declaration();
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_attribute_copying" );
  _cc_out << "this->";
  decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
  _cc_out << object_suffix << "  = s.";
  decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
  _cc_out << object_suffix << ";" << NL();
}

void
IIRScram_Attribute::_publish_cc_read_or_write( published_file &,
					       const string &, 
					       const string & ) {
  _report_undefined_scram_fn("_publish_cc_necessary_read_or_write(published_file &_cc_out, const char *, const char * )");
}

void
IIRScram_Attribute::_publish_cc_attribute_read_or_write( published_file &_cc_out,
							 SignalAttribute,
							 const string &objectSuffix,
							 const string &functionName,
							 const string &streamName) {
  IIR_Declaration *decl   = _get_actual_signal_declaration();

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_attribute_read_or_write" );

  decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
  _cc_out << "." << functionName << "(" << streamName << ");\n";
}

void 
IIRScram_Attribute::_publish_cc_bounds( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_bounds" );
  _cc_out << "defaultInfo";
}

void
IIRScram_Attribute::_publish_cc_sigtype( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_sigtype" );
  _report_undefined_scram_fn("_publish_cc_sigtype( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_type_cast( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_type_cast" );
  _report_undefined_scram_fn("_publish_cc_type_cast( _cc_out )");
}

void
IIRScram_Attribute::_publish_cc_type_attribute( published_file &_cc_out,
						const char *attributeString,
						IIR_Boolean checkSuffixFlag ){

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_type_attribute" );
  ASSERT(get_prefix() != NULL);
  ASSERT(get_prefix()->_is_resolved() == TRUE);
  
  if (get_prefix()->_is_array_type() == TRUE) {
    if (get_prefix()->_is_object() == TRUE) {
      if (get_prefix()->_is_access_type() == TRUE) {
	// publish the 'high of the object this access type is pointing
	IIR_AccessTypeDefinition *acc_type = (IIR_AccessTypeDefinition*)get_prefix()->get_subtype();
	if ((acc_type->get_designated_type()->_is_array_type() == TRUE) && (acc_type->get_designated_type()->_is_unconstrained_array_type() == FALSE)) {
	  _cc_out << "ArrayType::" << attributeString << OS("(");
	}
	else {
	  _cc_out << OS("(");
	  if(get_prefix()->_is_access_type() == TRUE) {
	    _cc_out << "(";
	    _cc_out << acc_type->get_designated_type()->_get_cc_type_name();
	    _cc_out << "&)";
	  }
	  get_prefix()->_publish_cc_lvalue( _cc_out );
	  _cc_out << ".all()" << CS(")") << "."
		  << attributeString << "_O" << OS("(");
	}
      }
      else {
	get_prefix()->_publish_cc_lvalue( _cc_out );
	_cc_out << "." << attributeString << "_O" << OS("(");
      }
    }
    else {
      _cc_out << get_prefix()->get_subtype()->_get_cc_type_name();
      _cc_out << "::" << attributeString << OS("(");
    }
    
    if(get_suffix() != NULL) {
      ASSERT(get_suffix()->_is_resolved() == TRUE);
      get_suffix()->_publish_cc_lvalue( _cc_out );
    }
    else {
      _cc_out << "IntegerType" << OS("(")
	      << "ObjectBase::VARIABLE," << NL()
	      << "UniversalInteger(1)," << NL()
	      << "SavantintegerType_info" 
	      << CS(")");
    }
    if ((get_prefix()->get_subtype()->_is_unconstrained_array_type() == FALSE)
	&& (get_prefix()->get_subtype()->_is_anonymous() == FALSE)
	&& (get_prefix()->_is_object() == FALSE)) {
      get_prefix()->get_subtype()->_publish_cc_object_type_info( _cc_out );
    }
    else {
      if ((get_prefix()->get_subtype()->_is_unconstrained_array_type()
	   == FALSE)
	  && (get_prefix()->get_subtype()->_is_anonymous() == FALSE)
	  && (get_prefix()->_is_access_type() == TRUE)) {
	IIR_AccessTypeDefinition *acc_type = (IIR_AccessTypeDefinition*) get_prefix()->get_subtype();
	_cc_out << "," << NL();
	_cc_out << acc_type->get_designated_type()->_get_cc_type_name();
	_cc_out << "_info";
      }
    }
    _cc_out << CS(")");
  }
  else {
    // Okay. The prefix is not a Array type!!
    ASSERT(get_prefix()->_is_type() == TRUE);
    // The check below is needed because we should not publish _info if the
    // attribute is referred to from one of the standard types    
    if (get_prefix()->_is_kernel_type() == TRUE) {
      if (checkSuffixFlag == FALSE) {
	_cc_out << get_prefix()->get_subtype()->_get_cc_type_name();
      }
      else {
	get_prefix()->get_subtype()->_publish_cc_kernel_type( _cc_out );
      }
      _cc_out << "::" << attributeString << OS("(");
      
      if (checkSuffixFlag == TRUE) {
	ASSERT (get_suffix() != NULL);
	ASSERT (get_suffix()->_is_resolved() == TRUE);
	if ((get_suffix()->_is_scalar_type() == TRUE) && 
	    (get_suffix()->_is_kernel_type() == FALSE) &&
	    (get_suffix()->_is_iir_declaration() == FALSE)) {
	  get_suffix()->_publish_cc_kernel_type( _cc_out );
	  _cc_out << OS("(") 
		  << "ObjectBase::VARIABLE," << NL();
	  get_suffix()->_publish_cc_lvalue( _cc_out );
	  _cc_out << CS(")");
	}
	else {
	  get_suffix()->_publish_cc_lvalue( _cc_out );
	  if (strcmp(attributeString, "VAL") == 0 ||
	      strcmp(attributeString, "POS") == 0) {
	    ASSERT(get_prefix()->get_subtype() != NULL);
	    get_prefix()->get_subtype()->_publish_cc_object_type_info( _cc_out );
	  }
	}
      }
      else {
	get_prefix()->get_subtype()->_publish_cc_object_type_info( _cc_out, FALSE );
      }
      _cc_out << CS(")");
    }
    else {
      // It is not from the standard types.
      get_prefix()->get_subtype()->_publish_cc_type_string( _cc_out );
      _cc_out << "::" << attributeString << OS("(");
      
      if (checkSuffixFlag == TRUE) {
	ASSERT (get_suffix() != NULL);
	ASSERT (get_suffix()->_is_resolved() == TRUE);
	if ((get_suffix()->_is_literal() == TRUE ||
	     get_suffix()->get_kind() == IIR_VARIABLE_DECLARATION)
	    && strcmp("VAL",attributeString) != 0
	    && strcmp("POS",attributeString) != 0){
	  if (get_prefix()->get_subtype()->_is_scalar_type() == TRUE &&
	      get_prefix()->get_subtype()->_is_kernel_type() == FALSE){
	    get_prefix()->get_subtype()->_publish_cc_kernel_type( _cc_out );
	  }
	  else{
	    _cc_out << get_prefix()->get_subtype()->_get_cc_type_name();
	  }
	  _cc_out << OS("(") 
		  <<"ObjectBase::VARIABLE," << NL();
	  if (get_suffix()->_is_literal() == TRUE){
	    switch(get_suffix()->get_kind()){
	    case IIR_INTEGER_LITERAL: 
	      _cc_out << "IntegerType";
	      break;
	    default:
	      break;
	    }
	  }
	  if (get_suffix()->get_kind() == IIR_VARIABLE_DECLARATION){
	    get_suffix()->get_subtype()->_publish_cc_type_string( _cc_out );
	  }
	  _cc_out << OS("(") 
		  << "ObjectBase::VARIABLE," << NL();
	  get_suffix()->_publish_cc_lvalue( _cc_out );
	  _cc_out << CS(")")
		  << "," << NL();
	  get_prefix()->get_subtype()->_publish_cc_lvalue( _cc_out );
	  _cc_out << "_info" 
		  << CS(")");
	}
	else {
	  if (get_suffix()->_is_kernel_type() == FALSE &&
	      get_suffix()->_is_scalar_type() == TRUE &&
	      get_suffix()->_is_iir_declaration() == FALSE){
	    get_suffix()->_publish_cc_kernel_type( _cc_out );
	    _cc_out << OS("(")
		    << "ObjectBase::VARIABLE, ";
	    get_suffix()->_publish_cc_lvalue( _cc_out );
	    _cc_out << CS(")");
	  }
	  else {
	    get_suffix()->_publish_cc_lvalue( _cc_out );
	  }
	}
	if (strcmp("VAL", attributeString) == 0 ||
	    strcmp("POS", attributeString) == 0) {
	  // _cc_out << ", "; // <-- breaks tc1327.vhd
	  get_prefix()->get_subtype()->_publish_cc_object_type_info( _cc_out );
	}
	_cc_out << CS(")");
      }	
      else {
	get_prefix()->get_subtype()->_publish_cc_object_type_info( _cc_out, FALSE );
	_cc_out << CS(")");
      }
    }
  }
}

void
IIRScram_Attribute::_check_and_add_necessary_decls() {
  IIR_Declaration         *declToCheck = _get_implicit_declaration();

  ASSERT ( get_prefix() != NULL );

  if (get_prefix()->get_kind() == IIR_INDEXED_NAME) {
    IIR_IndexedName *idxName = (IIR_IndexedName *) get_prefix();
    ASSERT ( idxName->get_kind() == IIR_INDEXED_NAME );
    ASSERT ( idxName->_get_prefix_declaration() != NULL );
      
    declToCheck = idxName->_get_prefix_declaration()->_get_implicit_declaration_for_attribute( (IIR_Attribute *) this );
  }
  

  if ((declToCheck != NULL) && (!cgen_sym_tab_ptr->in_scope(declToCheck))) {
    cgen_sym_tab_ptr->add_declaration(declToCheck);
  }
}

IIR_Declaration*
IIRScram_Attribute::_get_actual_signal_declaration() {
  ASSERT ( get_prefix() != NULL );
  ASSERT ( get_prefix()->_is_resolved() == TRUE );
  
  IIR_Declaration *decl   = NULL;
  IIR             *prefix = get_prefix();
  
  ASSERT ((prefix->_is_signal() == TRUE) || (prefix->_is_attribute() == TRUE));
  
  if (prefix->_is_signal() == TRUE) {
    if (prefix->_is_iir_name() == TRUE) {
      decl = (IIR_Declaration *) prefix->_get_prefix_declaration();
    }
    else {
      decl = (IIR_Declaration *) prefix;
    }
  }
  else {
    ASSERT (prefix->_is_attribute() == TRUE);
    decl = ((IIR_Attribute *) prefix)->_get_implicit_declaration();
  }

  ASSERT ( decl != NULL );
  ASSERT ( decl->_is_iir_declaration() == TRUE );
  ASSERT ( decl->get_subtype() != NULL );
  
  return decl;
}

void
IIRScram_Attribute::_publish_cc_signal_attribute( published_file &_cc_out,
						  const char *attributeString, 
						  IIR_Boolean checkSuffixFlag) {

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_signal_attribute" );

  ASSERT( get_prefix() != NULL );
  ASSERT(get_prefix()->_is_resolved() == TRUE );
  ASSERT(get_prefix()->_is_signal() == TRUE );
  ASSERT(_get_implicit_declaration() != NULL);
  ASSERT(_get_implicit_declaration()->_is_resolved() == TRUE);

  _check_and_add_necessary_decls();
  
  if(_is_currently_publishing_subprogram() == TRUE) {
    _cc_out << "processPtr->";
  }
  _cc_out << "locate" << attributeString << "Attribute(&(";
  get_prefix()->_publish_cc_lvalue( _cc_out );
  _cc_out << ")";

  _publish_cc_include( _cc_out, "tyvis/VHDLVTime.hh" );  
  if (checkSuffixFlag == TRUE) {
    _cc_out << ", ";
    if(get_suffix() != NULL) {
      ASSERT(get_suffix()->get_kind() == IIR_PHYSICAL_LITERAL);
      _cc_out << "VHDLVTime(";
      dynamic_cast<IIR_PhysicalLiteral *>(get_suffix())->_publish_cc_lvalue( _cc_out );
      _cc_out << ")";
    } else {
      _cc_out << "VHDLVTime::getVHDLVTimeZero()";
    }
  }
  
  _cc_out << ")";
}

void
IIRScram_Attribute::_publish_cc_necessary_signal_decl_in_state( published_file &_cc_out,
								SignalAttribute attrib, 
								const char *objectSuffix) {
  IIR_Declaration *decl   = _get_actual_signal_declaration();
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_signal_decl_in_state" );
  if ( decl->_is_published_attribute_in_state(attrib) == FALSE ){
    if( decl->_is_scalar_type() == TRUE ){
      _cc_out << _get_cc_signal_attribute_subelement_type( attrib );
    }
    else {
      _cc_out << decl->get_subtype()->_get_cc_type_name();
    }
    
    _cc_out << " ";
    
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << objectSuffix << ";" << NL();
    
    decl->_add_published_attribute_in_state(attrib);
  }
}

const string  
IIRScram_Attribute::_get_cc_signal_attribute_subelement_type( SignalAttribute attrib ){
  string retval;
  //  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_signal_attribute_subelement_type" );
  switch (attrib) {
  case EVENT:
  case ACTIVE:
  case STABLE:
  case QUIET:
  case TRANSACTION:
    retval = "EnumerationType";
    break;
  case LAST_EVENT:    
  case LAST_ACTIVE:
    retval = "PhysicalType";
    break;
  case LAST_VALUE:
    retval = _get_actual_signal_declaration()->get_subtype()->_get_cc_type_name();
    break;
  default:
    cerr << "Error : IIRScram_Attribute::_publish_cc_signal_attribute"
	 << "_subelement_type(SignalAttribute) - Unhandled case.\n";
    abort();
  }
  return retval;
}

void
IIRScram_Attribute::_publish_cc_signal_attribute_subelement_type_info( published_file &_cc_out,
								       SignalAttribute attrib ){

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_signal_attribute_subelement_type_info" );
  switch (attrib) {
  case EVENT:
  case ACTIVE:
  case STABLE:
  case QUIET:
    _cc_out << ", SavantbooleanType_info";
    break;
  case LAST_EVENT:    
  case LAST_ACTIVE:
    _cc_out << ", SavanttimeType_info";
    break;
  case LAST_VALUE:
    _get_actual_signal_declaration()->get_subtype()->_publish_cc_object_type_info( _cc_out );
    break;
  case TRANSACTION:
    _cc_out << ", SavantbitType_info";
    break;
  default:
    cerr << "Error : IIRScram_Attribute::_publish_cc_signal_attribute"
	 << "_subelement_type(SignalAttribute) - Unhandled case.\n";
  }
}

void
IIRScram_Attribute::_publish_cc_signal_attribute_subelement_type_suffix( published_file &_cc_out,
									 SignalAttribute attrib) {

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_signal_attribute_subelement_type_suffix" );
  switch (attrib) {
  case LAST_EVENT:
  case LAST_ACTIVE:
    _cc_out << "_lastevent";
    break;
  case EVENT:
  case STABLE:
  case ACTIVE:
  case QUIET:
    _cc_out << "_event";
    break;
  case LAST_VALUE:
  case TRANSACTION:
    break;
  default:
    cerr << "Error : IIRScram_Attribute::_publish_cc_signal_attribute"
	 << "_subelement_type_suffix(SignalAttribute) - Unhandled case.\n";
  }
}

void
IIRScram_Attribute::_publish_cc_necessary_signal_state_object_init( published_file &_cc_out,
								    SignalAttribute attrib, 
								    const char *objectSuffix, 
								    IIR_Boolean implicitSignal) {
  IIR_Declaration *decl = _get_actual_signal_declaration();
  if( decl->_is_published_attribute_in_constructor(attrib) == FALSE ){
    SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_signal_state_object_init" );
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << objectSuffix << "(ObjectBase::";
    
    if (implicitSignal == TRUE) {
      _cc_out << "IMPLICIT_SIGNAL";
    }
    else {
      _cc_out << "VARIABLE";
    }
    ASSERT ( decl->get_subtype() != NULL );
    ASSERT ( decl->get_subtype()->_is_iir_type_definition() == TRUE );
    
    IIR_TypeDefinition *st = decl->get_subtype();
    
    if ((st->_is_array_type() == TRUE) ||
	(st->_is_record_type() == TRUE)) {
      st->_publish_cc_object_type_info(_cc_out, TRUE, objectSuffix);
      _cc_out << ", ";
      st->_publish_cc_resolution_function_id(_cc_out);
    }
    if (st->_is_array_type() == TRUE) {
      if ((st->_is_unconstrained_array_type() == TRUE) ||
	  (st->_is_anonymous() == TRUE)) {
	_cc_out << ", ";
	st->_publish_cc_range( _cc_out );
      }
    }
    if (st->_is_scalar_type() == TRUE) {
      _publish_cc_signal_attribute_subelement_type_info( _cc_out, attrib );
    }
    _cc_out << "),\n";    
    decl->_add_published_attribute_in_constructor(attrib);
  }
}

void
IIRScram_Attribute::_publish_cc_necessary_signal_init( published_file &_cc_out,
						       SignalAttribute attrib, 
						       const char *objectSuffix, 
						       IIR_Boolean implicitSignal) {
  IIR_Declaration *decl = _get_actual_signal_declaration();
  int idx = 0;

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_necessary_signal_init" );
  
  if(decl->_is_published_attribute_in_initstate(attrib) == FALSE) {
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << objectSuffix << " = ";
    if (decl->_is_scalar_type() == TRUE) {
      _cc_out << _get_cc_signal_attribute_subelement_type( attrib )
	      << "(ObjectBase::";
      if (implicitSignal == FALSE) {
	_cc_out << "VARIABLE";
      }
      else {
	_cc_out << "IMPLICIT_SIGNAL";
      }
      _cc_out << ", ";
      _publish_cc_default_initial_universal_value( _cc_out, attrib );
      _publish_cc_signal_attribute_subelement_type_info( _cc_out, attrib );
      _cc_out << ");\n";
    }
    else {
      IIR_TypeDefinition *type = decl->get_subtype();
      do {
	idx++;
	_cc_out << type->_get_cc_type_name();
	_publish_cc_signal_attribute_subelement_type_suffix( _cc_out, attrib );
	_cc_out << "(ObjectBase::";
	if (implicitSignal == FALSE) {
	  _cc_out << "VARIABLE";
	}
	else {
	  _cc_out << "IMPLICIT_SIGNAL";
	}
	if ((type->_is_array_type() == TRUE) ||
	    (type->_is_record_type() == TRUE)) {
	  type->_publish_cc_object_type_info(_cc_out, TRUE, objectSuffix, TRUE);
	  _cc_out << ", ";
	  type->_publish_cc_resolution_function_id(_cc_out);
	}
	if (type->_is_array_type() == TRUE) {
	  if ((type->_is_unconstrained_array_type() == TRUE) ||
	      (type->_is_anonymous() == TRUE)) {
	    _cc_out << ", ";
	    type->_publish_cc_range( _cc_out );
	  }
	  type = type->_get_element_subtype();
	}
	_cc_out << ", 1, new ElementAssociation(Others, new ";
      } while (type->_is_array_type() == TRUE);
      
      _cc_out << _get_cc_signal_attribute_subelement_type( attrib )
	      << "(ObjectBase::";
      if (implicitSignal == FALSE) {
	_cc_out << "VARIABLE, ";
      }
      else {
	_cc_out << "IMPLICIT_SIGNAL, ";
      }      
      _publish_cc_default_initial_universal_value( _cc_out, attrib );
      _cc_out << "))";
      if (decl->_is_array_type() == TRUE) {
	while (idx > 1)  {
	  _cc_out << "))";
	  idx--;
	}
      }
      _cc_out << ");\n";
    }
    decl->_add_published_attribute_in_initstate(attrib);
  }
}

void
IIRScram_Attribute::_publish_cc_default_initial_universal_value( published_file &_cc_out,
								 SignalAttribute attrib ){

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_default_initial_universal_value" );
  switch (attrib) {
  case EVENT:
  case ACTIVE:
    _cc_out << "UniversalInteger(0)";
    break;
  case STABLE:
  case QUIET:
    _cc_out << "UniversalInteger(1)";
    break;
  case LAST_ACTIVE:
  case LAST_EVENT:
    _cc_out << "UniversalLongLongInteger(-1)";
    break;
  case LAST_VALUE: {
    IIR_Declaration *decl = _get_actual_signal_declaration();
    if (decl->get_subtype()->_is_scalar_type() == TRUE) {
      decl->get_subtype()->_publish_cc_universal_left( _cc_out );
    }
    else if (decl->get_subtype()->_is_array_type() == TRUE) {
      decl->get_subtype()->_get_element_subtype()->_publish_cc_universal_left( _cc_out );
    }
    break;
  }
  case TRANSACTION: 
    _cc_out << "UniveralInteger(0)";
    break;
  default:
    cerr << "Error : IIRScram_Attribute::_publish_cc_default_initial_value"
	 << "(SignalAttribute) - Unhandled case.\n";
  }
}
   
void
IIRScram_Attribute::_get_list_of_input_signals( set<IIR> *list ){
  if(get_prefix()->_is_signal() == TRUE) {
    get_prefix()->_get_list_of_input_signals(list);
  }
  else if(get_prefix()->_is_quantity() == TRUE &&
          _is_above_attribute() == TRUE) {
    _get_implicit_declaration()->_get_list_of_input_signals(list);
  }
}

void
IIRScram_Attribute::_add_decl_into_cgen_symbol_table() {
  if(get_prefix() != NULL) {
    get_prefix()->_add_decl_into_cgen_symbol_table();
  }
  if(get_suffix() != NULL) {
    get_suffix()->_add_decl_into_cgen_symbol_table();
  }

}

void
IIRScram_Attribute::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  if(_is_signal_attribute() == TRUE && _is_signal() == TRUE) {
    ASSERT(_get_implicit_declaration() != NULL);
    ASSERT(_get_implicit_declaration()->_is_resolved() == TRUE);
    //_get_implicit_declaration()->_build_sensitivity_list(sensitivity_list);
    IIR_Designator *node = sensitivity_list->first();
    while (node != NULL) {
      ASSERT(node->get_kind() == IIR_DESIGNATOR_EXPLICIT);
      if (((IIR_DesignatorExplicit *) node)->get_name() == this) {
	break;
      }
      node = sensitivity_list->successor(node);
    }
    if (node == NULL) {
      IIR_DesignatorExplicit *new_designator = new IIR_DesignatorExplicit;
      copy_location( this, new_designator );
      new_designator->set_name(this);
      sensitivity_list->append(new_designator);
    }
  }
  get_prefix()->_build_sensitivity_list(sensitivity_list);
  if(_has_suffix() == TRUE && get_suffix() != NULL) {
    get_suffix()->_build_sensitivity_list(sensitivity_list);
  }
}

IIR_Boolean
IIRScram_Attribute::_has_suffix() {
  return FALSE ;
}

void
IIRScram_Attribute::_publish_cc_init_fields_for_signals( published_file & ) {}

IIR_SignalDeclaration *
IIRScram_Attribute::_build_signal_declaration( const string &signal_name,
					       IIR_TypeDefinition *signal_type ){
  IIR_SignalDeclaration *retval;
  retval = new IIR_SignalDeclaration();
  copy_location( this, retval );
  retval->set_declarator( IIR_Identifier::get(signal_name));
  retval->set_subtype( signal_type );
  retval->_set_declarative_region( _get_current_declarative_region() );
  retval->_set_is_visible( FALSE );
  retval->_set_is_implicit( TRUE );
  if(get_kind() == IIR_ABOVE_ATTRIBUTE) { 
    copy_location(this, retval); 
    retval->_add_declaration();
  }
  return retval;
}

IIR_ConstantDeclaration *
IIRScram_Attribute::_build_constant_declaration( const string &constant_name,
						 IIR_TypeDefinition *constant_type ){
  IIR_ConstantDeclaration *retval;
  retval = new IIR_ConstantDeclaration;
  copy_location( this, retval );
  retval->set_declarator( IIR_Identifier::get( constant_name ));
  retval->set_subtype( constant_type );
  retval->_set_is_visible( FALSE );
  retval->_set_is_implicit( TRUE );
  return retval;
}


IIR_FunctionDeclaration *
IIRScram_Attribute::_build_function_declaration( const string &function_name,
						 IIR_TypeDefinition *return_value,
						 IIR_TypeDefinition *argument_type ){
  IIR_FunctionDeclaration *retval;
  retval = new IIR_FunctionDeclaration;
  copy_location( this, retval );
  retval->set_declarator( IIR_Identifier::get( function_name ));
  retval->set_return_type( return_value );
  retval->_set_declarative_region( _get_current_declarative_region() );
  retval->_set_is_visible( FALSE );
  retval->_set_is_implicit( TRUE );
  if( argument_type != NULL ){
    IIR_VariableInterfaceDeclaration *interface_decl = new IIR_VariableInterfaceDeclaration;
    copy_location( this, interface_decl );
    char *interface_name = "X";
    interface_decl->set_declarator( IIR_Identifier::get( interface_name, 
							 strlen( interface_name ) ) );
    interface_decl->set_subtype( argument_type  );
    retval->interface_declarations.append( interface_decl );
    retval->_set_is_visible( FALSE );
    retval->_set_is_implicit( TRUE );
  }
  return retval;
}

IIR_TypeDeclaration *
IIRScram_Attribute::_build_type_declaration( const string &decl_name, 
					     IIR_TypeDefinition *decl_type ){
  IIR_TypeDeclaration *retval = new IIR_TypeDeclaration();
  copy_location( this, retval );
  retval->set_type( decl_type );
  retval->set_declarator(IIR_Identifier::get( decl_name ));
  retval->_set_declarative_region( _get_current_declarative_region() );
  retval->_set_is_visible( FALSE );
  retval->_set_is_implicit( TRUE );
  return retval;
}

IIR_FreeQuantityDeclaration *
IIRScram_Attribute::_build_free_quantity_declaration( const string & quantity_name,
						      IIR_TypeDefinition *quantity_type){
  IIR_FreeQuantityDeclaration *retval = new IIR_FreeQuantityDeclaration;
  retval->set_declarator(IIR_Identifier::get(quantity_name));
  retval->set_value( ((IIR_FreeQuantityDeclaration*)get_prefix())->get_value() );
  retval->set_subtype( quantity_type );
  retval->_set_is_implicit( TRUE );
  return retval;
}

IIR_BranchQuantityDeclaration *
IIRScram_Attribute::_build_branch_quantity_declaration( const string &quantity_name,
                                                        IIR_TypeDefinition *quantity_type) {
  IIR_BranchQuantityDeclaration *retval = new IIR_BranchQuantityDeclaration;
  IIR_BranchQuantityDeclaration *my_decl = (IIR_BranchQuantityDeclaration *)get_prefix();
  retval->set_declarator(IIR_Identifier::get(quantity_name));
  retval->set_across_aspect_expression(my_decl->get_across_aspect_expression());
  retval->set_across_aspect_tolerance(my_decl->get_across_aspect_tolerance());
  retval->set_through_aspect_expression(my_decl->get_through_aspect_expression());
  retval->set_through_aspect_tolerance(my_decl->get_through_aspect_tolerance());
  retval->set_plus_terminal_name( my_decl->get_plus_terminal_name() );
  retval->set_minus_terminal_name( my_decl->get_minus_terminal_name() );
  retval->set_subtype(quantity_type);
  if(my_decl->_is_across_quantity()==TRUE) {
    retval->across_aspect_identifier_list.append(my_decl->across_aspect_identifier_list.first());
  }
  else {
    retval->through_aspect_designator_list.append(my_decl->through_aspect_designator_list.first());
  }
  retval->_set_is_implicit(TRUE);
  return retval;
}

void 
IIRScram_Attribute::_set_implicit_declaration( IIR_Declaration *implicit_decl ){
  implicit_declaration = implicit_decl;
}

IIR_Declaration *
IIRScram_Attribute::_get_implicit_declaration( ){
  return implicit_declaration;
}

void
IIRScram_Attribute::_publish_cc_implicit_signal_state_object_init( published_file &_cc_out ) {
  IIR_TypeDefinition* type_def = _get_explicit_signal_type();
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_implicit_signal_state_object_init" );
  ASSERT(_get_implicit_declaration()->_is_signal() == TRUE);
  _get_implicit_declaration()->_get_declarator()->_publish_cc_lvalue( _cc_out );
  if (this->get_kind() == IIR_ABOVE_ATTRIBUTE) {
      _cc_out << "(ObjectBase::SIGNAL,SavantbooleanType_info";
      if(type_def->_is_array_type() == TRUE &&
         type_def->_is_anonymous()) {
        _cc_out << ", ";
        type_def->_publish_cc_range(_cc_out);
      }
    }
  else {
      _cc_out << "(ObjectBase::IMPLICIT_SIGNAL";
      if ((type_def->_is_array_type() == TRUE) && 
	  (type_def->_is_anonymous() == TRUE)) {
	_cc_out << ", ";
	type_def->_publish_cc_range( _cc_out );
      }
  }
  _cc_out << "),\n";
}

void
IIRScram_Attribute::_publish_cc_implicit_signal_declaration( published_file &_cc_out ) {
  IIR_TypeDefinition* type_def = _get_explicit_signal_type();
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_implicit_signal_declaration" );
  if(type_def->_is_scalar_type() == TRUE) {
    _cc_out << _get_cc_signal_attribute_subelement_type();
  }
  else {
    type_def->_publish_cc_lvalue( _cc_out );
    _cc_out << _get_cc_signal_attribute_type_suffix();
  }
  _cc_out << " ";
  _get_implicit_declaration()->_get_declarator()->_publish_cc_lvalue( _cc_out );
  _cc_out << ";\n";
}

IIR_TypeDefinition*
IIRScram_Attribute::_get_explicit_signal_type() {
  IIR* prefix_decl = get_prefix();
  ASSERT((prefix_decl->_is_signal() == TRUE)  ||
         (prefix_decl->_is_attribute() == TRUE)||
         (prefix_decl->_is_quantity() == TRUE));

  if(prefix_decl->_is_signal() == TRUE) {
    return ((IIR_Declaration*)prefix_decl)->get_subtype();
  }
  else if(prefix_decl->_is_quantity() == TRUE) {
    return ( _get_implicit_declaration()->get_subtype() );
  }
  else {
    return ((IIR_Attribute*)prefix_decl)->_get_explicit_signal_type();
  }
}

const string
IIRScram_Attribute::_get_cc_signal_attribute_subelement_type( ){
  //  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_signal_attribute_subelement_type" );
  string retval;
  if((_is_quiet_attribute() == TRUE) ||
     (_is_active_attribute() == TRUE) ||
     (_is_stable_attribute() == TRUE)) {
    retval = "EnumerationType";
  }
  if(_is_transaction_attribute() == TRUE) {
    retval = "EnumerationType";
  }
  if(_is_delayed_attribute() == TRUE) {
    cerr << "Unimplemented Attribute 'DELAYED\n";
    abort();
  }
  if(_is_above_attribute() == TRUE) {  
    retval = "EnumerationType";
  }
  return retval;
}

void
IIRScram_Attribute::_publish_cc_declarator( published_file &_cc_out )
{
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_declarator" );

  //WORKAROUND #######SHOULD BE REMOVED ONCE PARSER FIXES THIS
  if(get_prefix()->_is_iir_declaration() == TRUE) {
    IIR_Declaration* decl = (IIR_Declaration*)get_prefix();
    if(decl->_get_declarative_region() != _get_implicit_declaration()->_get_declarative_region()) {
      if(_get_implicit_declaration()->_is_signal() == TRUE) {
	//	cerr << "work around for different declarative regions\nfor a signal and its attribute\n";
	_cc_out.get_stream() << 
	  *(_get_implicit_declaration()->_get_declarative_region()->_get_declarator());
	_cc_out << "_O";
	_cc_out.get_stream() << *(decl->get_declarator());
	_publish_cc_signal_attribute_suffix( _cc_out );
	return;
      }
    }
  }

  if (get_prefix()->get_kind() == IIR_INDEXED_NAME) {
    ASSERT ( get_prefix()->_get_prefix_declaration() != NULL );
    get_prefix()->_get_prefix_declaration()->_publish_cc_declarator( _cc_out );
  }
  else {
    get_prefix()->_publish_cc_declarator( _cc_out );
  }
  _publish_cc_signal_attribute_suffix( _cc_out );
}

void
IIRScram_Attribute::_publish_cc_signal_attribute_suffix( published_file &_cc_out ){
  //  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_signal_attribute_suffix" );
  ASSERT((_is_signal_attribute() == TRUE) || 
	 (_is_function_attribute() == TRUE));
  if (_is_quiet_attribute() == TRUE) {
    _cc_out << "_quiet";
  }
  if (_is_event_attribute() == TRUE) {
    _cc_out << "_event";
  }
  if (_is_lastevent_attribute() == TRUE) {
    _cc_out << "_last_event";
  }
  if (_is_active_attribute() == TRUE) {
    _cc_out << "_active";
  }
  if (_is_transaction_attribute() == TRUE) {
    _cc_out << "_transaction";
  }
  if (_is_stable_attribute() == TRUE) {
    _cc_out << "_stable";
  }
  if (_is_lastactive_attribute() == TRUE) {
    _cc_out << "_last_active";
  }
  if (_is_lastvalue_attribute() == TRUE) {
    _cc_out << "_last_value";
  }
  if(_is_above_attribute() == TRUE) {
    _cc_out << "_above";
  }    
}

const string  
IIRScram_Attribute::_get_cc_signal_attribute_type_suffix(){
  // Don't insert a scram_ref, this is a suffix
  string retval;
  ASSERT(_is_signal_attribute() == TRUE);
  if(_is_quiet_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_event";
    }
  }
  if(_is_stable_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_event";
    }
  }
  if(_is_transaction_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_transaction";
    }
  }
  if(_is_event_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_event";
    }
  }
  if(_is_active_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_event";
    }
  }
  if(_is_lastevent_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_lastevent";
    }
  }
  if(_is_lastactive_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_lastevent";
    }
  }
  if(_is_driving_attribute() == TRUE) {
    if(_get_explicit_signal_type()->_is_scalar_type() == FALSE) {
      retval = "_event";
    }
  }
  return retval;
}


void
IIRScram_Attribute::_clone( IIR *copy_into ){
  ASSERT( copy_into->_is_iir_attribute() == TRUE );
  IIR_Name::_clone( copy_into );
  IIR_Declaration *clonedecl;
  clonedecl = _get_implicit_declaration();
  if( clonedecl != NULL ){
    clonedecl = (IIR_Declaration*)clonedecl->_clone();
    ((IIR_Attribute *)copy_into)->_set_implicit_declaration(clonedecl);
  }
}

const string
IIRScram_Attribute::_get_cc_type_name(){
  string retval;

  IIR_TypeDefinition *type_def = _get_explicit_signal_type();
  ASSERT( type_def != 0 );
  //  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_type_nane" );
  if((type_def->_is_array_type() == TRUE) ||
     (type_def->_is_record_type() == TRUE)) {
    retval = type_def->_get_cc_type_name() + _get_cc_signal_attribute_type_suffix();
  }
  else {
    retval = _get_cc_signal_attribute_subelement_type();
  }
  return retval;
}

void
IIRScram_Attribute::_publish_cc_universal_value( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_universal_value" );
  ASSERT(_is_signal_attribute() == FALSE);
  ASSERT(this->get_kind() != IIR_RANGE_ATTRIBUTE);
  ASSERT(this->get_kind() != IIR_REVERSE_RANGE_ATTRIBUTE);
  _cc_out << "((";
  get_prefix()->_publish_cc_universal_type( _cc_out );
  _cc_out << "& )";
  _publish_cc_lvalue( _cc_out );
  _cc_out << ".readVal())";
}

void
IIRScram_Attribute::_publish_cc_universal_left( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_universal_left" );
  ASSERT(this->get_kind() == IIR_RANGE_ATTRIBUTE || this->get_kind() == IIR_REVERSE_RANGE_ATTRIBUTE);
  ASSERT(get_prefix() && (get_prefix()->_is_array_type() == TRUE));
  IIR_LeftAttribute *left_attr = new IIR_LeftAttribute();
  copy_location( this, left_attr );
  left_attr->set_prefix(get_prefix());
  left_attr->set_suffix(get_suffix());
  left_attr->_publish_cc_value( _cc_out );
  delete left_attr;
}

void
IIRScram_Attribute::_publish_cc_universal_right( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_universal_right" );
  ASSERT(this->get_kind() == IIR_RANGE_ATTRIBUTE || this->get_kind() == IIR_REVERSE_RANGE_ATTRIBUTE);
  ASSERT(get_prefix() && (get_prefix()->_is_array_type() == TRUE));
  IIR_RightAttribute *right_attr = new IIR_RightAttribute();
  copy_location( this, right_attr );
  right_attr->set_prefix(get_prefix());
  right_attr->set_suffix(get_suffix());
  right_attr->_publish_cc_value( _cc_out );
  delete right_attr;
}

void
IIRScram_Attribute::_publish_cc_value( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_value" );

  ASSERT(_is_signal_attribute() == FALSE);
  _publish_cc_universal_value( _cc_out );
}

void
IIRScram_Attribute::_publish_cc_as_for_loop( published_file &_cc_out, const char *forConstant ){
  IIR_ScalarTypeDefinition *range = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_as_for_loop" );
  
  if (get_prefix()->_is_array_type() == TRUE) {
    // Okay, extract the index constraint for this guy depending on the
    // suffix
    IIR_ArrayTypeDefinition *aType = (IIR_ArrayTypeDefinition *) get_prefix()->get_subtype();
    ASSERT ( aType->_get_index_subtype() != NULL );
    ASSERT ( aType->_get_index_subtype()->_is_scalar_type() == TRUE );
    range = (IIR_ScalarTypeDefinition *) aType->_get_index_subtype();
  }
  else {
    ASSERT ( get_prefix()->_is_scalar_type() == TRUE );
    if (get_prefix()->_is_object() == TRUE) {
      range = (IIR_ScalarTypeDefinition *) get_prefix()->get_subtype();
    }
    else {
      ASSERT ( get_prefix()->_is_iir_scalar_type_definition() == TRUE );
      range = (IIR_ScalarTypeDefinition *) get_prefix();
    }
  }
  
  ASSERT ( range != NULL );
  
  _cc_out << "for(int " << forConstant << " = ";

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << "0";
  }
  else {
    _cc_out << "(";
    range->get_left()->_publish_cc_universal_value( _cc_out );
    _cc_out << " - ";
    range->get_right()->_publish_cc_universal_value( _cc_out );
    _cc_out << ")";
  }
  
  _cc_out << "; (" << forConstant;
  
  if (range->_is_ascending_range() == TRUE) {
    _cc_out << " <= ";
  }
  else {
    _cc_out << " >= ";
  }

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << "(";
    range->get_right()->_publish_cc_universal_value( _cc_out );
    _cc_out << " - ";
    range->get_left()->_publish_cc_universal_value( _cc_out );
    _cc_out << ")";
  }
  else {
    _cc_out << "0";
  }

  _cc_out << "); ";

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << forConstant << "++";
  }
  else {
    _cc_out << forConstant << "--";
  }

  _cc_out << ")  ";
}

void 
IIRScram_Attribute::_resolve_suffix_base_type_of_prefix(){
  IIR *suffix_to_process = NULL;

  if( get_suffix() == NULL || get_suffix()->_is_resolved() == FALSE ){
    suffix_to_process = get_suffix();

    ASSERT( get_prefix()->_is_resolved() == TRUE );

    // The suffix must be an expression whose type is the base type of
    // of the prefix.
    IIR_TypeDefinition *prefix_type = get_prefix()->get_subtype();

    set<IIR_TypeDefinition> *suffix_types = suffix_to_process->_get_rval_set();
  
    IIR_TypeDefinition *current_type = suffix_types->get_element();
    while( current_type != NULL ){
      if( current_type->_is_compatible( prefix_type ) ){
	break;
      }
    
      current_type = suffix_types->get_next_element();
    }

    if( current_type != NULL ){
      ASSERT( current_type->_is_scalar_type() == TRUE );
      set_suffix( get_suffix()->_semantic_transform( current_type ) );
      get_suffix()->_type_check( current_type );
      set_suffix( get_suffix()->_rval_to_decl( current_type ) );
    }
    else{
      ostringstream err;
      err << "|" << *get_suffix() << "| is not a valid suffix for |" << *this << "|. ";
      err << "The suffix must be an expression whose type is the base type of the prefix.";
    
      report_error( this, err.str() );
    }
  }
}

void 
IIRScram_Attribute::_resolve_suffix_local_static_int(){
  
  // X a locally static expression of type universal integer, the
  // value of which must not exceed the dimensionality of of A
  // (defautls to 1)
  
  // Called by
  
  // IIR_LeftAttribute
  // IIR_RightAttribute
  // IIR_HighAttribute
  // IIR_LowAttribute
  // IIR_RangeAttribute
  // IIR_ReverseRangeAttribute
  // IIR_LengthAttribute
  // IIR_AscendingAttribute

  IIR *suffix_to_process = get_suffix();
  
  if( get_suffix() != NULL && get_suffix()->_is_resolved() == FALSE ){

    ASSERT( get_prefix()->_is_resolved() == TRUE );

    set<IIR_TypeDefinition> *suffix_types = suffix_to_process->_get_rval_set();
  
    IIR_TypeDefinition *current_type = suffix_types->get_element();
    while( current_type != NULL ){
      if( current_type == (IIR_TypeDefinition *)StandardPackage::get_savant_universal_integer() ){
	break;
      }
    
      current_type = suffix_types->get_next_element();
    }

    if( current_type != NULL ){
      _set_suffix( get_suffix()->_semantic_transform( current_type ) );
      get_suffix()->_type_check( current_type );
      _set_suffix( get_suffix()->_rval_to_decl( current_type ) );
    }
    else{
      ostringstream err;
      err << "|" << *get_suffix() << "| is not a valid suffix for |" << *this << "|.";
      err << " The suffix must be a locally static expression of type universal integer.";
    
      report_error( this, err.str() );
    }
    delete suffix_types;
  }
}

void 
IIRScram_Attribute::_resolve_suffix_non_negative_time(){
  // a static expression of type time that evaluates to a
  // nonnegative value
  
  // Used for:
  // IIR_DelayedAttribute
  // IIR_StableAttribute
  // IIR_QuietAttribute

  if( get_suffix() != NULL && get_suffix()->_is_resolved() == FALSE ){

    ASSERT( get_prefix()->_is_resolved() == TRUE );

    set<IIR_TypeDefinition> *suffix_types = get_suffix()->_get_rval_set();
    IIR_TypeDefinition *time_rval = StandardPackage::get_time_type();

    IIR_TypeDefinition *current_type = suffix_types->get_element();
    while( current_type != NULL ){
      if( current_type == time_rval ){
	break;
      }
    
      current_type = suffix_types->get_next_element();
    }

    if( current_type != NULL ){
      _set_suffix( get_suffix()->_semantic_transform( current_type ) );
      get_suffix()->_type_check( current_type );
      _set_suffix( get_suffix()->_rval_to_decl( current_type ) );
    }
    else{
      ostringstream err;
      err << "|" << *get_suffix() << "| is not a valid suffix for |" << *this << "|.";
      err << " The suffix must be a locally static expression of type universal integer.";
    
      report_error( this, err.str() );
    }
  }  
}

IIR_Boolean 
IIRScram_Attribute::_is_locally_static_primary(){
  ASSERT( get_prefix()->_is_resolved() == TRUE );
  if( _is_value_attribute() == TRUE &&
      get_kind() != IIR_USER_ATTRIBUTE && 
      get_kind() != IIR_PATH_NAME_ATTRIBUTE &&
      get_prefix()->_is_locally_static() == TRUE ){
    return TRUE;
  }
  else if( _is_function_attribute() == TRUE &&
	   get_kind() != IIR_USER_ATTRIBUTE &&
	   get_kind() != IIR_EVENT_ATTRIBUTE &&
	   get_kind() != IIR_ACTIVE_ATTRIBUTE &&
	   get_kind() != IIR_LAST_EVENT_ATTRIBUTE &&
	   get_kind() != IIR_LAST_ACTIVE_ATTRIBUTE &&
	   get_kind() != IIR_LAST_VALUE_ATTRIBUTE &&
	   get_kind() != IIR_DRIVING_ATTRIBUTE &&
	   get_kind() != IIR_DRIVING_VALUE_ATTRIBUTE &&
	   (get_prefix()->_is_subtype() == TRUE && get_prefix()->_is_locally_static() == TRUE ||
	    get_prefix()->_is_object() == TRUE && 
	    get_prefix()->get_subtype()->_is_locally_static() == TRUE ) &&
	   ( get_suffix() == NULL || get_suffix()->_is_locally_static() == TRUE )){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_Attribute::_is_globally_static_primary(){
  ASSERT( get_prefix()->_is_resolved() == TRUE );
  if( _is_value_attribute() == TRUE &&
      get_kind() != IIR_USER_ATTRIBUTE && 
      get_prefix()->_is_globally_static() == TRUE ){
    return TRUE;
  }
  else if( _is_function_attribute() == TRUE &&
	   get_kind() != IIR_USER_ATTRIBUTE &&
	   get_kind() != IIR_EVENT_ATTRIBUTE &&
	   get_kind() != IIR_ACTIVE_ATTRIBUTE &&
	   get_kind() != IIR_LAST_EVENT_ATTRIBUTE &&
	   get_kind() != IIR_LAST_ACTIVE_ATTRIBUTE &&
	   get_kind() != IIR_LAST_VALUE_ATTRIBUTE &&
	   get_kind() != IIR_DRIVING_ATTRIBUTE &&
	   get_kind() != IIR_DRIVING_VALUE_ATTRIBUTE &&
	   ((get_prefix()->_is_subtype() == TRUE && get_prefix()->_is_globally_static() == TRUE)||
	    ( (get_prefix()->_is_object() == TRUE || 
	       get_prefix()->get_kind() == IIR_FUNCTION_CALL ) &&  
	      get_prefix()->get_subtype()->_is_globally_static() == TRUE ) ) &&
	   ( get_suffix() == NULL || get_suffix()->_is_globally_static() == TRUE )){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void
IIRScram_Attribute::_publish_cc_extern_type_info( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_Attribute::_publish_cc_extern_type_info" );

  ASSERT ( get_prefix() != NULL );
  ASSERT ( get_prefix()->get_subtype() != NULL );
  ASSERT ( get_prefix()->get_subtype()->_is_resolved() == TRUE );
  
  get_prefix()->get_subtype()->_publish_cc_extern_type_info( _cc_out );
}

IIR_Boolean 
IIRScram_Attribute::_need_to_process_prefix_prefix_signal_attribute(){
  IIR_Boolean retval = FALSE;

  if( get_prefix()->_is_iir_name() == TRUE ){
    IIR_Name *as_name = (IIR_Name *)get_prefix();
    ASSERT( as_name->get_prefix()->get_subtype() != NULL );
    if( as_name->get_prefix()->get_subtype()->_is_array_type() == TRUE ||
	as_name->get_prefix()->get_subtype()->_is_record_type() == TRUE ){
      retval = TRUE;
    }
  }

  return retval;
}

void 
IIRScram_Attribute::_process_prefix_prefix( IIR_TypeDefinition * ){
  ASSERT( get_prefix()->_is_iir_name() == TRUE );
  IIR_Name *as_name = (IIR_Name *)get_prefix();
  ASSERT( as_name->get_prefix() != NULL );
  ASSERT( as_name->get_prefix()->get_subtype() != NULL );
  ASSERT( as_name->get_prefix()->get_subtype()->_is_array_type() == TRUE ||
	  as_name->get_prefix()->get_subtype()->_is_record_type() == TRUE );
  // build the 'active attribute for the prefix's prefix, too.
  ostringstream name;
  name << *as_name->get_prefix() << "_" << *_get_attribute_name();
  string name_string = name.str();
  IIR_TypeDefinition *type = StandardPackage::get_boolean_type();
  IIR_Declaration *prefix_prefix_decl = _build_function_declaration( name_string, type );

  prefix_prefix_decl->_set_attribute_name( (IIR_Attribute *)this );

  if( as_name->get_prefix()->_get_implicit_declarations() == NULL ){
    as_name->get_prefix()->_set_implicit_declarations(new set<IIR_Declaration>(prefix_prefix_decl));
  }
  else{
    IIR_Boolean found_one = FALSE;
    IIR_Declaration *current =as_name->get_prefix()->_get_implicit_declarations()->get_element();
    while( current != NULL ){
      if( IIR_TextLiteral::_cmp( current->get_declarator(), name_string ) == 0 ){
	found_one = TRUE;
	break;
      }
      current = as_name->get_prefix()->_get_implicit_declarations()->get_next_element();
    }

    if( found_one == TRUE ){
      delete prefix_prefix_decl;
    }
    else{
      as_name->get_prefix()->_get_implicit_declarations()->add( prefix_prefix_decl );
    }
  }
}

set<IIR_TypeDefinition> *
IIRScram_Attribute::_get_rval_set_high_low_left_right(IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *retval = NULL;

  set<IIR_TypeDefinition> *prefix_rvals = get_prefix()->_get_rval_set(constraint_function);
  if( prefix_rvals != NULL ){
    retval = new set<IIR_TypeDefinition>;
    IIR_TypeDefinition *current_rval = prefix_rvals->get_element();
    while( current_rval != NULL ){
      if( current_rval->_is_array_type() == TRUE ){
	retval->add( current_rval->_get_index_subtype() );
      }
      else if ( current_rval->_is_scalar_type() == TRUE ){
	retval->add( current_rval );
      }
      current_rval = prefix_rvals->get_next_element();
    }
  }

  delete prefix_rvals;

  if( retval != NULL && retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

IIR_TypeDefinition *
IIRScram_Attribute::_get_subtype_high_low_left_right(){
  ASSERT( get_prefix()->_is_resolved() == TRUE );

  IIR_TypeDefinition *prefix_rval = get_prefix()->get_subtype();

  if( prefix_rval->_is_array_type() == TRUE ){
    return prefix_rval->_get_index_subtype();
  }
  else if ( prefix_rval->_is_scalar_type() == TRUE ){
    return prefix_rval;
  }
  else{
    ostringstream err;
    err << "|" << *get_prefix() << "| must a scalar or an array type to use the "
	<< "attribute 'HIGH." << endl;
    report_error( this, err.str() );
    abort();
    return 0;
  }
}

IIR_TypeDefinition *
IIRScram_Attribute::_get_prefix_rval_range_attributes(){
  IIR_TypeDefinition *retval = NULL;

  set<IIR_TypeDefinition> *prefix_types = get_prefix()->_get_rval_set();
  if( prefix_types == NULL ){
    report_undefined_symbol( get_prefix() );
    return NULL;
  }
  
  prefix_types->reduce_set( &IIR::_is_array_type );

  switch( prefix_types->num_elements() ){
  case 0:{
    ostringstream err;
    err << "|" << *get_prefix() << "| has no array definition - "
	<< "'range is invalid here.";
    report_error( this, err.str() );
    break;
  }

  case 1:{
    retval = prefix_types->get_element();
    break;
  }

  default:{
    ostringstream err;
    err << "Cannot disambiguate |" << *get_prefix() << "| in it's usage.";
    report_error( this, err.str() );
  }
  }

  delete prefix_types;
  
  return retval;
}

ostream &
IIRScram_Attribute::_print( ostream &os ){
  get_prefix()->_print( os );
  os << "'";
  
  IIR_TextLiteral *attribute_name = _build_attribute_name();
  if( attribute_name != 0 ){
    os << *attribute_name;
    delete attribute_name;
  }

  if( get_suffix() != 0 ){
    os << "(" << *get_suffix() << ")";
  }

  return os;
}

IIR_ScalarTypeDefinition *
IIRScram_Attribute::_get_subtype_range_attribute(){
  IIR_ScalarTypeDefinition *retval = NULL;
  IIR_TypeDefinition *prefix_rval = _get_prefix_subtype();
  if( prefix_rval->_is_array_type() == FALSE ){
    ostringstream err;
    err << "Range attributes may only be applied to array types.";
    report_error( this, err.str() );
    return NULL;
  }

  if( get_suffix() == NULL ){
    retval = prefix_rval->_get_index_subtype();
  }
  else{
    IIR_IntegerLiteral *static_val = get_suffix()->_get_integer_static_value();
    if( static_val != NULL ){
      int val = static_val->_get_value();
      if( val > prefix_rval->_get_num_indexes() ){
	ostringstream err;
	err << "'range( " << val << " ) not allowed for " 
	    << prefix_rval->_get_num_indexes() << " dimensional array.";
	report_error( this, err.str() );
      }
      else{
	IIR_TypeDefinition *current_array_type = prefix_rval;
	int i;
	for( i = 1; i < val; i++ ){
	  current_array_type = current_array_type->_get_element_subtype();
	  ASSERT( current_array_type->_is_array_type() == TRUE );
	}
	retval = current_array_type->_get_index_subtype();
      }
      delete static_val;
    }
  }

  if( get_kind() == IIR_REVERSE_RANGE_ATTRIBUTE ){
    retval = (IIR_ScalarTypeDefinition *)((IIR *)retval)->_clone();
    IIR *tmp = retval->get_left();
    retval->set_left( retval->get_right() );
    retval->set_right( tmp );

    if( retval->get_direction() ){
      if( retval->get_direction() == StandardPackage::get_true_literal() ){
	retval->set_direction( StandardPackage::get_false_literal() );
      }
      else{
	ASSERT( retval->get_direction() == StandardPackage::get_false_literal() );
	retval->set_direction( StandardPackage::get_true_literal() );
      }
    } // else get_direction is null and we don't need to do anything.
  }
  return retval;
}


