#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc SaveImageDef {} {
    global saveimage

    set saveimage(base) ds9
    set saveimage(format) jpeg
    set saveimage(jpeg,quality) 75
    set saveimage(tiff,compress) none

    global tcl_platform

    set aa [msgcat::mc {An error has occurred while creating the image. Please be sure that the entire image window is visible on the screen.}]
    set bb [msgcat::mc {An error has occurred while creating the image. Please be sure that the ds9 window is in the upper left corner of the default screen and the entire window is visible.}]
    switch $tcl_platform(os) {
	Darwin {
	    switch [lindex [split $tcl_platform(osVersion) {.}] 0] {
		10 -
		11 {set saveimage(error) $bb}
		8 -
		9 -
		default {set saveimage(error) $aa}
	    }
	}
	default {set saveimage(error) $aa}
    }
}

proc SaveImageDialog {} {
    global saveimage
    global saveimagefbox
    global ed

    set w {.saveimage}

    set ed(ok) 0
    set ed(format) $saveimage(format)

    DialogCreate $w [msgcat::mc {Save Image}] ed(ok)

    # Param
    set f [ttk::frame $w.param]
    ttk::label $f.title -text [msgcat::mc {Format}]
    ttk::radiobutton $f.fits -text {FITS} -variable ed(format) -value fits
    ttk::radiobutton $f.gif -text {GIF} -variable ed(format) -value gif
    ttk::radiobutton $f.jpeg -text {JPEG} -variable ed(format) -value jpeg
    ttk::radiobutton $f.tiff -text {TIFF RGB 24-bit} \
	-variable ed(format) -value tiff
    ttk::radiobutton $f.png -text {PNG 24-bit (Portable Network Graphics)} \
	-variable ed(format) -value png
    ttk::radiobutton $f.ppm -text {PPM RGB (Portable Pixmap)} \
	-variable ed(format) -value ppm
    grid $f.fits -padx 2 -pady 2 -sticky w
    grid $f.gif -padx 2 -pady 2 -sticky w
    grid $f.jpeg -padx 2 -pady 2 -sticky w
    grid $f.tiff -padx 2 -pady 2 -sticky w
    grid $f.png -padx 2 -pady 2 -sticky w
    grid $f.ppm -padx 2 -pady 2 -sticky w

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	set saveimage(format) $ed(format)
	set saveimagefbox(types) [FormatToFBox $saveimage(format)]
 	FileLast saveimagefbox "$saveimage(base).$saveimage(format)"
 	set fn [SaveFileDialog saveimagefbox]

	if {$fn != {}} {
 	    set saveimage(base) [file rootname [file tail $fn]]

	    set ok 1
	    switch -- $saveimage(format) {
		fits {}
		gif {}
		jpeg {set ok [SaveJPEGParams saveimage(jpeg,quality)]}
		tiff {set ok [SaveTIFFParams saveimage(tiff,compress)]}
		png {}
		ppm {}
	    }

	    if {$ok} {
		SaveImage $fn
	    }
	}
    }

    set rr $ed(ok)
    unset ed
    return $rr
}

proc SaveImage {fn} {
    global ds9
    global current
    global saveimage
    global cube

    if {$fn == {} || ![$current(frame) has fits]} {
	return
    }

    # besure we are on top
    raise $ds9(top)

    # and no highlite
    $current(frame) highlite off
    # and refresh screen
    RealizeDS9

    switch -- $saveimage(format) {
	fits {$current(frame) save fits resample file "\{$fn\}"}
	default {SavePhoto $fn}
    }

    # reset
    switch -- $ds9(display) {
	single -
	blink {}
	tile {$current(frame) highlite on}
    }

    # and refresh screen
    RealizeDS9
}

# Support

proc SavePhoto {fn} {
    global ds9
    global saveimage
    global current

    set rr [catch {image create photo -format window -data $ds9(canvas)} ph]
    if {$rr != 0} {
	Error $saveimage(error)
	return $rr
    }

    switch -- $saveimage(format) {
	gif {
	    $ph write $fn -format $saveimage(format)]
	}
	jpeg {
	    $ph write $fn -format \
		[list $saveimage(format) -quality $saveimage(jpeg,quality)]
	}
	tiff {
	    $ph write $fn -format \
		[list $saveimage(format) \
		     -compression $saveimage(tiff,compress)]
	}
	png {$ph write $fn -format $saveimage(format)}
	ppm {$ph write $fn -format $saveimage(format)}
    }

    image delete $ph
    return 0
}

proc SaveJPEGParams {varname} {
    upvar $varname var
    global ed2

    set w {.savejpeg}

    set ed2(ok) 0
    set ed2(quality) $var

    DialogCreate $w {JPEG} ed2(ok)

    # Param
    set f [ttk::frame $w.param]
    slider $f.squality 0 100 [msgcat::mc {JPEG Quality Factor}] \
	ed2(quality) {}

    grid $f.squality -padx 2 -pady 2 -sticky ew
    grid columnconfigure $f 0 -weight 1

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed2(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed2(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed2(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed2(ok)
    DialogDismiss $w

    if {$ed2(ok)} {
	set var $ed2(quality)
    }

    set rr $ed2(ok)
    unset ed2
    return $rr
}

proc SaveTIFFParams {varname} {
    upvar $varname var
    global ed2

    set w {.savetiff}

    set ed2(ok) 0
    set ed2(compress) $var

    DialogCreate $w {TIFF} ed2(ok)

    # Param
    set f [ttk::frame $w.param]
    ttk::label $f.title -text [msgcat::mc {Compression}]
    ttk::radiobutton $f.none -text [msgcat::mc {None}] \
	-variable ed2(compress) -value none
    ttk::radiobutton $f.jpeg -text {JPEG} \
	-variable ed2(compress) -value jpeg
    ttk::radiobutton $f.packbits -text {Packbits} \
	-variable ed2(compress) -value packbits
    ttk::radiobutton $f.deflate -text {Deflate} \
	-variable ed2(compress) -value deflate
    grid $f.title -padx 2 -pady 2 -sticky w
    grid $f.none -padx 2 -pady 2 -sticky w
    grid $f.jpeg -padx 2 -pady 2 -sticky w
    grid $f.packbits -padx 2 -pady 2 -sticky w
    grid $f.deflate -padx 2 -pady 2 -sticky w

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed2(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed2(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed2(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed2(ok)
    DialogDismiss $w

    if {$ed2(ok)} {
	set var $ed2(compress)
    }

    set rr $ed2(ok)
    unset ed2
    return $rr
}

# Process Cmds

proc ProcessSaveImageCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    # we need to be realized
    ProcessRealizeDS9

    set format {}
    set param {}
    set fn [lindex $var $i]
    if {$fn == {}} {
	return
    }

    # backward compatibility
    switch $fn {
	fits -
	gif -
	tif -
	tiff -
	jpg -
	jpeg -
	png -
	ppm {
	    set format $fn
	    set fn {}
	    incr i
	}
	mpeg {
	    # backward compatibility
	    global movie
	    incr i
	    set fn [lindex $var $i]
	    if {$fn != {} && ([string is integer $fn])} {
		incr i
		set fn [lindex $var $i]
	    }
	    set movie(action) slice
	    Movie $fn
	}
    }

    # try again
    if {$fn == {}} {
	set fn [lindex $var $i]
	if {$fn == {}} {
	    return
	}

	if {$fn != {} && ([string is integer $fn] || 
			  $fn == {none} || 
			  $fn == {jpeg} || 
			  $fn == {backbits} || 
			  $fn == {deflate})} {
	    set param $fn
	    set fn {}
	    incr i
	}
    }

    # one last time
    if {$fn == {}} {
	set fn [lindex $var $i]
	if {$fn == {}} {
	    return
	}
    }

    global saveimage
    if {$format == {}} {
	set format [ExtToFormat $fn]
    }
    set saveimage(format) $format

    if {$param == {}} {
	set param [string tolower [lindex $var [expr $i+1]]]
	switch $saveimage(format) {
	    gif -
	    png -
	    ppm -
	    fits {}
	    jpeg {
		if {$param != {} && [string is integer $param]} {
		    set saveimage(jpeg,quality) $param
		    incr i
		}
	    }
	    tiff {
		switch $param {
		    none -
		    jpeg -
		    packbits -
		    deflate {
			set saveimage(tiff,compress) $param
			incr i
		    }
		}
	    }
	}
    }

    SaveImage $fn
    FileLast saveimagefbox $fn
}

