#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc BadVisualError {} {
    global ds9

    Error [msgcat::mc {Sorry, DS9 requires a Pseudocolor8, Truecolor8, Truecolor16, Truecolor24 visual be available}]
    exit
}

proc ProcessCommandLineFirst {} {
    global argc
    global argv
    global icolorbar
    global ds9
    global pds9

    set i 0
    while {$i < $argc} {
	set item [lindex $argv $i]

	switch -- $item {
	    -help {
		puts "For more information, use --help"
		QuitDS9
	    }
	    -debug {incr i; ProcessDebugTclCmd argv i}
	    -private {set icolorbar(private) true}
	    -title {
		incr i
		set ds9(title) [lindex $argv $i]
		set t "SAOImage $ds9(title)"
		wm title $ds9(top) "$t"
		wm iconname $ds9(top) "$t"
	    }
	    -language {
		incr i
		set pds9(language) [lindex $argv $i]
		set pds9(language,name) [LanguageToName $pds9(language)]
	    }
	    -msg {
		incr i
		set pds9(language,dir) [lindex $argv $i]
	    }
	    -xpa {incr i; ProcessXPACmd argv i}
	}
	incr i
    }
}

proc ProcessCommandLine {} {
    global argc
    global argv

    ProcessCommand $argv $argc
}

proc ds9Cmd {argv} {
    ProcessCommand $argv [llength $argv]
}

proc ProcessCommand {argv argc} {
    global ds9
    global pds9
    global help
    global current
    global contour
    global colorbar
    global debug
    global wcs
    global view
    global grid
    global bin
    global scale
    global minmax
    global zscale
    global iis
    global http

    set load 0
    set noopts 0
    set file(mode) fits
    set file(slice) 0
    set file(layer) {}
    set file(mosaic) wcs
    set i 0

    # Note: -help is processed previously for fv (temporary)
    set item {}

    while {$i < $argc} {
	set item [lindex $argv $i]

	switch -- $item {
	    -- {set noopts 1}
	    -? -
	    -help -
	    --help {HelpCommand}

	    -version {
		ProcessSendVersionCmd puts stdout {}
		QuitDS9
	    }

	    -2mass {incr i; Process2MASSCmd argv i}
	    -3d {incr i; Process3DCmd argv i}
	    -about {ProcessSendAboutCmd puts stdout {} {} {}}
	    -align {incr i; ProcessAlignCmd argv i}
	    -analysis -
	    -analysismenu {incr i; ProcessAnalysisCmd argv i {} {}}
	    -array {set file(mode) array}
	    -asinh {set scale(type) asinh; ChangeScale}
	    -bg -
	    -background {incr i; ProcessBgCmd argv i}
	    -backup {incr i; ProcessBackupCmd argv i}
	    -bin {incr i; ProcessBinCmd argv i}
	    -blink {incr i; ProcessBlinkCmd argv i}
	    -blue {set current(rgb) blue; RGBChannel}
	    -cat -
	    -catalog {incr i; ProcessCatalogCmd argv i}
	    -cd {incr i; ProcessCDCmd argv i}
	    -cmap {incr i; ProcessCmapCmd argv i}
	    -colorbar {incr i; ProcessColorbarCmd argv i}
	    -console {ProcessConsoleCmd argv i}
	    -contour {incr i; ProcessContourCmd argv i}
	    -nocontour {set contour(view) 0; ContourUpdate}
	    -crop {incr i; ProcessCropCmd argv i}
	    -crosshair {incr i; ProcessCrosshairCmd argv i}
	    -cursor {incr i; ProcessCursorCmd argv i}
	    -cube -
	    -datacube {incr i; ProcessCubeCmd argv i}
	    -debug {incr i; ProcessDebugCmd argv i}
	    -dss -
	    -dsssao -
	    -sao {incr i; ProcessSAOCmd argv i}
	    -dsseso -
	    -eso {incr i; ProcessESOCmd argv i}
	    -dssstsci -
	    -stsci {incr i; ProcessSTSCICmd argv i}
	    -export {incr i; ProcessExportCmd argv i}
	    -exit -
	    -quit {ProcessQuitCmd argv i}
	    -frame {incr i; ProcessFrameCmd argv i}
	    -fifo {
		incr i
		set fifo [lindex $argv $i]
		if {$fifo!="none"} {
		    set iis(ififo) ${fifo}i
		    set iis(ofifo) ${fifo}o
		} else {
		    set iis(ififo) none
		    set iis(ofifo) none
		}
	    }
	    -fifo_only {
		set iis(port) 0
		set iis(unix) none
	    }
	    -first {incr i; ProcessFIRSTCmd argv i}
	    -file -
	    -fits {set file(mode) fits}
	    -sfits {set file(mode) sfits}
	    -geometry {
		# already processed
	    }
	    -green {set current(rgb) green; RGBChannel}
	    -grid {incr i; ProcessGridCmd argv i}
	    -nogrid {set grid(view) 0; GridUpdate}
	    -header {incr i; ProcessHeaderCmd argv i}
	    -height {incr i; ProcessHeightCmd argv i}
	    -histequ {set scale(type) histequ; ChangeScale}
	    -horzgraph {
		# backward compatibility
		set view(graph,horz) 1
		UpdateView
	    }
	    -nohorzgraph {
		# backward compatibility
		set view(graph,horz) 0
		UpdateView
	    }
	    -iconify {incr i; ProcessIconifyCmd argv i}
	    -iis {incr i; ProcessIISCmd argv i}
	    -photo {set file(mode) photo}
	    -info {
		# backward compatibility
		set view(info) 1
		UpdateView
	    }
	    -noinfo {
		# backward compatibility
		set view(info) 0
		UpdateView
	    }
	    -invert {set colorbar(invert) 1; InvertColorbar}
	    -language {
		# already processed
		incr i
	    }
	    -lock {incr i; ProcessLockCmd argv i}
	    -linear {set scale(type) linear; ChangeScale}
	    -log {set scale(type) log; ChangeScale}
	    -lower {ProcessLowerCmd argv i}
	    -magnifier {incr i; ProcessMagnifierCmd argv i}
	    -nomagnifier {
		# backward compatibility
		set view(magnifier) 0
		UpdateView
	    }
	    -mask {incr i; set file(layer) [ProcessMaskCmd argv i]}
	    -match {incr i; ProcessMatchCmd argv i}
	    -datacube -
	    -mecube -
	    -medatacube {set file(mode) mecube}
	    -memf -
	    -multiframe {set file(mode) multiframe}
	    -minmax {incr i; ProcessMinMaxCmd argv i}
	    -minmaxmode {
		# backward compatibility
		incr i
		set minmax(mode) [lindex $argv $i]
	    }
	    -minmaxsample {
		# backward compatibility
		incr i
		set minmax(sample) [lindex $argv $i]
	    }
	    -mode {incr i; ProcessModeCmd argv i}

	    -mosaicimage {
		set file(mode) mosaicimage
		set aa [lindex $argv [expr $i+1]]
		switch -- $aa {
		    wfpc2 {incr i; set file(mosaic) wfpc2}
		    default {CommandMosaicType $aa file i}
		}
	    }
	    -mosaic {
		set file(mode) mosaic
		CommandMosaicType [lindex $argv [expr $i+1]] file i
	    }
	    -smosaic {
		set file(mode) smosaic
		CommandMosaicType [lindex $argv [expr $i+1]] file i
	    }

	    -mosaicimageiraf {
		# backward compatibility
		set file(mode) mosaicimage
		set file(mosaic) iraf
	    }
	    -mosaicimagewcs {
		# backward compatibility
		set file(mode) mosaicimage
		set file(mosaic) wcs
	    }
	    -mosaicimagewfpc2 {
		# backward compatibility
		set file(mode) mosaicimage
		set file(mosaic) wfpc2
	    }

	    -mosaiciraf {
		# backward compatibility
		set file(mode) mosaic
		set file(mosaic) iraf
	    }
	    -mosaicwcs {
		# backward compatibility
		set file(mode) mosaic
		set file(mosaic) wcs
	    }

	    -msg {
		# already processed
		incr i
	    }
	    -nameserver {incr i; ProcessNRESCmd argv i}
	    -nan {incr i; ProcessNanCmd argv i}
	    -nvss {incr i; ProcessNVSSCmd argv i}
	    -orient {incr i; ProcessOrientCmd argv i}
	    -pagesetup {incr i; ProcessPageSetupCmd argv i}
	    -pspagesetup {incr i; ProcessPSPageSetupCmd argv i}
	    -pan {incr i; ProcessPanCmd argv i}
	    -panner {
		# backward compatibility
		set view(panner) 1
		UpdateView
	    }
	    -nopanner {
		# backward compatibility
		set view(panner) 0
		UpdateView
	    }
	    -pixeltable {incr i; ProcessPixelTableCmd argv i}
	    -nopixeltable {PixelTableDestroyDialog}
	    -plot {incr i; ProcessPlotCmd argv i {} {}}
	    -port {incr i; set iis(port) [lindex $argv $i]}
	    -inet_only -
	    -port_only {
		set iis(ififo) none
		set iis(ofifo) none
		set iis(unix) none
	    }
	    -pow {set scale(type) pow; ChangeScale}
	    -prefs {incr i; ProcessPrefsCmd argv i}
	    -preserve {incr i; ProcessPreserveCmd argv i}
	    -print {incr i; ProcessPrintCmd argv i}
	    -psprint {incr i; ProcessPSPrintCmd argv i}
	    -private {
		#already processed
	    }
	    -raise {ProcessRaiseCmd argv i}
	    -red {set current(rgb) red; RGBChannel}
	    -region -
	    -regions -
	    -regionfile {incr i; ProcessRegionsCmd argv i {} {}}
	    -restore {incr i; ProcessRestoreCmd argv i}
	    -rgb {incr i; ProcessRGBCmd argv i}
	    -rgbcube {set file(mode) rgbcube}
	    -srgbcube {set file(mode) srgbcube}
	    -rgbimage {set file(mode) rgbimage}
	    -rgbarray {set file(mode) rgbarray}
	    -rotate {incr i; ProcessRotateCmd argv i}
	    -samp {incr i; ProcessSAMPCmd argv i}
	    -save -
	    -savefits {incr i; ProcessSaveCmd argv i}
	    -saveimage {incr i; ProcessSaveImageCmd argv i}
	    -movie -
	    -savempeg {incr i; ProcessMovieCmd argv i}
	    -scale -
	    -ztrans {incr i; ProcessScaleCmd argv i}
	    -scalelims -
	    -scalelimits {
		#backward compatibility
		incr i
		set scale(min) [lindex $argv $i]
		incr i
		set scale(max) [lindex $argv $i]
		ChangeScaleLimit
	    }
	    -scalemode {
		#backward compatibility
		incr i
		set scale(mode) [string tolower [lindex $argv $i]]
		ChangeScaleMode
	    }
	    -scalescope {
		#backward compatibility
		incr i
		set scale(scope) [string tolower [lindex $argv $i]]
		ChangeScaleScope
	    }
	    -shm {incr i; ProcessShmCmd argv i 1}
	    -single {ProcessSingleCmd argv i}
	    -sinh {set scale(type) sinh; ChangeScale}
	    -skyview {incr i; ProcessSkyViewCmd argv i}
	    -sleep {incr i; ProcessSleepCmd argv i}
	    -slice {set file(slice) 1}
	    -noslice {set file(slice) 0}
	    -smooth {incr i; ProcessSmoothCmd argv i}
	    -squared {set scale(type) squared; ChangeScale}
	    -sqrt {set scale(type) sqrt; ChangeScale}
	    -source {incr i; ProcessSourceCmd argv i}
	    -tcl {incr i; set pds9(tcl) [FromYesNo [lindex $argv $i]]}
	    -theme {incr i; ProcessThemeCmd argv i}
	    -threads -
	    -thread {incr i; ProcessThreadsCmd argv i}
	    -tile {incr i; ProcessTileCmd argv i}
	    -title {
		#already processed
		incr i
	    }
	    -unix {incr i; set iis(unix) [lindex $argv $i]}
	    -unix_only {
		set iis(ififo) none
		set iis(ofifo) none
		set iis(port) 0
	    }
	    -url {set file(mode) url}
	    -update {incr i; ProcessUpdateCmd argv i}
	    -vertgraph {
		#backward compatibility
		set view(graph,vert) 1
		UpdateView
	    }
	    -novertgraph {
		#backward compatibility
		set view(graph,vert) 0
		UpdateView
	    }
	    -coords -
	    -view {incr i; ProcessViewCmd argv i}
	    -visual {
		#already processed
	    }
	    -vo {incr i; ProcessVOCmd argv i}
	    -wcs {incr i; ProcessWCSCmd argv i {} {}}
	    -wcsformat {
		#backward compatibility
		incr i
		set wcs(format,) [lindex $argv $i]
	    }
	    -web {incr i; ProcessWebCmd argv i}
	    -width {incr i; ProcessWidthCmd argv i}
	    -xpa {incr i; ProcessXPACmd argv i}
	    -z1 {
		#backward compatibility
		incr i
		set scale(min) [lindex $argv $i]
		ChangeScaleLimit
	    }
	    -z2 {
		#backward compatibility
		incr i
		set scale(max) [lindex $argv $i]
		ChangeScaleLimit
	    }
	    -zscale {incr i; ProcessZScaleCmd argv i}
	    -zmax {set scale(mode) zmax; ChangeScaleMode}
	    -zoom {incr i; ProcessZoomCmd argv i}

	    default {
		# allow abc, -, and -[foo] but not -abc
		if {!$noopts && [regexp -- {^-[a-zA-Z]+} $item]} {
		    puts stderr "[msgcat::mc {Unknown command}]: $item"
		    puts stderr "[msgcat::mc {For more information, use --help}]"
		    return
		}

		if {$load == 0} {
		    StartLoad
		    incr load
		}

		switch $ds9(wm) {
		    x11 -
		    aqua {CommandLineLoad $item file argv i}
		    win32 {
			# if win32 and envoked via DOS shell
			# we must expand wildcards ourselves
			if {[catch {glob $item} fns]} {
			    # cygwin/double click/DOS Shell no wildcards
			    CommandLineLoad $item file argv i
			} else {
			    # DOS Shell with wildcards
			    foreach fn $fns {
				CommandLineLoad $fn file argv i
			    }
			}
		    }
		}

		FinishLoadPre
	    }
	}
	incr i
    }

    if {$load != 0} {
	FinishLoadPost
    }
}

proc CommandLineLoad {item filename argvname iname} {
    upvar $filename file
    upvar $argvname argv
    upvar $iname i

    global current

    if {$current(frame) != {}} {
	switch -- [$current(frame) get type] {
	    base {CommandLineLoadBase $item $filename $argvname $iname}
	    rgb {CommandLineLoadRGB $item $filename $argvname $iname}
	    3d {CommandLineLoad3D $item $filename $argvname $iname}
	}
    } else {
	CommandLineLoadBase $item $filename $argvname $iname
    }

    switch -- $file(mode) {
	array {FileLast arrayfbox $item}
	photo {FileLast photofbox $item}
	url {}
	default {FileLast fitsfbox $item}
    }
}

proc CommandLineLoadBase {item filename argvname iname} {
    upvar 2 $filename file
    upvar 2 $argvname argv
    upvar 2 $iname i

    global ds9

    switch -- $file(mode) {
	array {
	    MultiLoad $file(layer)
	    LoadArray $item $file(layer)
	}
	fits {
	    if {$file(slice)} {
		LoadSliceFits $item
	    } else {
		# under windows, a double click on a 
		# data file comes here
		switch $ds9(wm) {
		    x11 {}
		    aqua {}
		    win32 {
			# normalize for "abc" but not "-abc", "-", and "-[foo]"
			# otherwise, things get weird
			if {![regexp -- {^-} $item]} {
			    set item [file normalize [file nativename $item]]
			}				
		    }
		}
		MultiLoad $file(layer)
		LoadFits $item $file(layer)
	    }
	}
	sfits {
	    MultiLoad $file(layer)
	    incr i
	    LoadSFits $item [lindex $argv $i] $file(layer)
	}
	mecube {
	    MultiLoad
	    LoadMECubeFits $item
	}
	multiframe {
	    MultiLoad
	    LoadMultiFrameFits $item
	}
	mosaicimage {
	    MultiLoad $file(layer)
	    switch -- $file(mosaic) {
		iraf {LoadMosaicImageIRAFFits $item $file(layer)}
		wfpc2 {LoadMosaicImageWFPC2Fits $item}
		default {LoadMosaicImageWCSFits $file(mosaic) \
			     $item $file(layer)}
	    }
	}
	mosaic {
	    switch -- $file(mosaic) {
		iraf {LoadMosaicIRAFFits $item $file(layer)}
		default {LoadMosaicWCSFits $file(mosaic) \
			     $item $file(layer)}
	    }
	}
	smosaic {
	    incr i
	    switch -- $file(mosaic) {
		iraf {LoadMosaicIRAFSFits $item [lindex $argv $i] $file(layer)}
		default {LoadMosaicWCSSFits $file(mosaic) \
			     $item [lindex $argv $i] $file(layer)}
	    }
	}

	rgbcube {
	    CreateRGBFrame
	    LoadRGBCubeFits $item
	}
	srgbcube {
	    CreateRGBFrame
	    incr i
	    LoadRGBCubeSFits $item [lindex $argv $i]
	}
	rgbimage {
	    CreateRGBFrame
	    LoadRGBImageFits $item
	}
	rgbarray {
	    CreateRGBFrame
	    LoadRGBArray $item
	}

	url {
	    MultiLoad $file(layer)
	    LoadURL $item
	}

	photo {
	    if {$file(slice)} {
		LoadSlicePhoto $item
	    } else {
		MultiLoad
		LoadPhoto $item
	    }
	}
    }
}

proc CommandLineLoadRGB {item filename argvname iname} {
    upvar 2 $filename file
    upvar 2 $argvname argv
    upvar 2 $iname i

    switch -- $file(mode) {
	array {LoadArray $item}
	fits {LoadFits $item}
	sfits {
	    incr i
	    LoadSFits $item [lindex $argv $i]
	}
	mecube {LoadMECubeFits $item}
	multiframe {
	    # not supported
	}
	mosaicimage {
	    switch -- $file(mosaic) {
		iraf {LoadMosaicImageIRAFFits $item}
		wfpc2 {LoadMosaicImageWFPC2Fits $item}
		default {LoadMosaicImageWCSFits $file(mosaic) $item}
	    }
	}
	mosaic {
	    switch -- $file(mosaic) {
		iraf {LoadMosaicIRAFFits $item}
		default {LoadMosaicWCSFits $file(mosaic) $item}
	    }
	}
	smosaic {
	    incr i
	    switch -- $file(mosaic) {
		iraf {LoadMosaicIRAFSFits $item [lindex $argv $i]}
		default {LoadMosaicWCSSFits $file(mosaic) \
			     $item [lindex $argv $i]}
	    }
	}

	rgbcube {
	    MultiLoadRGB
	    LoadRGBCubeFits $item
	}
	srgbcube {
	    MultiLoadRGB
	    incr i
	    LoadRGBCubeSFits $item [lindex $argv $i]
	}
	rgbimage {
	    MultiLoadRGB
	    LoadRGBImageFits $item
	}
	rgbarray {
	    MultiLoadRGB
	    LoadRGBArray $item
	}

	url {LoadURL $item}

	photo {
	    MultiLoadRGB
	    LoadPhoto $item
	}
    }
}

proc CommandLineLoad3D {item filename argvname iname} {
    upvar 2 $filename file
    upvar 2 $argvname argv
    upvar 2 $iname i

    switch -- $file(mode) {
	array {
	    MultiLoad
	    LoadArray $item
	}
	fits {
	    if {$file(slice)} {
		LoadSliceFits $item
	    } else {
		MultiLoad
		LoadFits $item
	    }
	}
	sfits {
	    MultiLoad
	    incr i
	    LoadSFits $item [lindex $argv $i]
	}
	mecube {
	    MultiLoad
	    LoadMECubeFits $item
	}
	multiframe {
	    MultiLoad
	    LoadMultiFrameFits $item
	}
	mosaicimage {
	    MultiLoad
	    switch -- $file(mosaic) {
		iraf {LoadMosaicImageIRAFFits $item}
		wfpc2 {LoadMosaicImageWFPC2Fits $item}
		default {LoadMosaicImageWCSFits $file(mosaic) $item}
	    }
	}
	mosaic {
	    switch -- $file(mosaic) {
		iraf {LoadMosaicIRAFFits $item}
		default {LoadMosaicWCSFits $file(mosaic) $item}
	    }
	}
	smosaic {
	    incr i
	    switch -- $file(mosaic) {
		iraf {LoadMosaicIRAFSFits $item [lindex $argv $i]}
		default {LoadMosaicWCSSFits $file(mosaic) $item \
			     [lindex $argv $i]}
	    }
	}

	rgbcube {
	    # not supported
	}
	srgbcube {
	    # not supported
	}
	rgbimage {
	    # not supported
	}
	rgbarray {
	    # not supported
	}

	url {
	    MultiLoad
	    LoadURL $item
	}

	photo {
	    if {$file(slice)} {
		LoadSlicePhoto $item
	    } else {
		MultiLoad
		LoadPhoto $item
	    }
	}
    }
}

proc CommandMosaicType {sys filename iname} {
    upvar $iname i
    upvar $filename file

    switch $sys {
	iraf -
	wcs -
	wcsa -
	wcsb -
	wcsc -
	wcsd -
	wcse -
	wcsf -
	wcsg -
	wcsh -
	wcsi -
	wcsj -
	wcsk -
	wcsl -
	wcsm -
	wcsn -
	wcso -
	wcsp -
	wcsq -
	wcsr -
	wcss -
	wcst -
	wcsu -
	wcsv -
	wcsw -
	wcsx -
	wcsy -
	wcsz {
	    incr i
	    set ${filename}(mosaic) $sys
	}
	default {set ${filename}(mosaic) wcs}
    }
}
