// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "tcl.h"
#include <Xlib.h>

#include "frame.h"
#include "fitsimage.h"
#include "util.h"
#include "ps.h"

#include "alloc.h"
#include "allocgz.h"
#include "channel.h"
#include "mmap.h"
#include "share.h"
#include "socket.h"
#include "socketgz.h"
#include "var.h"
#include "contour.h"

#include "NaN.h"

// Frame Member Functions

Frame::Frame(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) : FrameBase(i, c, item)
{
  fits = NULL;
  mosaicCount = 0;
  sliceCount = 0;
  mosaicType = NOMOSAIC;
  mosaicSystem = WCS;
  contour = NULL;
  
  currentFits = NULL;
  keyFits = &fits;
  channelFits = &fits;
  currentMosaicCount = &mosaicCount;
  currentSliceCount = &sliceCount;
  currentScale = &frScale;
  currentMosaic = &mosaicType;
  currentMosaicSystem = &mosaicSystem;

  currentWCScdelt = &WCScdelt;
  currentWCSrot = &WCSrot;
  currentWCSorientation = &WCSorientation;
  currentWCSbb = &WCSbb;
  currentWCSmatrix = &WCSmatrix;

  currentcontour = &contour;

  cmapID = 1;
  bias = 0.5;
  contrast = 1.0;
  invert = 0;

  colorCount = 0;
  colorScale = NULL;
  colorCells = NULL;
  indexCells = NULL;

  iisSaveRoot = 0;
  iisInteractive = 0;
}

Frame::~Frame()
{
  if (colorScale)
    delete colorScale;

  if (colorCells)
    delete [] colorCells;

  if (indexCells)
    delete [] indexCells;
}

// Virtual Routines with base components

void Frame::reset()
{
  cmapID = 1;
  bias = 0.5;
  contrast = 1.0;
  invert = 0;

  FrameBase::reset();
}

void Frame::setBinCursor(const Vector& v)
{
  if (currentFits)
    currentFits->setBinCursor(cursor);
}

void Frame::analysisFits()
{
  FitsImage* ptr = fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->analysis();
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }

  resetClip(1);
}

void Frame::updateBin(const Matrix& mx)
{
  // Note: cursor is in USER coords, imageCenter() in IMAGE coords
  cursor = imageCenter();

  // update bin
  crosshair *= mx;

  updateMarkerCoords(&userMarkers, mx);
  updateMarkerCoords(&catalogMarkers, mx);

  // update contours
  // main contour
  if (hasContour())
    (*currentcontour)->updateCoords(mx);

  // aux contours
  {
    Contour* ptr=auxcontours.head();
    while (ptr) {
      ptr->updateCoords(mx);
      ptr=ptr->next();
    }
  }

  // delete any previous slices
  {
    FitsImage* ptr = (*channelFits)->nextSlice();
    (*channelFits)->setNextSlice(NULL);
    while (ptr) {
      FitsImage* tmp = ptr->nextSlice();
      delete ptr;
      ptr = tmp;
    }
  }

  // finish bin
  currentFits = *channelFits;
  *currentMosaicCount = 1;
  *currentSliceCount = 1;
  slice = 1;

  // bin data cube
  int bd = (*channelFits)->binDepth();
  if (bd > 1) {
    FitsImage* ptr = *channelFits;
    for (int i=1; i<bd; i++) {
      FitsImage* next = NULL;
      next = new FitsImageFitsNextHist(this, *channelFits, ptr->imageFile());
      if (next && next->isValid()) {
	ptr->setNextSlice(next);
	ptr = next;
	(*currentSliceCount)++;
      }
      else {
	if (next)
	  delete next;
	break;
      }
    }
  }

  updateBinFileNames();

  resetClip(0);
  align();
  updateNow(MATRIX); // because we have changed zoom

  // update markers call backs
  // wait til matrices have been updated so that any dialogs will print
  // the correct coords

  updateMarkerCBs(&userMarkers);
  updateMarkerCBs(&catalogMarkers);
}

BBox Frame::imageBBox()
{
  // returns imageBBox in IMAGE coords
  BBox r;
  if (fits) {
    switch (mosaicType) {
    case NOMOSAIC: 
      {
	int* params = fits->getDataParams(frScale.scanMode());
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	r = (BBox(xmin+1,ymin+1,xmax,ymax));
	if (DebugMosaic)
	  cerr << "Frame::imageBBox() NOMOSAIC ";
      }
      break;
    case IRAF:
      r = fits->getIRAFbb();
      if (DebugMosaic)
	cerr << "Frame::imageBBox() IRAF ";
      break;
    case WCSMOSAIC:
    case WFPC2:
      r = (*currentWCSbb);
      if (DebugMosaic)
	cerr << "Frame::imageBBox() WCS ";
      break;
    }
  }

  if (DebugMosaic)
    cerr << r << ' ' << r.size() << ' ' << r.center() << endl;
  return r;
}

void Frame::unloadFits()
{
  if (DebugPerf)
    cerr << "Frame::unloadFits" << endl;

  // clean up from iis if needed
  if (isIIS()) {
    frScale.setClipMode(FrScale::MINMAX);
    frScale.setMMMode(FrScale::SAMPLE);
    frScale.setULow(DEFAULTLOW);
    frScale.setUHigh(DEFAULTHIGH);
    setColorScale(FrScale::LINEARSCALE);
  }

  // reset user mode
  // if (frScale.clipMode() == FrScale::USERCLIP)
  // frScale.setClipMode(FrScale::MINMAX);

  // now, delete any fitsimages
  FitsImage* ptr = fits;
  while (ptr) {
    FitsImage* sptr = ptr->nextSlice();
    while (sptr) {
      FitsImage* stmp = sptr->nextSlice();
      delete sptr;
      sptr = stmp;
    }

    FitsImage* tmp = ptr->nextMosaic();
    delete ptr;
    ptr = tmp;
  }
  fits = NULL;

  // delete any contour
  if (contour)
    delete contour;
  contour = NULL;

  FrameBase::unloadFits();
}

void Frame::align()
{
  if (!wcsAlignment || !currentFits || !hasWCS(wcsSystem)) {
    wcsRotation = 0;
    wcsOrientation = NORMAL;
    wcsOrientationMatrix.identity();
    return;
  }

  if (hasWCS(wcsSystem)) {
    wcsRotation = currentFits->getWCSRotation(wcsSystem,wcsSky);
    wcsOrientation = currentFits->getWCSOrientation(wcsSystem,wcsSky);
  }
  else {
    wcsRotation = 0;
    wcsOrientation = NORMAL;
  }

  switch (wcsOrientation) {
  case NORMAL:
    wcsOrientationMatrix.identity();
    break;
  case XX:
    wcsOrientationMatrix = FlipX();
    break;
  case YY:
    wcsOrientationMatrix = FlipY();
    break;
  case XY:
    wcsOrientationMatrix = FlipXY();
    break;
  }

  // take into account any internal mosaic rotation/flip
  wcsRotation += currentFits->rotation();
  switch (currentFits->orientation()) {
  case NORMAL:
    break;
  case XX:
    wcsOrientationMatrix *= FlipX();
    break;
  case YY:
    wcsOrientationMatrix *= FlipY();
    break;
  case XY:
    wcsOrientationMatrix *= FlipXY();
    break;
  }
}

void Frame::updateMatrices()
{
  // very important! do this first
  FrameBase::updateMatrices();

  Matrix rgb; // alway identity

  FitsImage* ptr = fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->updateMatrices(mosaicType, rgb, refToUser, userToWidget, 
			   widgetToCanvas, userToPanner);
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }
}

void Frame::updateMagnifierMatrices(Matrix& refToMagnifier)
{
  FitsImage* ptr = fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->updateMagnifierMatrices(refToMagnifier);
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }
}

void Frame::updateColorCells(unsigned short* index, 
			     unsigned char* cells, int cnt)
{
  // the colorbar widget will pass us a pointer to the indexCells

  colorCount = cnt;
  if (indexCells)
    delete [] indexCells;
  indexCells = new unsigned short[cnt];
  if (!indexCells) {
    cerr << "Frame Internal Error: Unable to Alloc indexCells" << endl;
    exit(1);
  }
  memcpy(indexCells, index, cnt*sizeof(unsigned short));

  // copy the rgb vales to the colorCells array (for postscript printing)

  if (colorCells)
    delete [] colorCells;
  colorCells = new unsigned char[cnt*3];
  if (!colorCells) {
    cerr << "Frame Internal Error: Unable to Alloc colorCells" << endl;
    exit(1);
  }
  memcpy(colorCells, cells, cnt*3);
}

int Frame::isIIS() 
{
  return currentFits && currentFits->isIIS();
}

void Frame::psLevel1(PSColorSpace mode, int width, int height, float scale)
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  psLevel1Head(mode, width, height);

  if (!isMosaic())
    psLevel1Single(mode, width, height, scale);
  else {
    if (mosaicFast && *currentMosaic == IRAF)
      psLevel1MosaicFast(mode, width, height, scale);
    else
      psLevel1Mosaic(mode, width, height, scale);
  }
}

void Frame::psLevel2(PSColorSpace mode, int width, int height, float scale)
{
  // we need a colorScale before we can render
  if (!colorScale)
    return;

  psLevel2Head(mode, width, height);

  if (!isMosaic())
    psLevel2Single(mode, width, height, scale);
  else {
    if (mosaicFast && *currentMosaic == IRAF)
      psLevel2MosaicFast(mode, width, height, scale);
    else
      psLevel2Mosaic(mode, width, height, scale);
  }
}

void Frame::psLevel1Single(PSColorSpace mode, int width, int height, 
			   float scale)
{
  if (!currentFits)
    return;

  NoCompressAsciiHex filter;
  currentFits->updatePS(psMatrix(scale, width, height));
  double* m = currentFits->getmPSToData();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;
	    
      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);

	if (!isnand(value)) {
	  if (value <= ll) {
	    blue = table[0];
	    green = table[1];
	    red = table[2];
	  }
	  else if (value >= hh) {
	    blue = table[length*3];
	    green = table[length*3+1];
	    red = table[length*3+2];
	  }
	  else {
	    int l = (int)(((value - ll)/diff * length) + .5);
	    blue = table[l*3];
	    green = table[l*3+1];
	    red = table[l*3+2];
	  }
	}
	else {
	  blue = nanColor->blue;
	  green = nanColor->green;
	  red = nanColor->red;
	}
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }
    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void Frame::psLevel1Mosaic(PSColorSpace mode, int width, int height,
			   float scale)
{
  NoCompressAsciiHex filter;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  Matrix mm = psMatrix(scale, width, height);

  FitsImage* sptr = currentFits;
  while (sptr) {
    sptr->updatePS(mm);
    sptr = sptr->nextMosaic();
  }

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;
	    
      FitsImage* sptr = currentFits;
      while (sptr) {
	double* m = sptr->getmPSToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = sptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = sptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = sptr->getLowDouble();
	  double hh = sptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isnand(value)) {
	    if (value <= ll) {
	      blue = table[0];
	      green = table[1];
	      red = table[2];
	    }
	    else if (value >= hh) {
	      blue = table[length*3];
	      green = table[length*3+1];
	      red = table[length*3+2];
	    }
	    else {
	      int l = (int)(((value - ll)/diff * length) + .5);
	      blue = table[l*3];
	      green = table[l*3+1];
	      red = table[l*3+2];
	    }
	  }
	  else {
	    blue = nanColor->blue;
	    green = nanColor->green;
	    red = nanColor->red;
	  }
	  
	  break;
	}
	sptr = sptr->nextMosaic();
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }
    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void Frame::psLevel1MosaicFast(PSColorSpace mode, int width, int height,
			       float scale)
{
  if (!currentFits)
    return;

  NoCompressAsciiHex filter;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  Matrix mm = psMatrix(scale, width, height);
  FitsImage* sptr = currentFits;
  while (sptr) {
    sptr->updatePS(mm);
    sptr = sptr->nextMosaic();
  }

  sptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;
	    
      double x,y;

      if (sptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = sptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isnand(value))
	    if (value <= ll) {
	      blue = table[0];
	      green = table[1];
	      red = table[2];
	    }
	    else if (value >= hh) {
	      blue = table[length*3];
	      green = table[length*3+1];
	      red = table[length*3+2];
	    }
	    else {
	      int l = (int)(((value - ll)/diff * length) + .5);
	      blue = table[l*3];
	      green = table[l*3+1];
	      red = table[l*3+2];
	    }
	}
	else
	  sptr = NULL;
      }

      if (!sptr) {
	sptr = currentFits;
	while (sptr) {
	  m = sptr->getmPSToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = sptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = sptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = sptr->getLowDouble();
	    hh = sptr->getHighDouble();
	    diff = hh - ll;

	    if (!isnand(value)) {
	      if (value <= ll) {
		blue = table[0];
		green = table[1];
		red = table[2];
	      }
	      else if (value >= hh) {
		blue = table[length*3];
		green = table[length*3+1];
		red = table[length*3+2];
	      }
	      else {
		int l = (int)(((value - ll)/diff * length) + .5);
		blue = table[l*3];
		green = table[l*3+1];
		red = table[l*3+2];
	      }
	    }
	    else {
	      blue = nanColor->blue;
	      green = nanColor->green;
	      red = nanColor->red;
	    }

	    break;
	  }
	  sptr = sptr->nextMosaic();
	}
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }
    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void Frame::buildXImage(XImage* xmap, int x0, int y0, int x1, int y1,
			double* (FitsImage::*getMatrix)())
{
  if (!isMosaic())
    fillXImage(xmap, x0, y0, x1, y1, getMatrix);
  else {
    if (mosaicFast && *currentMosaic == IRAF)
      fillXImageMosaicFast(xmap, x0, y0, x1, y1, getMatrix);
    else
      fillXImageMosaic(xmap, x0, y0, x1, y1, getMatrix);
  }
}

// Fits Commands

void Frame::getFitsSizeCmd()
{
  if (fits)
    printVector(Vector(fits->width(),fits->height()),DEFAULT);
}

void Frame::getFitsSizeCmd(CoordSystem sys, SkyFormat format)
{
  if (fits)
    printVector(fits->mapLenFromRef(imageSize(), sys, format), FIXED);
}

void Frame::zoomToFitCmd(double s)
{
  if (!fits)
    return;

  centerImage();
  double z = s*calcZoom(Vector(options->width,options->height),imageSize());
  zoom_ = Vector(z,z);
  update(MATRIX);
}

void Frame::sliceCmd(int s)
{
  if (fits && (s>0) && (s<=sliceCount)) {
    slice = s;
    currentFits = fits;
    for (int i=1; i<s; i++)
      currentFits = currentFits->nextSlice();
  }

  // execute any update callbacks
  updateCBMarkers();

  update(BASE);
}

// Colormap Commands

void Frame::getColormapCmd()
{
  ostringstream str;
  str << cmapID << ' ' << bias << ' ' << contrast << ' ' << invert << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void Frame::getTypeCmd()
{
  Tcl_AppendResult(interp, "base", NULL);
}

// Pan Zoom Rotate Orient Commands

void Frame::getWCSZoomCmd(CoordSystem sys, Precision p)
{
  if (hasWCS(sys))
    printVector(currentFits->getWCScdelt(sys), p);
  else
    Tcl_AppendResult(interp, "1 1", NULL);
}

void Frame::wcsZoomCmd(CoordSystem sys, const Vector& a)
{
  if (hasWCS(sys)) {
    Vector b = currentFits->getWCScdelt(sys);
    b /= a;
    zoom_ *= b.abs();
  }

  update(MATRIX);
}

// IIS Commands

void Frame::iisCmd(int width, int height)
{
  unloadAllFits();

  // force clip mode to user between 1 to 200
  frScale.setClipMode(FrScale::USERCLIP);
  frScale.setMMMode(FrScale::SCAN);
  frScale.setULow(IISMIN);
  frScale.setUHigh(IISMAX);
  setColorScale(FrScale::IISSCALE);

  FitsImage* img = new FitsImageIIS(this, width, height);

  load(ALLOC,"",img);
}

void Frame::iisEraseCmd()
{
  if (currentFits)
    ((FitsImageIIS*)currentFits)->iisErase();
}

void Frame::iisGetCmd(char* ptr, int x, int y, int dx, int dy)
{
  if (currentFits)
    memcpy(ptr, ((FitsImageIIS*)currentFits)->iisGet(x,y), dx*dy);
}

void Frame::iisSetCmd(const char* data, int x, int y, int dx, int dy)
{
  if (currentFits)
    ((FitsImageIIS*)currentFits)->iisSet(x, y, data, dx*dy);
}

void Frame::iisWCSCmd(const Matrix& mx, const Vector& z, int zt)
{
  if (currentFits)
    ((FitsImageIIS*)currentFits)->iisWCS(mx, z, zt);
}

void Frame::getRGBChannelCmd()
{
  Tcl_AppendResult(interp, "red", NULL);
}

void Frame::getRGBViewCmd()
{
  Tcl_AppendResult(interp, "1 1 1", NULL);
}

void Frame::getRGBSystemCmd()
{
  Tcl_AppendResult(interp, "image", NULL);
}


