// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

%{
#define YYPARSE_PARAM fr
#define YYDEBUG 1

#define FR ((FrameBase*)fr)
#define FITSPTR (FR->findFits(globalTile))

#define DISCARD_(x) {yyclearin; mkDiscard(x);}

#include <math.h>
#include <string.h>
#include <iostream.h>

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "vector.h"
#include "list.h"
#include "basemarker.h"
#include "coord.h"

extern int mklex(void);
extern void mkerror(const char*);
extern void mkDiscard(int);

int mksign;
int mksign2;

static CoordSystem globalSystem;
static CoordSystem globalWCS;
static SkyFrame globalSky;
static CoordSystem localSystem;
static SkyFrame localSky;

static int globalTile;

static unsigned short globalProps;
static unsigned short localProps;

static int globalWidth;
static int localWidth;

static char globalColor[16];
static char localColor[16];

static char globalFont[32];
static char localFont[32];

static char globalText[80];
static char localText[80];

static char localComment[80];

static int globalLine1;
static int localLine1;
static int globalLine2;
static int localLine2;

static int globalVector;
static int localVector;

static int globalPoint;
static int localPoint;

static double globalTextAngle;
static double localTextAngle;

static CoordSystem globalRulerCoordSystem;
static CoordSystem localRulerCoordSystem;
static SkyFrame globalRulerSkyFrame;
static SkyFrame localRulerSkyFrame;
static CoordSystem globalRulerDistSystem;
static CoordSystem localRulerDistSystem;
static SkyFormat globalRulerDistFormat;
static SkyFormat localRulerDistFormat;

static CoordSystem globalCompassCoordSystem;
static SkyFrame globalCompassSkyFrame;
static char globalCompassNorth[80];
static char globalCompassEast[80];
static int globalCompassNArrow;
static int globalCompassEArrow;
static CoordSystem localCompassCoordSystem;
static SkyFrame localCompassSkyFrame;
static char localCompassNorth[80];
static char localCompassEast[80];
static int localCompassNArrow;
static int localCompassEArrow;

static int localCpanda;
static int localEpanda;
static int localBpanda;

static List<Vertex> polylist;
static List<Tag> taglist;
static List<CallBack> cblist;

static double aAnnuli[MAXANNULI];
static Vector aVector[MAXANNULI];
static int aNum;
static int aNumsao;
static int aStatus;
static int cStatus;
static Vector aCenter;
static double aAngles[MAXANGLES];
static int aAngNum;
static double aAngle;
static unsigned short aProps;
static char aColor[16];
static int aWidth;
static char aFont[32];
static char aText[80];
static char aComment[80];

static void setProps(unsigned short* props, unsigned short prop, int value);
static CoordSystem checkWCSSystem();
static SkyFrame checkWCSSky();

enum {CIRCLE,BOX,DIAMOND,CROSS,XPT,ARROW,BOXCIRCLE};
	
%}

%union {
  double real;
  int integer;
  char str[2048];
  double vector[3];
}

%type <real> numeric
%type <integer> yesno

%type <real> angle
%type <real> optangle
%type <real> value
%type <vector> vvalue
%type <real> sexagesimal
%type <vector> coord
%type <integer> coordSystem
%type <integer> wcsSystem
%type <integer> skyFrame
%type <integer> skyFormat
%type <integer> property
%type <integer> callBack
%type <integer> pointProp
%type <integer> numberof

%token <integer> INT
%token <real> REAL
%token <str> STRING

%token <integer> HOUR
%token <integer> MINUTE
%token <real> SECOND
%token <real> DEGREE
%token <real> ARCMINUTE
%token <real> ARCSECOND
%token <real> RADIAN
%token <real> PHYCOORD
%token <real> IMGCOORD

%token <str> SEXSTR

%token EOF_

%token AMPLIFIER_
%token ANNULUS_
%token ARCMIN_
%token ARCSEC_
%token ARROW_
%token B1950_
%token BACKGROUND_
%token BEGIN_
%token BOX_
%token BOXCIRCLE_
%token BPANDA_
%token CALLBACK_
%token CIRCLE_
%token COLOR_
%token COMPASS_
%token COMPOSITE_
%token CPANDA_
%token CROSS_
%token DEBUG_
%token DEGREES_
%token DELETE_
%token DETECTOR_
%token DIAMOND_
%token ECLIPTIC_
%token EDIT_
%token ELLIPSE_
%token END_
%token EPANDA_
%token FALSE_
%token FIELD_
%token FIXED_
%token FK4_
%token FK5_
%token FONT_
%token GALACTIC_
%token GLOBAL_
%token HIGHLITE_
%token ICRS_
%token IGNORE_
%token IMAGE_
%token INCLUDE_
%token J2000_
%token LINE_
%token LINEAR_
%token MOVE_
%token N_
%token NO_
%token OFF_
%token ON_
%token PHYSICAL_
%token PIE_
%token PIXELS_
%token POINT_
%token POLYGON_
%token PROJECTION_
%token PROPERTY_
%token ROTATE_
%token ROTBOX_
%token RULER_
%token SELECT_
%token SOURCE_
%token TAG_
%token TEXT_
%token TEXTANGLE_
%token TILE_
%token TRUE_
%token VECTOR_
%token VERSION_
%token UNHIGHLITE_
%token UNSELECT_
%token UPDATE_
%token WCS_
%token WCSA_
%token WCSB_
%token WCSC_
%token WCSD_
%token WCSE_
%token WCSF_
%token WCSG_
%token WCSH_
%token WCSI_
%token WCSJ_
%token WCSK_
%token WCSL_
%token WCSM_
%token WCSN_
%token WCSO_
%token WCSP_
%token WCSQ_
%token WCSR_
%token WCSS_
%token WCST_
%token WCSU_
%token WCSV_
%token WCSW_
%token WCSX_
%token WCSY_
%token WCSZ_
%token WCS0_
%token WIDTH_
%token X_
%token Y_
%token YES_

%token SYM_OR
%token SYM_AND
%token SYM_NOT
%token SYM_XOR

%%

start	: initGlobal commands postLocal
	;

commands: commands command terminator
	| command terminator
	;

command : /* empty */
	| DEBUG_ debug
	| VERSION_ {cerr << "DS9 Regions File 3.1" << endl;}

	| GLOBAL_ global comment
	| TILE_ INT {globalTile = $2;}
	| '#' TILE_ INT {globalTile = $3;}

	| coordSystem {globalSystem=(CoordSystem)$1;} comment
	| skyFrame {globalSystem=globalWCS; globalSky=(SkyFrame)$1;} comment
	| LINEAR_ {globalSystem=globalWCS; globalSky=NATIVEWCS;} comment

	| initLocal include shape
	| '#' initLocal include nonshape

	| generalComment
	;

generalComment: '#' {DISCARD_(1)} STRING
	;

comment : /* empty */
	| '#' {DISCARD_(1)} STRING
	;

shapeComment : /* empty */ postLocal
	| '#' {DISCARD_(1)} STRING postLocal {strncpy(localComment,$3,80)}
	| '#' local postLocal
	| '#' local {DISCARD_(1)} STRING postLocal
	  {strncpy(localComment,$4,80)}
	;

nonshapeComment : /* empty */ postLocal
	| {DISCARD_(1)} STRING postLocal {strncpy(localComment,$2,80)}
	| local postLocal
	| local {DISCARD_(1)} STRING postLocal {strncpy(localComment,$3,80)}
	;

terminator: '\n'
	| ';'
	| EOF_ {YYACCEPT;}
	;

numeric	: REAL {$$=$1;}
	| INT {$$=$1;}
	;

debug	: ON_ {yydebug=1;}
	| OFF_ {yydebug=0;}
	;

yesno	: INT {$$=($1 ? 1 : 0);}

	| YES_ {$$=1;}
	| Y_ {$$=1;}
	| ON_ {$$=1;}
	| TRUE_ {$$=1;}

	| NO_ {$$=0;}
	| N_ {$$=0;}
	| OFF_ {$$=0;}
	| FALSE_ {$$=0;}
	;

sp	: /* empty */
	| ','
	;

bp	: /* empty */
	| '('
	;

ep	: /* emtpy */
	| ')'
	;

conjuction : /* empty */ {cStatus = 0;}
	| '|' {cStatus = 1;}
	| '|' '|' {cStatus = 1;}
	;

optangle: /* empty */ {$$ = FR->mapAngleToRef(0,localSystem,localSky);}
	| angle {$$ = $1;}
	;

angle	: numeric {$$ = FR->mapAngleToRef(degToRad($1),localSystem,localSky);}
	| DEGREE {$$ = FR->mapAngleToRef(degToRad($1),localSystem,localSky);}
	| RADIAN {$$ = FR->mapAngleToRef($1,localSystem,localSky);}
	;

value	: numeric {$$ = FITSPTR->mapLenToRef($1, localSystem, DEGREES);}
	| PHYCOORD {$$ = FITSPTR->mapLenToRef($1, PHYSICAL);}
	| IMGCOORD {$$ = FITSPTR->mapLenToRef($1, IMAGE);}
	| DEGREE {$$ = FITSPTR->mapLenToRef($1, checkWCSSystem(), DEGREES);}
	| ARCMINUTE {$$ = FITSPTR->mapLenToRef($1, checkWCSSystem(), ARCMIN);}
	| ARCSECOND {$$ = FITSPTR->mapLenToRef($1, checkWCSSystem(), ARCSEC);}
	;

vvalue	: numeric sp numeric 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), localSystem, DEGREES);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| PHYCOORD sp PHYCOORD 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), PHYSICAL);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| IMGCOORD sp IMGCOORD 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), IMAGE);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| DEGREE sp DEGREE 
	{
	  Vector r=FITSPTR->mapLenToRef(Vector($1,$3),checkWCSSystem(),DEGREES);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| ARCMINUTE sp ARCMINUTE 
	{
	  Vector r=FITSPTR->mapLenToRef(Vector($1,$3),checkWCSSystem(),ARCMIN);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| ARCSECOND sp ARCSECOND 
	{
	  Vector r=FITSPTR->mapLenToRef(Vector($1,$3),checkWCSSystem(),ARCSEC);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	;

numberof: N_ '=' INT {$$ = $3;}
	;

sexagesimal: SEXSTR {$$ = parseDMS($1);}
	;

coord	: sexagesimal sp sexagesimal
	{
	  Vector r;
	  CoordSystem sys = checkWCSSystem();
	  SkyFrame sky = checkWCSSky();
	  if (sky == GALACTIC || sky == ECLIPTIC) 
	    r = FITSPTR->mapToRef(Vector($1,$3), sys, sky);
	  else
	    r = FITSPTR->mapToRef(Vector($1*360./24.,$3), sys, sky);

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| HOUR MINUTE SECOND {mksign2 = mksign;} sp INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(hmsToDegree(mksign2,$1,$2,$3),dmsToDegree(mksign,$6,$7,$8)),
	    checkWCSSystem(), checkWCSSky());
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| INT ARCMINUTE ARCSECOND {mksign2 = mksign;} sp 
	  INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(dmsToDegree(mksign2,$1,$2,$3),dmsToDegree(mksign,$6,$7,$8)),
	    checkWCSSystem(), checkWCSSky());
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| numeric sp numeric 
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), localSystem, localSky);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| DEGREE sp DEGREE
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), 
	    checkWCSSystem(), checkWCSSky());
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| IMGCOORD sp IMGCOORD
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), IMAGE);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| PHYCOORD sp PHYCOORD
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), PHYSICAL);
	  $$[0] = r[0];
	  $$[1] = r[1];
	}
	| IMGCOORD sp PHYCOORD
	{
	  Vector rx = FITSPTR->mapToRef(Vector($1,$1), IMAGE);
	  Vector ry = FITSPTR->mapToRef(Vector($3,$3), PHYSICAL);
	  $$[0] = rx[0];
	  $$[1] = ry[0];
	  $$[2] = 1;
	}
	| PHYCOORD sp IMGCOORD
	{
	  Vector rx = FITSPTR->mapToRef(Vector($1,$1), PHYSICAL);
	  Vector ry = FITSPTR->mapToRef(Vector($3,$3), IMAGE);
	  $$[0] = rx[0];
	  $$[1] = ry[0];
	  $$[2] = 1;
	}
	;

coordSystem :IMAGE_ {$$ = IMAGE;}
	| PHYSICAL_ {$$ = PHYSICAL;}
	| DETECTOR_ {$$ = DETECTOR;}
	| AMPLIFIER_ {$$ = AMPLIFIER;}
	| wcsSystem {$$ = $1; globalWCS = (CoordSystem)$1;}
	;

wcsSystem : WCS_ {$$ = WCS;}
	| WCSA_ {$$ = WCSA;}
	| WCSB_ {$$ = WCSB;}
	| WCSC_ {$$ = WCSC;}
	| WCSD_ {$$ = WCSD;}
	| WCSE_ {$$ = WCSE;}
	| WCSF_ {$$ = WCSF;}
	| WCSG_ {$$ = WCSG;}
	| WCSH_ {$$ = WCSH;}
	| WCSI_ {$$ = WCSI;}
	| WCSJ_ {$$ = WCSJ;}
	| WCSK_ {$$ = WCSK;}
	| WCSL_ {$$ = WCSL;}
	| WCSM_ {$$ = WCSM;}
	| WCSN_ {$$ = WCSN;}
	| WCSO_ {$$ = WCSO;}
	| WCSP_ {$$ = WCSP;}
	| WCSQ_ {$$ = WCSQ;}
	| WCSR_ {$$ = WCSR;}
	| WCSS_ {$$ = WCSS;}
	| WCST_ {$$ = WCST;}
	| WCSU_ {$$ = WCSU;}
	| WCSV_ {$$ = WCSV;}
	| WCSW_ {$$ = WCSW;}
	| WCSX_ {$$ = WCSX;}
	| WCSY_ {$$ = WCSY;}
	| WCSZ_ {$$ = WCSZ;}
	| WCS0_ {$$ = WCS0;}
	;

skyFrame : FK4_ {$$ = FK4;}
	| B1950_ {$$ = FK4;}
	| FK5_ {$$ = FK5;}
	| J2000_ {$$ = FK5;}
	| ICRS_ {$$ = ICRS;}
	| GALACTIC_ {$$ = GALACTIC;}
	| ECLIPTIC_ {$$ = ECLIPTIC;}
	;

skyFormat : DEGREES_ {$$=DEGREES;}
	| ARCMIN_ {$$=ARCMIN;}
	| ARCSEC_ {$$=ARCSEC;}
	;

property : SELECT_ {$$ = Marker::SELECT;}
	| EDIT_ {$$ = Marker::EDIT;}
	| MOVE_  {$$ = Marker::MOVE;}
	| ROTATE_ {$$ = Marker::ROTATE;}
	| DELETE_ {$$ = Marker::DELETE;}
	| HIGHLITE_ {$$ = Marker::HIGHLITE;}
	| INCLUDE_ {$$ = Marker::INCLUDE;}
	| SOURCE_ {$$ = Marker::SOURCE;}
	| FIXED_ {$$ = Marker::FIXED;}
	;

callBack : SELECT_ {$$ = CallBack::SELECTCB;}
	| UNSELECT_ {$$ = CallBack::UNSELECTCB;}
	| HIGHLITE_ {$$ = CallBack::HIGHLITECB;}
	| UNHIGHLITE_ {$$ = CallBack::UNHIGHLITECB;}
	| BEGIN_ MOVE_  {$$ = CallBack::MOVEBEGINCB;}
	| MOVE_  {$$ = CallBack::MOVECB;}
	| END_ MOVE_  {$$ = CallBack::MOVEENDCB;}
	| BEGIN_ EDIT_ {$$ = CallBack::EDITBEGINCB;}
	| EDIT_ {$$ = CallBack::EDITCB;}
	| END_ EDIT_ {$$ = CallBack::EDITENDCB;}
	| BEGIN_ ROTATE_ {$$ = CallBack::ROTATEBEGINCB;}
	| ROTATE_ {$$ = CallBack::ROTATECB;}
	| END_ ROTATE_ {$$ = CallBack::ROTATEENDCB;}
	| DELETE_ {$$ = CallBack::DELETECB;}
	| TEXT_ {$$ = CallBack::TEXTCB;}
	| COLOR_ {$$ = CallBack::COLORCB;}
	| WIDTH_ {$$ = CallBack::LINEWIDTHCB;}
	| PROPERTY_ {$$ = CallBack::PROPERTYCB;}
	| FONT_ {$$ = CallBack::FONTCB;}
	| UPDATE_ {$$ = CallBack::UPDATECB;}
	;

global	: global sp globalProperty
	| globalProperty
	;

globalProperty : property '=' yesno 
	{
	  setProps(&globalProps,$1,$3);
	  setProps(&localProps,$1,$3);
	}
	| COLOR_ '=' STRING 
	{
	  strncpy(globalColor,$3,16);
	  strncpy(localColor,$3,16);
	}
	| WIDTH_ '=' INT {globalWidth = localWidth = $3;}
	| FONT_ '=' STRING 
	{
	  strncpy(globalFont,$3,32);
	  strncpy(localFont,$3,32);
	}
	| TEXT_ '=' STRING 
	{
	  strncpy(globalText,$3,80);
	  strncpy(localText,$3,80);
	}
	| SOURCE_
	{
	  setProps(&globalProps,Marker::SOURCE,1);
	  setProps(&localProps,Marker::SOURCE,1);
	}
	| BACKGROUND_
	{
	  setProps(&globalProps,Marker::SOURCE,0);
	  setProps(&localProps,Marker::SOURCE,0);
	}
	| POINT_ '=' pointProp 
	{
	  globalPoint = localPoint = $3;
	}
	| LINE_ '=' INT INT 
	{
	  globalLine1 = localLine1 = $3;
	  globalLine2 = localLine2 = $4;
	}
	| VECTOR_ '=' INT 
	{
	  globalVector = localVector = $3;
	}
	| RULER_ '=' globalRuler
	{
	}
	| COMPASS_ '=' globalCompass STRING STRING INT INT
	{
	  strncpy(globalCompassNorth,$4,80);
	  strncpy(globalCompassEast,$5,80);
	  strncpy(localCompassNorth,$4,80);
	  strncpy(localCompassEast,$5,80);
	  globalCompassNArrow = localCompassNArrow = $6;
	  globalCompassEArrow = localCompassEArrow = $7;
	}
	| TEXTANGLE_ '=' angle
	{
	  globalTextAngle = localTextAngle = $3;
	}
	| WCS_ '=' wcsSystem {globalWCS = (CoordSystem)$3;}
	;

globalRuler : coordSystem skyFrame coordSystem skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = (CoordSystem)$1;
	  globalRulerSkyFrame = localRulerSkyFrame = (SkyFrame)$2;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$3;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$4;
	}
	| coordSystem coordSystem
	{
	  globalRulerCoordSystem = localRulerCoordSystem = (CoordSystem)$1;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$2;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	| coordSystem skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = (CoordSystem)$1;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFrame coordSystem
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = (SkyFrame)$1;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$2;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	| skyFrame skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = (SkyFrame)$1;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$2;
	}
	| LINEAR_ coordSystem
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$2;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	| LINEAR_ skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = IMAGE;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$1;
	}
	| PIXELS_
	{
	  globalRulerCoordSystem = localRulerCoordSystem = IMAGE;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = IMAGE;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	;

globalCompass : coordSystem skyFrame
	{
	  globalCompassCoordSystem = localCompassCoordSystem = (CoordSystem)$1;
	  globalCompassSkyFrame = localCompassSkyFrame = (SkyFrame)$2;
	}
	| coordSystem
	{
	  globalCompassCoordSystem = localCompassCoordSystem = (CoordSystem)$1;
	  globalCompassSkyFrame = localCompassSkyFrame = FK5;
	}
	| skyFrame
	{
	  globalCompassCoordSystem = localCompassCoordSystem = WCS;
	  globalCompassSkyFrame = localCompassSkyFrame = (SkyFrame)$1;
	}
	| LINEAR_
	{
	  globalCompassCoordSystem = localCompassCoordSystem = WCS;
	  globalCompassSkyFrame = localCompassSkyFrame = FK5;
	}
	;

initGlobal:{
	  // global properties
	  globalSystem = PHYSICAL;
	  globalWCS = WCS;
	  globalSky = NATIVEWCS;
	  globalTile = 1;
	  globalProps =
	    Marker::SELECT | Marker::EDIT | Marker::MOVE |
	    Marker::ROTATE | Marker::DELETE | Marker::HIGHLITE |
	    Marker::INCLUDE | Marker::SOURCE;
	  strcpy(globalColor,"green");
          globalWidth = 1;
	  strcpy(globalFont,"helvetica 10 normal");
	  strcpy(globalText,"");

	  // unique properties
	  globalLine1 = 0;
	  globalLine2 = 0;
	  globalVector = 1;
	  globalRulerCoordSystem = PHYSICAL;
	  globalRulerSkyFrame = FK5;
	  globalRulerDistSystem = PHYSICAL;
	  globalRulerDistFormat = DEGREES;
	  globalCompassCoordSystem = PHYSICAL;
	  globalCompassSkyFrame = FK5;
	  strcpy(globalCompassNorth,"N");
	  strcpy(globalCompassEast,"E");
	  globalCompassNArrow = 1;
	  globalCompassEArrow = 1;
	  globalPoint = BOXCIRCLE;
	  globalTextAngle=0;

	  aStatus = 0;
	  cStatus = 0;
	} 
	;

local	: local sp localProperty
	| localProperty
	;

localProperty : property '=' yesno {setProps(&localProps,$1,$3);}
	| COLOR_ '=' STRING {strncpy(localColor,$3,16);}
	| WIDTH_ '=' INT {localWidth = $3;}
	| FONT_ '=' STRING {strncpy(localFont,$3,32);}
	| TEXT_ '=' STRING {strncpy(localText,$3,80);}
	| TAG_ '=' STRING {taglist.append(new Tag($3));}
	| CALLBACK_ '=' callBack STRING STRING {cblist.append(
	    new CallBack(FR->getInterp(),(CallBack::Type)$3,$4,$5));}
	| SOURCE_ {setProps(&localProps,Marker::SOURCE,1);}
	| BACKGROUND_ {setProps(&localProps,Marker::SOURCE,0);}

	| POINT_ '=' pointProp {localPoint = $3;}
	| LINE_ '=' INT INT {localLine1=$3; localLine2=$4;}
	| VECTOR_ '=' INT {localVector=$3;}
	| RULER_ '=' localRuler
	| COMPASS_ '=' localCompass STRING STRING INT INT
	{
	  strncpy(localCompassNorth,$4,80);
	  strncpy(localCompassEast,$5,80);
	  localCompassNArrow = $6;
	  localCompassEArrow = $7;
	}
	| TEXTANGLE_ '=' angle {localTextAngle=$3;}
	| CPANDA_ '=' localCpanda
	| EPANDA_ '=' localEpanda
	| BPANDA_ '=' localBpanda
	;

localRuler : coordSystem skyFrame coordSystem skyFormat
	{
	  localRulerCoordSystem = (CoordSystem)$1;
	  localRulerSkyFrame = (SkyFrame)$2;
	  localRulerDistSystem = (CoordSystem)$3;
	  localRulerDistFormat = (SkyFormat)$4;
	}
	| coordSystem coordSystem
	{
	  localRulerCoordSystem = (CoordSystem)$1;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = (CoordSystem)$2;
	  localRulerDistFormat = DEGREES;
	}
	| coordSystem skyFormat
	{
	  localRulerCoordSystem = (CoordSystem)$1;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFrame coordSystem
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = (SkyFrame)$1;
	  localRulerDistSystem = (CoordSystem)$2;
	  localRulerDistFormat = DEGREES;
	}
	| skyFrame skyFormat
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = (SkyFrame)$1;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$2;
	}
	| LINEAR_ coordSystem
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = (CoordSystem)$2;
	  localRulerDistFormat = DEGREES;
	}
	| LINEAR_ skyFormat
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFormat
	{
	  localRulerCoordSystem = IMAGE;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$1;
	}
	| PIXELS_
	{
	  localRulerCoordSystem = IMAGE;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = IMAGE;
	  localRulerDistFormat = DEGREES;
	}
	;

localCompass : coordSystem skyFrame
	{
	  localCompassCoordSystem = (CoordSystem)$1;
	  localCompassSkyFrame = (SkyFrame)$2;
	}
	| coordSystem
	{
	  localCompassCoordSystem = (CoordSystem)$1;
	  localCompassSkyFrame = FK5;
	}
	| skyFrame
	{
	  localCompassCoordSystem = WCS;
	  localCompassSkyFrame = (SkyFrame)$1;
	}
	| LINEAR_
	{
	  localCompassCoordSystem = WCS;
	  localCompassSkyFrame = FK5;
	}
	;

localCpanda: {aNum=0; aAngNum=0;} '(' aAngs ')' '(' aRads ')' {localCpanda = 2;}
	| IGNORE_ {localCpanda=0;}
	;

localEpanda: {aNum=0; aAngNum=0, aAngle=0;} 
	  '(' aAngs ')' '(' vRads ')' '(' angle ')' {aAngle=$9;localEpanda=2;}
	| IGNORE_ {localEpanda=0;}
	;

localBpanda: {aNum=0; aAngNum=0, aAngle=0;} 
	  '(' aAngs ')' '(' vRads ')' '(' angle ')' {aAngle=$9;localBpanda=2;}
	| IGNORE_ {localBpanda=0;}
	;

initLocal : {
	  // needed for annulus, ellipse annulus, and box annulus
	  aNum = 2;

	  // composite (previous conjuction found?)
	  if (!cStatus) FR->resetCompositeMarker();

	  // global properties
	  localSystem = globalSystem;
	  localSky = globalSky;
	  localProps = globalProps;
	  strcpy(localColor,globalColor);
	  localWidth = globalWidth;
	  strcpy(localFont,globalFont);
	  strcpy(localText,globalText);
	  strcpy(localComment,"");
	  taglist.deleteAll();
	  cblist.deleteAll();

	  // unique properties
	  localLine1 = globalLine1;
	  localLine2 = globalLine2;
	  localVector = globalVector;
	  localPoint = globalPoint;
	  localRulerCoordSystem = globalRulerCoordSystem;
	  localRulerSkyFrame = globalRulerSkyFrame;
	  localRulerDistSystem = globalRulerDistSystem;
	  localRulerDistFormat = globalRulerDistFormat;
	  localCompassCoordSystem = globalCompassCoordSystem;
	  localCompassSkyFrame = globalCompassSkyFrame;
	  strcpy(localCompassNorth,globalCompassNorth);
	  strcpy(localCompassEast,globalCompassEast);
	  localCompassNArrow = globalCompassNArrow;
	  localCompassEArrow = globalCompassEArrow;
	  localTextAngle = globalTextAngle;
	  localCpanda = 1;
	  localEpanda = 1;
	  localBpanda = 1;
	}
	;

pointProp : CIRCLE_ {$$ = CIRCLE;}
	| BOX_ {$$ = BOX;}
	| DIAMOND_ {$$ = DIAMOND;}
	| CROSS_ {$$ = CROSS;}
	| X_ {$$ = XPT;}
	| ARROW_ {$$ = ARROW;}
	| BOXCIRCLE_ {$$ = BOXCIRCLE;}
	;

include	: /* empty */ {setProps(&localProps, Marker::INCLUDE, 1);}
	| '+' {setProps(&localProps, Marker::INCLUDE, 1);}
	| '-' {setProps(&localProps, Marker::INCLUDE, 0);}
	;

nonshape : TEXT_ bp coord ep conjuction nonshapeComment
	  {FR->createTextCmd(Vector($3),
	    localTextAngle,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| COMPOSITE_ bp coord sp optangle ep conjuction nonshapeComment
	  {FR->createCompositeCmd(Vector($3),
	    $5,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| VECTOR_ bp coord sp value sp angle ep conjuction nonshapeComment
	  {FR->createVectCmd(Vector($3),
	    $5,$7,
	    localVector,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| PROJECTION_ bp coord sp coord sp value ep conjuction nonshapeComment
	{
	  // hard coded into projection.tcl
	  FR->createProjectionCmd(Vector($3),
	    Vector($5),
	    $7,
	    "ProjectionPlot", "ProjectionPlotDelete",
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| RULER_ bp coord sp coord ep conjuction nonshapeComment
	  {FR->createRulerCmd(Vector($3),
	    Vector($5),
   	    localRulerCoordSystem, localRulerSkyFrame,
	    localRulerDistSystem, localRulerDistFormat,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| COMPASS_ bp coord sp value ep conjuction nonshapeComment
	  {FR->createCompassCmd(Vector($3), 
	   $5,
	   localCompassNorth, localCompassEast, 
	   localCompassNArrow, localCompassEArrow,
   	   localCompassCoordSystem, localCompassSkyFrame,
	   localColor,localWidth,localFont,
	   localText,localProps,localComment,taglist,cblist);}
	;

shape	: CIRCLE_ bp coord sp value ep conjuction shapeComment
	  {FR->createCircleCmd(Vector($3),
	    $5,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| ANNULUS_ bp coord sp value sp value ep conjuction shapeComment
	  {FR->createAnnulusCmd(Vector($3),
	    $5,$7,1,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| ANNULUS_ bp coord sp value sp value sp aRads ep conjuction 
	    shapeComment
	{
	  aAnnuli[0] = $5;
	  aAnnuli[1] = $7;
	  FR->createAnnulusCmd(Vector($3),
	    aNum,aAnnuli,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| ANNULUS_ bp coord sp value sp value sp numberof ep conjuction 
	    shapeComment
	  {FR->createAnnulusCmd(Vector($3),
	    $5,$7,$9,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| CPANDA_ bp coord sp angle sp angle sp INT sp 
	    value sp value sp INT ep conjuction shapeComment
	{
	  switch (localCpanda) {
	  case 0: /* ignore it */
	    break;
	  case 1: /* normal cpanda */
	    FR->createCpandaCmd(Vector($3),
	      $5,$7,$9,
	      $11,$13,$15,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case 2: /* one of our special pandas */
	    FR->createCpandaCmd(Vector($3),
	      aAngNum,aAngles,
	      aNum,aAnnuli,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  }
	}

	| ELLIPSE_ bp coord sp vvalue sp optangle ep conjuction shapeComment
	{
	  // for ellipse annulus
	  aStatus = 1;
	  aCenter = Vector($3);
	  aAngles[0] = $7;
	  aVector[0] = Vector($5);
	  aNumsao = 1;
	  strncpy(aColor,localColor,16);
	  aWidth = localWidth;
	  strncpy(aFont,localFont,32);
	  strncpy(aText,localText,80);
	  strncpy(aComment,localComment,80);
	  aProps = localProps;

	  FR->createEllipseCmd(Vector($3),
	    Vector($5),
	    $7,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| ELLIPSE_ bp coord sp vvalue sp vvalue sp optangle ep conjuction 
	    shapeComment
	{
	  // prefered syntax
	  FR->createEllipseAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),1,
	    $9,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| ELLIPSE_ bp coord sp vvalue sp vvalue sp 
	    numberof sp optangle ep conjuction shapeComment
	{
	  // prefered syntax
	  FR->createEllipseAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),$9,
	    $11,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| ELLIPSE_ bp coord sp vvalue sp vvalue sp
	    vRads sp optangle ep conjuction shapeComment
	{
	  // prefered syntax
	  aVector[0] = Vector($5);
	  aVector[1] = Vector($7);
	  FR->createEllipseAnnulusCmd(Vector($3),
	    aNum,aVector,
	    $11,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}

	| ELLIPSE_ bp coord sp vvalue sp optangle ep '&' '!' 
	  ELLIPSE_ bp coord sp vvalue sp optangle ep
	{	
	  // old saoimage syntax
	  aStatus = 2;
	  aVector[aNumsao++] = Vector($5);
	}
	| EPANDA_ bp coord sp angle sp angle sp INT sp 
	    vvalue sp vvalue sp INT sp optangle ep conjuction shapeComment
	{
	  switch (localEpanda) {
	  case 0: /* ignore it */
	    break;
	  case 1: /* normal epanda */
	    FR->createEpandaCmd(Vector($3),
	      $5,$7,$9,
	      Vector($11),Vector($13),$15,
	      $17,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case 2: /* one of our special pandas */
	    FR->createEpandaCmd(Vector($3),
	      aAngNum,aAngles,
	      aNum,aVector,
	      aAngle,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  }
	}

	| BOX_ bp coord sp vvalue sp optangle ep conjuction shapeComment
	{
	  // for box annulus
	  aStatus = 3;
	  aCenter = Vector($3);
	  aAngles[0] = $7;
	  aVector[0] = Vector($5);
	  aNumsao = 1;
	  strncpy(aColor,localColor,16);
	  aWidth = localWidth;
	  strncpy(aFont,localFont,32);
	  strncpy(aText,localText,80);
	  strncpy(aComment,localComment,80);
	  aProps = localProps;

	  FR->createBoxCmd(Vector($3),
	    Vector($5),
	    $7,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| ROTBOX_ bp coord sp vvalue sp optangle ep conjuction shapeComment
	  {FR->createBoxCmd(Vector($3),
	    Vector($5),
	    $7,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| BOX_ bp coord sp vvalue sp vvalue sp optangle ep conjuction 
	    shapeComment
	{
	  // prefered syntax
	  FR->createBoxAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),1,
	    $9,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| BOX_ bp coord sp vvalue sp vvalue sp 
	    vRads sp optangle ep conjuction shapeComment
	{
	  // prefered syntax
	  aVector[0] = Vector($5);
	  aVector[1] = Vector($7);
	  FR->createBoxAnnulusCmd(Vector($3),
	    aNum,aVector,
	    $11,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| BOX_ bp coord sp vvalue sp vvalue sp 
	    numberof sp optangle ep conjuction shapeComment
	{
	  // prefered syntax
	  FR->createBoxAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),$9,
	    $11,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}

	| BOX_ bp coord sp vvalue sp optangle ep '&' '!' 
	  BOX_ bp coord sp vvalue sp optangle ep
	{	
	  // old saoimage syntax
	  aStatus = 4;
	  aVector[aNumsao++] = Vector($5);
	}
	| BPANDA_ bp coord sp angle sp angle sp INT sp 
	    vvalue sp vvalue sp INT sp optangle ep conjuction shapeComment
	{
	  switch (localBpanda) {
	  case 0: /* ignore it */
	    break;
	  case 1: /* normal bpanda */
	    FR->createBpandaCmd(Vector($3),
	      $5,$7,$9,
	      Vector($11),Vector($13),$15,
	      $17,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case 2: /* one of our special pandas */
	    FR->createBpandaCmd(Vector($3),
	      aAngNum,aAngles,
	      aNum,aVector,
	      aAngle,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  }
	}

	| LINE_ bp coord sp coord ep conjuction shapeComment
	  {FR->createLineCmd(Vector($3),
	    Vector($5),
	    localLine1,localLine2,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}

	| POINT_ bp coord ep conjuction shapeComment 
	{
	  switch (localPoint) {
	  case CIRCLE:
	    FR->createCirclePointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case BOX:
	    FR->createBoxPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case DIAMOND:
	    FR->createDiamondPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case CROSS:
	    FR->createCrossPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case XPT:
	    FR->createExPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case ARROW:
	    FR->createArrowPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  case BOXCIRCLE:
	    FR->createBoxCirclePointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);
	    break;
	  }
	}
	| CIRCLE_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createCirclePointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| BOX_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createBoxPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| DIAMOND_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createDiamondPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| CROSS_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createCrossPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| X_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createExPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| ARROW_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createArrowPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| BOXCIRCLE_ POINT_ bp coord ep conjuction shapeComment
	  {FR->createBoxCirclePointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}

	| POLYGON_ {polylist.deleteAll();} bp polyNodes ep conjuction 
	    shapeComment
	  {FR->createPolygonCmd(polylist, 
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist,cblist);} 

	| PIE_ bp coord sp angle sp angle ep conjuction shapeComment
	| PIE_ bp coord sp angle sp angle sp aAngs ep conjuction shapeComment
	| PIE_ bp coord sp angle sp angle sp numberof ep conjuction 
	    shapeComment
	| FIELD_ bp ep conjuction shapeComment

 	| TEXT_ bp coord ep conjuction shapeComment
	  {FR->createTextCmd(Vector($3),
	    localTextAngle,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| TEXT_ bp coord sp STRING ep {strncpy(localText,$5,80);} conjuction 
	    shapeComment
	  {FR->createTextCmd(Vector($3),
	    localTextAngle,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| COMPOSITE_ bp coord sp optangle ep conjuction shapeComment
	  {FR->createCompositeCmd(Vector($3),
	    $5,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| VECTOR_ bp coord sp value sp angle ep conjuction shapeComment
	  {FR->createVectCmd(Vector($3),
	    $5,$7,
	    localVector,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| PROJECTION_ bp coord sp coord sp value ep conjuction shapeComment
	{
	  // hard coded into projection.tcl
	  FR->createProjectionCmd(Vector($3),
	    Vector($5),
	    $7,
	    "ProjectionPlot", "ProjectionPlotDelete",
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);
	}
	| RULER_ bp coord sp coord ep conjuction shapeComment
	  {FR->createRulerCmd(Vector($3),
	    Vector($5),
   	    localRulerCoordSystem, localRulerSkyFrame,
	    localRulerDistSystem, localRulerDistFormat,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist,cblist);}
	| COMPASS_ bp coord sp value ep conjuction shapeComment
	  {FR->createCompassCmd(Vector($3), 
	   $5,
	   localCompassNorth, localCompassEast, 
	   localCompassNArrow, localCompassEArrow,
   	   localCompassCoordSystem, localCompassSkyFrame,
	   localColor,localWidth,localFont,
	   localText,localProps,localComment,taglist,cblist);}
	;

polyNodes : polyNodes sp polyNode
	| polyNode
	;

polyNode : coord {polylist.append(new Vertex($1));}
	;

aRads	: aRads sp aRad
	| aRad
	;

aRad	: value 
	{
	  if (aNum < MAXANNULI)
	    aAnnuli[aNum++] = $1;
	}
	;

aAngs	: aAngs sp aAng
	| aAng
	;

aAng	: angle 
	{
	  if (aAngNum < MAXANGLES)
	    aAngles[aAngNum++] = $1;
	}
	;

vRads	: vRads sp vRad
	| vRad
	;

vRad	: value sp value {aVector[aNum++] = Vector($1,$3);}
	;

postLocal : /* empty */
	{
	  // old style annulus
	  switch (aStatus) {
	  case 0: // do nothing
	    break;
	  case 1: // we found just an ellipse, do nothing
	    break;
	  case 2: // ok we have an ellipse annulus
	    FR->markerDeleteLastCmd(); // delete the previous ellipse
	    FR->createEllipseAnnulusCmd(aCenter,
	      aNumsao,aVector,
	      aAngles[0],
	      aColor,aWidth,aFont,aText,aProps,aComment,taglist,cblist);
	    break;
	  case 3: // we found just a box, do nothing
	    break;
	  case 4: // ok, we have a box annulus
	    FR->markerDeleteLastCmd(); // delete the previous box
	    FR->createBoxAnnulusCmd(aCenter,
	      aNumsao,aVector,
	      aAngles[0],
	      aColor,aWidth,aFont,aText,aProps,aComment,taglist,cblist);
	    break;
	  }
	  aStatus = 0;
	}
	;
%%

static void setProps(unsigned short* props, unsigned short prop, int value)
{
  if (value)
    *props |= prop;
  else
    *props &= ~prop;
}

static CoordSystem checkWCSSystem()
{
  switch (localSystem) {
  case IMAGE:
  case PHYSICAL:
    return WCS;
  default:
   return localSystem;
  }
}

static SkyFrame checkWCSSky()
{
  switch (localSystem) {
  case IMAGE:
  case PHYSICAL:
    return NATIVEWCS;
  default:
   return localSky;
  }
}

