// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "basemarker.h"
#include "framebase.h"
#include "util.h"

// Base Markers Public

BaseMarker::BaseMarker(const BaseMarker& a) : Marker(a)
{
  startAng_ = a.startAng_;
  stopAng_ = a.stopAng_;

  numAnnuli_ = a.numAnnuli_;
  annuli_ = new Vector[a.numAnnuli_];
  for (int i=0; i<a.numAnnuli_; i++)
    annuli_[i] = a.annuli_[i];

  numPoints_ = a.numPoints_;

  vertices_ = NULL;
}

BaseMarker::BaseMarker(FrameBase* p, const Vector& ctr, 
		       double ang, int seg, 
		       const char* clr, int w, const char* f, 
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag, const List<CallBack>& cb)
  : Marker(p, ctr, ang, clr, w, f, t, prop, c, tag, cb)
{
  startAng_ = 0;
  stopAng_ = 360;

  numAnnuli_ = 0;
  annuli_ = NULL;

  numPoints_ = seg;
  vertices_ = NULL;
}

BaseMarker::~BaseMarker()
{
  if (annuli_)
    delete [] annuli_;
}

void BaseMarker::updateBBox()
{
  // generate handles
  updateHandles();

  // bound handles
  bbox = BBox(handle[0]);
  for (int i=1; i<numHandle; i++)
    bbox.bound(handle[i]);

  // make room for handles
  bbox.expand(3);

  // calculate overall bbox
  calcAllBBox();
}

void BaseMarker::updateCoords(const Matrix& mx)
{
  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] *= Scale(mx);

  Marker::updateCoords(mx);
}

void BaseMarker::ps(int mode)
{
  Marker::ps(mode);

  newVertices();

  for (int i=0; i<numAnnuli_; i++) {
    for (int j=0; j<numPoints_; j++) {
      Vector v =  vertices_[i][j] * parent->refToCanvas;

      ostringstream str;
      if (j==0)
	str << "newpath " << endl
	    << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
      else
	str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    }

    ostringstream str;
    str << "stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  if (!(properties & INCLUDE)) {
    Vector ll = vertices_[numAnnuli_][0] * parent->refToCanvas;
    Vector ur = vertices_[numAnnuli_][1] * parent->refToCanvas;

    psColor(mode, "red");

    ostringstream str;
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  deleteVertices();
}

void BaseMarker::setAnnuli(const Vector& r)
{
  annuli_[0] = r;
  updateBBox();

  doCallBack(CallBack::EDITCB);
}

void BaseMarker::setAnnuli(const Vector& r1, const Vector& r2, int rn)
{
  numAnnuli_ = rn+1;
  if (annuli_)
    delete [] annuli_;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = ((r2-r1)/rn)*i+r1;
  sortAnnuli();

  numHandle = 4 + numAnnuli_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void BaseMarker::setAnnuli(const Vector* r, int rn)
{
  numAnnuli_ = rn;
  if (annuli_)
    delete [] annuli_;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = r[i];
  sortAnnuli();

  numHandle = 4 + numAnnuli_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

int BaseMarker::insertAnnuli(Vector r)
{
  // new size array
  Vector* old = annuli_;
  annuli_ = new Vector[numAnnuli_+1];

  // copy old values
  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = old[i];

  // delete old
  if (old)
    delete [] old;

  // new size on end
  annuli_[numAnnuli_] = r;

  numAnnuli_++;
  numHandle++;

  updateBBox();

  // return handle number
  return numAnnuli_+4;
}

void BaseMarker::deleteAnnuli(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (numAnnuli_>2 && hh<numAnnuli_) {
      // new radii array
      Vector* old = annuli_;
      annuli_ = new Vector[numAnnuli_-1];

      // copy up to radii in question
      for (int i=0; i<hh; i++)
	annuli_[i] = old[i];

      // copy remainder
      for (int i=hh; i<numAnnuli_-1; i++)
	annuli_[i] = old[i+1];

      if (old)
	delete [] old;
      numAnnuli_--;

      numHandle = 4 + numAnnuli_;

      updateBBox();
      doCallBack(CallBack::EDITCB);
    }
  }
}

// Base Markers Private

void BaseMarker::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  newVertices();

  for (int i=0; i<numAnnuli_; i++) {
    XPoint* p = new XPoint[numPoints_];

    for (int j=0; j<numPoints_; j++) {
      Vector v = (vertices_[i][j] * mx).round();
      p[j].x = (short)v[0];
      p[j].y = (short)v[1];
    }

    XDRAWLINES(display, drawable, gc, p, numPoints_, CoordModeOrigin);
    delete [] p;
  }

  if (!(properties & INCLUDE)) {
    Vector ll = (vertices_[numAnnuli_][0] * mx).round();
    Vector ur = (vertices_[numAnnuli_][1] * mx).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }

  deleteVertices();
}

void BaseMarker::deleteVertices()
{
  if (vertices_) {
    for (int i=0; i<numAnnuli_+1; i++)
      if (vertices_[i])
	delete [] vertices_[i];
    delete [] vertices_;
  }
  vertices_ = NULL;
}

void BaseMarker::sortAnnuli()
{
  for (int i=0; i<numAnnuli_; i++)
    for (int j=i+1; j<numAnnuli_; j++)
      if (annuli_[i][0]>annuli_[j][0]) {
	Vector d = annuli_[i];
	annuli_[i] = annuli_[j];
	annuli_[j] = d;
      }
}

Matrix BaseMarker::fwdRefMatrix()
{
  if (properties & FIXED) {
    Vector cc = center * parent->refToCanvas;
    return Rotate(calcAngle()) * Translate(cc) * parent->canvasToRef;
  }
  else
    return Rotate(angle) * flip_ * Translate(center);
}

Matrix BaseMarker::fwdCanvasMatrix()
{
  if (properties & FIXED) {
    Vector cc = center * parent->refToCanvas;
    return Rotate(calcAngle()) * Translate(cc);
  }
  else
    return Rotate(angle) * flip_ * Translate(center) * parent->refToCanvas;
}

Matrix BaseMarker::bckRefMatrix()
{
  if (properties & FIXED) {
    Vector cc = center * parent->refToCanvas;
    return Translate(-cc) * Rotate(-calcAngle());
  }
  else
    return parent->canvasToRef * Translate(-center) * flip_ * Rotate(-angle);
}

Matrix BaseMarker::bckCanvasMatrix()
{
  if (properties & FIXED) {
    Vector cc = center * parent->refToCanvas;
    return parent->refToCanvas * Translate(-cc) * Rotate(-calcAngle());
  }
  else
    return Translate(-center) * flip_ * Rotate(-angle);
}
