// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <string.h>
#include <ctype.h>

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include "hist.h"
#include "util.h"

#include "fitsy.h"
#include "filter.h"

#define FILTERSIZE 262144
#define MULTWCS 27

FitsHist::FitsHist(FitsFile* fits, int w, int h, int d,
		   Matrix& m, Function func, Vector block)
  : width(w), height(h), depth(d)
{
  size = width*height*depth;

  xcol = NULL;
  ycol = NULL;
  zcol = NULL;

  fitsy = NULL;
  filter = NULL;

  valid_ = 0;

  if (!initHeader(fits, m))
    return;

  // we need to translate by another .5 for the offset from Data to Image
  Matrix mm = m * Translate(.5,.5);
  
  initLTMV(mm);
  initWCS(fits, mm, block);
  initWCSAlt(fits, mm, block);
  
  initFilter(fits);
  bin(fits, m, func, block);

  if (byteswap_)
    swap();

  deleteFilter();
  valid_ = 1;
}

FitsHist::~FitsHist()
{
  if (data_)
    delete [] (float*)data_;
}

int FitsHist::initHeader(FitsFile* fits, Matrix& m) 
{
  FitsHead* srcHead = fits->head();
  FitsTableHDU* srcHDU = (FitsTableHDU*)(srcHead->hdu());

  // make sure we have a table with columns, X, Y
  if (!fits->isTable())
    return 0;

  // get X column
  if (fits->pBinX())
    xcol = srcHDU->find(fits->pBinX());

  if (!xcol)
    return 0;

  // get Y column
  if (fits->pBinY())
    ycol = srcHDU->find(fits->pBinY());

  if (!ycol)
    return 0;

  // get Z column (if specified)
  if (fits->pBinZ() && depth > 1)
    zcol = srcHDU->find(fits->pBinZ());

  // create header
  head_ = new FitsHead(width, height, depth, -32);
  if (!head_->isValid())
    return 0;

  // OBJECT
  char* object = srcHead->getString("OBJECT");
  if (object) {
    head_->appendString("OBJECT", object, NULL);
    delete [] object;
  }

  // DATE-OBS
  char* dateobs = srcHead->getString("DATE-OBS");
  if (dateobs) {
    head_->appendString("DATE-OBS", dateobs, NULL);
    delete [] dateobs;
  }
  char* timeobs = srcHead->getString("TIME-OBS");
  if (timeobs) {
    head_->appendString("TIME-OBS", timeobs, NULL);
    delete [] timeobs;
  }
  char* dateend = srcHead->getString("DATE-END");
  if (dateend) {
    head_->appendString("DATE-END", dateend, NULL);
    delete [] dateend;
  }
  char* timeend = srcHead->getString("TIME-END");
  if (timeend) {
    head_->appendString("TIME-END", timeend, NULL);
    delete [] timeend;
  }

  return 1;
}

void FitsHist::initFilter(FitsFile* fits)
{
  FitsHead* srcHead = fits->head();

  const char* filtstr = fits->pFilter();
  if (filtstr && *filtstr) {

    ostringstream str;
    str << "bincols=(" << fits->pBinX() << ',' << fits->pBinY() << ')';

    if (byteswap_)
      str << ",convert=true";
    str << ends;

    if (!(fitsy = ft_headinit(srcHead->cards(), srcHead->headbytes())))
      cerr << "Error: Fitsy++ internal error- bad filter head" << endl;
    else {
      if (!(filter = FilterOpen((FITSHead)fitsy, (char*)filtstr, 
				(char*)str.str().c_str())))
	cerr << "Error: Fitsy++ internal error- unable to build filter"<<endl;
    }
  }
}

void FitsHist::deleteFilter()
{
  if (filter) {
    FilterClose((Filter)filter);
    filter = NULL;
  }
  if (fitsy) {
    ft_headfree((FITSHead)fitsy,0);
    fitsy = NULL;
  }
}

void FitsHist::bin(FitsFile* fits, Matrix& m, Function func, Vector block)
{
  FitsHead* srcHead = fits->head();
  FitsTableHDU* srcHDU = (FitsTableHDU*)(srcHead->hdu());

  // create image space
  float* dest = new float[size];
  memset(dest, 0, size*sizeof(float));

  // bin it up
  char* ptr = (char*)fits->data();
  int rowlen = srcHDU->width();
  int rows = srcHDU->rows();

  // third dimension
  Vector lim = depth > 1 ? fits->getBinZlim() : Vector();
  double& zmin = lim[0];
  double& zmax = lim[1];
  double zlength  = zmax-zmin;

  // filter
  int goodincr = 0;
  int goodindex = FILTERSIZE;
  int* good = NULL;
  if (filter)
    good = new int[FILTERSIZE];

  // matrix
  register double m00 = m.matrix(0,0);
  register double m10 = m.matrix(1,0);
  register double m20 = m.matrix(2,0);
  register double m01 = m.matrix(0,1);
  register double m11 = m.matrix(1,1);
  register double m21 = m.matrix(2,1);

  for (int i=0; i<rows; i++, ptr+=rowlen, goodindex++) {

    // incr filter block, if needed
    if (good && (goodindex>=FILTERSIZE)) {
      // for memory models that support internal paging
      // need at lease FILTERSIZE rows
      ptr = fits->page(ptr, rowlen*FILTERSIZE);

      int diff = srcHDU->rows() - (goodincr * FILTERSIZE);
      if (FilterEvents((Filter)filter, ptr, srcHDU->width(), 
		       (diff<FILTERSIZE) ? diff : FILTERSIZE, good)) {
	goodincr++;
	goodindex = 0;
      }
      else {
	if (good)
	  delete good;
	good = NULL;
	cerr << "Error: Fitsy++ internal error- filter failed" << endl;
      }
    }
    else {
      // for memory models that support internal paging
      // all we need is just one row
      ptr = fits->page(ptr, rowlen);
    }

    if (!good || (good && good[goodindex])) {
      register double x = xcol->value(ptr);
      register double y = ycol->value(ptr);

      register double X = x*m00 + y*m10 + m20;
      register double Y = x*m01 + y*m11 + m21;

      if (X >= 0 && X < width && Y >= 0 && Y < height) {
	if (!zcol)
	  dest[((int)Y)*width + (int)X]++;
	else {
	  int zz = (int)((zcol->value(ptr)-zmin)/zlength*depth);
	  if (zz>=0 && zz<depth)
	    dest[(zz*width*height) + ((int)Y)*width + (int)X]++;
	}
      }
    }
  }

  // for memory models that support internal paging
  fits->resetpage();

  // Average
  if (func==AVERAGE)
    for (int k=0; k<size; k++)
      dest[k] /= (block[0]*block[1]);
    
  if (good)
    delete good;

  data_ = dest;
}

void FitsHist::swap()
{
  if (!data_)
    return;

  // we now need to byteswap back to native form
  float* dest = (float*)data_;
  for (int i=0; i<size; i++) {
    const char* p = (char*)(dest+i);
    union {
      char c[4];
      float f;
    } u;
    u.c[3] = *p++;
    u.c[2] = *p++;
    u.c[1] = *p++;
    u.c[0] = *p;
    dest[i] = u.f;
  }
}

void FitsHist::initLTMV(Matrix& m) 
{
  head_->appendReal("LTM1_1", m[0][0], 10, NULL);
  head_->appendReal("LTM1_2", m[0][1], 10, NULL);
  head_->appendReal("LTM2_1", m[1][0], 10, NULL);
  head_->appendReal("LTM2_2", m[1][1], 10, NULL);
  head_->appendReal("LTV1"  , m[2][0], 10, NULL);
  head_->appendReal("LTV2"  , m[2][1], 10, NULL);
}

void FitsHist::initWCS(FitsFile* fits, Matrix& m, Vector block)
{
  FitsHead* srcHead = fits->head();

  // CTYPE
  {
    ostringstream str1;
    str1 << "TCTYP" << xcol->index() << ends;
    char* ctype1 = srcHead->getString(str1.str().c_str());
    if (!ctype1)
      ctype1 = srcHead->getString("CTYPE1");
    if (ctype1) {
      head_->appendString("CTYPE1", ctype1, NULL);
      delete [] ctype1;
    }

    ostringstream str2;
    str2 << "TCTYP" << ycol->index() << ends;
    char* ctype2 = srcHead->getString(str2.str().c_str());
    if (!ctype2)
      ctype2 = srcHead->getString("CTYPE2");
    if (ctype2) {
      head_->appendString("CTYPE2", ctype2, NULL);
      delete [] ctype2;
    }
  }

  // CUNIT
  {
    ostringstream str1;
    str1 << "TCUNI" << xcol->index() << ends;
    char* cunit1 = srcHead->getString(str1.str().c_str());
    if (!cunit1)
      cunit1 = srcHead->getString("CUNIT1");
    if (cunit1) {
      head_->appendString("CUNIT1", cunit1, NULL);
      delete [] cunit1;
    }

    ostringstream str2;
    str2 << "TCUNI" << ycol->index() << ends;
    char* cunit2 = srcHead->getString(str2.str().c_str());
    if (!cunit2)
      cunit2 = srcHead->getString("CUNIT2");
    if (cunit2) {
      head_->appendString("CUNIT2", cunit2, NULL);
      delete [] cunit2;
    }
  }

  // CRPIX
  {
    ostringstream str1;
    str1 << "TCRPX" << xcol->index() << ends;
    float crpix1 = srcHead->getReal(str1.str().c_str(), 0);
    if (!crpix1)
      crpix1 = srcHead->getReal("CRPIX1", 0);

    ostringstream str2;
    str2 << "TCRPX" << ycol->index() << ends;
    float crpix2 = srcHead->getReal(str2.str().c_str(), 0);
    if (!crpix2)
      crpix2 = srcHead->getReal("CRPIX2", 0);
    Vector pix = Vector(crpix1,crpix2) * m;

    if (crpix1)
      head_->appendReal("CRPIX1", pix[0], 10, NULL);
    if (crpix2)
      head_->appendReal("CRPIX2", pix[1], 10, NULL);
  }

  // CRVAL
  {
    ostringstream str1;
    str1 << "TCRVL" << xcol->index() << ends;
    float crval1 = srcHead->getReal(str1.str().c_str(), 0);
    if (!crval1)
      crval1 = srcHead->getReal("CRVAL1", 0);
    if (crval1)
      head_->appendReal("CRVAL1", crval1, 10, NULL);

    ostringstream str2;
    str2 << "TCRVL" << ycol->index() << ends;
    float crval2 = srcHead->getReal(str2.str().c_str(), 0);
    if (!crval2)
      crval2 = srcHead->getReal("CRVAL2", 0);
    if (crval2)
      head_->appendReal("CRVAL2", crval2, 10, NULL);
  }

  // CDELT
  {
    ostringstream str1;
    str1 << "TCDLT" << xcol->index() <<ends;
    float cdelt1 = srcHead->getReal(str1.str().c_str(), 0);
    if (!cdelt1)
      cdelt1 = srcHead->getReal("CDELT1", 0);

    ostringstream str2;
    str2 << "TCDLT" << ycol->index() <<ends;
    float cdelt2 = srcHead->getReal(str2.str().c_str(), 0);
    if (!cdelt2)
      cdelt2 = srcHead->getReal("CDELT2", 0);
    Vector delt = Vector(cdelt1*block[0],cdelt2*block[1]);

    if (cdelt1)
      head_->appendReal("CDELT1", delt[0], 10, NULL);
    if (cdelt2)
      head_->appendReal("CDELT2", delt[1], 10, NULL);
  }

  // CROTA
  {
    ostringstream str1;
    str1 << "TCROT" << xcol->index() <<ends;
    float crota1 = srcHead->getReal(str1.str().c_str(), 0);
    if (!crota1)
      crota1 = srcHead->getReal("CROTA1", 0);
    if (crota1)
      head_->appendReal("CROTA1", crota1, 10, NULL);

    ostringstream str2;
    str2 << "TCROT" << ycol->index() <<ends;
    float crota2 = srcHead->getReal(str2.str().c_str(), 0);
    if (!crota2)
      crota2 = srcHead->getReal("CROTA2", 0);
    if (crota2)
      head_->appendReal("CROTA2", crota2, 10, NULL);
  }

  // PC
  {
    ostringstream str1;
    str1 << "TP" << xcol->index() << "_" <<  xcol->index() << ends;
    float pc11 = srcHead->getReal(str1.str().c_str(), 0);
    if (!pc11)
      pc11 = srcHead->getReal("PC1_1", 0);
    if (pc11)
      head_->appendReal("PC1_1", pc11, 10, NULL);

    ostringstream str2;
    str2 << "TP" << xcol->index() << "_" <<  ycol->index() << ends;
    float pc12 = srcHead->getReal(str2.str().c_str(), 0);
    if (!pc12)
      pc12 = srcHead->getReal("PC1_2", 0);
    if (pc12)
      head_->appendReal("PC1_2", pc12, 10, NULL);

    ostringstream str3;
    str3 << "TP" << ycol->index() << "_" <<  xcol->index() << ends;
    float pc21 = srcHead->getReal(str3.str().c_str(), 0);
    if (!pc21)
      pc21 = srcHead->getReal("PC2_1", 0);
    if (pc21)
      head_->appendReal("PC2_1", pc21, 10, NULL);

    ostringstream str4;
    str4 << "TP" << ycol->index() << "_" <<  ycol->index() << ends;
    float pc22 = srcHead->getReal(str4.str().c_str(), 0);
    if (!pc22)
      pc22 = srcHead->getReal("PC2_2", 0);
    if (pc22)
      head_->appendReal("PC2_2", pc22, 10, NULL);
  }

  // CD
  {
    ostringstream str1;
    str1 << "TC" << xcol->index() << "_" <<  xcol->index() << ends;
    float cd11 = srcHead->getReal(str1.str().c_str(), 0);
    if (!cd11)
      cd11 = srcHead->getReal("CD1_1", 0);

    ostringstream str2;
    str2 << "TC" << xcol->index() << "_" <<  ycol->index() << ends;
    float cd12 = srcHead->getReal(str2.str().c_str(), 0);
    if (!cd12)
      cd12 = srcHead->getReal("CD1_2", 0);

    ostringstream str3;
    str3 << "TC" << ycol->index() << "_" <<  xcol->index() << ends;
    float cd21 = srcHead->getReal(str3.str().c_str(), 0);
    if (!cd21)
      cd21 = srcHead->getReal("CD2_1", 0);

    ostringstream str4;
    str4 << "TC" << ycol->index() << "_" <<  ycol->index() << ends;
    float cd22 = srcHead->getReal(str4.str().c_str(), 0);
    if (!cd22)
      cd22 = srcHead->getReal("CD2_2", 0);
    Matrix cd = 
      Matrix(cd11*block[0], cd12*block[0], cd21*block[1], cd22*block[1], 0, 0);

    if (cd11)
      head_->appendReal("CD1_1", cd[0][0], 10, NULL);
    if (cd12)
      head_->appendReal("CD1_2", cd[0][1], 10, NULL);
    if (cd21)
      head_->appendReal("CD2_1", cd[1][0], 10, NULL);
    if (cd22)
      head_->appendReal("CD2_2", cd[1][1], 10, NULL);
  }

  // RADESYS
  {
    ostringstream str;
    str << "RADE" << xcol->index() << ends;
    char* radesys = srcHead->getString(str.str().c_str());
    if (!radesys)
      radesys = srcHead->getString("RADESYS");
    if (!radesys)
      radesys = srcHead->getString("RADECSYS");
    if (radesys) {
      head_->appendString("RADESYS", radesys, NULL);
      delete [] radesys;
    }
  }

  // EQUINOX
  {
    ostringstream str;
    str << "EQUI" << xcol->index() << ends;
    float equinox = srcHead->getReal(str.str().c_str(), 0);
    if (!equinox)
      equinox = srcHead->getReal("EQUINOX", 0);
    if (equinox)
      head_->appendReal("EQUINOX", equinox, 10, NULL);
  }

  // MJDOBS
  {
    ostringstream str;
    str << "MJDOB" << xcol->index() << ends;
    float mjdobs = srcHead->getReal(str.str().c_str(), 0);
    if (!mjdobs)
      mjdobs =  srcHead->getReal("MJD-OBS", 0);
    if (mjdobs)
      head_->appendReal("MJD-OBS", mjdobs, 10, NULL);
  }

  // PV (TV) xcol
  for (int m=0; m<=9; m++) {
    ostringstream str;
    str << "TV" << xcol->index() << "_" << m << ends;
    if (srcHead->find(str.str().c_str())) {
      float val = srcHead->getReal(str.str().c_str(), 0);
      ostringstream str;
      str << "PV1_" << m << ends;
      head_->appendReal(str.str().c_str(), val, 10, NULL);
    }
  }

  // PV (TV) ycol
  for (int m=0; m<=9; m++) {
    ostringstream str;
    str << "TV" << ycol->index() << "_" << m << ends;
    if (srcHead->find(str.str().c_str())) {
      float val = srcHead->getReal(str.str().c_str(), 0);
      ostringstream str;
      str << "PV2_" << m << ends;
      head_->appendReal(str.str().c_str(), val, 10, NULL);
    }
  }

  // PS (TS) xcol
  for (int m=0; m<=9; m++) {
    ostringstream str;
    str << "TS" << xcol->index() << "_" << m << ends;
    if (srcHead->find(str.str().c_str())) {
      float val = srcHead->getReal(str.str().c_str(), 0);
      ostringstream str;
      str << "PS1_" << m << ends;
      head_->appendReal(str.str().c_str(), val, 10, NULL);
    }
  }

  // PS (TS) ycol
  for (int m=0; m<=9; m++) {
    ostringstream str;
    str << "TS" << ycol->index() << "_" << m << ends;
    if (srcHead->find(str.str().c_str())) {
      float val = srcHead->getReal(str.str().c_str(), 0);
      ostringstream str;
      str << "PS2_" << m << ends;
      head_->appendReal(str.str().c_str(), val, 10, NULL);
    }
  }
}

void FitsHist::initWCSAlt(FitsFile* fits, Matrix& m, Vector block)
{
  FitsHead* srcHead = fits->head();
  char w[2];
  w[1] = '\0';

  for (int i=1; i<MULTWCS; i++) {
    w[0] = '@'+i;

    // CTYPE
    {
      ostringstream str1;
      str1 << "TCTYP" << xcol->index() << w << ends;
      char* ctype1 = srcHead->getString(str1.str().c_str());
      if (ctype1) {
	head_->appendString("CTYPE1", ctype1, NULL);
	delete [] ctype1;
      }

      ostringstream str2;
      str2 << "TCTYP" << ycol->index() << w << ends;
      char* ctype2 = srcHead->getString(str2.str().c_str());
      if (ctype2) {
	head_->appendString("CTYPE2", ctype2, NULL);
	delete [] ctype2;
      }
    }

    // CUNIT
    {
      ostringstream str1;
      str1 << "TCUNI" << xcol->index() << w << ends;
      char* cunit1 = srcHead->getString(str1.str().c_str());
      if (cunit1) {
	head_->appendString("CUNIT1", cunit1, NULL);
	delete [] cunit1;
      }

      ostringstream str2;
      str2 << "TCUNI" << ycol->index() << w << ends;
      char* cunit2 = srcHead->getString(str2.str().c_str());
      if (cunit2) {
	head_->appendString("CUNIT2", cunit2, NULL);
	delete [] cunit2;
      }
    }

    // CRPIX
    {
      ostringstream str1;
      str1 << "TCRPX" << xcol->index() << w << ends;
      float crpix1 = srcHead->getReal(str1.str().c_str(), 0);

      ostringstream str2;
      str2 << "TCRPX" << ycol->index() << w << ends;
      float crpix2 = srcHead->getReal(str2.str().c_str(), 0);
      Vector pix = Vector(crpix1,crpix2) * m;

      if (crpix1)
	head_->appendReal("CRPIX1", pix[0], 10, NULL);
      if (crpix2)
	head_->appendReal("CRPIX2", pix[1], 10, NULL);
    }

    // CRVAL
    {
      ostringstream str1;
      str1 << "TCRVL" << xcol->index() << w << ends;
      float crval1 = srcHead->getReal(str1.str().c_str(), 0);
      if (crval1)
	head_->appendReal("CRVAL1", crval1, 10, NULL);

      ostringstream str2;
      str2 << "TCRVL" << ycol->index() << w << ends;
      float crval2 = srcHead->getReal(str2.str().c_str(), 0);
      if (crval2)
	head_->appendReal("CRVAL2", crval2, 10, NULL);
    }

    // CDELT
    {
      ostringstream str1;
      str1 << "TCDLT" << xcol->index() << w <<ends;
      float cdelt1 = srcHead->getReal(str1.str().c_str(), 0);

      ostringstream str2;
      str2 << "TCDLT" << ycol->index() << w << ends;
      float cdelt2 = srcHead->getReal(str2.str().c_str(), 0);
      Vector delt = Vector(cdelt1*block[0],cdelt2*block[1]);

      if (cdelt1)
	head_->appendReal("CDELT1", delt[0], 10, NULL);
      if (cdelt2)
	head_->appendReal("CDELT2", delt[1], 10, NULL);
    }

    // CROTA
    {
      ostringstream str1;
      str1 << "TCROT" << xcol->index() << w <<ends;
      float crota1 = srcHead->getReal(str1.str().c_str(), 0);
      if (crota1)
	head_->appendReal("CROTA1", crota1, 10, NULL);

      ostringstream str2;
      str2 << "TCROT" << ycol->index() << w << ends;
      float crota2 = srcHead->getReal(str2.str().c_str(), 0);
      if (crota2)
	head_->appendReal("CROTA2", crota2, 10, NULL);
    }

    // PC
    {
      ostringstream str1;
      str1 << "TP" << xcol->index() << "_" <<  xcol->index() << w << ends;
      float pc11 = srcHead->getReal(str1.str().c_str(), 0);
      if (pc11)
	head_->appendReal("PC1_1", pc11, 10, NULL);

      ostringstream str2;
      str2 << "TP" << xcol->index() << "_" <<  ycol->index() << w << ends;
      float pc12 = srcHead->getReal(str2.str().c_str(), 0);
      if (pc12)
	head_->appendReal("PC1_2", pc12, 10, NULL);

      ostringstream str3;
      str3 << "TP" << ycol->index() << "_" <<  xcol->index() << w << ends;
      float pc21 = srcHead->getReal(str3.str().c_str(), 0);
      if (pc21)
	head_->appendReal("PC2_1", pc21, 10, NULL);

      ostringstream str4;
      str4 << "TP" << ycol->index() << "_" <<  ycol->index() << w << ends;
      float pc22 = srcHead->getReal(str4.str().c_str(), 0);
      if (pc22)
	head_->appendReal("PC2_2", pc22, 10, NULL);
    }

    // CD
    {
      ostringstream str1;
      str1 << "TC" << xcol->index() << "_" <<  xcol->index() << w << ends;
      float cd11 = srcHead->getReal(str1.str().c_str(), 0);

      ostringstream str2;
      str2 << "TC" << xcol->index() << "_" <<  ycol->index() << w << ends;
      float cd12 = srcHead->getReal(str2.str().c_str(), 0);

      ostringstream str3;
      str3 << "TC" << ycol->index() << "_" <<  xcol->index() << w << ends;
      float cd21 = srcHead->getReal(str3.str().c_str(), 0);

      ostringstream str4;
      str4 << "TC" << ycol->index() << "_" <<  ycol->index() << w << ends;
      float cd22 = srcHead->getReal(str4.str().c_str(), 0);
      Matrix cd = 
	Matrix(cd11*block[0],cd12*block[0],cd21*block[1],cd22*block[1],0,0);

      if (cd11)
	head_->appendReal("CD1_1", cd[0][0], 10, NULL);
      if (cd12)
	head_->appendReal("CD1_2", cd[0][1], 10, NULL);
      if (cd21)
	head_->appendReal("CD2_1", cd[1][0], 10, NULL);
      if (cd22)
	head_->appendReal("CD2_2", cd[1][1], 10, NULL);
    }

    // RADESYS
    {
      ostringstream str;
      str << "RADE" << xcol->index() << w << ends;
      char* radesys = srcHead->getString(str.str().c_str());
      if (radesys) {
	head_->appendString("RADESYS", radesys, NULL);
	delete [] radesys;
      }
    }

    // EQUINOX
    {
      ostringstream str;
      str << "EQUI" << xcol->index() << w << ends;
      float equinox = srcHead->getReal(str.str().c_str(), 0);
      if (equinox)
	head_->appendReal("EQUINOX", equinox, 10, NULL);
    }

    // MJDOBS
    {
      ostringstream str;
      str << "MJDOB" << xcol->index() << w << ends;
      float mjdobs = srcHead->getReal(str.str().c_str(), 0);
      if (mjdobs)
	head_->appendReal("MJD-OBS", mjdobs, 10, NULL);
    }

    // PV (TV) xcol
    for (int m=0; m<=9; m++) {
      ostringstream str;
      str << "TV" << xcol->index() << "_" << m << w << ends;
      if (srcHead->find(str.str().c_str())) {
	float val = srcHead->getReal(str.str().c_str(), 0);
	ostringstream str;
	str << "PV1_" << m << w << ends;
	head_->appendReal(str.str().c_str(), val, 10, NULL);
      }
    }

    // PV (TV) ycol
    for (int m=0; m<=9; m++) {
      ostringstream str;
      str << "TV" << ycol->index() << "_" << m << w << ends;
      if (srcHead->find(str.str().c_str())) {
	float val = srcHead->getReal(str.str().c_str(), 0);
	ostringstream str;
	str << "PV2_" << m << w << ends;
	head_->appendReal(str.str().c_str(), val, 10, NULL);
      }
    }

    // PS (TS) xcol
    for (int m=0; m<=9; m++) {
      ostringstream str;
      str << "TS" << xcol->index() << "_" << m << w << ends;
      if (srcHead->find(str.str().c_str())) {
	float val = srcHead->getReal(str.str().c_str(), 0);
	ostringstream str;
	str << "PS1_" << m << w << ends;
	head_->appendReal(str.str().c_str(), val, 10, NULL);
      }
    }

    // PS (TS) ycol
    for (int m=0; m<=9; m++) {
      ostringstream str;
      str << "TS" << ycol->index() << "_" << m << w << ends;
      if (srcHead->find(str.str().c_str())) {
	float val = srcHead->getReal(str.str().c_str(), 0);
	ostringstream str;
	str << "PS2_" << m << w << ends;
	head_->appendReal(str.str().c_str(), val, 10, NULL);
      }
    }
  }
}

FitsHistNext::FitsHistNext(FitsFile* prev)
{
  head_ = prev->head();
  manageHead_ = 0;
  primary_ = prev->primary();
  managePrimary_ = 0;
  ext_ = prev->ext();

  data_ = (char*)prev->data() + head_->pixbytes();
  valid_ = 1;
  return;
}
