#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# SaveImage

proc SaveImageDialog {} {
    global ed
    global menu
    global saveimage
    global saveimagefbox

    set w ".saveimage"

    set ed(ok) 0
    set ed(format) $saveimage(format)

    DialogCreate $w "Save Image as" -borderwidth 2
    frame $w.format -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.format $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    label $w.format.title -text "Format"
    radiobutton $w.format.fits -text FITS \
	-selectcolor $menu(selectcolor) \
	-variable ed(format) -value fits
    radiobutton $w.format.jpeg -text JPEG \
	-selectcolor $menu(selectcolor) \
	-variable ed(format) -value jpeg
    radiobutton $w.format.tiff -text "TIFF RGB 24-bit" \
	-selectcolor $menu(selectcolor) \
	-variable ed(format) -value tiff
    radiobutton $w.format.png -text "PNG 24-bit (Portable Network Graphics)" \
	-selectcolor $menu(selectcolor) \
	-variable ed(format) -value png
    radiobutton $w.format.ppm -text "PPM RGB (Portable Pixmap)" \
	-selectcolor $menu(selectcolor) \
	-variable ed(format) -value ppm
    radiobutton $w.format.mpeg -text "MPEG-1 (Movie)" \
	-selectcolor $menu(selectcolor) \
	-variable ed(format) -value mpeg

    grid $w.format.fits -ipadx 4 -sticky w
    grid $w.format.jpeg -ipadx 4 -sticky w
    grid $w.format.tiff -ipadx 4 -sticky w
    grid $w.format.png -ipadx 4 -sticky w
    grid $w.format.ppm -ipadx 4 -sticky w
    grid $w.format.mpeg -ipadx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	set saveimage(format) $ed(format)

	switch -- $saveimage(format) {
	    fits {
		set fn "$saveimage(base).fits"
		set saveimagefbox(types) {
		    {"FITS files"   *.fits}
		    {"All files"    *}
		}
	    }
	    jpeg {
		set fn "$saveimage(base).jpg"
		set saveimagefbox(types) {
		    {"JPEG files"   *.jpg}
		    {"All files"    *}
		}
	    }
	    tiff {
		set fn "$saveimage(base).tif"
		set saveimagefbox(types) {
		    {"TIFF files"   *.tif }
		    {"All files"    *}
		}
	    }
	    png {
		set fn "$saveimage(base).png"
		set saveimagefbox(types) {
		    {"PNG files"    *.png }
		    {"All files"    *}
		}
	    }
	    ppm	{
		set fn "$saveimage(base).ppm"
		set saveimagefbox(types) {
		    {"PPM files"    *.ppm }
		    {"All files"    *}
		}
	    }
	    mpeg {
		set fn "$saveimage(base).mpg"
		set saveimagefbox(types) {
		    {"MPEG files"    *.mpg }
		    {"All files"    *}
		}
	    }
	}
	FileLast saveimagefbox $fn
	set fn [SaveFileDialog saveimagefbox]
	set saveimage(base) [file rootname [file tail $fn]]

	set saveimage(ok) 1
	if {$fn != {}} {
	    switch -- $saveimage(format) {
		fits {}
		jpeg {set saveimage(ok) \
			  [SaveJPEGParams saveimage(jpeg,quality)]}
		tiff {set saveimage(ok) \
			  [SaveTIFFParams saveimage(tiff,compression)]}
		png {}
		ppm {}
		mpeg {set saveimage(ok) \
			  [SaveMPEGParams saveimage(mpeg,quality)]}
	    }
	}

	if {$fn != {} && $saveimage(ok)} {
	    SaveImage $fn
	}
    }
    destroy $w 
    unset ed
}

proc SaveImage {fn} {
    global ds9
    global current
    global saveimage
    global cube

    if {$fn == {}} {
	return
    }

    # besure we are on top
    raise $ds9(top)

    # we want a white background
    set bg [$ds9(canvas) cget -background]
    $ds9(canvas) configure -background white

    # and no highlite
    $current(frame) highlite off

    switch -- $saveimage(format) {
	fits -
	jpeg -
	tiff -
	png -
	ppm {
	    set depth [$current(frame) get fits depth]
	    if {$ds9(display,mode) == "single" && $depth > 1} {
		set slice [$current(frame) get fits slice]
		
		# loop thru datacube
		for {set ii 1} {$ii <= $depth} {incr ii} {
		    set ff "[file root $fn]_$ii[file extension $fn]"

		    set cube(slice) $ii
		    $current(frame) update fits slice $ii
		    ContourUpdateNow

		    set r [SaveImageDoit $ff]
		    if {$r} {
			Error "Error $r has occurred while creating image"
			break
		    }
		}

		# reset current slice
		$current(frame) update fits slice $slice
		ContourUpdateNow

	    } else {
		set r [SaveImageDoit $fn]
		if {$r} {
		    Error "Error $r has occurred while creating image"
		}
	    }
	}
	mpeg {
	    set depth [$current(frame) get fits depth]
	    set slice [$current(frame) get fits slice]
		
	    # loop thru datacube
	    set saveimage(mpeg,first) 1
	    for {set ii 1} {$ii <= $depth} {incr ii} {

		set cube(slice) $ii
		$current(frame) update fits slice $ii
		ContourUpdateNow
		
		set r [SaveImageDoit $fn]
		if {$r} {
		    Error "Error $r has occurred while creating image"
		    break
		}

		set saveimage(mpeg,first) 0
	    }
	    mpeg close

	    # reset current slice
	    set cube(slice) $slice
	    $current(frame) update fits slice $slice
	    ContourUpdateNow
	}
    }

    FileLastFull saveimagefbox $fn

    # reset
    $ds9(canvas) configure -background $bg
    switch -- $ds9(display,mode) {
	single -
	blink {}
	tile {$current(frame) highlite on}
    }
}

proc SaveImageDoit {fn} {
    global ds9
    global saveimage
    global current

    RealizeDS9

    set r \
	[catch {image create photo -format window -data $ds9(canvas)} ph]
    if {$r != 0} {
	return $r
    }

    switch -- $saveimage(format) {
	fits {
	    $current(frame) save fits resample file "\{$fn\}"
	}
	jpeg {
	    $ph write $fn -format \
		[list $saveimage(format) -quality $saveimage(jpeg,quality)]
	}
	tiff {
	    $ph write $fn -format \
		[list $saveimage(format) \
		     -compression $saveimage(tiff,compression)]
	}
	png {$ph write $fn -format $saveimage(format)}
	ppm {$ph write $fn -format $saveimage(format)}
	mpeg {
	    if {$saveimage(mpeg,first)} {
		set w [image width $ph]
		set h [image height $ph]
		mpeg create $fn $w $h 25 30 $saveimage(mpeg,quality)
	    }
	    mpeg add $ph
	}
    }

    image delete $ph
    return 0
}

proc ProcessSaveImageCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global saveimage
    
    set saveimage(format) [string tolower [lindex $var $i]]
    incr i
    set fn [file normalize [lindex $var $i]]
    SaveImage $fn
}

# SaveFits

proc SaveFitsDialog {} {
    set fn [SaveFileDialog savefitsfbox]
    SaveFits $fn
}

proc SaveFits {fn} {
    global current

    RealizeDS9
    if {$fn != {}} {
	$current(frame) save fits image file "\{$fn\}"
    }
    FileLastFull savefitsfbox $fn
}

proc ProcessSaveFitsCmd {varname iname} {
    upvar $varname var
    upvar $iname i
    SaveFits [file normalize [lindex $var $i]]
}

# SaveMPEG

proc SaveMPEGDialog {} {
    global savempeg

    set fn [SaveFileDialog savempegfbox]
    if {$fn != {}} {
	if {[SaveMPEGParams savempeg(quality)]} {
	    SaveMPEG $fn
	}
    }
}

proc SaveMPEG {fn} {
    global ds9
    global current
    global savempeg

    if {$fn == {}} {
	return
    }
    
    # we need single mode
    if {$ds9(display,mode) != "single"} {
	set modesav $ds9(display,mode)
	set ds9(display,user) single
	DisplayMode
    }

    # loop thru all active frames
    set first 1
    set framesav $current(frame)

    foreach f $ds9(active) {
	set ds9(next) $f
	GotoFrame
	
	RealizeDS9

	set r \
	    [catch {image create photo -format window -data $ds9(canvas)} ph]
	if {$r != 0} {
	    Error "An Error has occurred while creating image"
	    break
	}

	if {$first} {
	    set w [image width $ph]
	    set h [image height $ph]
	    mpeg create "$fn" $w $h 25 30 $savempeg(quality)
	    set first 0
	}
	mpeg add $ph

	image delete $ph
    }
    mpeg close

    if {[info exists modesav]} {
	set ds9(display,user) $modesav
	DisplayMode
    }

    set ds9(next) $framesav
    GotoFrame
    FileLastFull savefitsfbox $fn
}

proc ProcessSaveMPEGCmd {varname iname} {
    upvar $varname var
    upvar $iname i
    SaveMPEG [file normalize [lindex $var $i]]
}

# Support

proc SaveJPEGParams {varname} {
    upvar $varname var

    global eds
    global menu

    set w ".savejpeg"

    set eds(ok) 0
    set eds(quality) $var

    DialogCreate $w "JPEG" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    scale $w.param.quality -from 0 -to 100 -length 300 \
	-variable eds(quality) \
	-orient horizontal -label "JPEG Quality Factor" \
	-tickinterval 25 -showvalue true -resolution 1

    grid $w.param.quality -ipadx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set eds(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set eds(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set eds(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w eds(ok)
    DialogDismiss $w

    set r 0
    if {$eds(ok)} {
	set var $eds(quality)
	set r 1
    }
    destroy $w 
    unset eds

    return $r
}

proc SaveTIFFParams {varname} {
    upvar $varname var

    global eds
    global menu

    set w ".savetiff"

    set eds(ok) 0
    set eds(compression) $var

    DialogCreate $w "TIFF" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    label $w.param.title -text "Compression:"
    radiobutton $w.param.none -text "None" \
	-selectcolor $menu(selectcolor) \
	-variable eds(compression) -value none
    radiobutton $w.param.jpeg -text "JPEG" \
	-selectcolor $menu(selectcolor) \
	-variable eds(compression) -value jpeg
    radiobutton $w.param.packbits -text "Packbits" \
	-selectcolor $menu(selectcolor) \
	-variable eds(compression) -value packbits
    radiobutton $w.param.deflate -text "Deflate" \
	-selectcolor $menu(selectcolor) \
	-variable eds(compression) -value deflate

    grid $w.param.title -ipadx 4 -sticky w
    grid $w.param.none -ipadx 4 -sticky w
    grid $w.param.jpeg -ipadx 4 -sticky w
    grid $w.param.packbits -ipadx 4 -sticky w
    grid $w.param.deflate -ipadx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set eds(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set eds(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set eds(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w eds(ok)
    DialogDismiss $w

    set r 0
    if {$eds(ok)} {
	set var $eds(compression)
	set r 1
    }
    destroy $w 
    unset eds

    return $r
}

proc SaveMPEGParams {varname} {
    upvar $varname var

    global eds
    global menu

    set w ".savempeg"

    set eds(ok) 0
    set eds(quality) $var

    DialogCreate $w "MPEG" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    scale $w.param.quality -bd 1 -from 2 -to 31 -length 300 \
	-variable eds(quality) \
	-orient horizontal -label "MPEG Quality Factor" \
	-tickinterval 4 -showvalue true -resolution 1

    grid $w.param.quality -ipadx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set eds(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set eds(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set eds(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w eds(ok)
    DialogDismiss $w

    set r 0
    if {$eds(ok)} {
	set var $eds(quality)
	set r 1
    }
    destroy $w 
    unset eds

    return $r
}
