#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Open

proc OpenFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadFits $fn
    FinishLoad
}

proc OpenDataCubeFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadDataCubeFits $fn
    FinishLoad
}

proc OpenMosaicImageIRAFFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadMosaicImageIRAFFits $fn
    FinishLoad
}

proc OpenMosaicIRAFFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadMosaicIRAFFits $fn
    FinishLoad
}

proc OpenMosaicImageWCSFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    set sys [MosaicWCSDialog]
    if {$sys != {}} {
	StartLoad
	LoadMosaicImageWCSFits $sys $fn
	FinishLoad
    }
}

proc OpenMosaicImageNextWCSFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    set sys [MosaicWCSDialog]
    if {$sys != {}} {
	StartLoad
	LoadMosaicImageNextWCSFits $sys $fn
	FinishLoad
    }
}

proc OpenMosaicWCSFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    set sys [MosaicWCSDialog]
    if {$sys != {}} {
	StartLoad
	LoadMosaicWCSFits $sys $fn
	FinishLoad
    }
}

proc OpenMosaicImageWFPC2Fits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadMosaicImageWFPC2Fits $fn
    FinishLoad
}

proc OpenRGBImageFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadRGBImageFits $fn
    FinishLoad
}

proc OpenRGBCubeFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadRGBCubeFits $fn
    FinishLoad
}

proc OpenArray {} {
    set fn [OpenFileDialog arrayfbox]
    if {[string length $fn] == 0} {
	return
    }

    # do we have an array spec tag'd on
    if {![regexp -nocase {(.*)(\[.*\])} $fn foo base ext]} {
	set ext [ArrayDialog 1]
	if {$ext != {}} {
	    append fn "$ext"
	} else {
	    return
	}
    }

    StartLoad
    LoadArray $fn
    FinishLoad
}

proc OpenRGBArray {} {
    set fn [OpenFileDialog arrayfbox]
    if {[string length $fn] == 0} {
	return
    }

    # do we have an array spec tag'd on
    if {![regexp -nocase {(.*)(\[.*\])} $fn foo base ext]} {
	set ext [ArrayDialog 3]
	if {$ext != {}} {
	    append fn "$ext"
	} else {
	    return
	}
    }

    StartLoad
    LoadRGBArray $fn
    FinishLoad
}

proc OpenURL {} {
    global http

    set fn [URLDialog http(last)]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadURL $fn
    FinishLoad
}

proc OpenVar {} {
    set fn [OpenFileDialog fitsvarfbox]
    if {[string length $fn] == 0} {
	return
    }

    if {![catch {set ch [open "$fn"]}]} {
	fconfigure $ch -translation binary -encoding binary
	global vardata
	set vardata [read -nonewline $ch]
	close $ch

	StartLoad
	LoadVar vardata $fn {}
	FinishLoad
    }
}

# Load

proc LoadFits {fn} {
    global current
    global ds9
    global loadParam
    global marker

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    ConvertFile $fn
    # save load type, since ProcessLoad will clear loadParam
    if {$loadParam(load,type) == "mmapincr"} {
	set mmap 1
    } else {
	set mmap 0
    }
    ProcessLoad

    # now autoload markers
    if {$ds9(automarker) && $mmap} {
	# now, load fits[REGION] if present
	set id [string first "\[" $fn]
	if {$id > 0} {
	    set base [string range $fn 0 [expr $id-1]]
	} else {
	    set base $fn
	}

	set reg "${base}\[REGION\]"
	if {[$current(frame) fitsy has ext "\"$reg\""]} {
	    RealizeDS9
	    catch {
		$current(frame) marker load fits "\"$reg\"" \
		    $marker(color) $marker(width) \
		    "\{$marker(font) $marker(font,size) $marker(font,style)\}"
	    }
	}
    }
}

proc LoadSFits {hdr fn} {
    global current
    global ds9
    global loadParam
    global marker

    set loadParam(load,type) smmap
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    set loadParam(file,name) $fn
    set loadParam(file,header) $hdr
    ProcessLoad
}

proc LoadDataCubeFits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) {data cube}
    ConvertFile $fn
    ProcessLoad
}

proc LoadMosaicImageIRAFFits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic image iraf}
    ConvertFile $fn
    ProcessLoad
}

proc LoadMosaicIRAFFits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic iraf}
    ConvertFile $fn
    ProcessLoad
}

proc LoadMosaicIRAFSFits {hdr fn} {
    global loadParam

    set loadParam(load,type) smmap
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic iraf}
    set loadParam(file,name) $fn
    set loadParam(file,header) $hdr
    ProcessLoad
}

proc LoadMosaicImageWCSFits {sys fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic image $sys]
    ConvertFile $fn
    ProcessLoad
}

proc LoadMosaicImageNextWCSFits {sys fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic image next $sys]
    ConvertFile $fn
    ProcessLoad
}

proc LoadMosaicWCSFits {sys fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic $sys]
    ConvertFile $fn
    ProcessLoad
}

proc LoadMosaicWCSSFits {sys hdr fn} {
    global loadParam

    set loadParam(load,type) smmap
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic $sys]
    set loadParam(file,name) $fn
    set loadParam(file,header) $hdr
    ProcessLoad
}

proc LoadMosaicImageWFPC2Fits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic image wfpc2}
    ConvertFile $fn
    ProcessLoad
}

proc LoadRGBImageFits {fn} {
    global loadParam
    global current
    global message

    if {[$current(frame) get type] == "rgb"} {
	set loadParam(load,type) mmapincr
	set loadParam(file,type) fits
	set loadParam(file,mode) {rgb image}
	ConvertFile $fn
	ProcessLoad
    } else {
	Error "$message(error,fits,rgb)"
    }
}

proc LoadRGBCubeFits {fn} {
    global loadParam
    global current
    global message

    if {[$current(frame) get type] == "rgb"} {
	set loadParam(load,type) mmapincr
	set loadParam(file,type) fits
	set loadParam(file,mode) {rgb cube}
	ConvertFile $fn
	ProcessLoad
    } else {
	Error "$message(error,fits,rgb)"
    }
}

proc LoadRGBCubeSFits {hdr fn} {
    global loadParam
    global current
    global message

    if {[$current(frame) get type] == "rgb"} {
	set loadParam(load,type) smmap
	set loadParam(file,type) fits
	set loadParam(file,mode) {rgb cube}
	set loadParam(file,name) $fn
	set loadParam(file,header) $hdr
	ProcessLoad
    } else {
	Error "$message(error,fits,rgb)"
    }
}

proc LoadRGBArray {fn} {
    global loadParam
    global current
    global message

    # if no zdim is present, insert one
    set exp {.*\[.*zdim[ ]*=[ ]*[0-9]+}
    if {![regexp $exp $fn]} {
	set i [string last "\]" $fn]
        set fn "[string range $fn 0 [expr $i-1]],zdim=3\]"
    }

    if {[$current(frame) get type] == "rgb"} {
	set loadParam(load,type) mmapincr
	set loadParam(file,type) array
	set loadParam(file,mode) {rgb cube}
	ConvertFile $fn
	ProcessLoad
    } else {
	Error "$message(error,fits,rgb)"
    }
}

proc LoadVar {varname fn mode} {
    global loadParam

    set loadParam(load,type) var
    set loadParam(var,name) $varname
    set loadParam(file,type) fits
    set loadParam(file,name) "$fn"
    set loadParam(file,mode) $mode
    ProcessLoad
}

proc LoadArray {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(file,type) array
    set loadParam(file,mode) {}
    ConvertFile $fn
    ProcessLoad
}

proc LoadShared {filetype mode idtype id fn} {
    global loadParam

    set loadParam(load,type) shared
    set loadParam(shared,idtype) $idtype
    set loadParam(shared,id) $id
    set loadParam(file,type) $filetype
    set loadParam(file,name) $fn
    set loadParam(file,mode) $mode
    ProcessLoad
}

proc LoadURL {url} {
    if {[string length $url] == 0} {
	return
    }

    ParseURL $url r
    switch -- $r(scheme) {
	ftp {LoadFTP $r(authority) $r(path)}
	file {LoadFits $r(path)}
	http -
	default {LoadHTTP $url}
    }
}

proc LoadFTP {host path} {
    global loadParam
    global ds9
    global debug

    set ftp [ftp::Open $host "ftp" "ds9@" -mode passive]
    if {$ftp > -1} {
	set fn "$ds9(tmpdir)/[file tail $path]"
	set ftp::VERBOSE $debug(tcl,ftp)
	set "ftp::ftp${ftp}(Output)" FTPLog
	ftp::Type $ftp binary
	if [ftp::Get $ftp $path $fn] {
	    # alloc it because we are going to delete it after load
	    set loadParam(load,type) allocgz
	    set loadParam(file,type) fits
	    set loadParam(file,mode) {}
	    ConvertFile $fn
	    ProcessLoad
	}

	ftp::Close $ftp

	if [file exists $fn] {
	    catch {file delete -force $fn}
	}
    }
}

proc LoadHTTP {url} {
    global loadParam
    global ds9
    global message

    ParseURL $url r
    set fn "$ds9(tmpdir)/[file tail $r(path)]"

    set code 200
    set meta {}
    set mime "application/fits"
    set encoding {}

    set ch [open $fn w]
    set token [http::geturl $url -channel $ch -binary 1 -headers "[ProxyHTTP]"]
    catch {close $ch}

    upvar #0 $token t

    # Code
    set code [http::ncode $token]

    # Meta
    set meta $t(meta)

    # Mime-type
    # we want to strip and extra info after ';'
    regexp -nocase {([^;])*} $t(type) mime

    # Content-Encoding
    foreach {name value} $meta {
	if {[regexp -nocase ^content-encoding $name]} {
	    switch -- $value {
		gzip -
		x-gzip {set encoding gzip}
		compress -
		Z {set encoding compress}
		pack -
		z {set encoding pack}
		default {}
	    }
	}
    }

    HTTPLog $token
    http::cleanup $token

    global debug
    if {$debug(tcl,hv)} {
	puts "Load HTTP: fn $fn : code $code : meta $meta : mime $mime : encoding $encoding"
    }

    switch -- "$mime" {
	"image/fits" -
	"application/fits" {}

	"application/fits-image" -
	"application/fits-table" -
	"application/fits-group" {}

	"image/x-fits" -
	"binary/x-fits" -
	"application/x-fits" {}

	"image/x-gfits" -
	"binary/x-gfits" -
	"image/gz-fits" -
	"display/gz-fits" {set encoding gzip}

	"image/x-cfits" -
	"binary/x-cfits" {set encoding compress}

	default {
	    Error "$message(error,fits,mime) $mime"
	    return
	}
    }

    # alloc it because we are going to delete it after load
    set loadParam(load,type) allocgz
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    set loadParam(file,name) $fn

    # may have to convert the file, based on content-encoding
    switch -- "$encoding" {
	compress {
	    catch {set ch [open "| uncompress < $fn " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
	pack {
	    catch {set ch [open "| pcat $fn " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
    }

    ProcessLoad

    if {[file exists $fn]} {
	catch {file delete -force $fn}
    }
}

# Support

proc MultiLoad {fc} {
    global ds9
    upvar $fc frameCount

    global debug
    if {$debug(tcl,layout)} {
	puts "MultiLoad"
    }

    # create a new frame
    if {$frameCount != 0} {
	CreateFrame  
    }

    # go into tile mode if more than one
    if {$frameCount >= 1 && $ds9(display,user) != "tile"} {
	set ds9(display,user) tile
	DisplayMode
    }
    incr frameCount
}

proc MultiRGBLoad {fc} {
    global ds9
    upvar $fc frameCount

    global debug
    if {$debug(tcl,layout)} {
	puts "MultiRGBLoad"
    }

    # create a new frame
    CreateRGBFrame  

    # go into tile mode if more than one
    if {$frameCount >= 1 && $ds9(display,user) != "tile"} {
	set ds9(display,user) tile
	DisplayMode
    }
    incr frameCount
}

proc URLDialog {prevURL} {
    upvar $prevURL result

    if {[EntryDialog "URL" "Enter World Wide Web Location (URL):" 80 result]} {
	return $result
    } else {
	return ""
    }
}

# Header

proc DisplayHeaderMenu {} {
    global current

    set prim {}
    set cnt [$current(frame) get fits ext count]

    if {$cnt > 0} {
	global slb
	set slb(count) 0

	for {set ii 1} {$ii <= $cnt} {incr ii} {
	    if {[$current(frame) has fits ext $ii]} {
		set fn [$current(frame) get fits file name ext $ii]
		set b [string first {[} $fn]
		if {$b > 0} {
		    set fn [string range $fn 0 [expr $b-1]]
		}
		       
		if {$prim != $fn} {
		    set prim $fn

		    incr slb(count)
		    set slb($slb(count),item) $prim
		    set slb($slb(count),value) "-$ii"
		}
	    }

	    incr slb(count)
	    set slb($slb(count),item) \
		[$current(frame) get fits file name ext $ii]
	    set slb($slb(count),value) $ii
	}

	if {$slb(count) == 1} {
 	    DisplayHeader $current(frame) 1 $slb(1,item)
	} elseif {$slb(count) > 1} {
	    SLBDialog slb {Select Header} 40

	    if {$slb(value) != {}} {
	        if {$slb(value) != -1} {
		    DisplayHeader $current(frame) $slb(value) \
		        $slb([expr $slb(value)+1],item)
	        } else {
		    DisplayHeader $current(frame) $slb(value) $slb(1,item)
	        }
            }
	}

	unset slb
    }
}

proc DisplayHeader {frame which title} {
    global current
    global st

    set tt "hd[string range $frame end end]:$which"

    SimpleTextDialog $tt $title 80 40 insert top \
	[$current(frame) get fits header $which]

    # create a special text tag for keywords
    $st($tt,text) tag configure keyword -font {courier 12} -foreground blue
    
    # color tag keywords
    set stop [$st($tt,text) index end]
    for {set ii 1.0} {$ii<$stop} {set ii [expr $ii+1]} {
	$st($tt,text) tag add keyword $ii "$ii +8 chars"
    }
}

proc DestroyHeader {frame} {
    global st

    set ttt "hd[string range $frame end end]"

    foreach x [array names st] {
	set f [split $x ,]
	if {[lindex $f 1] == "top"} {
	    set tt [lindex $f 0]
	    set fff [split $tt :]
	    if {[lindex $fff 0] == $ttt} {
		if {[info exists st($tt,top)] && [winfo exist $st($tt,top)]} {
		    SimpleTextDestroy $tt
		}
	    }
	}
    }
}

proc UpdateFitsMenu {} {
    global ds9
    global current
    global crosshair
    global menu
 
    global debug
    if {$debug(tcl,update)} {
	puts "UpdateFitsMenu"
    }


    if {$current(frame) != ""} {
	$ds9(mb).file entryconfig $menu(file,open) -state normal
	$ds9(mb).file entryconfig $menu(file,openother) -state normal
	$ds9(mb).file entryconfig $menu(file,saveimage) -state normal
	$ds9(mb).file entryconfig $menu(file,savefits) -state normal
	$ds9(mb).file entryconfig $menu(file,savempeg) -state normal
	$ds9(mb).file entryconfig $menu(file,header) -state disabled
	$ds9(mb).file entryconfig $menu(file,print) -state normal
	$ds9(mb).frame entryconfig $menu(frame,matchframes) -state normal
	$ds9(mb).frame entryconfig $menu(frame,lockcrosshairs) -state normal
	$ds9(mb).frame entryconfig $menu(frame,datacube) -state normal
	$ds9(mb).frame entryconfig $menu(frame,clear) -state normal

	if {[$current(frame) get type] == "rgb"} {
	    $ds9(mb).file.open entryconfig "Open RGB Fits Image..." \
		-state normal
	    $ds9(mb).file.open entryconfig "Open RGB Fits Cube..." \
		-state normal
	    $ds9(mb).file.open entryconfig "Open RGB Array..." \
		-state normal
	} else {
	    $ds9(mb).file.open entryconfig "Open RGB Fits Image..." \
		-state disabled
	    $ds9(mb).file.open entryconfig "Open RGB Fits Cube..." \
		-state disabled
	    $ds9(mb).file.open entryconfig "Open RGB Array..." \
		-state disabled
	}

	$ds9(buttons).file.open configure -state normal
	$ds9(buttons).file.saveimage configure -state normal
	$ds9(buttons).file.savefits configure -state normal
	$ds9(buttons).file.savempeg configure -state normal
	$ds9(buttons).file.header configure -state disabled
	$ds9(buttons).file.print configure -state normal
	$ds9(buttons).frame.clear configure -state normal

	if {[$current(frame) has fits]} {
	    $ds9(mb).file entryconfig $menu(file,header) -state normal
	    $ds9(buttons).file.header configure -state normal
	}
    } else {
	$ds9(mb).file entryconfig $menu(file,open) -state disabled
	$ds9(mb).file entryconfig $menu(file,openother) -state disabled
	$ds9(mb).file entryconfig $menu(file,saveimage) -state disabled
	$ds9(mb).file entryconfig $menu(file,savefits) -state disabled
	$ds9(mb).file entryconfig $menu(file,savempeg) -state disabled
	$ds9(mb).file entryconfig $menu(file,header) -state disabled
	$ds9(mb).file entryconfig $menu(file,print) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,matchframes) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,lockcrosshairs) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,datacube) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,clear) -state disabled

	$ds9(buttons).file.open configure -state disabled
	$ds9(buttons).file.saveimage configure -state disabled
	$ds9(buttons).file.savefits configure -state disabled
	$ds9(buttons).file.savempeg configure -state disabled
	$ds9(buttons).file.header configure -state disabled
	$ds9(buttons).file.print configure -state disabled
	$ds9(buttons).frame.clear configure -state disabled
    }
}

proc ProcessLoad {} {
    global current
    global loadParam
    global message

    switch -- $loadParam(load,type) {
	alloc -
	allocgz {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
		           $loadParam(load,type)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	channel {
	    fconfigure $loadParam(channel,name) -translation binary \
		-encoding binary
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type) \
			   $loadParam(channel,name)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	mmap -
	mmapincr {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	smmap {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,header)\" \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	shared {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type) \
			   $loadParam(shared,idtype) \
			   $loadParam(shared,id)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	sshared {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type) \
			   $loadParam(shared,idtype) \
			   $loadParam(shared,hdr) \
			   $loadParam(shared,id)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	socket -
	socketgz {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type) \
			   $loadParam(socket,id)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
	var {
	    if [catch {$current(frame) load $loadParam(file,type) \
			   $loadParam(file,mode) \
			   \"$loadParam(file,name)\" \
			   $loadParam(load,type) \
			   $loadParam(var,name)}] {
		Error "$message(error,fits,load) $loadParam(file,type) $loadParam(file,mode) image $loadParam(file,name)"
	    }
	}
    }

    unset loadParam
}

proc StartLoad {} {
    SetWatchCursor
}

proc FinishLoad {} {
    global loadParam
    global current

    GridUpdate
    ContourUpdate

    # do after updategrid and updatecontour
    UpdateDS9
    RefreshInfoBox $current(frame)
    UnsetWatchCursor

    # if header(s) were open, remove them
    DestroyHeader $current(frame)

    # do we have a data cube?
    if [$current(frame) has fits cube] {
	CubeDialog
    }
}

proc ConvertFile {filename} {
    global ds9
    global loadParam

    # default
    set loadParam(file,name) $filename

    foreach t $ds9(fileProcs) {
	if [$t $filename] {
	    return
	}
    }
}

# File Types

proc Stdin {filename} {
    global loadParam

    if {[string index $filename 0] == "-"} {
	set loadParam(load,type) allocgz
	set loadParam(file,name) "stdin[string range $filename 1 end]"
	return 1
    } elseif {[string range $filename 0 4] == "stdin" || 
	      [string range $filename 0 4] == "STDIN"} {
	set loadParam(load,type) allocgz
	set loadParam(file,name) "stdin[string range $filename 5 end]"
	return 1
    }

    return 0
}

proc CompressFits {filename} {
    global loadParam

    if { [regexp {(.*)\.Z($|\[)} $filename matched root] } {
	if [catch {set ch [open "| uncompress < $root.Z " r]}] {
	    return 0
	}
	set loadParam(load,type) channel
	set loadParam(file,name) "$filename"
	set loadParam(channel,name) $ch
	return 1
    }

    return 0
}

proc PackFits {filename} {
    global loadParam

    if { [regexp {(.*)\.z($|\[)} $filename matched root] } {
	if [catch {set ch [open "| pcat $root.z " r]}] {
	    return 0
	}
	set loadParam(load,type) channel
	set loadParam(file,name) "$filename"
	set loadParam(channel,name) $ch
	return 1
    }

    return 0
}

proc GzipFits {filename} {
    global loadParam

    if { [regexp {(.*)\.gz($|\[)} $filename matched root] } {
	set loadParam(load,type) allocgz
	set loadParam(file,name) "$filename"
	return 1
    }

    return 0
}

proc FTZFits {filename} {
    global loadParam

    if { [regexp -nocase {(.*)\.(ftz)($|\[)} $filename matched root ext] } {
	set loadParam(load,type) allocgz
	set loadParam(file,name) "$filename"
	return 1
    }

    return 0
}

proc ExternalFits {fn} {
    global extFits
    global loadParam

    foreach id [array names extFits] {
	if {[string match $id "$fn"]} {
	    regsub -all {\$filename} $extFits($id) "$fn" result
	    set cmd "| $result"
	    if [catch {set ch [open "$cmd" r]}] {
		return 0
	    }
	    set loadParam(load,type) channel
	    set loadParam(file,name) "$fn"
	    set loadParam(channel,name) $ch
	    return 1
	}
    }
    return 0
}

proc ArrayDialog {depth} {
    global env
    global ed

    set w ".arr"
    set length 300

    set ed(ok) 0
    set ed(x) 512
    set ed(y) 512
    set ed(z) $depth
    set ed(bitpix) -32
    set ed(skip) 0
    set ed(arch) bigendian

    if {[info exists env(DS9_ARRAY)]} {
	if {[regexp {.*(dims.?=)([0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(x) $item
	    set ed(y) $item
	}
	if {[regexp {.*(dim.?=)([0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(x) $item
	    set ed(y) $item
	}
	if {[regexp {.*(xdim.?=)([0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(x) $item
	}
	if {[regexp {.*(ydim.?=)([0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(y) $item
	}
	if {[regexp {.*(zdim.?=)([0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(z) $item
	}
	if {[regexp {.*(bitpix.?=)(-?[0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(bitpix) $item
	}
	if {[regexp {.*(skip.?=)(-?[0-9]+)} $env(DS9_ARRAY) foo f1 item]} {
	    set ed(skip) $item
	}
	if {[regexp {.*arch.?=bigendian} $env(DS9_ARRAY) foo item]} {
	    set ed(arch) $item
	}
	if {[regexp {.*arch.?=littleendian} $env(DS9_ARRAY) foo item]} {
	    set ed(arch) $item
	}
    }

    DialogCreate $w "Load Array" -borderwidth 2
    frame $w.arr
    frame $w.arr.dim -relief groove -borderwidth 2
    frame $w.arr.bitpix -relief groove -borderwidth 2
    frame $w.arr.skip -relief groove -borderwidth 2
    frame $w.arr.arch -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.arr.dim $w.arr.bitpix $w.arr.skip $w.arr.arch -side top \
	-ipadx 4 -ipady 4 -fill x -expand true
    pack $w.arr -fill x -expand true
    pack $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    label $w.arr.dim.title -text "Dimension"
    label $w.arr.dim.dummy -width 4
    entry $w.arr.dim.x -textvariable ed(x) -width 6
    entry $w.arr.dim.y -textvariable ed(y) -width 6
    entry $w.arr.dim.z -textvariable ed(z) -width 6
    grid $w.arr.dim.title $w.arr.dim.x $w.arr.dim.y $w.arr.dim.z \
	$w.arr.dim.dummy -padx 4 -sticky w

    label $w.arr.bitpix.title -text "Pixel Size"
    label $w.arr.bitpix.dummy -width 9
    radiobutton $w.arr.bitpix.char -text Char -variable ed(bitpix) -value 8
    radiobutton $w.arr.bitpix.short -text Short -variable ed(bitpix) -value 16
    radiobutton $w.arr.bitpix.ushort -text UShort -variable ed(bitpix) \
	-value -16
    radiobutton $w.arr.bitpix.long -text Long -variable ed(bitpix) -value 32
    radiobutton $w.arr.bitpix.float -text Float -variable ed(bitpix) -value -32
    radiobutton $w.arr.bitpix.double -text Double -variable ed(bitpix) \
	-value -64
    grid $w.arr.bitpix.title $w.arr.bitpix.char \
	$w.arr.bitpix.dummy -padx 4 -sticky w
    grid x $w.arr.bitpix.short -padx 4 -sticky w
    grid x $w.arr.bitpix.ushort -padx 4 -sticky w
    grid x $w.arr.bitpix.long -padx 4 -sticky w
    grid x $w.arr.bitpix.float -padx 4 -sticky w
    grid x $w.arr.bitpix.double -padx 4 -sticky w

    label $w.arr.skip.t1 -text "Skip First"
    label $w.arr.skip.t2 -text "bytes"
    label $w.arr.skip.dummy -width 4
    entry $w.arr.skip.skip -textvariable ed(skip) -width 6
    grid $w.arr.skip.t1 $w.arr.skip.skip $w.arr.skip.t2 \
	$w.arr.skip.dummy -padx 4 -sticky w

    label $w.arr.arch.title -text "Architecture"
    label $w.arr.arch.dummy -width 0
    radiobutton $w.arr.arch.big -text "Big-Endian" \
	-variable ed(arch) -value bigendian
    radiobutton $w.arr.arch.little -text "Little-Endian" \
	-variable ed(arch) -value littleendian
    grid $w.arr.arch.title $w.arr.arch.big $w.arr.arch.dummy -padx 4 -sticky w
    grid x $w.arr.arch.little -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    set r {}
    if {$ed(ok)} {
	set r "\[xdim=$ed(x),ydim=$ed(y),zdim=$ed(z),bitpix=$ed(bitpix),skip=$ed(skip),arch=$ed(arch)\]"
    }

    # we want to destroy this window

    destroy $w 

    unset ed
    return $r
}

proc MosaicWCSDialog {} {
    global env
    global ed
    global menu

    set w ".wcs"
    set length 300

    set ed(ok) 0
    set ed(sys) wcs
    set ed(label) WCS

    DialogCreate $w "Load Mosaic" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    label $w.param.title -text "Select Coordinate System "
    menubutton $w.param.sys -textvariable ed(label) \
	-menu $w.param.sys.m -relief raised -width 10
    grid $w.param.title $w.param.sys

    menu $w.param.sys.m -tearoff 0 -selectcolor $menu(selectcolor)
    $w.param.sys.m add radiobutton -label "WCS" -variable ed(sys) \
	-value "wcs" -command [list set ed(label) WCS]
    $w.param.sys.m add separator
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$w.param.sys.m add radiobutton -label "WCS $l" -variable ed(sys) \
	    -value "wcs$l" -command [list set ed(label) "WCS $l"]
    }

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    set r {}
    if {$ed(ok)} {
	set r $ed(sys)
    }

    # we want to destroy this window
    destroy $w 

    unset ed
    return $r
}

proc ProcessShmCmd {varname iname flag fcName} {
    upvar $varname var
    upvar $iname i
    upvar $fcName fc

    global loadParam
    global current
    global ds9
    global message

    StartLoad

    set done 0
    while {!$done} {

	# defaults
	set loadParam(load,type) shared
	set loadParam(file,type) fits
	set loadParam(file,mode) {}

	set nn [lindex $var [expr $i+4]]
	if {$nn == {} || [string range $nn 0 0] == "-"} {
	    set def 1
	} else {
	    set def 0
	}

	# do we have a rgb frame
	if {[$current(frame) get type] == "rgb"} {
	    set base 0
	} else {
	    set base 1
	}

	switch -- [lindex $var $i] {
	    key -
	    shmid {
		if {$flag && $base} {
		    MultiLoad fc
		}
		set loadParam(shared,idtype) [lindex $var $i]
		set loadParam(shared,id) [lindex $var [expr $i+1]]
		set loadParam(file,name) [lindex $var [expr $i+2]]
		incr i 2
	    }

	    fits {
		if {$flag && $base} {
		    MultiLoad fc
		}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    sfits {
		if {$flag && $base} {
		    MultiLoad fc
		}
		set loadParam(load,type) sshared
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,hdr) [lindex $var [expr $i+2]]
		set loadParam(shared,id) [lindex $var [expr $i+3]]
		set loadParam(file,name) [lindex $var [expr $i+4]]
		incr i 4
	    }

	    mosaicimage {
		if {$flag && $base} {
		    MultiLoad fc
		}

		if {$def} {
		    set loadParam(file,mode) {mosaic image iraf}
		    set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		    set loadParam(shared,id) [lindex $var [expr $i+2]]
		    set loadParam(file,name) [lindex $var [expr $i+3]]
		    incr i 3
		} else {
		    set loadParam(file,mode) \
			[list mosaic image [lindex $var [expr $i+1]]]
		    set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		    set loadParam(shared,id) [lindex $var [expr $i+3]]
		    set loadParam(file,name) [lindex $var [expr $i+4]]
		    incr i 4
		}
	    }
	    mosaicimagenext {
		if {$def} {
		    set loadParam(file,mode) {mosaic image next wcs}
		    set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		    set loadParam(shared,id) [lindex $var [expr $i+2]]
		    set loadParam(file,name) [lindex $var [expr $i+3]]
		    incr i 3
		} else {
		    set loadParam(file,mode) \
			[list mosaic image next [lindex $var [expr $i+1]]]
		    set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		    set loadParam(shared,id) [lindex $var [expr $i+3]]
		    set loadParam(file,name) [lindex $var [expr $i+4]]
		    incr i 4
		}
	    }
	    mosaic {
		if {$def} {
		    set loadParam(file,mode) {mosaic iraf}
		    set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		    set loadParam(shared,id) [lindex $var [expr $i+2]]
		    set loadParam(file,name) [lindex $var [expr $i+3]]
		    incr i 3
		} else {
		    set loadParam(file,mode) \
			[list mosaic [lindex $var [expr $i+1]]]
		    set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		    set loadParam(shared,id) [lindex $var [expr $i+3]]
		    set loadParam(file,name) [lindex $var [expr $i+4]]
		    incr i 4
		}
	    }
	    smosaic {
		set loadParam(load,type) sshared
		set loadParam(file,mode) \
		    [list mosaic [lindex $var [expr $i+1]]]
		set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		set loadParam(shared,hdr) [lindex $var [expr $i+3]]
		set loadParam(shared,id) [lindex $var [expr $i+4]]
		set loadParam(file,name) [lindex $var [expr $i+5]]
		incr i 5
	    }

	    mosaicimageiraf {
		# backward compatibility
		if {$flag && $base} {
		    MultiLoad fc
		}
		set loadParam(file,mode) {mosaic image iraf}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaiciraf {
		# backward compatibility
		set loadParam(file,mode) {mosaic iraf}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicimagewcs {
		# backward compatibility
		if {$flag && $base} {
		    MultiLoad fc
		}
		set loadParam(file,mode) {mosaic image wcs}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicimagenextwcs {
		# backward compatibility
		set loadParam(file,mode) {mosaic image next wcs}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicwcs {
		# backward compatibility
		set loadParam(file,mode) {mosaic wcs}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicimagewfpc2 {
		# backward compatibility
		if {$flag && $base} {
		    MultiLoad fc
		}
		set loadParam(file,mode) {mosaic image wfpc2}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }

	    rgbcube {
		if {$flag} {
		    MultiRGBLoad fc
		}
		if {[$current(frame) get type] == "rgb"} {
		    set loadParam(file,mode) {rgb cube}
		} else {
		    Error "$message(error,fits,rgb)"
		}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    srgbcube {
		if {$flag} {
		    MultiRGBLoad fc
		}

		set loadParam(load,type) sshared
		if {[$current(frame) get type] == "rgb"} {
		    set loadParam(file,mode) {rgb cube}
		} else {
		    Error "$message(error,fits,rgb)"
		}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,hdr) [lindex $var [expr $i+2]]
		set loadParam(shared,id) [lindex $var [expr $i+3]]
		set loadParam(file,name) [lindex $var [expr $i+4]]
		incr i 4
	    }
	    rgbimage {
		if {$flag} {
		    MultiRGBLoad fc
		}

		if {[$current(frame) get type] == "rgb"} {
		    set loadParam(file,mode) {rgb image}
		} else {
		    Error "$message(error,fits,rgb)"
		}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    rgbarray {
		if {$flag} {
		    MultiRGBLoad fc
		}

		if {[$current(frame) get type] == "rgb"} {
		    set loadParam(file,type) array
		    set loadParam(file,mode) {rgb cube}
		} else {
		    Error "$message(error,fits,rgb)"
		}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    array {
		if {$flag && $base} {
		    MultiLoad fc
		}

		set loadParam(file,type) array
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }

	    default {
		if {$flag && $base} {
		    MultiLoad fc
		}

		set loadParam(shared,idtype) key
		set loadParam(shared,id) [lindex $var $i]
		set loadParam(file,name) [lindex $var [expr $i+1]]
		incr i 1
	    }
	}

	ProcessLoad

	# more to come?
	incr i
	if {([lindex $var $i] == "-shm") || 
	    ([lindex $var $i] == "shm")} {
	    set done 0
	    incr i
	} else {
	    set done 1
	    incr i -1
	}
    }
    FinishLoad
}

proc ProcessPreserveCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    global scale
    global panzoom
    global marker

    switch -- [string tolower [lindex $var $i]] {
	scale {
	    incr i
	    set scale(preserve) [FromYesNo [lindex $var $i]]
	    PreserveScale
	}
	pan {
	    incr i
	    set panzoom(preserve) [FromYesNo [lindex $var $i]]
	    PreservePan
	}
	marker -
	regions {
	    incr i
	    set marker(preserve) [FromYesNo [lindex $var $i]]
	    MarkerPreserve
	}
    }
}
