#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc DSSDialog {} {
    global dss
    global menu

    if [winfo exists $dss(top)] {
	raise $dss(top)
	return
    }

    set w $dss(top)
    IMGSVRInit dss "DSS Server" DSSExec DSSAck DSSUpdate

    # custom dialog
    label $w.param.wtitle -text "Width"
    entry $w.param.w -textvariable dss(width) -width 14
    label $w.param.htitle -text "Height"
    entry $w.param.h -textvariable dss(height) -width 14
    label $w.param.format -textvariable dss(rformat) \
	-width 10 -relief groove

    grid $w.param.wtitle $w.param.w $w.param.htitle $w.param.h \
	$w.param.format -padx 4 -pady 1 -sticky w

    # menu
    IMGSVRFileMenu dss
    AREditMenu dss
    $dss(mb) add cascade -label "DSS Server" -menu $dss(mb).server
    NSVRServerMenu dss
    $dss(mb) add cascade -label Survey -menu $dss(mb).survey
    IMGSVRPrefsMenu dss

    menu $dss(mb).server -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).server add radiobutton -label "DSS@SAO" \
	-variable dss(server) -value sao -command DSSServer
    $dss(mb).server add radiobutton -label "DSS@STSCI" \
	-variable dss(server) -value stsci -command DSSServer
    $dss(mb).server add radiobutton -label "DSS@ESO" \
	-variable dss(server) -value eso -command DSSServer

    menu $dss(mb).survey -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).survey add radiobutton -label "DSS1" \
	-variable dss(survey) -value dss
    $dss(mb).survey add radiobutton -label "DSS2-red" \
	-variable dss(survey) -value dss2red
    $dss(mb).survey add radiobutton -label "DSS2-blue" \
	-variable dss(survey) -value dss2blue

    IMGSVRUpdate dss 1

    DSSServer
}

proc DSSDestroyDialog {} {
    global dss

    set varname dss
    IMGSVRDestroy $varname
}

proc DSSExec {} {
    global dss

    switch -- $dss(server) {
	sao {DSSSAO}
	stsci {DSSSTSCI}
	eso {DSSESO}
    }
}

proc DSSSAO {} {
    global dss

    if {$dss(save)} {
	set compress no
	set dss(fn) [SaveFileDialog dssfbox]
	if {$dss(fn) == {}} {
	    return
	}

    } else {
	set compress gzip
	set dss(fn) [tmpnam dss ".fits.gz"]
    }

    if {$dss(width)>60} {
	set dss(width) 60
    }
    if {$dss(height)>60} {
	set dss(height) 60
    }
    set query [http::formatQuery r $dss(x) d $dss(y) e J2000 \
		   w $dss(width) h $dss(height) \
		   c $compress]

    IMGSVRLoad dss "http://archive.harvard.edu/archive/dss" $query
}

proc DSSESO {} {
    global dss

    switch -- $dss(survey) {
	dss {set survey "DSS1"}
	dss2red {set survey "DSS2-red"}
	dss2blue {set survey "DSS2-blue"}
    }

    if {$dss(save)} {
	set mime "application/x-fits"
	set dss(fn) [SaveFileDialog dssfbox]
	if {$dss(fn) == {}} {
	    return
	}
    } else {
	set mime "display/gz-fits"
	set dss(fn) [tmpnam dss ".fits.gz"]
    }

    if {$dss(width)>40} {
	set dss(width) 40
    }
    if {$dss(height)>40} {
	set dss(height) 40
    }
    set query [http::formatQuery ra $dss(x) dec $dss(y) equinox J2000\
		   x $dss(width) y $dss(height) \
		   mime-type "$mime" \
		   Sky-Survey $survey]

    # Load image
    # we can't use -query because eso needs a GET not a POST

    IMGSVRLoad dss "http://archive.eso.org/dss/dss?$query" {}
}

proc DSSSTSCI {} {
    global dss

    switch -- $dss(survey) {
	dss {set survey "1"}
	dss2red {set survey "2r"}
	dss2blue {set survey "2b"}
    }

    if {$dss(save)} {
	set compress none
	set dss(fn) [SaveFileDialog dssfbox]
	if {$dss(fn) == {}} {
	    return
	}

    } else {
	set compress gz
	set dss(fn) [tmpnam dss ".fits.gz"]
    }

    if {$dss(width)>60} {
	set dss(width) 60
    }
    if {$dss(height)>60} {
	set dss(height) 60
    }
    set query [http::formatQuery r $dss(x) d $dss(y) e J2000\
		   w $dss(width) h $dss(height) \
		   f fits c $compress \
		   v $survey]

    IMGSVRLoad dss "http://stdatu.stsci.edu/cgi-bin/dss_search" $query
}

proc DSSAck {} {
set msg {Acknowledgements for the DSS 

The Digitized Sky Surveys were produced at the Space Telescope Science
Institute under U.S.  Government grant NAG W-2166. The images of these
surveys are based on photographic data obtained using the Oschin
Schmidt Telescope on Palomar Mountain and the UK Schmidt Telescope.
The plates were processed into the present compressed digital form
with the permission of these institutions.

The National Geographic Society - Palomar Observatory Sky Atlas
(POSS-I) was made by the California Institute of Technology with
grants from the National Geographic Society.

The Second Palomar Observatory Sky Survey (POSS-II) was made by the
California Institute of Technology with funds from the National
Science Foundation, the National Geographic Society, the Sloan
Foundation, the Samuel Oschin Foundation, and the Eastman Kodak
Corporation.

The Oschin Schmidt Telescope is operated by the California Institute
of Technology and Palomar Observatory.

The UK Schmidt Telescope was operated by the Royal Observatory
Edinburgh, with funding from the UK Science and Engineering Research
Council (later the UK Particle Physics and Astronomy Research
Council), until 1988 June, and thereafter by the Anglo-Australian
Observatory. The blue plates of the southern Sky Atlas and its
Equatorial Extension (together known as the SERC-J), as well as the
Equatorial Red (ER), and the Second Epoch [red] Survey (SES) were all
taken with the UK Schmidt.
}

    SimpleTextDialog dssack Acknowledgement 80 40 insert top $msg
}

proc DSSUpdate {} {
    global dss
    global current

    set size [$current(frame) get fits size $dss(system) $dss(rformat)]
    set dss(width) [lindex $size 0]
    set dss(height) [lindex $size 1]
}

proc DSSServer {} {
    global dss

    switch -- $dss(server) {
	sao {$dss(mb) entryconfig "Survey" -state disabled}
	stsci {$dss(mb) entryconfig "Survey" -state normal}
	eso {$dss(mb) entryconfig "Survey" -state normal}
    }
}

proc ProcessDSSCmd {varname iname flag fcName} {
    upvar $varname var
    upvar $iname i
    upvar $fcName fc

    global current
    global dss

    # do we have a rgb frame
    if {[$current(frame) get type] == "rgb"} {
	set base 0
    } else {
	set base 1
    }

    switch -- [string tolower [lindex $var $i]] {
	server {
	    incr i
	    set dss(server) [lindex $var $i]
	}
	survey {
	    incr i
	    set dss(survey) [lindex $var $i]
	    switch -- $dss(survey) {
		dss {}
		dss2red -
		dss2blue {set dss(server) eso}
	    }
	}
	size {
	    incr i
	    set dss(width) [lindex $var $i]
	    incr i
	    set dss(height) [lindex $var $i]
	}
	width {
	    incr i
	    set dss(width) [lindex $var $i]
	}
	height {
	    incr i
	    set dss(height) [lindex $var $i]
	}
	coord {
	    if {$flag} {
		MultiLoad fc
	    }
	    incr i
	    set dss(x) [lindex $var $i]
	    incr i
	    set dss(y) [lindex $var $i]
	    DSSCmd
	}
	x {
	    incr i
	    set dss(x) [lindex $var $i]
	}
	y {
	    incr i
	    set dss(y) [lindex $var $i]
	}
	name {
	    if {$flag && $base} {
		MultiLoad fc
	    }
	    incr i
	    set dss(name) [lindex $var $i]
	    DSSCmd
	}
	default {
	    if {$flag && $base} {
		MultiLoad fc
	    }
	    set dss(name) [lindex $var $i]
	    DSSCmd
	}
    }
}

proc DSSCmd {} {
    global dss

    if {$dss(width) == ""} {
	set dss(width) 15
    }
    if {$dss(height) == ""} {
	set dss(height) 15
    }
    set dss(mode) current
    set dss(sync) 1
    
    DSSDialog
    IMGSVRApply dss 1
}

