#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

set cat(def) { \
	   {{SAO Star Catalog J2000} catsao {I/131A}} \
	   {{HST Guide Star Catalog} cathst {I/254}} \
	   {{GSC 2.2 Catalog} catgsc {I/271}} \
	   {{Tycho-2 Catalog} cattycho {I/259}} \
	   {{Hipparcos AC 2000.2 Catalog} catac {I/275}} \
	   {{USNO-A2.0 Catalog} catua2 {I/252}} \
	   {{USNO-B1.0 Catalog} catub1 {I/284}} \
	   {{USNO UCAC2 Catalog} catucac2 {I/289}} \
	   {- - -} \
	   {{2MASS All-Sky Catalog of Point Sources} cat2mass {II/246}}\
	   {{DENIS Database} catdenis {B/denis}} \
	   {- - -} \
	   {{Second ROSAT PSPC Catalog} catrosat {IX/30}} \
	   {{ROSAT All-Sky Survey Faint Source Catalog} catrosf {IX/29}}\
	   {- - -} \
	   {{FIRST Survey Catalog} catfirst {VIII/71}} \
}

proc CATDialog {varname catalog title action} {
    upvar #0 $varname var
    global $varname

    global ds9
    global menu
    global cat

    # main dialog
    set var(top) ".${varname}"
    set var(mb) ".${varname}mb"

    if [winfo exists $var(top)] {
	raise $var(top)
	return
    }

    # CAT variables
    lappend cat(current) $varname

    # AR variables
    set var(status) {}
    set var(sync) 0
    set var(proc,next) CATServer

    # CAT variables
    set var(dir) ${varname}dir
    set var(catdb) ${varname}catdb
    set var(tbldb) ${varname}tbldb
    set var(symdb) ${varname}symdb
    set var(symdl) ${varname}symdl
    set var(fltdl) ${varname}fltdl

    set var(nameserver) $cat(nameserver)
    set var(server) $cat(server)
    set var(system) $cat(system)
    set var(sky) $cat(sky)
    set var(skyformat) $cat(skyformat)
    set var(rformat) $cat(rformat)
    set var(width) $cat(width)
    set var(height) $cat(height)
    set var(max) $cat(max)
    set var(allcols) 0

    set var(name) {}
    set var(x) {}
    set var(y) {}
    set var(state) 0

    set var(blink) 0
    set var(blink,count) 0
    set var(blink,marker) {}

    CATSet $varname $catalog $title
    CATSymDBInit $varname

    # top
    set w $var(top)
    set t "Catalog Tool"

    # create the window
    toplevel $w -colormap $ds9(main)
    wm title $w $t
    wm iconname $w $t
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW "CATDestroy $varname"

    # menu
    $w configure -menu $var(mb)
    menu $var(mb) -tearoff 0

    # file
    $var(mb) add cascade -label File -menu $var(mb).file
    menu $var(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).file add cascade -label "Catalogs" \
	-menu $var(mb).file.cat
    $var(mb).file add separator
    $var(mb).file add command -label "Load..." -command "CATLoad $varname"
    $var(mb).file add command -label "Save..." -command "CATSave $varname"
    $var(mb).file add separator
    $var(mb).file add command -label "Display Catalog Header..." \
	-command "CATHeader $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Filter -command "CATTable $varname"
    $var(mb).file add command -label Clear -command "CATClear $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Retrieve -command "CATApply $varname 0"
    $var(mb).file add command -label Cancel -command "ARCancel $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Acknowledgement -command "CATAck $varname"
    $var(mb).file add separator
    $var(mb).file add command -label "Update from Current Frame" \
	-command "CATUpdate $varname"
    $var(mb).file add command -label "Update from Current Crosshair" \
	-command "CATCrosshair $varname"
    $var(mb).file add separator
    $var(mb).file add command -label "Copy to Regions" \
	-command "CATPlotRegions $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Print -command "CATPrint $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Close -command "CATDestroy $varname"

    menu $var(mb).file.cat -tearoff 0 -selectcolor $menu(selectcolor)
    CATCatalogMenu $varname

    # edit
    AREditMenu $varname

    # catalog
    $var(mb) add cascade -label "Catalog Server" -menu $var(mb).server
    menu $var(mb).server -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).server add radiobutton -label "CDS" \
	-variable ${varname}(server) -value cds
    $var(mb).server add radiobutton -label "SAO" \
	-variable ${varname}(server) -value sao
    $var(mb).server add radiobutton -label "CADC" \
	-variable ${varname}(server) -value cadc
    $var(mb).server add radiobutton -label "ADAC" \
	-variable ${varname}(server) -value adac
    $var(mb).server add radiobutton -label "IUCAA" \
	-variable ${varname}(server) -value iucaa
    $var(mb).server add radiobutton -label "BEJING" \
	-variable ${varname}(server) -value bejing
    $var(mb).server add radiobutton -label "CAMBRIDGE UK" \
	-variable ${varname}(server) -value cambridge
    $var(mb).server add radiobutton -label "UKIRT HAWAII" \
	-variable ${varname}(server) -value ukirt

    # name server
    NSVRServerMenu $varname

    # coordinate
    $var(mb) add cascade -label Coord -menu $var(mb).coord
    menu $var(mb).coord -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).coord add radiobutton -label "Equatorial B1950" \
	-variable ${varname}(sky) -value fk4 \
	-command "CATCoord $varname"
    $var(mb).coord add radiobutton -label "Equatorial J2000" \
	-variable ${varname}(sky) -value fk5 \
	-command "CATCoord $varname"
    $var(mb).coord add radiobutton -label "ICRS" \
	-variable ${varname}(sky) -value icrs \
	-command "CATCoord $varname"
    $var(mb).coord add separator
    $var(mb).coord add radiobutton -label "Degrees" \
	-variable ${varname}(skyformat) -value degrees \
	-command "CATCoord $varname"
    $var(mb).coord add radiobutton -label "Sexagesimal" \
	-variable ${varname}(skyformat) -value sexagesimal \
	-command "CATCoord $varname"

    # size
    $var(mb) add cascade -label Size -menu $var(mb).size
    menu $var(mb).size -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).size add radiobutton -label "Degrees" \
	-variable ${varname}(rformat) -value degrees \
	-command "CATCoord $varname"
    $var(mb).size add radiobutton -label "ArcMin" \
	-variable ${varname}(rformat) -value arcmin \
	-command "CATCoord $varname"
    $var(mb).size add radiobutton -label "ArcSec" \
	-variable ${varname}(rformat) -value arcsec \
	-command "CATCoord $varname"

    # symbol
    global $var(symdb)
    set flt $var(symdb)
    set sn [starbase_colnum $var(symdb) shape]
    set cn [starbase_colnum $var(symdb) color]

    $var(mb) add cascade -label Symbol -menu $var(mb).symbol
    menu $var(mb).symbol -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).symbol add radiobutton -label {Circle Point} \
	-variable ${flt}(1,$sn) -value {circle point} \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label {Box Point} \
	-variable ${flt}(1,$sn) -value {box point} \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label {Diamond Point} \
	-variable ${flt}(1,$sn) -value {diamond point} \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label {Cross Point} \
	-variable ${flt}(1,$sn) -value {cross point} \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label {X Point} \
	-variable ${flt}(1,$sn) -value {x point} \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label {Arrow Point} \
	-variable ${flt}(1,$sn) -value {arrow point} \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label {Boxcircle Point} \
	-variable ${flt}(1,$sn) -value {boxcircle point} \
	-command "CATPlot $varname"
    $var(mb).symbol add separator
    $var(mb).symbol add radiobutton -label Circle \
	-variable ${flt}(1,$sn) -value circle \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Ellipse \
	-variable ${flt}(1,$sn) -value ellipse \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Box \
	-variable ${flt}(1,$sn) -value box \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Text \
	-variable ${flt}(1,$sn) -value text \
	-command "CATPlot $varname"
    $var(mb).symbol add separator
    $var(mb).symbol add radiobutton -label White \
	-variable ${flt}(1,$cn) -value white \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Black \
	-variable ${flt}(1,$cn) -value black \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Red \
	-variable ${flt}(1,$cn) -value red \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Green \
	-variable ${flt}(1,$cn) -value green \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Blue \
	-variable ${flt}(1,$cn) -value blue \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Cyan \
	-variable ${flt}(1,$cn) -value cyan \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Magenta \
	-variable ${flt}(1,$cn) -value magenta \
	-command "CATPlot $varname"
    $var(mb).symbol add radiobutton -label Yellow \
	-variable ${flt}(1,$cn) -value yellow \
	-command "CATPlot $varname"
    $var(mb).symbol add separator
    $var(mb).symbol add command -label "Advanced..." \
	-command "CATSymDialog $varname"

    # dialog
    frame $w.param -relief groove -borderwidth 2
    frame $w.tbl -relief groove -borderwidth 2
    frame $w.status -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2

    pack $w.buttons $w.status -side bottom -fill x -ipadx 4 -ipady 4
    pack $w.param -side top -fill x  -ipadx 4 -ipady 4
    pack $w.tbl -side top -fill both -expand true -ipadx 4 -ipady 4

    # param
    frame $w.param.f
    pack $w.param.f -side left

    label $w.param.f.ttitle -text "Catalog"
    label $w.param.f.title -textvariable ${varname}(title) \
	-relief groove -borderwidth 2 -width 40 -anchor w

    label $w.param.f.tcat -text "Id"
    label $w.param.f.cat -textvariable ${varname}(catalog) \
	-relief groove -borderwidth 2 -width 14 -anchor w

    label $w.param.f.nametitle -text "Name"
    entry $w.param.f.name -textvariable ${varname}(name) -width 40

    set var(xname) [label $w.param.f.xtitle -text "" \
			-font {symbol 12} -width 1]
    entry $w.param.f.x -textvariable ${varname}(x) -width 14
    set var(yname) [label $w.param.f.ytitle -text "" \
			-font {symbol 12} -width 1]
    entry $w.param.f.y -textvariable ${varname}(y) -width 14
    label $w.param.f.system -textvariable ${varname}(sky) \
	-width 10 -relief groove

    label $w.param.f.wtitle -text "Width"
    entry $w.param.f.w -textvariable ${varname}(width) -width 14
    label $w.param.f.htitle -text "Height"
    entry $w.param.f.h -textvariable ${varname}(height) -width 14
    label $w.param.f.format -textvariable ${varname}(rformat) \
	-width 10 -relief groove

    label $w.param.f.mfilter -text "Filter"
    entry $w.param.f.filter -textvariable ${varname}(filter) -width 40
    button $w.param.f.bfilter -text "Edit" \
	-command "CATEditDialog $varname filter $var(catdb)"

    label $w.param.f.msort -text "Sort"
    menubutton $w.param.f.sort -textvariable ${varname}(sort) \
	-menu $w.param.f.sort.m -relief raised -bd 2 -width 14
    radiobutton $w.param.f.isort -text "Incr" \
	-variable ${varname}(sort,dir) -value "-increasing" \
	-selectcolor $menu(selectcolor) -command "CATTable $varname"
    radiobutton $w.param.f.dsort -text "Decr" \
	-variable ${varname}(sort,dir) -value "-decreasing" \
	-selectcolor $menu(selectcolor) -command "CATTable $varname"

    label $w.param.f.mtitle -text "Max Rows"
    entry $w.param.f.max -textvariable ${varname}(max) -width 14

    checkbutton $w.param.f.all -selectcolor $menu(selectcolor) \
	-text "All Columns" -variable ${varname}(allcols) \
	-command "CATApply $varname 0"

    label $w.param.f.ftitle -text "Found" 
    set var(found) [label $w.param.f.found  \
			-relief groove -borderwidth 2 -width 14 -anchor w]

    grid rowconfigure $w.param.f 0 -pad 4
    grid rowconfigure $w.param.f 1 -pad 4
    grid rowconfigure $w.param.f 2 -pad 4
    grid rowconfigure $w.param.f 3 -pad 4
    grid rowconfigure $w.param.f 4 -pad 4
    grid rowconfigure $w.param.f 5 -pad 4
    grid rowconfigure $w.param.f 6 -pad 4

    grid $w.param.f.ttitle $w.param.f.title - - -padx 4 -pady 1 -sticky w
    grid $w.param.f.tcat $w.param.f.cat - - -padx 4 -pady 1 -sticky w
    grid $w.param.f.nametitle $w.param.f.name - - -padx 4 -pady 1 -sticky w
    grid $w.param.f.xtitle $w.param.f.x $w.param.f.ytitle $w.param.f.y \
	$w.param.f.system -padx 4 -pady 1 -sticky w
    grid $w.param.f.wtitle $w.param.f.w $w.param.f.htitle \
	$w.param.f.h $w.param.f.format -padx 4 -pady 1 -sticky w
    grid $w.param.f.mfilter $w.param.f.filter - - $w.param.f.bfilter \
	-padx 4 -pady 1 -sticky w
    grid $w.param.f.msort $w.param.f.sort $w.param.f.isort $w.param.f.dsort \
	-padx 4 -pady 1 -sticky w
    grid $w.param.f.mtitle $w.param.f.max $w.param.f.ftitle $w.param.f.found \
	$w.param.f.all -padx 4 -pady 1 -sticky w

    # table
    set var(tbl) [table $var(top).tbl.t \
		      -state disabled \
		      -usecommand 0 \
		      -variable $var(tbldb) \
		      -colorigin 1 \
		      -roworigin 0 \
		      -cols $cat(mincols) \
		      -rows $cat(minrows) \
		      -width -1 \
		      -height -1 \
		      -maxwidth 300 \
		      -maxheight 300 \
		      -titlerows 1 \
		      -xscrollcommand [list $var(top).tbl.xscroll set]\
		      -yscrollcommand [list $var(top).tbl.yscroll set]\
		      -selecttype row \
		      -selectmode extended \
		      -browsecommand [list CATSelectCB $varname] \
		     ]

    scrollbar $var(top).tbl.yscroll -command [list $var(tbl) yview] \
	-orient vertical
    scrollbar $var(top).tbl.xscroll -command [list $var(tbl) xview] \
	-orient horizontal

    grid $var(tbl) $var(top).tbl.yscroll -sticky news
    grid $var(top).tbl.xscroll -stick news
    grid rowconfigure $var(top).tbl 0 -weight 1
    grid columnconfigure $var(top).tbl 0 -weight 1

    # status
    label $w.status.item -textvariable ${varname}(status)
    pack $w.status.item -anchor w -pady 4

    # buttons
    set var(apply) [button $w.buttons.apply -text Retrieve \
			-command "CATApply $varname 0"]
    set var(cancel) [button $w.buttons.cancel -text Cancel \
			 -command "ARCancel $varname" -state disabled]
    button $w.buttons.filter -text Filter -command "CATTable $varname"
    button $w.buttons.clear -text Clear -command "CATClear $varname"
    button $w.buttons.close -text Close -command "CATDestroy $varname"
    pack $w.buttons.apply $w.buttons.cancel $w.buttons.filter \
	$w.buttons.clear $w.buttons.close -side left -padx 10 -expand true

    # needs to go after sort menu button is defined
    CATSortMenu $varname

    CATCoord $varname
    ARStatus $varname {}

    switch -- $action {
	apply {CATApply $varname 0}
	sync {CATApply $varname 1}
	search {CATSearch $varname}
    }
}

# Dialog Commands

proc CATDestroy {varname} {
    upvar #0 $varname var
    global $varname
    global $var(catdb)
    global $var(tbldb)
    global $var(dir)
    global $var(symdb)
    global $var(symdl)
    global cat

    upvar #0 $var(symdl) svar
    if [info exists svar(top)] {
	if [winfo exists $svar(top)] {
	    CATSymDestroy $var(symdl)
	}
    }

    ARDestroy $varname
    catch {unset $var(symdb)}
    catch {unset $var(dir)}
    catch {unset $var(tbldb)}
    catch {unset $var(catdb)}
    catch {unset var}

    set i [lsearch $cat(current) $varname]
    if {$i>=0} {
	set cat(current) [lreplace $cat(current) $i $i]
    }
}

proc CATApply {varname sync} {
    upvar #0 $varname var
    global $varname

    if {$var(catalog) == {}} {
	Error "No Catalog specified"
	return
    }

    set var(sync) $sync

    ARStatus $varname {}

    $var(mb).file entryconfig Retrieve -state disabled
    $var(mb).file entryconfig Cancel -state normal

    $var(apply) configure -state disabled
    $var(cancel) configure -state normal

    if {$var(name) != {}} {
	set var(system) wcs
	set var(sky) fk5
	ARCoord $varname

	NSVRServer $varname
    } else {
	CATServer $varname
    }
}

proc CATLoad {varname} {
    upvar #0 $varname var
    global $varname

    set fn [OpenFileDialog catfbox]
    if {$fn != {}} {
	CATLoadFn $varname $fn
    }
}

proc CATLoadFn {varname fn} {
    upvar #0 $varname var
    global $varname
    global $var(catdb)

    CATClear $varname
    CATSet $varname {} $fn

    set var(name) {}
    set var(x) {}
    set var(y) {}
    set var(width) {}
    set var(height) {}

    starbase_read $var(catdb) $fn
    CATSortMenu $varname
    CATTable $varname
}

proc CATSave {varname} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)

    set fn [SaveFileDialog catfbox]
    if {$fn != {}} {
	starbase_write $var(tbldb) $fn
    }
}

proc CATClear {varname} {
    upvar #0 $varname var
    global $varname
    global $var(catdb)
    global $var(tbldb)
    global current

    catch {unset $var(catdb)}
    catch {unset $var(tbldb)}
    set db $var(tbldb)
    set ${db}(Nrows) {}

    $var(tbl) selection clear all

    if {$current(frame) != {}} {
	$current(frame) marker catalog $varname delete
    }

    CATSortMenu $varname
    set var(filter) {}
    set var(sort) {}
}

proc CATCoord {varname} {
    upvar #0 $varname var
    global $varname

    ARCoord $varname
    CATUpdate $varname
}

proc CATUpdate {varname} {
    upvar #0 $varname var
    global $varname

    global current
    if {$current(frame) != {} } {
	if {[$current(frame) has wcs equatorial $var(system)]} {
	    set coord [$current(frame) get cursor \
			   $var(system) $var(sky) $var(skyformat)]
	    set var(x) [lindex $coord 0]
	    set var(y) [lindex $coord 1]

	    set size [$current(frame) get fits size \
			  $var(system) $var(rformat)]
	    set var(width) [lindex $size 0]
	    set var(height) [lindex $size 1]
	    set var(name) {}
	}
    }
}

proc CATCrosshair {varname} {
    upvar #0 $varname var
    global $varname

    global current
    if {$current(frame) != {} } {
	if {[$current(frame) has wcs equatorial $var(system)]} {
	    set coord [$current(frame) get crosshair \
			   $var(system) $var(sky) sexagesimal]
	    set var(x) [lindex $coord 0]
	    set var(y) [lindex $coord 1]
	    set var(name) {}
	}
    }
}

proc CATAck {varname} {
    upvar #0 $varname var
    global $varname

set msg {Acknowledgements

This research has made use of the VizieR catalogue access tool, CDS,
Strasbourg, France. VizieR is a joint effort of CDS (Centre de Données
astronomiques de Strasbourg) and ESA-ESRIN (Information Systems
Division).
}
    SimpleTextDialog ${varname}ack Acknowledgement 80 10 insert top $msg
}

proc CATPrint {varname} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)

    global ps
    global message

    if {[PrintDialog txt]} { 
	if {$ps(dest) == "file"} {
	    catch {set ch [open "| cat > $ps(filename)" w]}
	} else {
	    catch {set ch [open "| $ps(cmd)" w]}
	}

	if {$ch != {}} {
	    starbase_writefp $var(tbldb) $ch
	    close $ch
	} else {
	    Error "$message(error,dialog,print)"
	    return
	}
    }
}

proc CATHeader {varname} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)

    SimpleTextDialog ${varname}hdr "$var(title) Header" \
	80 20 insert top [CATGetHeader $varname]
}

proc CATGetHeader {varname} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)

    if [CATValidDB $var(tbldb)] {
	set hdr {}
	set nl [expr $${var(tbldb)}(HLines) - 2]
	for {set ll 1} {$ll <= $nl} {incr ll} {
	    append hdr "[expr $${var(tbldb)}(H_$ll)]\n"
	}
	return $hdr
    }
    return {}
}

# Other procedures

proc CATServer {varname} {
    upvar #0 $varname var
    global $varname

    if {($var(x) != {}) && 
	($var(y) != {}) && 
	($var(width) != {}) && 
	($var(height) != {})} {

	ARStatus $varname "Searching $var(title)"
	CATCDS $varname
    } else {
	Message "Please specify width, height, and either name or (ra,dec)"

	$var(mb).file entryconfig Retrieve -state normal
	$var(mb).file entryconfig Cancel -state disabled

	$var(apply) configure -state normal
	$var(cancel) configure -state disabled
    }
}

proc CATCDS {varname} {
    upvar #0 $varname var
    global $varname

    global http

    switch -- $var(server) {
	cds {set site "vizier.u-strasbg.fr"}
	sao {set site "vizier.cfa.harvard.edu"}
	cadc {set site "vizier.hia.nrc.ca"}
	adac {set site "vizier.nao.ac.jp"}
	iucaa {set site "urania.iucaa.ernet.in"}
	moscow {set site "www.inasan.rssi.ru"}
	bejing {set site "data.bao.ac.cn"}
	cambridge {set site "archive.ast.cam.ac.uk"}
	ukirt {set site "www.ukirt.jach.hawaii.edu"}
    }

    set cgidir "viz-bin"
    set script "asu-tsv"

    set url "http://$site/$cgidir/$script"
    
    # defaults
    set query {}
    append query "-source=$var(catalog)&"

    # source
    append query "-c.ra=$var(x)&"
    append query "-c.dec=$var(y)&"
    switch -- $var(sky) {
	fk4 {append query "-c.eq=B1950&"}
	fk5 {append query "-c.eq=J2000&"}
	icrs {append query "-c.eq=J2000&"}
	galactic {append query "-c.eq=Gal&"}
    }

    switch -- $var(rformat) {
	degrees {append query "-c.bd=$var(width)x$var(height)&"}
	arcmin {append query "-c.bm=$var(width)x$var(height)&"}
	arcsec {append query "-c.bs=$var(width)x$var(height)&"}
    }

    # output
    append query "-out.max=$var(max)&"
    if {$var(allcols)} {
	append query "-out.all&"
    }

    set out [http::formatQuery "Tab-Separated-Values"]
    append query "-out.form=$out&"

    set cols [http::formatQuery _ra _dec]
    append query "-out.add=$cols&"
    append query "-oc.form=dec"

    # clear previous db
    global $var(catdb)
    catch {unset $var(catdb)}

    # geturl
    if {$var(sync)} {
	set token [http::geturl $url?$query \
		       -handler [list CATCDSReader $var(catdb)] \
		       -headers "[ProxyHTTP]"]
	set var(state) 1
	set var(token) $token

	CATCDSFinish $varname $token
    } else {
	set token [http::geturl $url?$query \
		       -handler [list CATCDSReader $var(catdb)] \
		       -command [list CATCDSFinish $varname]\
		       -headers "[ProxyHTTP]"]

	set var(state) 1
	set var(token) $token
    }
}

proc CATCDSReader {t sock token} {
    upvar #0 $t T
    global $t

    set result 0

    if { ![info exists ${t}(state)]  } {
	set T(state) 0
    }

    switch -- $T(state) {
	0 {
	    # init db
	    fconfigure $sock -blocking 1
	    set T(Nrows) 0
	    set T(Ncols) 0
	    set T(Header) {}
	    set T(HLines) 0

	    set T(state) 1
	}

	1 {
	    # process header
	    incr ${t}(HLines)
	    set n $T(HLines)
	    if {[gets $sock line] == -1} {
		set T(state) -1
		set T(HLines) [expr $T(HLines) - 1]
		set T(Nrows) 0
		set T(Ncols) 0
		return
	    }

	    set result [string length "$line"]
	    set T(H_$n) $line
	    if {[regexp -- {^ *(-)+ *(\t *(-)+ *)*} $line]} {
		# remove units line
		unset T(H_$n)
		incr ${t}(HLines) -1
		incr n -1
		set T(H_$n) $line
		
		# clean up header column name
		set hh $T(H_[expr $n-1])
		regsub -all {\[} $hh {} hh
		regsub -all {\]} $hh {} hh
		set T(H_[expr $n-1]) $hh

		# cols
		set T(Header) [split $T(H_[expr $n-1]) "\t"]
		set T(Dashes) [split $T(H_$n) "\t"]
		set T(Ndshs) [llength $T(Dashes)]
		
		starbase_colmap $t
		set T(state) 2
	    }
	}

	2 { 
	    # process table
	    if {[gets $sock line] == -1} {
		set T(state) 0
	    } else {
		set result [string length "$line"]
		set line [string trim $line]
		if {$line != {}} {
		    # check for beginning of another table
		    if {[string range $line 0 0] == "#"} {
			set T(state) 3
			break
		    }

		    # ok, save it
		    incr ${t}(Nrows)
		    set r $T(Nrows)

		    set NCols [starbase_ncols $t]
		    set c 1
		    foreach val [split $line "\t"] {
			set T($r,$c) $val
			incr c
		    }
		    for {} {$c <= $NCols} {incr c} {
			set T($r,$c) {}
		    }
		}
	    }
	}

	3 {
	    # finished, eat everything else
	    if {[gets $sock line] == -1} {
		set T(state) 0
	    }
	}
    }

    return $result
}

proc CATCDSFinish {varname token} {
    upvar #0 $varname var
    global $varname

    HTTPLog $token
    if {$var(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    ARError $varname $code
	    return
	}

	ARStatus $varname {Done}
	ARReset $varname

	CATSortMenu $varname
 	CATTable $varname
    } else {
	ARStatus $varname {Cancelled}
	ARReset $varname
    }
}

proc CATTable {varname} {
    upvar #0 $varname var
    global $varname
    global $var(catdb)
    global cat
    global current

    if {![CATValidDB $var(catdb)]} {
	Error "No catalog data loaded"
	return
    }

    # delete any previous tbldb
    set db ${varname}tbldb
    global $db
    catch {unset $db}

    # clear the selection
    $var(tbl) selection clear all

    # clear regions
    if {$current(frame) != {}} {
	$current(frame) marker catalog $varname delete
    }

    if {$var(filter) == {} && $var(sort) == {}} {
	set var(tbldb) $var(catdb)
    } else {
	set var(tbldb) ${varname}tbldb
	global $var(tbldb)
	if {![CATFltSort $varname]} {
	    Error "Unable to evaluate filter $var(filter)"
	    catch {unset $var(tbldb)}
	    set var(tbldb) $var(catdb)
	}
    }

    global $var(tbldb)
    $var(tbl) configure -variable $var(tbldb)
    $var(found) configure -textvariable ${var(tbldb)}(Nrows)

#    starbase_writefp $var(catdb) stdout
#    starbase_writefp $var(tbldb) stdout

    if {[starbase_nrows $var(tbldb)] == 0} {
	Message "No Items Found"
	return
    }

    set nr [starbase_nrows $var(tbldb)]
    set nc [starbase_ncols $var(tbldb)]

    if { $nc > $cat(mincols)} {
	$var(tbl) configure -cols $nc
    } else {
	$var(tbl) configure -cols $cat(mincols)
    }

    if {$nr > $cat(minrows)} {
	$var(tbl) configure -rows $nr
    } else {
	$var(tbl) configure -rows $cat(minrows)
    }

    CATPlot $varname

    if {$nr >= $var(max)} {
	Message "$nr rows of data have been downloaded. More may be available. You may wish to adjust the maximum allowed."
    }
}

proc CATPlot {varname} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)

    global current
    if {$current(frame) == {}} {
	return
    }

    # do we have a db?
    if {![CATValidDB $var(tbldb)]} {
	return
    }

    ARStatus $varname {Plotting Regions...}

    if {[$current(frame) has wcs equatorial wcs]} {
	# delete any previous 
	$current(frame) marker catalog $varname delete

	# load
	global reg
	set reg [CATReg $varname 1]
	if {$reg != {}} {
	    $current(frame) marker catalog command ds9 var reg
	}
    }
    ARStatus $varname {Done}
}

proc CATPlotRegions {varname} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)

    global current
    if {$current(frame) == {}} {
	return
    }

    # do we have a db?
    if {![CATValidDB $var(tbldb)]} {
	return
    }

    ARStatus $varname {Generating Regions...}

    if {[$current(frame) has wcs equatorial wcs]} {
	# load
	global reg
	set reg [CATReg $varname 0]
	if {$reg != {}} {
	    $current(frame) marker command ds9 var reg
	}
    }

    ARStatus $varname {Done}
}

proc CATSet {varname catalog title} {
    upvar #0 $varname var
    global $varname

    set var(catalog) $catalog
    set var(title) $title
    set var(filter) {}
    set var(sort) {}
    set var(sort,dir) "-increasing"
}

proc CATValidDB {varname} {
    upvar #0 $varname var
    global $varname

    if {[info exists var(Nrows)] && 
	[info exists var(Ncols)] &&
	[info exists var(Header)]} {
	return 1
    } else {
	return 0
    }
}

proc CATCatalogMenu {varname} {
    upvar #0 $varname var
    global $varname
    global cat
    global ds9

    foreach m $cat(def) {
	set l [lindex $m 0]
	set w [lindex $m 1]
	set c [lindex $m 2]

	if {$varname != {}} {
	    if {$l != "-"} {
		$var(mb).file.cat add command \
		    -label $l -command [list CATCatalogCmd $varname $c $l]
	    } else {
		$var(mb).file.cat add separator
	    }
	} else {
	    if {$l != "-"} {
		$ds9(mb).analysis.cat add command \
		    -label $l -command [list CATDialog $w $c $l apply]
	    } else {
		$ds9(mb).analysis.cat add separator
	    }
	}
    }

    if {$varname != {}} {
	$var(mb).file.cat add separator
	$var(mb).file.cat add command -label "Other Catalogs..." \
	    -command "CATSearch $varname"
    } else {
	$ds9(mb).analysis.cat add separator
	$ds9(mb).analysis.cat add command -label "Other Catalogs..." \
	    -command "CATOther search"
    }
}

proc CATCatalogCmd {varname catalog title} {
    upvar #0 $varname var
    global $varname

    CATClear $varname
    CATSet $varname $catalog $title
    CATApply $varname 0
}

proc CATSortMenu {varname} {
    upvar #0 $varname var
    global $varname
    global $var(catdb)
    global menu

    set m $var(top).param.f.sort.m
    catch {destroy $m}

    menu $m -tearoff 0 -selectcolor $menu(selectcolor)
    $m add command -label {} -command "CATSortCmd $varname {}"
    if [CATValidDB $var(catdb)] {
	foreach col [starbase_columns $var(catdb)] {
	    $m add command -label $col -command "CATSortCmd $varname \{$col\}"
	}
    }
}

proc CATSortCmd {varname val} {
    upvar #0 $varname var
    global $varname

    set ${varname}(sort) $val
    CATTable $varname
}

# Tcl Commands

proc CATButton {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    set marker(motion) none

    # else, see if we are on a marker
    set id [$which get marker catalog id $x $y]
    if {$id != 0} {
	$which marker catalog $id highlite only
	$which marker catalog $id move back
	return
   }

    # see if any markers are highlited
    if {[$which get marker catalog highlite number]>0} {
	$which marker catalog unhighlite all
	return
    }
}

proc CATShift {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }


    if {[$which marker catalog highlite toggle $x $y]} {
	set marker(motion) none
	return
    }

    # else, start a region highlite
    $which region highlite begin $x $y
    set marker(motion) shiftregion
}

proc CATMotion {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    switch -- $marker(motion) {
	none {}
	region -
	shiftregion {$which region highlite motion $x $y}
    }
}

proc CATRelease {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    switch -- $marker(motion) {
	none {}
	region {$which region highlite end}
	shiftregion {$which region highlite shift end}
    }

    unset marker(motion)
}

# Callbacks

proc CATHighliteCB {tag id} {
    set t [split $tag .]
    set varname [lindex $t 0]
    set row [lindex $t 1]

    upvar #0 $varname var
    global $varname

    if [info exists ${varname}(top)] {
	if {$var(blink) == 0} {
	    if [info exists ${varname}(tbl)] {
		$var(tbl) selection set $row,1
		$var(tbl) see $row,1
	    }
	}
    }
}

proc CATUnhighliteCB {tag id} {
    set t [split $tag .]
    set varname [lindex $t 0]
    set row [lindex $t 1]

    upvar #0 $varname var
    global $varname

    if [info exists ${varname}(top)] {
	if {$var(blink) == 0} {
	    if [info exists ${varname}(tbl)] {
		$var(tbl) selection clear $row,1
	    }
	}
    }
}

proc CATSelectCB {varname} {
    upvar #0 $varname var
    global $varname

    global current
    if {$current(frame) == {}} {
	return
    }

    $current(frame) marker catalog $varname unhighlite

    # init timer vars
    set var(blink,count) 0
    set var(blink,marker) {}

    set row 0
    foreach ss [$var(tbl) curselection] {
	set rr [lindex [split $ss ,] 0]
	if {$rr != $row} {
	    lappend ${varname}(blink,marker) "\{${varname}.${rr}\}"
	    set row $rr
	}
    }

    # start timer, if needed
    if {!$var(blink)} {
	set var(blink) 1
	CATSelectTimer $varname
    }
}

proc CATSelectTimer {varname} {
    upvar #0 $varname var
    global $varname

    global current

    if {$current(frame) == {}} {
	set var(blink) 0
	set var(blink,count) 0
	set var(blink,marker) {}
	return
    }

    switch -- $var(blink) {
	0 {
	    set var(blink) 0
	    set var(blink,count) 0
	    set var(blink,marker) {}
	}
	1 {
	    foreach mm $var(blink,marker) {
		$current(frame) marker catalog $mm highlite
	    }

	    incr ${varname}(blink,count)
	    if {$var(blink,count) < 5} {
		set var(blink) 2
	    } else {
		set var(blink) 0
	    }

	    after 250 [list CATSelectTimer $varname]
	}
	2 {
	    foreach mm $var(blink,marker) {
		$current(frame) marker catalog $mm unhighlite
	    }
	    set var(blink) 1

	    after 250 [list CATSelectTimer $varname]
	}
    }
}

# Other interface

proc CATOther {action} {
    global cat

    incr cat(id)
    set varname "catoth$cat(id)"
    CATDialog $varname {} {} $action
}

proc CATClearFrame {} {
    global current

    if {$current(frame) != {}} {
	$current(frame) marker catalog delete all
    }
}

proc CATEditDialog {varname which db} {
    upvar #0 $varname var
    global $varname
    global ed
    global menu

    set w ".${varname}edit"
    set mb ".${varname}editmb"

    set ed(ok) 0
    set ed(text) $w.param.txt

    DialogCreate $w Edit -borderwidth 2

    $w configure -menu $mb
    menu $mb -tearoff 0

    # file
    $mb add cascade -label File -menu $mb.file
    menu $mb.file -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.file add command -label OK -command {set ed(ok) 1}
    $mb.file add separator
    $mb.file add command -label Load -command "CATEditLoad"
    $mb.file add command -label Save -command "CATEditSave"
    $mb.file add separator
    $mb.file add command -label Cancel -command {set ed(ok) 0}

    # edit
    $mb add cascade -label Edit -menu $mb.edit
    menu $mb.edit -tearoff 0
    $mb.edit add command -label "Undo" -command "$ed(text) edit undo"
    $mb.edit add command -label "Redo" -command "$ed(text) edit redo"
    $mb.edit add separator
    $mb.edit add command -label "Cut" -command "tk_textCut $ed(text)"
    $mb.edit add command -label "Copy" -command "tk_textCopy $ed(text)"
    $mb.edit add command -label "Paste" -command "tk_textPaste $ed(text)"

    global $db
    # column
    $mb add cascade -label Column -menu $mb.col
    catch {destroy $mb.col}
    menu $mb.col -tearoff 0 -selectcolor $menu(selectcolor)
    if [CATValidDB $db] {
	foreach col [starbase_columns $db] {
	    $mb.col add command -label "$col" \
		-command "$ed(text) insert insert \{\$$col\}"
	}
    }

    # operator
    $mb add cascade -label Operator -menu $mb.op
    menu $mb.op -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.op add command -label {-} \
	-command "$ed(text) insert insert {-}"
    $mb.op add command -label {!} \
	-command "$ed(text) insert insert {!}"
    $mb.op add command -label {(} \
				   -command "$ed(text) insert insert {(}"
	$mb.op add command -label {)} \
	-command "$ed(text) insert insert {)}"
    $mb.op add separator
    $mb.op add command -label {*} \
	-command "$ed(text) insert insert {*}"
    $mb.op add command -label {/} \
	-command "$ed(text) insert insert {/}"
    $mb.op add command -label {%} \
	-command "$ed(text) insert insert {%}"
    $mb.op add command -label {+} \
	-command "$ed(text) insert insert {+}"
    $mb.op add command -label {-} \
	-command "$ed(text) insert insert {-}"
    $mb.op add separator
    $mb.op add command -label {<} \
	-command "$ed(text) insert insert {<}"
    $mb.op add command -label {>} \
	-command "$ed(text) insert insert {>}"
    $mb.op add command -label {<=} \
	-command "$ed(text) insert insert {<=}"
    $mb.op add command -label {>=} \
	-command "$ed(text) insert insert {>=}"
    $mb.op add command -label {==} \
	-command "$ed(text) insert insert {==}"
    $mb.op add command -label {!=} \
	-command "$ed(text) insert insert {!=}"
    $mb.op add separator
    $mb.op add command -label {&&} \
	-command "$ed(text) insert insert {&&}"
    $mb.op add command -label {||} \
	-command "$ed(text) insert insert {||}"

    # operator
    $mb add cascade -label "Math Function" -menu $mb.math
    menu $mb.math -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.math add command -label {acos} \
	-command "$ed(text) insert insert {acos()}"
    $mb.math add command -label {asin} \
	-command "$ed(text) insert insert {asin()}"
    $mb.math add command -label {atan} \
	-command "$ed(text) insert insert {atan()}"
    $mb.math add command -label {atan2} \
	-command "$ed(text) insert insert {atan2(,)}"
    $mb.math add command -label {ceil} \
	-command "$ed(text) insert insert {ceil()}"
    $mb.math add command -label {cos} \
	-command "$ed(text) insert insert {cos()}"
    $mb.math add command -label {cosh} \
	-command "$ed(text) insert insert {cosh()}"
    $mb.math add command -label {exp} \
	-command "$ed(text) insert insert {exp()}"
    $mb.math add command -label {floor} \
	-command "$ed(text) insert insert {floor()}"
    $mb.math add command -label {fmod} \
	-command "$ed(text) insert insert {fmod(,)}"
    $mb.math add command -label {hypot} \
	-command "$ed(text) insert insert {hypot(,)}"
    $mb.math add command -label {log} \
	-command "$ed(text) insert insert {log()}"
    $mb.math add command -label {log10} \
	-command "$ed(text) insert insert {log10()}"
    $mb.math add command -label {pow} \
	-command "$ed(text) insert insert {pow(,)}"
    $mb.math add command -label {sin} \
	-command "$ed(text) insert insert {sin()}"
    $mb.math add command -label {sinh} \
	-command "$ed(text) insert insert {sinh()}"
    $mb.math add command -label {sqrt} \
	-command "$ed(text) insert insert {sqrt()}"
    $mb.math add command -label {tan} \
	-command "$ed(text) insert insert {tan()}"
    $mb.math add command -label {tanh} \
	-command "$ed(text) insert insert {tanh()}"
    $mb.math add command -label {abs} \
	-command "$ed(text) insert insert {abs()}"
    $mb.math add command -label {double} \
	-command "$ed(text) insert insert {double()}"
    $mb.math add command -label {int} \
	-command "$ed(text) insert insert {int()}"
    $mb.math add command -label {round} \
	-command "$ed(text) insert insert {round()}"

    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -fill x -ipadx 4 -ipady 4

    text $w.param.txt \
	-height 10 \
	-width 60 \
	-font {courier 12} \
	-yscrollcommand "$w.param.yscroll set" \
	-xscrollcommand "$w.param.xscroll set" \
	-undo true \
	-wrap none 

    scrollbar $w.param.yscroll -command [list $ed(text) yview] \
	-orient vertical
    scrollbar $w.param.xscroll -command [list $ed(text) xview] \
	-orient horizontal

    grid $ed(text) $w.param.yscroll -sticky news
    grid $w.param.xscroll -stick news
    grid rowconfigure $w.param 0 -weight 1
    grid columnconfigure $w.param 0 -weight 1

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok $w.buttons.cancel -side left -padx 10 -expand true

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    $ed(text) insert end $var($which)
    $ed(text) see end

    DialogCenter $w
    DialogWait $w ed(ok) $w.buttons.ok

    if {$ed(ok)} {
	set flt [$ed(text) get 1.0 end]
	catch {regsub {\n} $flt " " $flt}
	set var($which) [string trim $flt]
    }

    DialogDismiss $w
    catch {destroy $mb}
    unset ed
}

proc CATEditSave {} {
    global ed

    set fn [SaveFileDialog catfltfbox]
    if {$fn != {}} {
	if [catch {open $fn w} fp] {
	    Error "Unable to open $fn: $fp"
	    return
	}
	set flt [$ed(text) get 1.0 end]
	catch {regsub {\n} $flt " " $flt}
	puts $fp [string trim $flt]
	catch {close $fp}
    }
}

proc CATEditLoad {} {
    global ed

    set fn [OpenFileDialog catfltfbox]
    if {$fn != {}} {
	if [catch {open $fn r} fp] {
	    Error "Unable to open $fn: $fp"
	    return
	}
	$ed(text) delete 1.0 end
	$ed(text) insert end [read -nonewline $fp]
	$ed(text) see end
	catch {close $fp}
    }
}

proc ProcessCatalogCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global cat
    global message

    # we need to be realized
    RealizeDS9

    set cvarname {}

    # determine catalog name
    set item [string tolower [lindex $var $i]]
    switch -- $item {
	cds {
	    incr i
	    set item [string tolower [lindex $var $i]]
	    foreach m $cat(def) {
		set l [lindex $m 0]
		set w [lindex $m 1]
		set c [lindex $m 2]
		
		if {"cat${item}" == $w} {
		    CATDialog $w $c $l apply
		    return
		}
	    }

	    # not a default, assume other name
	    incr cat(id)
	    set catname "catoth$cat(id)"
	    CATDialog $catname $item $item sync
	    return
	}
	file -
	load {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    if {$fn != {}} {
		incr cat(id)
		set catname "catoth$cat(id)"
		CATDialog $catname {} {} none
		CATLoadFn $catname $fn
		FileLastFull catfbox $fn
	    }
	    return
	}

	name -
	coordinate -
	size -
	save -
	header -
	filter -
	clear -
	retreive -
	cancel -
	print -
	close -
	server -
	symbol -
	sort -
	maxrows -
	allcols {set cvarname [lindex $cat(current) end]}

	default {
	    set cvarname [lindex $var $i]
	    incr i
	}
    }

    # we should have a catalog now
    global $cvarname
    upvar #0 $cvarname cvar

    if {![info exists cvar(top)]} {
	Error "$message(error,cat,window) $cvarname"
	return
    }
    if {![winfo exists $cvar(top)]} {
	Error "$message(error,cat,window) $cvarname"
	return
    }

    # now, process it
    set item [string tolower [lindex $var $i]]
    switch -- $item {
	name {
	    incr i
	    set cvar(name) [lindex $var $i]
	}
	coordinate {
	    incr i
	    set cvar(x) [lindex $var $i]
	    incr i
	    set cvar(y) [lindex $var $i]
	    incr i
	    set cvar(sky) [lindex $var $i]
	}
	size {
	    incr i
	    set cvar(width) [lindex $var $i]
	    incr i
	    set cvar(height) [lindex $var $i]
	    incr i
	    set cvar(rformat) [lindex $var $i]
	}
	save {
	    global $cvar(tbldb)
	    incr i
	    starbase_write $cvar(tbldb) [lindex $var $i]
	}
	header {}
	filter {
	    incr i
	    set item [lindex $var $i]
	    switch -- $item {
		load {
		    incr i
		    set fn [lindex $var $i]
		    if [catch {open $fn r} fp] {
			Error "Unable to open $fn: $fp"
			return
		    }
		    set flt [read -nonewline $fp]
		    catch {regsub {\n} $flt " " $flt}
		    set cvar(filter) [string trim $flt]
		    catch {close $fp}
		}
		default {
		    set cvar(filter) $item
		}
	    }
	    CATTable $cvarname
	}
	clear {CATClear $cvarname}
	retreive {CATApply $cvarname 0}
	cancel {ARCancel $cvarname}
	print {CATPrint $cvarname}
	close {CATDestroy $cvarname}
	server {
	    incr i
	    set cvar(server) [lindex $var $i]
	}
	symbol {
	    global $cvar(symdb)
	    incr i
	    switch -- [lindex $var $i] {
		shape {
		    incr i
		    starbase_set $cvar(symdb) 1 \
			[starbase_colnum $cvar(symdb) shape] [lindex $var $i]
		}
		color {
		    incr i
		    starbase_set $cvar(symdb) 1 \
			[starbase_colnum $cvar(symdb) color] [lindex $var $i]
		}
		load {
		    incr i
		    starbase_read $cvar(symdb) [lindex $var $i]
		}
	    }
	    CATPlot $cvarname
	}
	sort {
	    incr i
	    set cvar(sort) [lindex $var $i]
	    incr i
	    switch -- [lindex $var $i] {
		incr {
		    set cvar(sort,dir) "-increasing"
		}
		decr {
		    set cvar(sort,dir) "-decreasing"
		}
	    }
	    CATTable $cvarname
	}
	maxrows {
	    incr i
	    set cvar(max) [lindex $var $i]
	}
	allcols {
	    set cvar(allcols) 1
	    CATApply $cvarname 0
	}
    }
}
