// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "text.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "rotstr.h"

#include <Xutil.h>

Text::Text(const Text& a) : Marker(a) {}

Text::Text(FrameBase* p, const Vector& ctr, double a,
	   const char* clr, int w, const char* f, 
	   const char* t, unsigned short prop, const char* c,
	   const List<Tag>& tag) 
  : Marker(p, ctr, clr, w, f, t, prop, c, tag)
{
  angle = a;

  if (!parent->isIIS())
    flip = FlipY();

  strcpy(type,"text");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Text::updateBBox()
{
  Vector c = center * parent->refToCanvas;
  bbox = BBox(c,c);

  if (text && font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    Vector s(width, metrics.linespace);
    s /= parent->getZoom();
    Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

    // generate handles

    handle[0] = Vector(-s[0],-s[1])/2 * m;
    handle[1] = Vector( s[0],-s[1])/2 * m;
    handle[2] = Vector( s[0], s[1])/2 * m;
    handle[3] = Vector(-s[0], s[1])/2 * m;

    // bbox

    for (int i=0; i<4; i++)
      bbox.bound(handle[i]);
  }
  else
    bbox.expand(5);

  // make room for handles

  bbox.expand(3);
  allBBox = bbox;
}

void Text::draw(Drawable drawable)
{
  renderHandles(drawable);
  render(drawable, parent->refToWidget, parent->getZoom(), SRC);
}

void Text::drawXOR()
{
  render(Tk_WindowId(parent->tkwin), parent->refToWindow, 
	 parent->getZoom(), XOR);
}

void Text::drawMagnifier(const Matrix& m)
{
  render(parent->magnifierPixmap, m, parent->getMagnifierZoom(), SRC);
}

#if __GNUC__ >= 3
void Text::ps(int mode)
{
  if (text && font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    double a = angle + parent->getRotation();
    Vector v = center * parent->refToCanvas;
    Matrix m = Translate(-v) * Rotate(-a) * 
      Translate(-width/2.,metrics.descent) *
      Rotate(a) * Translate(v);

    ostringstream str;
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
      
    Tcl_DStringFree(&psFont);

    psLineNoDash();
    psColor(mode, colorName);

    str << "gsave "
        << (v*m).TkCanvasPs(parent->canvas) << " moveto" << endl
      	<< radToDeg(a) << " rotate " 
	<< '(' << psQuote(text) << ')' 
	<< " show grestore" << endl << ends;
    
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}
#else
void Text::ps(int mode)
{
  if (text && font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    double a = angle + parent->getRotation();
    Vector v = center * parent->refToCanvas;
    Matrix m = Translate(-v) * Rotate(-a) * 
      Translate(-width/2.,metrics.descent) *
      Rotate(a) * Translate(v);

    char buf[512];
    ostrstream str(buf,512);
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
      
    Tcl_DStringFree(&psFont);

    psLineNoDash();
    psColor(mode, colorName);

    str << "gsave "
        << (v*m).TkCanvasPs(parent->canvas) << " moveto" << endl
      	<< radToDeg(a) << " rotate " 
	<< '(' << psQuote(text) << ')' 
	<< " show grestore" << endl << ends;
    
    Tcl_AppendResult(parent->interp, buf, NULL);
  }
}
#endif

// list

void Text::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] 
		<< ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];

#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Text::listProperties(ostream& str, char delim)
{
  if (delim == ';')
    return;

  if (strcmp("green",colorName) ||
      strcmp("helvetica 10 normal",getFont()) ||
      text[0] ||
      !(properties&SELECT) ||
      !(properties&EDIT) ||
      !(properties&MOVE) ||
      !(properties&ROTATE) ||
      !(properties&DELETE) ||
      !(properties&HIGHLITE) ||
      (properties&FIXED) ||
      !(properties&SOURCE) ||
      (angle != 0) ||
      (comment && *comment)) {

    str << " #";
    if (angle != 0)
      str << " textangle=" << radToDeg(angle);

    listProps(str);
  }
}

void Text::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      if (properties&INCLUDE)
	str << '+';
      else
	str << '-';

      Vector v = ptr->mapFromRef(center,IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] 
	  << ", \"" << text << "\")";

      listSAOtngPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      if (properties&INCLUDE)
	str << '+';
      else
	str << '-';

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<", \""
	      << text << "\")";
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec
	      << ", \"" << text << "\")";
	}
	break;
      }

      listSAOtngPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

// private

void Text::render(Drawable drawable, const Matrix& matrix, double zoom,
		  RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }
  setLineNoDash();

  if (font)
    XSetFont(display, gc, Tk_FontId(font));

  double a = angle+parent->getRotation();
  if (text && font)
    if (a>-.01 && a<+.01)
      drawText(drawable, matrix);
    else
      drawRotText(drawable, matrix);
}

void Text::drawText(Drawable drawable, const Matrix& matrix)
{
  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(font, &metrics);

  Vector t = (center * matrix) * 
    Translate(-Tk_TextWidth(font, text, strlen(text))/2., 
	      (metrics.ascent-metrics.descent)/2.);

  XDrawString(display, drawable, gc, (int)t[0], (int)t[1], 
	      text, strlen(text));
}

void Text::drawRotText(Drawable drawable, const Matrix& matrix)
{
  XDrawRotString(display, drawable, gc, 
		 parent->baseXImage->byte_order, 
		 parent->getBlackColor(), parent->getWhiteColor(),
		 text, font, angle+parent->getRotation(), center*matrix);
}

