// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "rotstr.h"
#include "util.h"

#include <Xutil.h>
#include <string.h>

void XDrawRotString(Display* display, Drawable drawable, GC gc,
		    int byte_order, int black, int white,
		    const char* text, Tk_Font font, double angle, 
		    const Vector& v)
{
  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(font, &metrics);
  int width = Tk_TextWidth(font, text, strlen(text));
  int height = metrics.linespace;

  // calculate bounding box
  int srcw = width+4;
  int srch = height+2;
  Vector corner = (Vector(srcw,srch)-Vector(1,1))/2;
  BBox bb;
  bb.bound(-corner * Rotate(angle));
  bb.bound( corner * Rotate(angle));
  bb.bound(Vector( corner[0],-corner[1]) * Rotate(angle));
  bb.bound(Vector(-corner[0], corner[1]) * Rotate(angle));
  Vector dstsize = bb.size();
  int dstw = (int)dstsize[0];
  int dsth = (int)dstsize[1];

  // create bitmap
  Pixmap srcbmap = XCreatePixmap(display, drawable, srcw, srch, 1);
  Pixmap dstbmap = XCreatePixmap(display, drawable, dstw, dsth, 1);

  // create bitmap gc
  GC bgc = XCreateGC(display, srcbmap, 0L, NULL);
  XGCValues values;
  XGetGCValues(display,gc,GCFont,&values);
  XChangeGC(display,bgc,GCFont,&values);
#ifndef _WIN32
  XSetBackground(display, bgc, 0);
#else
  XSetBackground(display, bgc, black);
#endif

  // clear bitmaps
#ifndef _WIN32
  XSetForeground(display, bgc, 0);
#else
  XSetForeground(display, bgc, black);
#endif
  XFillRectangle(display, srcbmap, bgc, 0, 0, srcw, srch);
  XFillRectangle(display, dstbmap, bgc, 0, 0, dstw, dsth);

#ifndef _WIN32
  XSetForeground(display, bgc, 1);
#else
  XSetForeground(display, bgc, white);
#endif
  XDrawString(display, srcbmap, bgc, 2, metrics.ascent, text, strlen(text));

  // create xmaps
  XImage* srcxmap = XGETIMAGE(display, srcbmap, 0, 0, srcw, srch,
			      AllPlanes, XYPixmap);
  XImage* dstxmap = XGETIMAGE(display, dstbmap, 0, 0, dstw, dsth,
			      AllPlanes, XYPixmap);

  // rotate text from srcxmap to dstxmap
  {
    char* src = srcxmap->data;
    char* dst = dstxmap->data;
    int srcl = srcxmap->bytes_per_line;
    int dstl = dstxmap->bytes_per_line;
    Matrix m = 
      Translate(-Vector(srcw,srch)/2) *
      Rotate(angle) *
      Translate(Vector(dstw,dsth)/2);
    Matrix mm = m.invert();

    for (int j=0; j<dsth; j++) {
      for (int i=0; i<dstw; i++) {

	Vector s = Vector(i,j) * mm;
	if (s[0]>=0 && s[0]<srcw && s[1]>=0 && s[1]<srch) {

	  int soff = ((int)s[1])*srcl+(((int)s[0])/8);
	  int sshift = ((int)s[0])%8;

#ifndef _WIN32
	  if (byte_order) {
#else
	  if (1) {
#endif
	    // NOTE: ximage gave us the bitmap with bytes MSB, so we must
	    // 128>>shift instead of 1<<shift.
	    if (*(src+soff) & (128>>sshift)) {
	      int doff = j*dstl+(i/8);
	      int dshift = i%8;
	      *(dst+doff) |= 128>>dshift;
	    }
	  }
	  else {
	    if (*(src+soff) & (1<<sshift)) {
	      int doff = j*dstl+(i/8);
	      int dshift = i%8;
	      *(dst+doff) |= 1<<dshift;
	    }
	  }
	}
      }
    }
  }

  // put rotated text into dstbmap
  XPutImage(display, dstbmap, bgc, dstxmap, 0, 0, 0, 0, dstw, dsth);

  // create new GC
  GC ggc = XCreateGC(display, drawable, 0L, NULL);
  XCopyGC(display, gc, GCForeground, ggc);
  XSetStipple(display, ggc, dstbmap);
  XSetFillStyle(display, ggc, FillStippled);

  // calculate rect location
  BBox vv(v,v);

  Matrix m = Rotate(angle) * Translate(v);
  vv.bound(-corner * m);
  vv.bound( corner * m);
  vv.bound(Vector( corner[0],-corner[1]) * m);
  vv.bound(Vector(-corner[0], corner[1]) * m);

  Vector t = vv.ll;

  // stencil in the text
  XSetTSOrigin(display, ggc, (int)t[0], (int)t[1]);
  XFillRectangle(display, drawable, ggc, (int)t[0], (int)t[1], dstw, dsth);
  
  // clean up
  XFreePixmap(display, srcbmap);
  XFreePixmap(display, dstbmap);
  XFreeGC(display, bgc);
  XFreeGC(display, ggc);
  XDestroyImage(srcxmap);
  XDestroyImage(dstxmap);
}
