// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

/*
 *
 * NaN.h -- include file to deal with IEEE NaN values
 *
 * Idea taken from fitsio2.h in the cfitsio package by William Pence (HEASARC),
 * to whom grateful acknowledgement is made.
 *
 * These functions tests whether the float value is a reserved IEEE
 * value such as a Not-a-Number (NaN), or underflow, overflow, or
 * infinity.   The functions returns 1 if the value is a NaN, overflow
 * or infinity; otherwise it returns 0.
 *
 * isNaNf tests floats, isNaNd tests doubles.
 *
 */
#ifndef	__nan_h
#define	__nan_h

#define NATIVE             0 /* generic machine using IEEE formats */
#define ALPHA              1
#define IBMPC              2

/* the following are used to determine what type machine we are running on */
#if defined(__alpha)
 
#define NAN_MACHINE ALPHA
#define BYTESWAPPED 1
 
#elif defined(__linux__) && ( defined(__i386__) || defined(__i486__) || defined(__i586__) )

/*  IBM PC running linux */
#define NAN_MACHINE IBMPC
#define BYTESWAPPED 1

#elif defined(_MSC_VER) || defined(__BORLANDC__) || defined(__TURBOC__) || defined(_WIN32)

/*  IBM PC running DOS or Windows */
#define NAN_MACHINE IBMPC       
#define BYTESWAPPED 1

#elif defined(_NI_mswin_)

/*  LabWindows/CVI with Windows 3.x, 95, or NT  */
#define NAN_MACHINE IBMPC       
#define BYTESWAPPED 1

#elif  defined(__EMX__)

/*  IBM PC running OS/2 */
#define NAN_MACHINE IBMPC
#define BYTESWAPPED 1

#else

/*  assume machine uses the same IEEE formats as used in FITS files */
#define NAN_MACHINE NATIVE
#define BYTESWAPPED 0
 
#endif
 
/* byte-swapping determine which short int we will mask for NaN test */
#if (BYTESWAPPED == 1)

#define FNANOFF 1
#define DNANOFF 3

#else

#define FNANOFF 0
#define DNANOFF 0

#endif

#define FNANMASK   0x7F80 /* mask bits 1 - 8; all set on NaNs */
                                     /* all 0 on underflow  or 0. */
 
#define DNANMASK   0x7FF0 /* mask bits 1 - 11; all set on NaNs */
                                     /* all 0 on underflow  or 0. */
 
inline int isNaNf(float f)
{
  union {
    float f;
    short x[2];
  } a;
  a.f = f;

  return ((a.x[FNANOFF] & FNANMASK) == FNANMASK);
}

inline int isNaNd(double d)
{
  union {
    double d;
    short x[4];
  } a;
  a.d = d;

  return ((a.x[DNANOFF] & DNANMASK) == DNANMASK);
}

float getNaNf(void);
double getNaNd(void);

#endif
