// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framebase.h"
#include "frame.h"
#include "util.h"
#include "fitsimage.h"
#include "nan.h"

#define BLT 1
#if BLT == 1
#include "blt.h"
#endif

void FrameBase::bltHist(char* xname, char* yname)
{
#if BLT == 1
  int num = 256;

  if (!currentFits)
    return;

  if (DebugPerf)
    cerr << "bltHist...";

  double* histogramX = currentScale->histogramX();
  double* histogramY = currentScale->histogramY();

  if (!histogramX || !histogramY) {
    histogramX = currentScale->initHistogramX(num);
    histogramY = currentScale->initHistogramY(num);

    for (int i=0; i<num; i++)
      histogramX[i] = (double)i/(num-1)*
	(currentScale->max()-currentScale->min()) + currentScale->min();
    
    for (int j=0; j<num; j++)
      histogramY[j] = 0;

    if (currentScale->max() > currentScale->min()) {
      FitsImage* ptr = *channelFits;
      while (ptr) {
	ptr->bin(histogramY, num, currentScale->min(), currentScale->max(), 
		 currentScale->scanMode());
	ptr = ptr->next();
      }
    }
  }

  Blt_Vector* xx;
  if (Blt_GetVector(interp, xname, &xx) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xx, histogramX, num, num*sizeof(double), TCL_STATIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* yy;
  if (Blt_GetVector(interp, yname, &yy) != TCL_OK) 
    goto error;

  if (Blt_ResetVector(yy, histogramY, num, num*sizeof(double), TCL_STATIC) 
      != TCL_OK)
    goto error;

  if (DebugPerf)
    cerr << "end" << endl;

  return;

 error:
  result = TCL_ERROR;
  return;

#endif
}

void FrameBase::bltProjection(char* xname, char* yname, 
			      char* xcname, char* ycname,
			      CoordSystem sys, SkyFrame sky,
			      const Vector& p1, const Vector& p2, 
			      double width, int avg)
{
#if BLT == 1

  if (!currentFits)
    return;

  // the number of pixels sampled is the greater of either dimension
  Vector ap1 = p1;
  Vector ap2 = p2;
  int xnum = ::abs(ap2[0]-ap1[0]);
  int ynum = ::abs(ap2[1]-ap1[1]);
  int num = (xnum > ynum ? xnum : ynum)+1;

  double* x = (double*)malloc(num*sizeof(double));
  double* y = (double*)malloc(num*sizeof(double));
  double* xc = (double*)malloc(num*sizeof(double));
  double* yc = (double*)malloc(num*sizeof(double));

  switch (*currentMode) {
  case SINGLE:
    bltProjSingle(x, y, xc, yc, num, sys, sky, p1, p2, width, avg);
    break;
  case MOSAIC:
    bltProjMosaic(x, y, xc, yc, num, sys, sky, p1, p2, width, avg);
    break;
  }

  Blt_Vector* xx;
  if (Blt_GetVector(interp, xname, &xx) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xx, x, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* yy;
  if (Blt_GetVector(interp, yname, &yy) != TCL_OK) 
    goto error;

  if (Blt_ResetVector(yy, y, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* xxc;
  if (Blt_GetVector(interp, xcname, &xxc) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xxc, xc, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* yyc;
  if (Blt_GetVector(interp, ycname, &yyc) != TCL_OK) 
    goto error;

  if (Blt_ResetVector(yyc, yc, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  return;

 error:
  result = TCL_ERROR;
  return;

#endif
}

void FrameBase::bltProjSingle(double* x, double* y, 
			      double* xc, double* yc, int num,
			      CoordSystem sys, SkyFrame sky,
			      const Vector& p1, const Vector& p2, 
			      double width, int avg)
{
#if BLT == 1

  int* cnt = new int[num];
  Vector s = (p2-p1)/(num-1);

  FitsImage* ptr = currentFits;
  Matrix mm = ptr->getRefToData();

  for (int i=0; i<num; i++) {
    Vector t = p1 + s*i;

    x[i] = i+1;
    Vector tt = ptr->mapFromRef(t,sys,sky);
    xc[i] = tt[0];
    yc[i] = tt[1];

    // check for nan
    double v = ptr->getValueDouble(t*mm);
    if (!isNaNd(v)) {
      y[i] = v;
      cnt[i] = 1;
    }
    else
      y[i] = getNaNd();

    Vector ps = (p2-p1).normalize();
    Vector ss = Vector(-ps[1],ps[0]);

    for (int j=1; j<width; j++) {
      Vector tt = p1 + s*i + ss*j;
      double value = ptr->getValueDouble(tt*mm);
      // check for nan
      if (!isNaNd(value) && !isNaNd(y[i])) {
	y[i] += value;
	cnt[i]++;
      }
      else
	y[i] = getNaNd();
    }
  }

  // average if needed
  if (avg)
    for (int i=0; i<num; i++)
      if (!isNaNd(y[i]))
	y[i] /= cnt[i];

  // convert nand's to low
  for (int j=0; j<num; j++)
    if (isNaNd(y[j]))
      y[j] = ptr->getLowDouble();

  delete [] cnt;

#endif
}

void FrameBase::bltProjMosaic(double* x, double* y, 
			      double* xc, double* yc, int num,
			      CoordSystem sys, SkyFrame sky,
			      const Vector& p1, const Vector& p2, 
			      double width, int avg)
{
#if BLT == 1

  int* cnt = new int[num];
  Vector s = (p2-p1)/(num-1);

  for (int i=0; i<num; i++) {
    Vector t = p1 + s*i;

    x[i] = i+1;

    FitsImage* ptr = *channelFits;
    while (ptr) {
      Vector tt = ptr->mapFromRef(t,sys,sky);
      xc[i] = tt[0];
      yc[i] = tt[1];

      int* params = ptr->getDataParams(currentScale->scanMode());
      int& srcw = params[0];
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      Vector z = t * ptr->getRefToData();

      if (z[0]>=xmin && z[0]<xmax && z[1]>=ymin && z[1]<ymax) {

	// check for nan
	double v = ptr->getValueDouble(z);
	if (!isNaNd(v)) {
	  y[i] = v;
	  cnt[i] = 1;
	}
	else
	  y[i] = getNaNd();

	Vector ps = (p2-p1).normalize();
	Vector ss = Vector(-ps[1],ps[0]);

	for (int j=1; j<width; j++) {
	  Vector tt = p1 + s*i + ss*j;

	  FitsImage* pptr = *channelFits;
	  while (pptr) {
	    int* pparams = pptr->getDataParams(currentScale->scanMode());
	    int& ssrcw = pparams[0];
	    int& xxmin = pparams[1];
	    int& xxmax = pparams[2];
	    int& yymin = pparams[3];
	    int& yymax = pparams[4];

	    Vector zz = tt * pptr->getRefToData();
		
	    if (zz[0]>=xxmin && zz[0]<xxmax && 
		zz[1]>=yymin && zz[1]<yymax) {

	      double vvalue = pptr->getValueDouble(zz);
	      // check for nan
	      if (!isNaNd(vvalue) && !isNaNd(y[i])) {
		y[i] += vvalue;
		cnt[i]++;
	      }
	      else
		y[i] = getNaNd();

	      break;
	    }
	    else
	      pptr = pptr->next();
	  }

	  if (!pptr)
	    y[i] = getNaNd();
	}

	break;
      }
      else
	ptr = ptr->next();
    }

    if (!ptr)
      y[i] = getNaNd();
  }

  // average if needed
  if (avg)
    for (int i=0; i<num; i++)
      if (!isNaNd(y[i]))
	y[i] /= cnt[i];

      // convert nand's to low
  for (int j=0; j<num; j++)
    if (isNaNd(y[j]))
      y[j] = currentScale->low();

  delete [] cnt;

#endif
}

void FrameBase::bltCut(char* xname, char* yname, 
		       Orientation axis, const Vector& rr)
{
#if BLT == 1

  if (!currentFits)
    return;

  // Vector r is in Widget coords
  Vector r = rr;

  int size;
  if (axis == XX)
    size = options->width;
  else
    size = options->height;

  long length = (size+1) * 2;
  double* x = (double*)malloc(sizeof(double)*length);
  double* y = (double*)malloc(sizeof(double)*length);

  if (*currentCount == 0) {
    for (int i=0; i<=size; i++) {
      x[i*2] = i;
      x[i*2+1] = i;
      y[i*2] = 0;
      y[i*2+1] = 0;
    }
  }
  else {
    switch (*currentMode) {
    case SINGLE:
      bltCutSingle(x, y, size, r, axis);
      break;
    case MOSAIC:
      bltCutMosaic(x, y, size, r, axis);
      break;
    }
  }

  Blt_Vector* xx;
  if (Blt_GetVector(interp, xname, &xx) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xx, x, length, length*sizeof(double), TCL_DYNAMIC) != 
      TCL_OK)
    goto error;

  Blt_Vector* yy;
  if (Blt_GetVector(interp, yname, &yy) != TCL_OK)
    goto error;

  if (Blt_ResetVector(yy, y, length, length*sizeof(double), TCL_DYNAMIC) != 
      TCL_OK)
    goto error;

  return;

 error:
    result = TCL_ERROR;
    return;

#endif
}

void FrameBase::bltCutSingle(double* x, double* y, int size, Vector& r, 
			 Orientation axis)
{
  Matrix m = currentFits->getWidgetToData();
  int* params = currentFits->getDataParams(currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  double prev = currentScale->low();

  for (int i=0; i<=size; i++) {
    double v = currentScale->low();

    Vector img;
    if (axis == XX)
      img = Vector(1+i,r[1]) * m;
    else
      img = Vector(r[0],1+i) * m;

    if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
      double value = currentFits->getValueDouble(img);

      if (!isNaNd(value))
	v = value;
    }

    x[2*i] = i;
    x[2*i +1] = i;
    
    y[2*i] = prev;
    y[2*i +1] = v;
    prev = v;
  }
}

void FrameBase::bltCutMosaic(double* x, double* y, int size, Vector& r, 
			 Orientation axis)
{
  double prev = currentScale->low();

  for (int i=0; i<=size; i++) {
    double v = currentScale->low();

    // find y first

    FitsImage* ptr = *channelFits;
    while (ptr) {

      Matrix& m = ptr->getWidgetToData();
      int* params = ptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      Vector img;
      if (axis == XX)
	img = Vector(1+i,r[1]) * m;
      else
	img = Vector(r[0],1+i) * m;

      if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
	double value = ptr->getValueDouble(img);

	if (!isNaNd(value))
	  v = value;

	break;
      }
      else
	ptr = ptr->next();
    }

    x[2*i] = i;
    x[2*i +1] = i;
    
    y[2*i] = prev;
    y[2*i +1] = v;
    prev = v;
  }
}

