// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "ellipse.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Ellipse::Ellipse(const Ellipse& a) : BaseEllipse(a) {}

Ellipse::Ellipse(FrameBase* p, const Vector& ctr, const Vector& r, double a, 
		 const char* clr,  int w, const char* f, const char* t, 
		 unsigned short prop, const char* c, const List<Tag>& tag)
  : BaseEllipse(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = 1;
  radii = new Vector[1];
  radii[0] = r;

  strcpy(type,"ellipse");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Ellipse::edit(const Vector& v, int h)
{
  radii[0] = (v * Translate(-center) * flip * Rotate(-angle));

  updateBBox();
  doCallBack(&editCB);
}

void Ellipse::setRadius(const Vector& r)
{
  radii[0] = r;
  updateBBox();

  doCallBack(&editCB);
}

// private

void Ellipse::updateHandles()
{
  Vector r = radii[0];
  if (properties & FIXED)
    r /= parent->getZoom();

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  handle[0] = Vector(-r[0],-r[1]) * m;
  handle[1] = Vector( r[0],-r[1]) * m;
  handle[2] = Vector( r[0], r[1]) * m;
  handle[3] = Vector(-r[0], r[1]) * m;
}

// list

void Ellipse::list(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector r = ptr->mapLenFromRef(radii[0],sys);
      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r[0] << ',' << r[1] << ',' 
	  << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	Vector r = ptr->mapLenFromRef(radii[0],sys,ARCSEC);
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		<< r[0] << "\"" << ',' << r[1] << "\"" << ',' 
		<< radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ',' 
		<< r[0] << "\"" << ',' << r[1] << "\"" << ',' 
		<< radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	Vector r = ptr->mapLenFromRef(radii[0],sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << r[0] << ',' << r[1] << ',' 
	    << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Ellipse::listCiao(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // radius is always in image coords

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listCiaoPre(str);

      Vector r = ptr->mapLenFromRef(radii[0],PHYSICAL);
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r[0] << ',' << r[1] << ',' << radToDeg(angle) << ')'
	  << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listCiaoPre(str);

      Vector r = ptr->mapLenFromRef(radii[0],sys,ARCMIN);
      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
#if __GNUC__ >= 3
      string x(buf);
      istringstream wcs(x);
#else
      istrstream wcs(buf,64);
#endif
      wcs >> ra >> dec;
      str << type << '(' << ra << ',' << dec << ',' 
	  << r[0] << '\'' << ',' << r[1] << '\'' << ',' 
	  << radToDeg(angle) << ')'
	  << delim;
    }
    break;
  }
}

void Ellipse::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      listProsCoordSystem(str,sys,sky);
      str << "; ";
      Vector r = ptr->mapLenFromRef(radii[0],IMAGE);
      Vector v = ptr->mapFromRef(center,sys); 
      str << type << ' ' << setprecision(8) << v << r << radToDeg(angle)
	  << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listProsCoordSystem(str,sys,sky);
      str << "; ";

      // display wcs radius in ARCSEC
      Vector r = ptr->mapLenFromRef(radii[0],sys,ARCSEC);

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << type << ' ' << setprecision(8) << v[0] << "d " << v[1] <<"d " 
	      << r[0] << "\" " << r[1] << "\" " << radToDeg(angle);
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    str << type << ' ' << ra << ' ' << dec+1 << ' ' 
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);
	  else
	    str << type << ' ' << ra << ' ' << dec << ' ' 
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);
	}
	break;
      }

      str << delim;
    }
    else
      str << "";
    break;
  }
}

void Ellipse::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // radius is always in image coords

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listSAOtngPre(str,delim);

      Vector v = ptr->mapFromRef(center,IMAGE);
      Vector r = ptr->mapLenFromRef(radii[0],IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r[0] << ',' << r[1] << ',' << radToDeg(angle) << ')';

      listSAOtngPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listSAOtngPre(str,delim);

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  Vector r = ptr->mapLenFromRef(radii[0],IMAGE);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	      << r[0] << ',' << r[1] << ',' << radToDeg(angle) << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  Vector r = ptr->mapLenFromRef(radii[0],IMAGE);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec << ',' 
	      << r[0] << ',' << r[1] << ',' << radToDeg(angle) << ')';
	}
	break;
      }

      listSAOtngPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Ellipse::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // all coords are in image coords

  if (!(properties&INCLUDE))
    str << '-';

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << "ellipse(" << setprecision(8) << v[0] << ',' << v[1] << ',' 
      << radii[0][0] << ',' << radii[0][1] << ',' << radToDeg(angle) << ')';

  str << delim;
}
