// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

%{
#define YYPARSE_PARAM fr
#define YYDEBUG 1

#define FR ((FrameBase*)fr)
#define FITSPTR (FR->findFits(globalTile))

#define DISCARD_(x) {yyclearin; mkDiscard(x);}

#include <math.h>
#include <string.h>
#include <iostream.h>

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "vector.h"
#include "list.h"
#include "basemarker.h"
#include "coord.h"

extern int mklex(void);
extern void mkerror(const char*);
extern void mkDiscard(int);

int mksign;
int mksign2;

static CoordSystem globalSystem;
static CoordSystem globalWCS;
static SkyFrame globalSky;
static CoordSystem localSystem;
static SkyFrame localSky;

static int globalTile;

static unsigned short globalProps;
static unsigned short localProps;

static int globalWidth;
static int localWidth;

static char globalColor[16];
static char localColor[16];

static char globalFont[32];
static char localFont[32];

static char globalText[80];
static char localText[80];

static char localComment[80];

static int globalLine1;
static int localLine1;
static int globalLine2;
static int localLine2;

static int globalPoint;
static int localPoint;

static double globalTextAngle;
static double localTextAngle;

static CoordSystem globalRulerCoordSystem;
static CoordSystem localRulerCoordSystem;
static SkyFrame globalRulerSkyFrame;
static SkyFrame localRulerSkyFrame;
static CoordSystem globalRulerDistSystem;
static CoordSystem localRulerDistSystem;
static SkyFormat globalRulerDistFormat;
static SkyFormat localRulerDistFormat;

static CoordSystem globalCompassCoordSystem;
static SkyFrame globalCompassSkyFrame;
static char globalCompassNorth[80];
static char globalCompassEast[80];
static int globalCompassNArrow;
static int globalCompassEArrow;
static CoordSystem localCompassCoordSystem;
static SkyFrame localCompassSkyFrame;
static char localCompassNorth[80];
static char localCompassEast[80];
static int localCompassNArrow;
static int localCompassEArrow;

static int localPanda;

static List<Vertex> polylist;
static List<Tag> taglist;

static double aAnnuli[MAXANNULI];
static Vector aVector[MAXANNULI];
static int aNum;
static int aNumsao;
static int aStatus;
static Vector aCenter;
static double aAngle[MAXANGLES];
static int aAngNum;
static unsigned short aProps;
static char aColor[16];
static int aWidth;
static char aFont[32];
static char aText[80];
static char aComment[80];

static void setProps(unsigned short* props, unsigned short prop, int value);
static CoordSystem checkWCSSystem();
static SkyFrame checkWCSSky();

enum {CIRCLE,BOX,DIAMOND,CROSS,XPT,ARROW,BOXCIRCLE};
	
%}

%union {
  double real;
  int integer;
  char str[256];
  double vector[3];
}

%type <real> numeric
%type <integer> yesno

%type <real> angle
%type <real> optangle
%type <real> value
%type <vector> vvalue
%type <real> sexagesimal
%type <vector> coord
%type <integer> coordSystem
%type <integer> wcsSystem
%type <integer> skyFrame
%type <integer> skyFormat
%type <integer> property
%type <integer> pointProp
%type <integer> numberof

%token <integer> INT
%token <real> REAL
%token <str> STRING

%token <integer> HOUR
%token <integer> MINUTE
%token <real> SECOND
%token <real> DEGREE
%token <real> ARCMINUTE
%token <real> ARCSECOND
%token <real> RADIAN
%token <real> PHYCOORD
%token <real> IMGCOORD

%token <str> SEXSTR

%token AMPLIFIER_
%token ANNULUS_
%token ARCMIN_
%token ARCSEC_
%token ARROW_
%token B1950_
%token BACKGROUND_
%token BOX_
%token BOXCIRCLE_
%token CIRCLE_
%token COLOR_
%token COMPASS_
%token CROSS_
%token DEBUG_
%token DEGREES_
%token DELETE_
%token DETECTOR_
%token DIAMOND_
%token ECLIPTIC_
%token EDIT_
%token ELLIPSE_
%token FALSE_
%token FIELD_
%token FIXED_
%token FK4_
%token FK5_
%token FONT_
%token GALACTIC_
%token GLOBAL_
%token HIGHLITE_
%token ICRS_
%token IGNORE_
%token IMAGE_
%token INCLUDE_
%token J2000_
%token LINE_
%token LINEAR_
%token MOVE_
%token N_
%token NO_
%token OFF_
%token ON_
%token PANDA_
%token PHYSICAL_
%token PIE_
%token PIXELS_
%token POINT_
%token POLYGON_
%token PROJECTION_
%token ROTATE_
%token ROTBOX_
%token RULER_
%token SELECT_
%token SOURCE_
%token TAG_
%token TEXT_
%token TEXTANGLE_
%token TILE_
%token TRUE_
%token VERSION_
%token WCS_
%token WCSA_
%token WCSB_
%token WCSC_
%token WCSD_
%token WCSE_
%token WCSF_
%token WCSG_
%token WCSH_
%token WCSI_
%token WCSJ_
%token WCSK_
%token WCSL_
%token WCSM_
%token WCSN_
%token WCSO_
%token WCSP_
%token WCSQ_
%token WCSR_
%token WCSS_
%token WCST_
%token WCSU_
%token WCSV_
%token WCSW_
%token WCSX_
%token WCSY_
%token WCSZ_
%token WIDTH_
%token X_
%token Y_
%token YES_

%%

start	: initGlobal commands postLocal
	;

commands: commands command terminator
	| command terminator
	;

command : /* empty */
	| DEBUG_ debug
	| VERSION_ {cerr << "DS9 Regions File 3.0" << endl;}
	| GLOBAL_ global comment
	| TILE_ INT {globalTile = $2;}
	| coordSystem {globalSystem = (CoordSystem)$1;} comment
	| skyFrame {globalSystem = WCS; globalSky = (SkyFrame)$1;} comment
	| LINEAR_ {globalSystem = globalWCS; globalSky = NATIVEWCS;} comment
	| initLocal include shape
	| generalComment
	;

generalComment: '#' {DISCARD_(0)} STRING
	;

comment : /* empty */
	| '#' {DISCARD_(1)} STRING
	;

shapeComment : /* empty */ postLocal
	| '#' {DISCARD_(1)} STRING postLocal 
	  {strncpy(localComment,$3,80);}
	| '#' local postLocal
	| '#' local {DISCARD_(1)} STRING postLocal
	  {strncpy(localComment,$4,80);}
	;

terminator: '\n'
	| ';'
	;

numeric	: REAL {$$=$1;}
	| INT {$$=$1;}
	;

debug	: ON_ {yydebug=1;}
	| OFF_ {yydebug=0;}
	;

yesno	: INT {$$=($1 ? 1 : 0);}

	| YES_ {$$=1;}
	| Y_ {$$=1;}
	| ON_ {$$=1;}
	| TRUE_ {$$=1;}

	| NO_ {$$=0;}
	| N_ {$$=0;}
	| OFF_ {$$=0;}
	| FALSE_ {$$=0;}
	;

sp	: /* empty */
	| ','
	;

bp	: /* empty */
	| '('
	;

ep	: /* emtpy */
	| ')'
	;

optangle: /* empty */ {$$ = FR->mapAngleToRef(0,localSystem);}
	| angle {$$ = $1;}
	;

angle	: numeric {$$ = FR->mapAngleToRef(degToRad($1),localSystem);}
	| DEGREE {$$ = FR->mapAngleToRef(degToRad($1),localSystem);}
	| RADIAN {$$ = FR->mapAngleToRef($1,localSystem);}
	;

value	: numeric {$$ = FITSPTR->mapLenToRef($1, localSystem, DEGREES);}
	| PHYCOORD {$$ = FITSPTR->mapLenToRef($1, PHYSICAL);}
	| IMGCOORD {$$ = FITSPTR->mapLenToRef($1, IMAGE);}
	| DEGREE {$$ = FITSPTR->mapLenToRef($1, checkWCSSystem(), DEGREES);}
	| ARCMINUTE {$$ = FITSPTR->mapLenToRef($1, checkWCSSystem(), ARCMIN);}
	| ARCSECOND {$$ = FITSPTR->mapLenToRef($1, checkWCSSystem(), ARCSEC);}
	;

vvalue	: numeric sp numeric 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), localSystem, DEGREES);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| PHYCOORD sp PHYCOORD 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), PHYSICAL);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| IMGCOORD sp IMGCOORD 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), IMAGE);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| DEGREE sp DEGREE 
	{
	  Vector r=FITSPTR->mapLenToRef(Vector($1,$3),checkWCSSystem(),DEGREES);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| ARCMINUTE sp ARCMINUTE 
	{
	  Vector r=FITSPTR->mapLenToRef(Vector($1,$3),checkWCSSystem(),ARCMIN);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| ARCSECOND sp ARCSECOND 
	{
	  Vector r=FITSPTR->mapLenToRef(Vector($1,$3),checkWCSSystem(),ARCSEC);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	;

numberof: N_ '=' INT {$$ = $3;}
	;

sexagesimal: SEXSTR {$$ = parseDMS($1);}
	;

coord	: sexagesimal sp sexagesimal
	{
	  Vector r;
	  CoordSystem sys = checkWCSSystem();
	  SkyFrame sky = checkWCSSky();
	  if (sky == GALACTIC || sky == ECLIPTIC) 
	    r = FITSPTR->mapToRef(Vector($1,$3), sys, sky);
	  else
	    r = FITSPTR->mapToRef(Vector($1*360./24.,$3), sys, sky);

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| HOUR MINUTE SECOND {mksign2 = mksign;} sp INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(hmsToDegree(mksign2,$1,$2,$3),dmsToDegree(mksign,$6,$7,$8)),
	    checkWCSSystem(), checkWCSSky());
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| INT ARCMINUTE ARCSECOND {mksign2 = mksign;} sp 
	  INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(dmsToDegree(mksign2,$1,$2,$3),dmsToDegree(mksign,$6,$7,$8)),
	    checkWCSSystem(), checkWCSSky());
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| numeric sp numeric 
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), localSystem, localSky);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| DEGREE sp DEGREE
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), 
	    checkWCSSystem(), checkWCSSky());
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| IMGCOORD sp IMGCOORD
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), IMAGE);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| PHYCOORD sp PHYCOORD
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), PHYSICAL);
	  $$[0] = r[0];
	  $$[1] = r[1];
	}
	| IMGCOORD sp PHYCOORD
	{
	  Vector rx = FITSPTR->mapToRef(Vector($1,$1), IMAGE);
	  Vector ry = FITSPTR->mapToRef(Vector($3,$3), PHYSICAL);
	  $$[0] = rx[0];
	  $$[1] = ry[0];
	  $$[2] = 1;
	}
	| PHYCOORD sp IMGCOORD
	{
	  Vector rx = FITSPTR->mapToRef(Vector($1,$1), PHYSICAL);
	  Vector ry = FITSPTR->mapToRef(Vector($3,$3), IMAGE);
	  $$[0] = rx[0];
	  $$[1] = ry[0];
	  $$[2] = 1;
	}
	;

coordSystem :IMAGE_ {$$ = IMAGE;}
	| PHYSICAL_ {$$ = PHYSICAL;}
	| DETECTOR_ {$$ = DETECTOR;}
	| AMPLIFIER_ {$$ = AMPLIFIER;}
	| wcsSystem {$$ = $1;}
	;

wcsSystem : WCS_ {$$ = WCS;}
	| WCSA_ {$$ = WCSA;}
	| WCSB_ {$$ = WCSB;}
	| WCSC_ {$$ = WCSC;}
	| WCSD_ {$$ = WCSD;}
	| WCSE_ {$$ = WCSE;}
	| WCSF_ {$$ = WCSF;}
	| WCSG_ {$$ = WCSG;}
	| WCSH_ {$$ = WCSH;}
	| WCSI_ {$$ = WCSI;}
	| WCSJ_ {$$ = WCSJ;}
	| WCSK_ {$$ = WCSK;}
	| WCSL_ {$$ = WCSL;}
	| WCSM_ {$$ = WCSM;}
	| WCSN_ {$$ = WCSN;}
	| WCSO_ {$$ = WCSO;}
	| WCSP_ {$$ = WCSP;}
	| WCSQ_ {$$ = WCSQ;}
	| WCSR_ {$$ = WCSR;}
	| WCSS_ {$$ = WCSS;}
	| WCST_ {$$ = WCST;}
	| WCSU_ {$$ = WCSU;}
	| WCSV_ {$$ = WCSV;}
	| WCSW_ {$$ = WCSW;}
	| WCSX_ {$$ = WCSX;}
	| WCSY_ {$$ = WCSY;}
	| WCSZ_ {$$ = WCSZ;}
	;

skyFrame : FK4_ {$$ = FK4;}
	| B1950_ {$$ = FK4;}
	| FK5_ {$$ = FK5;}
	| J2000_ {$$ = FK5;}
	| ICRS_ {$$ = ICRS;}
	| GALACTIC_ {$$ = GALACTIC;}
	| ECLIPTIC_ {$$ = ECLIPTIC;}
	;

skyFormat : DEGREES_ {$$=DEGREES;}
	| ARCMIN_ {$$=ARCMIN;}
	| ARCSEC_ {$$=ARCSEC;}
	;

property : SELECT_ {$$ = Marker::SELECT;}
	| EDIT_ {$$ = Marker::EDIT;}
	| MOVE_  {$$ = Marker::MOVE;}
	| ROTATE_ {$$ = Marker::ROTATE;}
	| DELETE_ {$$ = Marker::DELETE;}
	| HIGHLITE_ {$$ = Marker::HIGHLITE;}
	| INCLUDE_ {$$ = Marker::INCLUDE;}
	| SOURCE_ {$$ = Marker::SOURCE;}
	| FIXED_ {$$ = Marker::FIXED;}
	;

global	: global sp globalProperty
	| globalProperty
	;

globalProperty : property '=' yesno 
	{
	  setProps(&globalProps,$1,$3);
	  setProps(&localProps,$1,$3);
	}
	| COLOR_ '=' STRING 
	{
	  strncpy(globalColor,$3,16);
	  strncpy(localColor,$3,16);
	}
	| WIDTH_ '=' INT {globalWidth = localWidth = $3;}
	| FONT_ '=' STRING 
	{
	  strncpy(globalFont,$3,32);
	  strncpy(localFont,$3,32);
	}
	| TEXT_ '=' STRING 
	{
	  strncpy(globalText,$3,80);
	  strncpy(localText,$3,80);
	}
	| SOURCE_
	{
	  setProps(&globalProps,Marker::SOURCE,1);
	  setProps(&localProps,Marker::SOURCE,1);
	}
	| BACKGROUND_
	{
	  setProps(&globalProps,Marker::SOURCE,0);
	  setProps(&localProps,Marker::SOURCE,0);
	}
	| POINT_ '=' pointProp 
	{
	  globalPoint = localPoint = $3;
	}
	| LINE_ '=' INT INT 
	{
	  globalLine1 = localLine1 = $3;
	  globalLine2 = localLine2 = $4;
	}
	| RULER_ '=' globalRuler
	{
	}
	| COMPASS_ '=' globalCompass STRING STRING INT INT
	{
	  strncpy(globalCompassNorth,$4,80);
	  strncpy(globalCompassEast,$5,80);
	  strncpy(localCompassNorth,$4,80);
	  strncpy(localCompassEast,$5,80);
	  globalCompassNArrow = localCompassNArrow = $6;
	  globalCompassEArrow = localCompassEArrow = $7;
	}
	| TEXTANGLE_ '=' angle
	{
	  globalTextAngle = localTextAngle = $3;
	}
	| WCS_ '=' wcsSystem {globalWCS = (CoordSystem)$3;}
	;

globalRuler : coordSystem skyFrame coordSystem skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = (CoordSystem)$1;
	  globalRulerSkyFrame = localRulerSkyFrame = (SkyFrame)$2;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$3;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$4;
	}
	| coordSystem coordSystem
	{
	  globalRulerCoordSystem = localRulerCoordSystem = (CoordSystem)$1;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$2;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	| coordSystem skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = (CoordSystem)$1;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFrame coordSystem
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = (SkyFrame)$1;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$2;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	| skyFrame skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = (SkyFrame)$1;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$2;
	}
	| LINEAR_ coordSystem
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = (CoordSystem)$2;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	| LINEAR_ skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = WCS;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFormat
	{
	  globalRulerCoordSystem = localRulerCoordSystem = IMAGE;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = WCS;
	  globalRulerDistFormat = localRulerDistFormat = (SkyFormat)$1;
	}
	| PIXELS_
	{
	  globalRulerCoordSystem = localRulerCoordSystem = IMAGE;
	  globalRulerSkyFrame = localRulerSkyFrame = FK5;
	  globalRulerDistSystem = localRulerDistSystem = IMAGE;
	  globalRulerDistFormat = localRulerDistFormat = DEGREES;
	}
	;

globalCompass : coordSystem skyFrame
	{
	  globalCompassCoordSystem = localCompassCoordSystem = (CoordSystem)$1;
	  globalCompassSkyFrame = localCompassSkyFrame = (SkyFrame)$2;
	}
	| coordSystem
	{
	  globalCompassCoordSystem = localCompassCoordSystem = (CoordSystem)$1;
	  globalCompassSkyFrame = localCompassSkyFrame = FK5;
	}
	| skyFrame
	{
	  globalCompassCoordSystem = localCompassCoordSystem = WCS;
	  globalCompassSkyFrame = localCompassSkyFrame = (SkyFrame)$1;
	}
	| LINEAR_
	{
	  globalCompassCoordSystem = localCompassCoordSystem = WCS;
	  globalCompassSkyFrame = localCompassSkyFrame = FK5;
	}
	;

initGlobal:{
	  // global properties
	  globalSystem = PHYSICAL;
	  globalWCS = WCS;
	  globalSky = NATIVEWCS;
	  globalTile = 1;
	  globalProps =
	    Marker::SELECT | Marker::EDIT | Marker::MOVE |
	    Marker::ROTATE | Marker::DELETE | Marker::HIGHLITE |
	    Marker::INCLUDE | Marker::SOURCE;
	  strcpy(globalColor,"green");
          globalWidth = 1;
	  strcpy(globalFont,"helvetica 10 normal");
	  strcpy(globalText,"");

	  // unique properties
	  globalLine1 = 0;
	  globalLine2 = 0;
	  globalRulerCoordSystem = PHYSICAL;
	  globalRulerSkyFrame = FK5;
	  globalRulerDistSystem = PHYSICAL;
	  globalRulerDistFormat = DEGREES;
	  globalCompassCoordSystem = PHYSICAL;
	  globalCompassSkyFrame = FK5;
	  strcpy(globalCompassNorth,"N");
	  strcpy(globalCompassEast,"E");
	  globalCompassNArrow = 1;
	  globalCompassEArrow = 1;
	  globalPoint = BOXCIRCLE;
	  globalTextAngle=0;

	  aStatus = 0;
	} 
	;

local	: local sp localProperty
	| localProperty
	;

localProperty : property '=' yesno {setProps(&localProps,$1,$3);}
	| COLOR_ '=' STRING {strncpy(localColor,$3,16);}
	| WIDTH_ '=' INT {localWidth = $3;}
	| FONT_ '=' STRING {strncpy(localFont,$3,32);}
	| TEXT_ '=' STRING {strncpy(localText,$3,80);}
	| TAG_ '=' STRING {taglist.append(new Tag($3));}
	| SOURCE_ {setProps(&localProps,Marker::SOURCE,1);}
	| BACKGROUND_ {setProps(&localProps,Marker::SOURCE,0);}

	| POINT_ '=' pointProp {localPoint = $3;}
	| LINE_ '=' INT INT {localLine1=$3; localLine2=$4;}
	| RULER_ '=' localRuler
	| COMPASS_ '=' localCompass STRING STRING INT INT
	{
	  strncpy(localCompassNorth,$4,80);
	  strncpy(localCompassEast,$5,80);
	  localCompassNArrow = $6;
	  localCompassEArrow = $7;
	}
	| TEXTANGLE_ '=' angle {localTextAngle=$3;}
	| PANDA_ '=' localPanda
	;

localRuler : coordSystem skyFrame coordSystem skyFormat
	{
	  localRulerCoordSystem = (CoordSystem)$1;
	  localRulerSkyFrame = (SkyFrame)$2;
	  localRulerDistSystem = (CoordSystem)$3;
	  localRulerDistFormat = (SkyFormat)$4;
	}
	| coordSystem coordSystem
	{
	  localRulerCoordSystem = (CoordSystem)$1;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = (CoordSystem)$2;
	  localRulerDistFormat = DEGREES;
	}
	| coordSystem skyFormat
	{
	  localRulerCoordSystem = (CoordSystem)$1;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFrame coordSystem
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = (SkyFrame)$1;
	  localRulerDistSystem = (CoordSystem)$2;
	  localRulerDistFormat = DEGREES;
	}
	| skyFrame skyFormat
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = (SkyFrame)$1;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$2;
	}
	| LINEAR_ coordSystem
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = (CoordSystem)$2;
	  localRulerDistFormat = DEGREES;
	}
	| LINEAR_ skyFormat
	{
	  localRulerCoordSystem = WCS;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$2;
	}
	| skyFormat
	{
	  localRulerCoordSystem = IMAGE;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = WCS;
	  localRulerDistFormat = (SkyFormat)$1;
	}
	| PIXELS_
	{
	  localRulerCoordSystem = IMAGE;
	  localRulerSkyFrame = FK5;
	  localRulerDistSystem = IMAGE;
	  localRulerDistFormat = DEGREES;
	}
	;

localCompass : coordSystem skyFrame
	{
	  localCompassCoordSystem = (CoordSystem)$1;
	  localCompassSkyFrame = (SkyFrame)$2;
	}
	| coordSystem
	{
	  localCompassCoordSystem = (CoordSystem)$1;
	  localCompassSkyFrame = FK5;
	}
	| skyFrame
	{
	  localCompassCoordSystem = WCS;
	  localCompassSkyFrame = (SkyFrame)$1;
	}
	| LINEAR_
	{
	  localCompassCoordSystem = WCS;
	  localCompassSkyFrame = FK5;
	}
	;

localPanda: {aNum=0; aAngNum=0;} '(' aAngs ')' '(' aRads ')' {localPanda = 2;}
	| IGNORE_ {localPanda=0;}
	;

initLocal : {
	  // needed for annulus, ellipse annulus, and box annulus
	  aNum = 2;

	  // global properties
	  localSystem = globalSystem;
	  localSky = globalSky;
	  localProps = globalProps;
	  strcpy(localColor,globalColor);
	  localWidth = globalWidth;
	  strcpy(localFont,globalFont);
	  strcpy(localText,globalText);
	  strcpy(localComment,"");
	  taglist.deleteAll();

	  // unique properties
	  localLine1 = globalLine1;
	  localLine2 = globalLine2;
	  localPoint = globalPoint;
	  localRulerCoordSystem = globalRulerCoordSystem;
	  localRulerSkyFrame = globalRulerSkyFrame;
	  localRulerDistSystem = globalRulerDistSystem;
	  localRulerDistFormat = globalRulerDistFormat;
	  localCompassCoordSystem = globalCompassCoordSystem;
	  localCompassSkyFrame = globalCompassSkyFrame;
	  strcpy(localCompassNorth,globalCompassNorth);
	  strcpy(localCompassEast,globalCompassEast);
	  localCompassNArrow = globalCompassNArrow;
	  localCompassEArrow = globalCompassEArrow;
	  localTextAngle = globalTextAngle;
	  localPanda = 1;
	}
	;

pointProp : CIRCLE_ {$$ = CIRCLE;}
	| BOX_ {$$ = BOX;}
	| DIAMOND_ {$$ = DIAMOND;}
	| CROSS_ {$$ = CROSS;}
	| X_ {$$ = XPT;}
	| ARROW_ {$$ = ARROW;}
	| BOXCIRCLE_ {$$ = BOXCIRCLE;}
	;

include	: /* empty */ {setProps(&localProps, Marker::INCLUDE, 1);}
	| '+' {setProps(&localProps, Marker::INCLUDE, 1);}
	| '-' {setProps(&localProps, Marker::INCLUDE, 0);}
	;

shape	: CIRCLE_ bp coord sp value ep shapeComment
	  {FR->createCircleCmd(Vector($3),$5,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}

	| ELLIPSE_ bp coord sp vvalue sp optangle ep shapeComment
	{
	  // for ellipse annulus
	  aStatus = 1;
	  aCenter = Vector($3);
	  aAngle[0] = $7;
	  aVector[0] = Vector($5);
	  aNumsao = 1;
	  strncpy(aColor,localColor,16);
	  aWidth = localWidth;
	  strncpy(aFont,localFont,32);
	  strncpy(aText,localText,80);
	  strncpy(aComment,localComment,80);
	  aProps = localProps;

	  FR->createEllipseCmd(Vector($3),Vector($5),$7,
	  localColor,localWidth,localFont,localText,localProps,localComment,
	  taglist);
	}

	| BOX_ bp coord sp vvalue sp optangle ep shapeComment
	{
	  // for box annulus
	  aStatus = 3;
	  aCenter = Vector($3);
	  aAngle[0] = $7;
	  aVector[0] = Vector($5);
	  aNumsao = 1;
	  strncpy(aColor,localColor,16);
	  aWidth = localWidth;
	  strncpy(aFont,localFont,32);
	  strncpy(aText,localText,80);
	  strncpy(aComment,localComment,80);
	  aProps = localProps;

	  FR->createBoxCmd(Vector($3),Vector($5),$7,
	    localColor,localWidth,localFont,localText,localProps,localComment,
	    taglist);
	}
	| ROTBOX_ bp coord sp vvalue sp optangle ep shapeComment
	  {FR->createBoxCmd(Vector($3),Vector($5),$7,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}

	| POLYGON_ {polylist.deleteAll();} bp polyNodes ep shapeComment
	  {FR->createPolygonCmd(polylist, localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);} 

	| LINE_ bp coord sp coord ep shapeComment
	  {FR->createLineCmd(Vector($3),Vector($5),localLine1,localLine2,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}

	| TEXT_ bp coord ep shapeComment
	  {FR->createTextCmd(Vector($3),localTextAngle,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| TEXT_ bp coord sp STRING ep {strncpy(localText,$5,80);} shapeComment
	  {FR->createTextCmd(Vector($3),localTextAngle,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}

	| RULER_ bp coord sp coord ep shapeComment
	  {FR->createRulerCmd(Vector($3),Vector($5),
   	   localRulerCoordSystem, localRulerSkyFrame,
	   localRulerDistSystem, localRulerDistFormat,
	   localColor,localWidth,localFont,
	   localText,localProps,localComment,taglist);}
	| COMPASS_ bp coord sp value ep shapeComment
	  {FR->createCompassCmd(Vector($3), $5,
	   localCompassNorth, localCompassEast, 
	   localCompassNArrow, localCompassEArrow,
   	   localCompassCoordSystem, localCompassSkyFrame,
	   localColor,localWidth,localFont,
	   localText,localProps,localComment,taglist);}
	| PROJECTION_ bp coord sp coord sp value ep shapeComment
	{
	  // hard coded into projection.tcl
	  FR->createProjectionCmd(Vector($3),Vector($5),$7,
	  "ProjectionPlot", "DeleteProjectionPlot",
	  localColor,localWidth,localFont,
	  localText,localProps,localComment,taglist);
	}

	| POINT_ bp coord ep shapeComment 
	{
	  switch (localPoint) {
	  case CIRCLE:
	    FR->createCirclePointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case BOX:
	    FR->createBoxPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case DIAMOND:
	    FR->createDiamondPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case CROSS:
	    FR->createCrossPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case XPT:
	    FR->createExPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case ARROW:
	    FR->createArrowPointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case BOXCIRCLE:
	    FR->createBoxCirclePointCmd(Vector($3),
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  }
	}
	| CIRCLE_ POINT_ bp coord ep shapeComment
	  {FR->createCirclePointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| BOX_ POINT_ bp coord ep shapeComment
	  {FR->createBoxPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| DIAMOND_ POINT_ bp coord ep shapeComment
	  {FR->createDiamondPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| CROSS_ POINT_ bp coord ep shapeComment
	  {FR->createCrossPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| X_ POINT_ bp coord ep shapeComment
	  {FR->createExPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| ARROW_ POINT_ bp coord ep shapeComment
	  {FR->createArrowPointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| BOXCIRCLE_ POINT_ bp coord ep shapeComment
	  {FR->createBoxCirclePointCmd(Vector($4),
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}

	| ANNULUS_ bp coord sp value sp value ep shapeComment
	  {FR->createAnnulusCmd(Vector($3),$5,$7,1,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}
	| ANNULUS_ bp coord sp value sp value sp aRads ep
	   shapeComment
	{
	  aAnnuli[0] = $5;
	  aAnnuli[1] = $7;
	  FR->createAnnulusCmd(Vector($3),aAnnuli,aNum,
	  localColor,localWidth,localFont,localText,localProps,localComment,
	  taglist);
	}
	| ANNULUS_ bp coord sp value sp value sp numberof ep shapeComment
	  {FR->createAnnulusCmd(Vector($3),$5,$7,$9,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);}

	| ELLIPSE_ bp coord sp vvalue sp vvalue sp optangle ep shapeComment
	{
	  // prefered syntax
	  FR->createEllipseAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),1,$9,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);
	}
	| ELLIPSE_ bp coord sp vvalue sp vvalue sp 
	    numberof sp optangle ep shapeComment
	{
	  // prefered syntax
	  FR->createEllipseAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),$9,$11,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);
	}
	| ELLIPSE_ bp coord sp vvalue sp vvalue sp
	    vRads sp optangle ep shapeComment
	{
	  // prefered syntax
	  aVector[0] = Vector($5);
	  aVector[1] = Vector($7);
	  FR->createEllipseAnnulusCmd(Vector($3),$11,aNum,
	    aVector,localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);
	}

	| ELLIPSE_ bp coord sp vvalue sp optangle ep '&' '!' 
	  ELLIPSE_ bp coord sp vvalue sp optangle ep
	{	
	  // old saoimage syntax
	  aStatus = 2;
	  aVector[aNumsao++] = Vector($5);
	}

	| BOX_ bp coord sp vvalue sp vvalue sp optangle ep shapeComment
	{
	  // prefered syntax
	  FR->createBoxAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),1,$9,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);
	}
	| BOX_ bp coord sp vvalue sp vvalue sp 
	    vRads sp optangle ep shapeComment
	{
	  // prefered syntax
	  aVector[0] = Vector($5);
	  aVector[1] = Vector($7);
	  FR->createBoxAnnulusCmd(Vector($3),$11,aNum,
	  aVector,localColor,localWidth,localFont,
	  localText,localProps,localComment,taglist);
	}
	| BOX_ bp coord sp vvalue sp vvalue sp 
	    numberof sp optangle ep shapeComment
	{
	  // prefered syntax
	  FR->createBoxAnnulusCmd(Vector($3),
	    Vector($5),Vector($7),$9,$11,
	    localColor,localWidth,localFont,
	    localText,localProps,localComment,taglist);
	}

	| BOX_ bp coord sp vvalue sp optangle ep '&' '!' 
	  BOX_ bp coord sp vvalue sp optangle ep
	{	
	  // old saoimage syntax
	  aStatus = 4;
	  aVector[aNumsao++] = Vector($5);
	}

	| PANDA_ bp coord sp angle sp angle sp INT sp 
	    value sp value sp INT ep shapeComment
	{
	  switch (localPanda) {
	  case 0: /* ignore it */
	    break;
	  case 1: /* normal panda */
	    FR->createPandaCmd(Vector($3),$5,$7,$9,$11,$13,$15,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  case 2: /* one of our special pandas */
	    FR->createPandaCmd(Vector($3),aAngle,aAngNum,aAnnuli,aNum,
	      localColor,localWidth,localFont,
	      localText,localProps,localComment,taglist);
	    break;
	  }
	}

	| PIE_ bp coord sp angle sp angle ep shapeComment
	| PIE_ bp coord sp angle sp angle sp aAngs ep shapeComment
	| PIE_ bp coord sp angle sp angle sp numberof ep shapeComment
	| FIELD_ bp ep shapeComment
	;

polyNodes : polyNodes sp polyNode
	| polyNode
	;

polyNode : coord {polylist.append(new Vertex($1));}
	;

aRads	: aRads sp aRad
	| aRad
	;

aRad	: value 
	{
	  if (aNum < MAXANNULI)
	    aAnnuli[aNum++] = $1;
	}
	;

aAngs	: aAngs sp aAng
	| aAng
	;

aAng	: angle 
	{
	  if (aAngNum < MAXANGLES)
	    aAngle[aAngNum++] = $1;
	}
	;

vRads	: vRads sp vRad
	| vRad
	;

vRad	: value sp value {aVector[aNum++] = Vector($1,$3);}
	;

postLocal : /* empty */
	{
	  // old style annulus
	  switch (aStatus) {
	  case 0: // do nothing
	    break;
	  case 1: // we found just an ellipse, do nothing
	    break;
	  case 2: // ok we have an ellipse annulus
	    FR->markerDeleteLastCmd(); // delete the previous ellipse
	    FR->createEllipseAnnulusCmd(aCenter,aAngle[0],aNumsao,aVector,
	      aColor,aWidth,aFont,aText,aProps,aComment,taglist);
	    break;
	  case 3: // we found just a box, do nothing
	    break;
	  case 4: // ok, we have a box annulus
	    FR->markerDeleteLastCmd(); // delete the previous box
	    FR->createBoxAnnulusCmd(aCenter,aAngle[0],aNumsao,aVector,
	      aColor,aWidth,aFont,aText,aProps,aComment,taglist);
	    break;
	  }
	  aStatus = 0;
	}
	;
%%

static void setProps(unsigned short* props, unsigned short prop, int value)
{
  if (value)
    *props |= prop;
  else
    *props &= ~prop;
}

static CoordSystem checkWCSSystem()
{
  switch (localSystem) {
  case IMAGE:
  case PHYSICAL:
    return WCS;
  default:
   return localSystem;
  }
}

static SkyFrame checkWCSSky()
{
  switch (localSystem) {
  case IMAGE:
  case PHYSICAL:
    return NATIVEWCS;
  default:
   return localSky;
  }
}

