// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "contour.h"
#include "framebase.h"
#include "util.h"
#include "nan.h"

Contour::Contour(FrameBase* p, const char* c, int w, int m) 
  : parent(p), lineWidth(w)
{
  colorName = dupstr(c);
  color = parent->getColor(colorName);
  scale = NULL;
  method = (Method)m;

  previous_ = NULL;
  next_ = NULL;
}

Contour::Contour(FrameBase* p, const char* c, int w, const List<Vertex>& cont) 
  : parent(p), lineWidth(w)
{
  contours_ = cont;
  colorName = dupstr(c);
  color = parent->getColor(colorName);
  scale = NULL;
  method = BLOCK;

  previous_ = NULL;
  next_ = NULL;
}

Contour::~Contour()
{
  if (colorName)
    delete [] colorName;

  if (scale)
    delete scale;
}

void Contour::render(Pixmap pmap, const Matrix& m, const BBox& bb)
{
  // bb is in ref coords

  if (contours_.head()) {
    XSetForeground(parent->display, parent->contourGC, color);
    int l = lineWidth>=1 ? lineWidth : 1;
    XSetLineAttributes(parent->display, parent->contourGC, 
		       l, LineSolid, CapButt, JoinMiter);

    Vector u1 = contours_.current()->vector;
    while (contours_.next()) {
      Vector u2 = contours_.current()->vector;
      if (u1[0] != DBL_MAX && u2[0] != DBL_MAX) {
	Vector v1 = u1 * m;
	Vector v2 = u2 * m;
	BBox b = bb*m;
	if (b.isIn(v1) || b.isIn(v2))
	  XDRAWLINE(parent->display, pmap, parent->contourGC, 
		    (int)v1[0], (int)v1[1], (int)v2[0], (int)v2[1]);
      }

      u1 = u2;
    }
  }
}

#if __GNUC__ >= 3
void Contour::ps(int mode)
{
  if (contours_.head()) {
    {
    ostringstream str;
      switch ((FrameBase::PSColorSpace)mode) {
      case FrameBase::BW:
      case FrameBase::GRAY:
	psColorGray(colorName, str);
	str << " setgray";
	break;
      case FrameBase::RGB:
	psColorRGB(colorName, str);
	str << " setrgbcolor";
	break;
      case FrameBase::CMYK:
	psColorCMYK(colorName, str);
	str << " setcmykcolor";
	break;
      }
      str << endl << ends;
      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    }  
  
    {
      ostringstream str;
      str << lineWidth << " setlinewidth" << endl << ends;
      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    }

    // output stroke paths

    int first = 1;
    do {
    ostringstream str;
      Vector u = contours_.current()->vector;
      if (u[0] != DBL_MAX) {
	Vector v = u * parent->refToCanvas;
	if (first) {
	  str << "newpath " << endl
	      << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
	  first = 0;
	}
	else
	  str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;
      }
      else {
	str << "stroke" << endl << ends;
	first = 1;
      }
      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    } while (contours_.next());
  }
}
#else
void Contour::ps(int mode)
{
  char buf[256];
  if (contours_.head()) {
    {
    ostrstream str(buf,256);
      switch ((FrameBase::PSColorSpace)mode) {
      case FrameBase::BW:
      case FrameBase::GRAY:
	psColorGray(colorName, str);
	str << " setgray";
	break;
      case FrameBase::RGB:
	psColorRGB(colorName, str);
	str << " setrgbcolor";
	break;
      case FrameBase::CMYK:
	psColorCMYK(colorName, str);
	str << " setcmykcolor";
	break;
      }
      str << endl << ends;
      Tcl_AppendResult(parent->interp, buf, NULL);
    }  
  
    {
      ostrstream str(buf,256);
      str << lineWidth << " setlinewidth" << endl << ends;
      Tcl_AppendResult(parent->interp, buf, NULL);
    }

    // output stroke paths

    int first = 1;
    do {
    ostrstream str(buf,256);
      Vector u = contours_.current()->vector;
      if (u[0] != DBL_MAX) {
	Vector v = u * parent->refToCanvas;
	if (first) {
	  str << "newpath " << endl
	      << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
	  first = 0;
	}
	else
	  str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;
      }
      else {
	str << "stroke" << endl << ends;
	first = 1;
      }
      Tcl_AppendResult(parent->interp, buf, NULL);
    } while (contours_.next());
  }
}
#endif

void Contour::updateCoords(const Matrix& m)
{
  if (contours_.head())
    do {
      Vector& v = contours_.current()->vector;
      if (v[0] != DBL_MAX && v[0] != DBL_MAX)
	v *= m;
    } while (contours_.next());
}

void Contour::setLineWidth(int w)
{
  lineWidth = w;
}

void Contour::setColor(const char* clr)
{
  if (colorName)
    delete [] colorName;

  colorName = dupstr(clr);
  color = parent->getColor(colorName);
}

