// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "colorscale.h"
#include "util.h"

ColorScale::ColorScale(int s)
{
  size_ = s;
  psColors_ = new unsigned char[s*3];
  psIndex_ = new unsigned short[s];

  memset(psColors_, '0', size_*3);
  memset(psIndex_, '0', size_*sizeof(unsigned short));
}

ColorScale::~ColorScale()
{
  if (psColors_)
    delete [] psColors_;

  if (psIndex_)
    delete [] psIndex_;
}

LinearScale::LinearScale(int s, unsigned short* indexCells, 
			 unsigned char* colorCells, int count) : ColorScale(s)
{
  for (int i=0; i<s; i++) {
    int l = (int)((double)i/s * count);
    psIndex_[i] = indexCells[l];
    memcpy(psColors_+i*3, colorCells+l*3,3);
  }
}

LogScale::LogScale(int s, unsigned short* indexCells, 
		   unsigned char* colorCells, int count) : ColorScale(s)
{
  for (int i=0; i<s; i++) {
    int l = (i>0) ? (int)((log10(float(i))/log10(float(s)))*count) : 0;
    psIndex_[i] = indexCells[l];
    memcpy(psColors_+i*3, colorCells+l*3,3);
  }
}

SquaredScale::SquaredScale(int s, unsigned short* indexCells, 
			   unsigned char* colorCells, int count) 
  : ColorScale(s)
{
  for(int i=0; i<s; i++) {
    double a = (double)i/s;
    int l = (int)(a*a * count);
    psIndex_[i] = indexCells[l];
    memcpy(psColors_+i*3, colorCells+l*3,3);
  }
}

SqrtScale::SqrtScale(int s, unsigned short* indexCells, 
		     unsigned char* colorCells, int count) : ColorScale(s)
{
  for(int i=0; i<s; i++) {
    int l = (int)(sqrt((double)i/s) * count);
    psIndex_[i] = indexCells[l];
    memcpy(psColors_+i*3, colorCells+l*3,3);
  }
}

IISScale::IISScale(unsigned short* indexCells, 
		   unsigned char* colorCells, int count) : ColorScale(IISSIZE)
{
  {
    for (int i=0; i<IISCOLORS; i++) {
      int l = (int)((double)i/IISCOLORS * count);
      psIndex_[i] = indexCells[l];
      memcpy(psColors_+i*3, colorCells+l*3,3);
    }
  }

  {
    for (int i=IISCOLORS; i<IISSIZE; i++)
      psIndex_[i] = i;
  }

  setRGBColor(psColors_+200*3,"black");
  setRGBColor(psColors_+201*3,"black");
  setRGBColor(psColors_+202*3,"white");
  setRGBColor(psColors_+203*3,"red");
  setRGBColor(psColors_+204*3,"green");
  setRGBColor(psColors_+205*3,"blue");
  setRGBColor(psColors_+206*3,"yellow");
  setRGBColor(psColors_+207*3,"cyan");
  setRGBColor(psColors_+208*3,"magenta");
}

void IISScale::setRGBColor(unsigned char* ptr, const char* colorName)
{
  if (!strcmp(colorName,"black")) {
    *(ptr+0) = 000;
    *(ptr+1) = 000;
    *(ptr+2) = 000;
  }
  else if (!strcmp(colorName,"white")) {
    *(ptr+0) = 255;
    *(ptr+1) = 255;
    *(ptr+2) = 255;
  }
  else if (!strcmp(colorName,"red")) {
    *(ptr+0) = 000;
    *(ptr+1) = 000;
    *(ptr+2) = 255;
  }
  else if (!strcmp(colorName,"green")) {
    *(ptr+0) = 000;
    *(ptr+1) = 255;
    *(ptr+2) = 000;
  }
  else if (!strcmp(colorName,"blue")) {
    *(ptr+0) = 255;
    *(ptr+1) = 000;
    *(ptr+2) = 000;
  }
  else if (!strcmp(colorName,"cyan")) {
    *(ptr+0) = 255;
    *(ptr+1) = 255;
    *(ptr+2) = 000;
  }
  else if (!strcmp(colorName,"magenta")) {
    *(ptr+0) = 255;
    *(ptr+1) = 000;
    *(ptr+2) = 255;
  }
  else if (!strcmp(colorName,"yellow")) {
    *(ptr+0) = 000;
    *(ptr+1) = 255;
    *(ptr+2) = 255;
  }
  else {
    cerr << "setRGBColor Internal Error: Unable to Parser color: "
	 << colorName << endl;
    *(ptr+0) = 000;
    *(ptr+1) = 000;
    *(ptr+2) = 000;
  }
}

HistEquScale::HistEquScale(int s, unsigned short* indexCells, 
			   unsigned char* colorCells, int count, 
			   double* hist, int histsize)
  : ColorScale(s)
{
  // if no histogram, return linear distribution
  if (!hist)
    for (int i=0; i<s; i++) {
      int l = (int)((double)i/s * count);
      psIndex_[i] = indexCells[l];
      memcpy(psColors_+i*3, colorCells+l*3,3);
    }
  else
    for (int i=0; i<s; i++) {
      int l = (int)(hist[i*histsize/s]*count);
      psIndex_[i] = indexCells[l];
      memcpy(psColors_+i*3, colorCells+l*3,3);
    }
}
