// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <limits.h>
#include <string.h>
#include <time.h>

#include "sao.h"

// LIColor

istream& operator>>(istream& s, LIColor& c)
{
  char paren, comma;
  s >> paren >> c.x >> comma >> c.y >> paren;
  return s;
}

ostream& operator<<(ostream& s, LIColor& c)
{
  s << "(" << c.x << "," << c.y << ")";
  return s;
}

// SAOColorMap

SAOColorMap::SAOColorMap()
{
  red = NULL;
  green = NULL;
  blue = NULL;
}

SAOColorMap::~SAOColorMap()
{
  LIColor* ptr;

  ptr = red;
  while (ptr) {
    LIColor* tmp = ptr->getNext();
    delete ptr;
    ptr = tmp;
  }

  ptr = green;
  while (ptr) {
    LIColor* tmp = ptr->getNext();
    delete ptr;
    ptr = tmp;
  }

  ptr = blue;
  while (ptr) {
    LIColor* tmp = ptr->getNext();
    delete ptr;
    ptr = tmp;
  }
}

int SAOColorMap::load()
{
  // Determine if we can read the file

  ifstream f(fileName);
  if (!f)
    return 0;
  
  // Parse file

  LIColor** which = &red;
  char buf[128];
  while (!f.eof()) {
    f.getline(buf,128);     // read next line

    if (buf[0] == '#')      // check for comments
      ;                     // do nothing

    else if (!strncmp(buf,"PSEUDOCOLOR",11)) // check for PSEUDOCOLOR
      ;                     // do nothing

    // Identify a color

    else if (!strncmp(buf,"RED",3))
      which = &red;
    else if (!strncmp(buf,"GREEN",5))
      which = &green;
    else if (!strncmp(buf,"BLUE",4))
      which = &blue;

    // And parse color

    else if (buf[0] == '(')
      loadColor(buf, which);
  }

  // Are we valid?

  if (red && green && blue)
    return 1;  // we found at least one LIColor for each RGB
  else
    return 0;  // something is missing, bailout
}

int SAOColorMap::save(const char* fn)
{
  ofstream f(fn);
  if (!f)
    return 0;

  f << *this;

  return 1;
}

void SAOColorMap::loadColor(char* buf, LIColor** c)
{
#if __GNUC__ >= 3
  string x(buf);
  istringstream str(x);
#else
  istrstream str(buf);
#endif

  while (1) {
    LIColor* li = new LIColor();         // create new LIColor
    if (!li) {
      cerr << "SAOColorMap Interal Error: Unable to Create LIColor." << endl;
      return;
    }
    str >> *li;                          // and try to read into it
    if (!str.fail()) {                   // successful read
      if (*c) {                          // is this the first?
	LIColor* head = *c;              // advance to end of link list
	while (head && head->getNext())
	  head = head->getNext();

	head->setNext(li);               // and add to end of list
      }
      else
	*c = li;                         // yes, it is
    }
    else {              // read failed, we are at the end of the line
      delete li;        // clean up
      break;            // and bail out
    }
  }
}

unsigned char SAOColorMap::getColorChar(int i, int count, LIColor* c)
{
  float x = (float)i/count;
  LIColor* head = c;
  LIColor* tail = NULL;

  while (head && (head->getX() < x))
    if (head) {
      tail = head;
      head = head->getNext();
    }

  if (tail && head) {       // interpolate between head and tail
    float m = (head->getY() - tail->getY()) / (head->getX() - tail->getX());
    if (m) {
      float y = m * (x - tail->getX()) + tail->getY(); //point slope form
      return (unsigned char)(y * UCHAR_MAX);
    }
    else
      return (unsigned char)(head->getY() * UCHAR_MAX);

  }
  else if (!tail && head)   // return first LIColor
    return (unsigned char)(head->getY() * UCHAR_MAX);

  else if (tail && !head)   // return last LIColor
    return (unsigned char)(tail->getY() * UCHAR_MAX);

  else
    return 0;               // there is something very wrong-- bail out
}

unsigned short SAOColorMap::getColorShrt(int i, int count, LIColor* c)
{
  float x = (float)i/count;
  LIColor* head = c;
  LIColor* tail = NULL;

  while (head && (head->getX() < x))
    if (head) {
      tail = head;
      head = head->getNext();
    }

  if (tail && head) {       // interpolate between head and tail
    float m = (head->getY() - tail->getY()) / (head->getX() - tail->getX());
    if (m) {
      float y = m * (x - tail->getX()) + tail->getY(); //point slope form
      return (unsigned short)(y * USHRT_MAX);
    }
    else
      return (unsigned short)(head->getY() * USHRT_MAX);

  }
  else if (!tail && head)   // return first LIColor
    return (unsigned short)(head->getY() * USHRT_MAX);

  else if (tail && !head)   // return last LIColor
    return (unsigned short)(tail->getY() * USHRT_MAX);

  else
    return 0;               // there is something very wrong-- bail out
}

ostream& operator<<(ostream& s, SAOColorMap& c)
{
  s << "# SAOimage color table" << endl;

  time_t timeOfDay = time(NULL);
  char* str = ctime(&timeOfDay);
  if (str)
    s << "# " << str;

  s << "PSEUDOCOLOR" << endl;

  LIColor* li;
  s << "RED:" << endl;
  for (li=c.red; li; li=li->getNext())
    s << *li;
  s << endl;

  s << "GREEN:" << endl;
  for (li=c.green; li; li=li->getNext())
    s << *li;
  s << endl;

  s << "BLUE:" << endl;
  for (li=c.blue; li; li=li->getNext())
    s << *li;
  s << endl;

  return s;
}
