# Samizdat message content model
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'
require 'fileutils'
require 'mahoro'

class Content

  def initialize(id, login, title=nil, format=nil, body=nil)
    @id = id.kind_of?(Integer) ? id : nil
    @login = login

    if @id.kind_of? Integer
      title ||= rdf.get_property(@id, 'dc::title')
      format ||= rdf.get_property(@id, 'dc::format')
    end

    @title = title
    self.format = format   # also sets @inline and @cacheable

    if cacheable? and @id.kind_of? Integer
      @html_full = rdf.get_property(@id, 's::htmlFull')
      @html_short = rdf.get_property(@id, 's::htmlShort')
    end

    if inline?
      if body
        self.body = body
      elsif @id.kind_of? Integer
        self.body = rdf.get_property(@id, 's::content')
      end

    else
      @file = ContentFile.new(self)
    end
  end

  # set content id, move file to new id if necessary
  #
  def id=(id)
    @id = id
    @file.id = id if @file.kind_of?(ContentFile)
  end

  attr_reader :id, :login, :title, :format, :plugin

  # store checks for format, inline?, cacheable?, plugin
  #
  def format=(format)
    @format = format
    @format = nil unless config['format'].values.flatten.include? @format
    @format.untaint

    @inline = (@format.nil? or config['format']['inline'].include?(@format))
    @cacheable = (@inline and @format != 'application/x-squish')

    plugin_api = @inline ? 'content_inline' : 'content_file'
    @plugin = config.plugins.find(plugin_api, @format)
  end

  # true if content is rendered by Samizdat and not linked to a file
  #
  def inline?
    @inline
  end

  # HTML rendering of all inline messages except RDF queries is cached in
  # database (which implies that it doesn't depend on Resource object)
  #
  def cacheable?
    @cacheable
  end

  # content body (+nil+ if not inline)
  attr_reader :body

  # set body, force Unix newlines
  #
  def body=(body)
    @body = body.kind_of?(String) ? body.gsub(/\r\n/, "\n") : nil
  end

  # format content using matching content rendering plugin
  #
  def render(request, mode, body = @body)
    return '' if (inline? and body.nil?) or @plugin.nil?

    case @plugin.api
    when 'content_inline'
      begin
        config.xhtml.sanitize(@plugin.render(request, mode, body))

      rescue Samizdat::SanitizeError => e
        raise UserError, CGI.escapeHTML(e.message).untaint
      end

    when 'content_file'
      @plugin.render(request, mode, self)
    end
  end

  def html_full
    return nil unless cacheable?
    return @html_full if @html_full

    render(nil, :full)
  end

  def html_short
    return nil unless cacheable?
    return @html_short if @html_short

    body_short = limit_string(@body, config['limit']['short'])
    if body_short and body_short.size < @body.size
      render(nil, :short, body_short)
    end
  end

  # update html_full and html_short for a message
  # (running inside transaction is assumed)
  #
  def update_html
    return unless cacheable? and @id.kind_of? Integer

    @html_full = nil
    @html_short = nil

    @html_full = html_full
    db.do 'UPDATE Message SET html_full = ? WHERE id = ?', @html_full, @id

    @html_short = html_short
    if @html_short
      db.do 'UPDATE Message SET html_short = ? WHERE id = ?', @html_short, @id
    else
      db.do 'UPDATE Message SET html_short = NULL WHERE id = ?', @id
    end
  end

  # re-render html_full and html_short for a single messages
  #
  # command line:
  #
  # SAMIZDAT_SITE=samizdat SAMIZDAT_URI=/ ruby -r samizdat/engine -e 'Content.regenerate_html(1)'
  #
  def Content.regenerate_html(id)
    db.transaction do |db|
      message = Message.cached(id)
      Content.new(message.id, message.creator.login).update_html
    end   # transaction
  end

  # re-render html_full and html_short for all messages
  #
  # command line:
  #
  # SAMIZDAT_SITE=samizdat SAMIZDAT_URI=/ ruby -r samizdat/engine -e 'Content.regenerate_all_html'
  #
  def Content.regenerate_all_html
    db.select_all('SELECT id FROM Message WHERE content IS NOT NULL') do |id,|
      Content.regenerate_html(id)
    end
    cache.flush
  end

  # ContentFile object for a non-inline message
  attr_reader :file

  # initialize ContentFile from new upload
  #
  def file=(file)
    @file = ContentFile.new(self, file)
    @body = nil
  end
end


class ContentFile

  # check if size is within the configured limit
  #
  def ContentFile.validate_size(size)
    # todo: fine-grained size limits
    if size > config['limit']['content']
      raise UserError, sprintf(
        _('Uploaded file is larger than %s bytes limit'),
        config['limit']['content'])
    end
  end

  def initialize(content, file = nil)
    @id = (content.id or upload_id)
    @login = content.login

    if file   # new upload
      save(file)
      content.format = @format
    else
      @format = content.format
    end

    @plugin = content.plugin

    if file
      @plugin.new_file(self)
    end
  end

  attr_reader :id, :login, :format

  def ContentFile.extension(format)
    config['file_extension'][format] or format.sub(%r{\A.*/(x-)?}, '')
  end

  def extension
    ContentFile.extension(@format)
  end

  # relative path to file holding multimedia message content
  #
  def location(id = @id)
    validate_id(id)

    # security: keep format and creator login controlled (see untaint in
    # path())
    '/' + @login + '/' + id.to_s + '.' + extension
  end

  # multimedia message content filename
  #
  def path(id = @id)
    File.join(config.content_dir, location(id).untaint)
  end

  def href(request)
    File.join(request.content_location, location)
  end

  def size
    filename = path
    return nil unless File.exists? filename

    File.size(filename)
  end

  def exists?
    File.exists?(path)
  end

  # move content file to a new id
  #
  def id=(id)
    validate_id(id)

    File.rename(path, path(id))
    @plugin.move_file(self, id)

    @id = id
  end

  def delete
    File.delete(path)
    @plugin.delete_file(self)
  end

  private

  # id component of the file path of a new upload
  #
  def upload_id
    'upload'
  end

  def validate_id(id)
    id.kind_of?(Integer) or upload_id == id or raise RuntimeError,
      "Unexpected file upload id (#{id.inspect})"
  end

  # detect and validate format from upload
  #
  def detect_file_format(file)
    @@mahoro ||= Mahoro.new(Mahoro::MIME)

    format =
      if file.respond_to?(:path) and file.path
        @@mahoro.file(file.path)
      elsif file.kind_of?(StringIO)
        @@mahoro.buffer(file.string)
      elsif file.kind_of?(String)
        @@mahoro.buffer(file)
      end

    format.nil? and raise RuntimeError,
      _('Failed to detect content type of the uploaded file')

    config['format'].values.flatten.include?(format) or
      raise UserError,
        sprintf(_("Format '%s' is not supported"), CGI.escapeHTML(format))

    format.untaint
  end

  def mkdir(dir)
    File.exists?(dir) or FileUtils.mkdir_p(dir)
  end

  def save(file)
    @format = detect_file_format(file)

    destination = self.path
    mkdir(File.dirname(destination))

    case file
    when Tempfile   # copy large files directly
      FileUtils.cp(file.path, destination)
    when StringIO
      File.open(destination, 'w') {|f| f.write(file.read) }
    else
      raise RuntimeError, "Unexpected file class '#{file.class}'"
    end
  end
end
