# Samizdat engine helper functions
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

# debug output
#
def log(msg)
  $stderr << 'Samizdat: ' << msg << "\n"
end

# log a verbose report on exception, return unique error id that can be used to
# find this report in the log
#
def log_exception(error, request = nil)
  error_id = digest(Time.now.to_s + rand.to_s)

  error_report = Time.now.to_s +
    %{: #{error.class}: #{error.to_s}\nError ID: #{error_id}\n}
  error_report << %{CGI parameters: #{@request.dump_params}\n} if
    request.respond_to?(:dump_params)
  error_report << %{Backtrace:\n#{error.backtrace.join("\n")}\n} if
    error.respond_to?(:backtrace)

  log(error_report)
  error_id
end

# expire timeout for permanent cookie
#
def forever
  60 * 60 * 24 * 365 * 5   # 5 years
end

# generate uniform hash
#
def digest(value)
  Digest::MD5.hexdigest(value)   # todo: configurable digest function
end

# generate a random hash from a given seed
#
def random_digest(seed = '')
  digest(seed.to_s + Time.now.to_s + rand().to_s)
end

# parse YAML data or return empty hash
#
def yaml_hash(yaml)
  YAML.load(yaml.to_s) or {}
end

# check for readable file in all directories in sequence, return the first one
# found or nil
#
def find_file(file, dirs)
  if found = dirs.find {|dir| File.readable?(dir + file) }
    found + file
  end
end

# shortcut for page length limit config option
#
def limit_page
  config['limit']['page']
end

# Squish condition to exclude hidden messages
#
def exclude_hidden
  " AND ?hidden = 'false'"
end

# Squish filter to exclude hidden messages
#
def filter_hidden
  " FILTER ?hidden = 'false'"
end

def ellipsis
  "\xe2\x80\xa6"
end

# return _string_ truncated to the _limit_, with ellipsis if necessary
#
def limit_string(string, limit=config['limit']['title'])
  return nil unless string.kind_of? String

  $KCODE = 'utf8'
  if string.jsize > limit
    string.each_char[0, limit - 1].join + ellipsis
  else
    string
  end
end

# transform date to a standard string representation
#
def format_date(date)
  date = date.to_time if date.methods.include? 'to_time'   # duck
  date = date.strftime '%Y-%m-%d %H:%M' if date.kind_of? Time
  date
end

# default language on the site is the first language in the languages list
#
def default_language
  config['locale']['languages'][0]
end

# true if _url_ can be an absolute URL (i.e. contains network root component)
#
def absolute_url?(url)
  url =~ %r{//}
end
