#!/usr/bin/env ruby
#
# Samizdat template tests
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'test/unit'
require 'rexml/document'
require 'test/util'
require 'samizdat/engine'

class TestSession
  def initialize
    @id = 1
    @login = 'test'
    @full_name = 'Test'
    @options = {}
    @base = 'http://localhost' + uri_prefix
    @template = Template.new(self)
    bindtextdomain('samizdat', nil, 'C', 'utf-8')
  end

  def cookie(name)
    nil
  end

  attr_accessor :id, :login, :full_name, :options, :base, :template, :moderator
end

class TestMessage
  def initialize(params)
    @id = params.delete(:id)
    @params = {
      :title => '',
      :content => '',
      :date => Time.now,
      :lang => nil,
      :format => nil,
      :creator => nil,
      :creator_login => 'guest',
      :creator_name => 'guest',
      :desc => nil,
      :parent => nil,
      :current => nil,
      :open => false,
      :nversions => 0,
      :translations => [],
      :nreplies => 0,
      :focuses => [],
      :nrelated => 0,
      :html_full => nil,
      :html_short => nil,
      :inline? => true
    }
    @params.update(params)
  end

  attr_reader :id

  def method_missing(name)
    @params[name]
  end
end

class TC_Template < Test::Unit::TestCase
  def setup
    # create a dummy session
    @s = TestSession.new
    @t = @s.template
  end

  def teardown
    @t = nil
    @s = nil
  end

  def test_box
    root = parse(@t.box('Title', 'Content.'))
    assert_equal 'box', root.attributes['class']
    assert_equal ['Title'], elements(root, 'div[@class="box-title"]')
    assert_equal ['Content.'], elements(root, 'div[@class="box-content"]')

    root = parse(@t.box(nil, 'Content.'))
    assert_equal [], elements(root, 'div[@class="box-title"]')
    assert_equal ['Content.'], elements(root, 'div[@class="box-content"]')
  end

  def test_form_field_label
    root = parse(@t.form_field(:label, 'some_field', 'Label'))
    assert_equal 'label', root.attributes['class']
    assert_equal 'f_some_field', root.elements['label'].attributes['for']
    assert_equal ['Label'], elements(root, 'label')
  end

  def test_form_field_textarea
    root = parse(@t.form_field(:textarea, 't_a', "Some\nText."))
    assert_equal 't_a', root.attributes['name']
    assert_equal "Some\nText.", root.text.strip
  end

  def test_form_field_select
    root = parse(@t.form_field(:select, 't_select',
      ['a1', ['a2', 'Label 2']]))
    assert_equal 't_select', root.attributes['name']
    assert_equal ['a1'], elements(root, 'option[@value="a1"]')
    assert_equal ['Label 2'], elements(root, 'option[@value="a2"]')
  end

  def test_form_field_submit
    root = parse(@t.form_field(:submit, 't_submit', 'Click'))
    assert_equal 't_submit', root.attributes['name']
    assert_equal 'Click', root.attributes['value']
    assert_equal 'submit', root.attributes['type']
    assert_equal 'submit', root.attributes['class']
  end

  def test_form_field_input
    root = parse(@t.form_field(:password, 'passwd', 'secret'))
    assert_equal 'passwd', root.attributes['name']
    assert_equal 'secret', root.attributes['value']
    assert_equal 'password', root.attributes['type']
  end

  def test_form
    root = parse(@t.form('login.rb', [:password, 'passwd'], [:submit]))
    assert_equal 'login.rb', root.attributes['action']
    assert_equal 'post', root.attributes['method']
    assert_equal ['password', 'submit'], elements(root, 'div/input', 'type')

    root = parse(@t.form('message.rb', [:file, 't_file']))
    assert_equal 'message.rb', root.attributes['action']
    assert_equal 'post', root.attributes['method']
    assert_equal 'multipart/form-data', root.attributes['enctype']
    assert_equal ['file'], elements(root, 'div/input', 'type')
  end

  def test_member_line
    @s.id = nil
    @t = Template.new(@s)
    root = parse('<p>'+@t.member_line+'</p>')
    assert_equal ['member.rb', 'login.rb'], elements(root, '//a', 'href')
  end

  def test_member_line_registered
    root = parse('<p>'+@t.member_line+'</p>')
    assert_equal [@s.id.to_s, 'member.rb', 'logout.rb'],
      elements(root, '//a', 'href')
  end

  def test_message_buttons
    m = TestMessage.new(:id => 'upload', :focuses => [],
      :creator_name => 'Test', :title => 'Title')
    root = parse(@t.message_buttons(m))
    assert_equal ['title', 'parent', 'reply', 'edit'],
      elements(root, 'form/div/input', 'name')
  end

  def test_message_info_date
    m = TestMessage.new(:id => 'upload', :date => Time.now,
      :full_name => 'Test')
    @t.message_info(m, :full)
    m = TestMessage.new(:id => 'upload', :date => '2004-03-11',
      :full_name => 'Test')
    @t.message_info(m, :full)
    m = TestMessage.new(:id => 'upload', :date => nil, :full_name => 'Test')
    @t.message_info(m, :full)
    date = 'current time'
    class << date   # check duck typing
      def to_time
        Time.now
      end
    end
    m = TestMessage.new(:id => 'upload', :date => date, :full_name => 'Test')
    @t.message_info(m, :full) =~ /<\/a> on (.*)$/
    assert_not_equal date, $1
  end

  def test_message_info_parent
    m = TestMessage.new(:id => 'upload', :creator_name => 'Test', :parent => 2)
    assert_equal true,
      @t.message_info(m, :full).split(",\n ")[1].include?('href="2"')
  end

  def test_message_content
    m = TestMessage.new({})
    assert_equal nil, @t.message_content(m, :list)

    m = TestMessage.new(:content => "line 1\n\nline 2")
    root = parse(@t.box(nil, @t.message_content(m, :full)))
    assert_equal ['line 1', 'line 2'], elements(root, 'div/p')

    m = TestMessage.new(:content => 'test', :format => 'text/plain')
    assert_equal 'pre',
      parse(@t.message_content(m, :full)).name

    m = TestMessage.new(:format => 'application/x-squish',
      :content => 'SELECT')
    root = parse(@t.message_content(m, :full))
    assert_equal ['query', nil, 'run'], elements(root, 'div/*', 'name')

    m = TestMessage.new(:format => 'image/jpeg',
      :title => 'A', :content => 'anarchy.png')
    root = parse(@t.message_content(m, :full))
    assert_equal 'A', root.attributes['alt']
    assert_equal 'anarchy.png', root.attributes['src']

    m = TestMessage.new(:format => 'application/pdf')
    root = parse(@t.message_content(m, :full))
    assert_equal ['application/pdf content'], elements(root, 'a')
  end

  def test_simple_page
    root = parse(@t.page('Test Title', 'Some text.'))
    assert_equal ['Test Title'], elements(root, '//div[@class="box-title"]')
  end

  def test_multi_page
    root = parse( @t.page('Title',
      [ ['Title 1', 'Body 1.'], ['Title 2', 'Body 2.']]))
    assert_equal ['Title'], elements(root, '/html/head/title')
    main = root.elements['/html/body/div[@id="main"]']
    assert_equal ['Title 1', 'Title 2'],
      elements(main, 'div/div[@class="box-title"]')
    assert_equal ['Body 1.', 'Body 2.'],
      elements(main, 'div/div[@class="box-content"]')
  end
end
