/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 *
 * $Id: saildisp.cpp,v 1.38 2004/10/26 19:30:50 jeremy_laine Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "saildisp.h"

/*****************************************************************************
 
                          CSailDisp class
 
*****************************************************************************/

/** Constructs a generic view area for a sail.
 */
CSailDisp::CSailDisp()
{
    // initialise data
    m = CMatrix::id(3);
    //setSail(sail);
    zoom=0.8;
}


/** Calculates logical viewport rectangle to match
 * the ratio of the device viewport
 */
void CSailDisp::calcLRect()
{
    // avoid division by zero errors
    if ((vRect.height() == 0) || (vRect.width() == 0))
    {
        lRect.min = center;
        lRect.max = center;
        return;
    }

    real viewAspect = vRect.width() / vRect.height();
    CRect3d sRect = sailBase.boundingRect();
    real sailAspect = sRect.width() / sRect.height();

    // the viewing area may not match the proportions of the sail
    // so we set the logical viewport accordingly
    if (sailAspect > viewAspect)
    {
        // we are limited by the width of the window, grow logical viewport's height
        lRect.min = CVector3d(sRect.min.x(),sRect.min.y()*(sailAspect/viewAspect),0);
        lRect.max = CVector3d(sRect.max.x(),sRect.max.y()*(sailAspect/viewAspect),0);
    }
    else
    {
        // we are limited by the height of the window, grow logical viewport's width
        lRect.min = CVector3d(sRect.min.x()*(viewAspect/sailAspect),sRect.min.y(),0);
        lRect.max = CVector3d(sRect.max.x()*(viewAspect/sailAspect),sRect.max.y(),0);
    }
    /*
        qDebug("-------");
        qDebug("sRect w: %f, h: %f",sRect.width(), sRect.height());
        qDebug("vRect w: %f, h: %f",vRect.width(), vRect.height());
        qDebug("lRect w: %f, h: %f",lRect.width(), lRect.height());
        qDebug("zoom : %f",zoom);
    */
    lRect = lRect*(1/zoom)+center;
    /*
        qDebug("lRect w: %f, h: %f",lRect.width(), lRect.height());
        qDebug("-------");
    */
}


/** Rotates local copy of the sail by a given azimuth and elevation.
 */
void CSailDisp::calcSailDisp()
{
    sailDisp = sailBase.rotate( sailBase.boundingRect().center(), m);
}


/** Resets display zoom and center to their default values.
 */
void CSailDisp::resetZoomCenter()
{
    center=CPoint3d(0,0,0);
    zoom = 0.8;
    calcLRect();
}


/** Converts screen coordinates to logical coordinates.
 */
CPoint3d CSailDisp::screenToLogical( const int x, const int y ) const
{
    // avoid division by zero
    if ((vRect.width()==0)||(vRect.height()==0))
        return center;

    return center + CVector3d( lRect.width() * ( real(x) / vRect.width() - 0.5 ),
                               lRect.height() * ( 0.5 - real(y) / vRect.height() ), 0);
}


/** Sets the azimuth and elevation view angles.
 *
 * @param azimuth azimuth in degrees
 * @param elevation elevation in degrees
 */
void CSailDisp::setAngle( real azimuth, real elevation )
{
    m = CMatrix::rot3d(0, PI/180*elevation) * CMatrix::rot3d(1, PI/180*azimuth);
    calcSailDisp();
}


/** Sets the center of the display.
 *
 * @param newCenter the new center
 */
void CSailDisp::setCenter( CPoint3d newCenter )
{
    center = newCenter;
    calcLRect();
}


/** Sets the sail that is to be displayed.
 */
void CSailDisp::setSail( const CSail &sail )
{
    sailBase = sail;
    sailBase = sailBase + CVector3d( -sailBase.boundingRect().center() );
    calcSailDisp();
    calcLRect();
}


/** Set the size of the viewing rectangle.
 */
void CSailDisp::setVRect(const int width, const int height)
{
    vRect.min=CPoint3d(0,0,0);
    vRect.max=CPoint3d(width,height);
    calcLRect();
}



/** Sets the zoom factor
 *
 * @param newZoom
 */
void CSailDisp::setZoom(real newZoom)
{
    zoom = newZoom;
    calcLRect();
}

/** Zooms IN by a factor 2.
 */
void CSailDisp::zoomIn()
{
    setZoom(zoom * 2);
}


/** Zooms OUT by a factor 2.
 */
void CSailDisp::zoomOut()
{
    setZoom(zoom / 2);
}
