#! /usr/bin/env python

'''
Python Interface to SAGA Tools Library
----------
- Category : Import/Export
- Name     : PostgreSQL
- ID       : db_pgsql

Description
----------
Accessing PostgreSQL databases.
'''

from PySAGA.helper import Tool_Wrapper

def List_PostgreSQL_Connections(CONNECTIONS=None, Verbose=2):
    '''
    List PostgreSQL Connections
    ----------
    [db_pgsql.0]\n
    Lists all PostgreSQL sources.\n
    Arguments
    ----------
    - CONNECTIONS [`output table`] : Connections

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '0', 'List PostgreSQL Connections')
    if Tool.is_Okay():
        Tool.Set_Output('CONNECTIONS', CONNECTIONS)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_0(CONNECTIONS=None, Verbose=2):
    '''
    List PostgreSQL Connections
    ----------
    [db_pgsql.0]\n
    Lists all PostgreSQL sources.\n
    Arguments
    ----------
    - CONNECTIONS [`output table`] : Connections

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '0', 'List PostgreSQL Connections')
    if Tool.is_Okay():
        Tool.Set_Output('CONNECTIONS', CONNECTIONS)
        return Tool.Execute(Verbose)
    return False

def Connect_to_PostgreSQL(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Connect to PostgreSQL
    ----------
    [db_pgsql.1]\n
    Connect to PostgreSQL data source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '1', 'Connect to PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_1(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Connect to PostgreSQL
    ----------
    [db_pgsql.1]\n
    Connect to PostgreSQL data source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '1', 'Connect to PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def Disconnect_from_PostgreSQL(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TRANSACT=None, Verbose=2):
    '''
    Disconnect from PostgreSQL
    ----------
    [db_pgsql.2]\n
    Disconnect PostgreSQL data source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '2', 'Disconnect from PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_2(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TRANSACT=None, Verbose=2):
    '''
    Disconnect from PostgreSQL
    ----------
    [db_pgsql.2]\n
    Disconnect PostgreSQL data source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '2', 'Disconnect from PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def Disconnect_All(TRANSACT=None, Verbose=2):
    '''
    Disconnect All
    ----------
    [db_pgsql.3]\n
    Disconnects all PostgreSQL connections.\n
    Arguments
    ----------
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '3', 'Disconnect All')
    if Tool.is_Okay():
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_3(TRANSACT=None, Verbose=2):
    '''
    Disconnect All
    ----------
    [db_pgsql.3]\n
    Disconnects all PostgreSQL connections.\n
    Arguments
    ----------
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '3', 'Disconnect All')
    if Tool.is_Okay():
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def Begin_Transaction(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, SAVEPOINT=None, Verbose=2):
    '''
    Begin Transaction
    ----------
    [db_pgsql.4]\n
    Begins a transaction, which will be finished later with a commit or rollback. Tries to add a save point, if already in transaction mode.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - SAVEPOINT [`text`] : Save Point. Default: SAVEPOINT_01

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '4', 'Begin Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_4(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, SAVEPOINT=None, Verbose=2):
    '''
    Begin Transaction
    ----------
    [db_pgsql.4]\n
    Begins a transaction, which will be finished later with a commit or rollback. Tries to add a save point, if already in transaction mode.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - SAVEPOINT [`text`] : Save Point. Default: SAVEPOINT_01

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '4', 'Begin Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def CommitRollback_Transaction(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TRANSACT=None, SAVEPOINT=None, Verbose=2):
    '''
    Commit/Rollback Transaction
    ----------
    [db_pgsql.5]\n
    Execute a commit or rollback on open transactions with PostgreSQL source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1
    - SAVEPOINT [`text`] : Save Point

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '5', 'Commit/Rollback Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TRANSACT', TRANSACT)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_5(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TRANSACT=None, SAVEPOINT=None, Verbose=2):
    '''
    Commit/Rollback Transaction
    ----------
    [db_pgsql.5]\n
    Execute a commit or rollback on open transactions with PostgreSQL source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1
    - SAVEPOINT [`text`] : Save Point

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '5', 'Commit/Rollback Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TRANSACT', TRANSACT)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def Execute_SQL(TABLES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, SQL=None, OUTPUT=None, STOP=None, Verbose=2):
    '''
    Execute SQL
    ----------
    [db_pgsql.6]\n
    Execute SQL commands on a connected PostgreSQL source. Separate different commands with a semicolon (';').\n
    Arguments
    ----------
    - TABLES [`output table list`] : Tables
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - SQL [`long text`] : SQL Statement. Default: CREATE TABLE myTable1 (Col1 VARCHAR(255) PRIMARY KEY, Col2 INTEGER);
INSERT INTO myTable1 (Col1, Col2) VALUES('First Value', 1);
DROP TABLE myTable1;

    - OUTPUT [`choice`] : Output. Available Choices: [0] none [1] message window [2] tables Default: 2
    - STOP [`boolean`] : Stop on Error. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '6', 'Execute SQL')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('SQL', SQL)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('STOP', STOP)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_6(TABLES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, SQL=None, OUTPUT=None, STOP=None, Verbose=2):
    '''
    Execute SQL
    ----------
    [db_pgsql.6]\n
    Execute SQL commands on a connected PostgreSQL source. Separate different commands with a semicolon (';').\n
    Arguments
    ----------
    - TABLES [`output table list`] : Tables
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - SQL [`long text`] : SQL Statement. Default: CREATE TABLE myTable1 (Col1 VARCHAR(255) PRIMARY KEY, Col2 INTEGER);
INSERT INTO myTable1 (Col1, Col2) VALUES('First Value', 1);
DROP TABLE myTable1;

    - OUTPUT [`choice`] : Output. Available Choices: [0] none [1] message window [2] tables Default: 2
    - STOP [`boolean`] : Stop on Error. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '6', 'Execute SQL')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('SQL', SQL)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('STOP', STOP)
        return Tool.Execute(Verbose)
    return False

def List_Tables(TABLES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    List Tables
    ----------
    [db_pgsql.10]\n
    Lists all tables of an PostgreSQL data source.\n
    Arguments
    ----------
    - TABLES [`output table`] : Tables
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '10', 'List Tables')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_10(TABLES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    List Tables
    ----------
    [db_pgsql.10]\n
    Lists all tables of an PostgreSQL data source.\n
    Arguments
    ----------
    - TABLES [`output table`] : Tables
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '10', 'List Tables')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def List_Table_Fields(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, VERBOSE=None, Verbose=2):
    '''
    List Table Fields
    ----------
    [db_pgsql.11]\n
    Loads table information from PostgreSQL data source.\n
    Arguments
    ----------
    - TABLE [`output table`] : Field Description
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - VERBOSE [`boolean`] : Verbose. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '11', 'List Table Fields')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('VERBOSE', VERBOSE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_11(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, VERBOSE=None, Verbose=2):
    '''
    List Table Fields
    ----------
    [db_pgsql.11]\n
    Loads table information from PostgreSQL data source.\n
    Arguments
    ----------
    - TABLE [`output table`] : Field Description
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - VERBOSE [`boolean`] : Verbose. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '11', 'List Table Fields')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('VERBOSE', VERBOSE)
        return Tool.Execute(Verbose)
    return False

def Import_Table(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, Verbose=2):
    '''
    Import Table
    ----------
    [db_pgsql.12]\n
    Imports a table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Table
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '12', 'Import Table')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_12(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, Verbose=2):
    '''
    Import Table
    ----------
    [db_pgsql.12]\n
    Imports a table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Table
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '12', 'Import Table')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def Export_Table(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TABLE_PK=None, TABLE_NN=None, TABLE_UQ=None, NAME=None, EXISTS=None, Verbose=2):
    '''
    Export Table
    ----------
    [db_pgsql.13]\n
    Exports a table to a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`input table`] : Table
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TABLE_PK [`table fields`] : Primary Key
    - TABLE_NN [`table fields`] : Not Null
    - TABLE_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '13', 'Export Table')
    if Tool.is_Okay():
        Tool.Set_Input ('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TABLE_PK', TABLE_PK)
        Tool.Set_Option('TABLE_NN', TABLE_NN)
        Tool.Set_Option('TABLE_UQ', TABLE_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_13(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TABLE_PK=None, TABLE_NN=None, TABLE_UQ=None, NAME=None, EXISTS=None, Verbose=2):
    '''
    Export Table
    ----------
    [db_pgsql.13]\n
    Exports a table to a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`input table`] : Table
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TABLE_PK [`table fields`] : Primary Key
    - TABLE_NN [`table fields`] : Not Null
    - TABLE_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '13', 'Export Table')
    if Tool.is_Okay():
        Tool.Set_Input ('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TABLE_PK', TABLE_PK)
        Tool.Set_Option('TABLE_NN', TABLE_NN)
        Tool.Set_Option('TABLE_UQ', TABLE_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        return Tool.Execute(Verbose)
    return False

def Drop_Table(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, Verbose=2):
    '''
    Drop Table
    ----------
    [db_pgsql.14]\n
    Deletes a table from a PostgreSQL database.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '14', 'Drop Table')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_14(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, Verbose=2):
    '''
    Drop Table
    ----------
    [db_pgsql.14]\n
    Deletes a table from a PostgreSQL database.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '14', 'Drop Table')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def Import_Table_from_Query(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TABLES=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query
    ----------
    [db_pgsql.15]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TABLES [`text`] : Tables
    - FIELDS [`text`] : Fields
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '15', 'Import Table from Query')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_15(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TABLES=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query
    ----------
    [db_pgsql.15]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TABLES [`text`] : Tables
    - FIELDS [`text`] : Fields
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '15', 'Import Table from Query')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def Import_Table_from_Query_GUI(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, WHERE=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query (GUI)
    ----------
    [db_pgsql.16]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - WHERE [`text`] : Where
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '16', 'Import Table from Query (GUI)')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_16(TABLE=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, WHERE=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query (GUI)
    ----------
    [db_pgsql.16]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - WHERE [`text`] : Where
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '16', 'Import Table from Query (GUI)')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def Import_Shapes(SHAPES=None, COLLECTION=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, Verbose=2):
    '''
    Import Shapes
    ----------
    [db_pgsql.20]\n
    Imports shapes from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - COLLECTION [`output shapes list`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '20', 'Import Shapes')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_20(SHAPES=None, COLLECTION=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, Verbose=2):
    '''
    Import Shapes
    ----------
    [db_pgsql.20]\n
    Imports shapes from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - COLLECTION [`output shapes list`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '20', 'Import Shapes')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def Export_Shapes(SHAPES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, SHAPES_PK=None, SHAPES_NN=None, SHAPES_UQ=None, NAME=None, EXISTS=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Export Shapes
    ----------
    [db_pgsql.21]\n
    Exports shapes to a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - SHAPES_PK [`table fields`] : Primary Key
    - SHAPES_NN [`table fields`] : Not Null
    - SHAPES_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name. if empty shapes layers's name is used as table name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '21', 'Export Shapes')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('SHAPES_PK', SHAPES_PK)
        Tool.Set_Option('SHAPES_NN', SHAPES_NN)
        Tool.Set_Option('SHAPES_UQ', SHAPES_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_21(SHAPES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, SHAPES_PK=None, SHAPES_NN=None, SHAPES_UQ=None, NAME=None, EXISTS=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Export Shapes
    ----------
    [db_pgsql.21]\n
    Exports shapes to a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - SHAPES_PK [`table fields`] : Primary Key
    - SHAPES_NN [`table fields`] : Not Null
    - SHAPES_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name. if empty shapes layers's name is used as table name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '21', 'Export Shapes')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('SHAPES_PK', SHAPES_PK)
        Tool.Set_Option('SHAPES_NN', SHAPES_NN)
        Tool.Set_Option('SHAPES_UQ', SHAPES_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def Update_Shapes_SRID(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Update Shapes SRID
    ----------
    [db_pgsql.22]\n
    Change the SRID of all geometries in the user-specified column and table.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '22', 'Update Shapes SRID')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_22(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Update Shapes SRID
    ----------
    [db_pgsql.22]\n
    Change the SRID of all geometries in the user-specified column and table.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '22', 'Update Shapes SRID')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def Import_Shapes_with_Joined_Data_GUI(SHAPES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GEO_TABLE=None, GEO_KEY=None, JOIN_TABLE=None, JOIN_KEY=None, WHERE=None, Verbose=2):
    '''
    Import Shapes with Joined Data (GUI)
    ----------
    [db_pgsql.23]\n
    Imports shapes with joined data from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GEO_TABLE [`choice`] : Geometry Table. Available Choices: Default: 0
    - GEO_KEY [`choice`] : Key. Available Choices: Default: 0
    - JOIN_TABLE [`choice`] : Join Table. Available Choices: Default: 0
    - JOIN_KEY [`choice`] : Key. Available Choices: Default: 0
    - WHERE [`text`] : Where

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '23', 'Import Shapes with Joined Data (GUI)')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GEO_TABLE', GEO_TABLE)
        Tool.Set_Option('GEO_KEY', GEO_KEY)
        Tool.Set_Option('JOIN_TABLE', JOIN_TABLE)
        Tool.Set_Option('JOIN_KEY', JOIN_KEY)
        Tool.Set_Option('WHERE', WHERE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_23(SHAPES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GEO_TABLE=None, GEO_KEY=None, JOIN_TABLE=None, JOIN_KEY=None, WHERE=None, Verbose=2):
    '''
    Import Shapes with Joined Data (GUI)
    ----------
    [db_pgsql.23]\n
    Imports shapes with joined data from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GEO_TABLE [`choice`] : Geometry Table. Available Choices: Default: 0
    - GEO_KEY [`choice`] : Key. Available Choices: Default: 0
    - JOIN_TABLE [`choice`] : Join Table. Available Choices: Default: 0
    - JOIN_KEY [`choice`] : Key. Available Choices: Default: 0
    - WHERE [`text`] : Where

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '23', 'Import Shapes with Joined Data (GUI)')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GEO_TABLE', GEO_TABLE)
        Tool.Set_Option('GEO_KEY', GEO_KEY)
        Tool.Set_Option('JOIN_TABLE', JOIN_TABLE)
        Tool.Set_Option('JOIN_KEY', JOIN_KEY)
        Tool.Set_Option('WHERE', WHERE)
        return Tool.Execute(Verbose)
    return False

def Import_Shapes_with_Joined_Data(SHAPES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GEO_TABLE=None, GEO_KEY=None, JOIN_TABLE=None, JOIN_KEY=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Shapes with Joined Data
    ----------
    [db_pgsql.24]\n
    Imports shapes with joined data from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GEO_TABLE [`choice`] : Geometry Table. Available Choices: Default: 0
    - GEO_KEY [`choice`] : Key. Available Choices: Default: 0
    - JOIN_TABLE [`choice`] : Join Table. Available Choices: Default: 0
    - JOIN_KEY [`choice`] : Key. Available Choices: Default: 0
    - FIELDS [`text`] : Fields
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '24', 'Import Shapes with Joined Data')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GEO_TABLE', GEO_TABLE)
        Tool.Set_Option('GEO_KEY', GEO_KEY)
        Tool.Set_Option('JOIN_TABLE', JOIN_TABLE)
        Tool.Set_Option('JOIN_KEY', JOIN_KEY)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_24(SHAPES=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GEO_TABLE=None, GEO_KEY=None, JOIN_TABLE=None, JOIN_KEY=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Shapes with Joined Data
    ----------
    [db_pgsql.24]\n
    Imports shapes with joined data from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GEO_TABLE [`choice`] : Geometry Table. Available Choices: Default: 0
    - GEO_KEY [`choice`] : Key. Available Choices: Default: 0
    - JOIN_TABLE [`choice`] : Join Table. Available Choices: Default: 0
    - JOIN_KEY [`choice`] : Key. Available Choices: Default: 0
    - FIELDS [`text`] : Fields
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '24', 'Import Shapes with Joined Data')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GEO_TABLE', GEO_TABLE)
        Tool.Set_Option('GEO_KEY', GEO_KEY)
        Tool.Set_Option('JOIN_TABLE', JOIN_TABLE)
        Tool.Set_Option('JOIN_KEY', JOIN_KEY)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def Import_Raster(GRIDS=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, WHERE=None, MULTIPLE=None, Verbose=2):
    '''
    Import Raster
    ----------
    [db_pgsql.30]\n
    Imports grids from a PostGIS database.\n
    Arguments
    ----------
    - GRIDS [`output grid list`] : Grids
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - WHERE [`text`] : Where
    - MULTIPLE [`choice`] : Multiple Bands. Available Choices: [0] single grids [1] grid collection [2] automatic Default: 2

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '30', 'Import Raster')
    if Tool.is_Okay():
        Tool.Set_Output('GRIDS', GRIDS)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('MULTIPLE', MULTIPLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_30(GRIDS=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, WHERE=None, MULTIPLE=None, Verbose=2):
    '''
    Import Raster
    ----------
    [db_pgsql.30]\n
    Imports grids from a PostGIS database.\n
    Arguments
    ----------
    - GRIDS [`output grid list`] : Grids
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - WHERE [`text`] : Where
    - MULTIPLE [`choice`] : Multiple Bands. Available Choices: [0] single grids [1] grid collection [2] automatic Default: 2

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '30', 'Import Raster')
    if Tool.is_Okay():
        Tool.Set_Output('GRIDS', GRIDS)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('MULTIPLE', MULTIPLE)
        return Tool.Execute(Verbose)
    return False

def Export_Raster(GRIDS=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GRID_SYSTEM=None, TABLE=None, NAME=None, GRID_NAME=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Export Raster
    ----------
    [db_pgsql.31]\n
    Exports grids to a PostGIS database.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Bands
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GRID_SYSTEM [`grid system`] : Grid System
    - TABLE [`choice`] : Add to Table. Available Choices: Default: 0
    - NAME [`text`] : Table Name
    - GRID_NAME [`boolean`] : Band Name Field. Default: 1
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '31', 'Export Raster')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GRID_SYSTEM', GRID_SYSTEM)
        Tool.Set_Option('TABLE', TABLE)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('GRID_NAME', GRID_NAME)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_31(GRIDS=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GRID_SYSTEM=None, TABLE=None, NAME=None, GRID_NAME=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Export Raster
    ----------
    [db_pgsql.31]\n
    Exports grids to a PostGIS database.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Bands
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GRID_SYSTEM [`grid system`] : Grid System
    - TABLE [`choice`] : Add to Table. Available Choices: Default: 0
    - NAME [`text`] : Table Name
    - GRID_NAME [`boolean`] : Band Name Field. Default: 1
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '31', 'Export Raster')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GRID_SYSTEM', GRID_SYSTEM)
        Tool.Set_Option('TABLE', TABLE)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('GRID_NAME', GRID_NAME)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def Update_Raster_SRID(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TABLES=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Update Raster SRID
    ----------
    [db_pgsql.32]\n
    Change the SRID of all rasters in the user-specified column and table.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TABLES [`choice`] : Tables. Available Choices: Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '32', 'Update Raster SRID')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_32(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, TABLES=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Update Raster SRID
    ----------
    [db_pgsql.32]\n
    Change the SRID of all rasters in the user-specified column and table.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - TABLES [`choice`] : Tables. Available Choices: Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '32', 'Update Raster SRID')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def Import_Single_Raster_Band(GRID=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, RID=None, Verbose=2):
    '''
    Import Single Raster Band
    ----------
    [db_pgsql.33]\n
    Imports grids from a PostGIS database.\n
    Arguments
    ----------
    - GRID [`output data object`] : Grid
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - RID [`text`] : Band Identifier

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '33', 'Import Single Raster Band')
    if Tool.is_Okay():
        Tool.Set_Output('GRID', GRID)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('RID', RID)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_33(GRID=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, DB_TABLE=None, RID=None, Verbose=2):
    '''
    Import Single Raster Band
    ----------
    [db_pgsql.33]\n
    Imports grids from a PostGIS database.\n
    Arguments
    ----------
    - GRID [`output data object`] : Grid
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - DB_TABLE [`text`] : Table
    - RID [`text`] : Band Identifier

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '33', 'Import Single Raster Band')
    if Tool.is_Okay():
        Tool.Set_Output('GRID', GRID)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('RID', RID)
        return Tool.Execute(Verbose)
    return False

def Export_Grid_Collection(GRIDS=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GRID_SYSTEM=None, NAME=None, EXISTS=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Export Grid Collection
    ----------
    [db_pgsql.34]\n
    Exports a grid collection to a PostGIS database.\n
    Arguments
    ----------
    - GRIDS [`input grid collection`] : Grid Collection
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GRID_SYSTEM [`grid system`] : Grid System
    - NAME [`text`] : Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort [1] replace Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '34', 'Export Grid Collection')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GRID_SYSTEM', GRID_SYSTEM)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_34(GRIDS=None, PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, GRID_SYSTEM=None, NAME=None, EXISTS=None, CRS_CODE=None, CRS_AUTHORITY=None, Verbose=2):
    '''
    Export Grid Collection
    ----------
    [db_pgsql.34]\n
    Exports a grid collection to a PostGIS database.\n
    Arguments
    ----------
    - GRIDS [`input grid collection`] : Grid Collection
    - PG_HOST [`text`] : Host. Default: 127.0.0.1
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User
    - PG_PWD [`text`] : Password
    - GRID_SYSTEM [`grid system`] : Grid System
    - NAME [`text`] : Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort [1] replace Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '34', 'Export Grid Collection')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        Tool.Set_Option('GRID_SYSTEM', GRID_SYSTEM)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        return Tool.Execute(Verbose)
    return False

def Create_Database(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Create Database
    ----------
    [db_pgsql.35]\n
    Creates a new PostgreSQL Database.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database. Default: dbname
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '35', 'Create Database')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_35(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Create Database
    ----------
    [db_pgsql.35]\n
    Creates a new PostgreSQL Database.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database. Default: dbname
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '35', 'Create Database')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def Drop_Database(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Drop Database
    ----------
    [db_pgsql.36]\n
    Deletes a PostgreSQL Database.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database. Default: dbname
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '36', 'Drop Database')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_36(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Drop Database
    ----------
    [db_pgsql.36]\n
    Deletes a PostgreSQL Database.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database. Default: dbname
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '36', 'Drop Database')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

