/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file RutilT.cxx
    \author Romain BONDUE
    \date 05/07/2005 */
#include <iostream>
#include <vector>
#include <memory>
#include <exception>

extern "C"{
#include <gtk/gtk.h>
}

#include "Exceptions.h"
#include "ErrorsCode.h"
#include "CLIOpt.h"
#include "GtkGUI.h"
#include "Module.h"
#include "NetTools.h"
#include "UserData.h"
#include "IDriver.h"



namespace
{
    void ApplyProfile (nsCore::ICLIModule& Module,
                       const std::string& ProfileName)
                                                    throw (nsErrors::CException)
    {
        unsigned I (0);
        for ( ; I < Module.GetNbProfile() && Module.GetProfile (I).GetName() !=
                                                            ProfileName ; ++I);
        if (I == Module.GetNbProfile())
            throw (nsErrors::CException (std::string ("Profile not found : ") +
                                     ProfileName, nsErrors::ProfileNotFound));
        Module.ApplyProfile (I);

    } // ApplyProfile()


    inline void UseDhcp (const nsCore::ICLIModule& Module,
                  const std::string& IfaceName) throw (nsErrors::CException)
    {
        nsNet::CInterface Iface (IfaceName);
        Iface.SetIPFromDHCP (Module.GetDriver().GetSSID());

    } // UseDhcp()


    void CLICore (const nsCore::CCLIOpt Opt) throw (nsErrors::CException,
                                                    std::bad_alloc)
    {
        if (Opt.Profile() || Opt.Dhcp())
        {
            if (Opt.IfaceName().empty())
                throw (nsErrors::CException ("Iface name missing.",
                                             nsErrors::InvalidArguments));
            std::vector<nsWireless::CDeviceInfo> DeviceVec;
            nsCore::GetWirelessDevice (DeviceVec);
            unsigned I (0);
            for ( ; I < DeviceVec.size() && DeviceVec [I].GetDeviceName()
                                                    != Opt.IfaceName(); ++I);
            if (I == DeviceVec.size())
                throw nsErrors::CException (std::string ("Can't find interface"
                            " : ") + Opt.IfaceName(), nsErrors::NoDeviceFound);

            std::auto_ptr<nsCore::ICLIModule> pModule
                                        (nsCore::MakeCLIModule (DeviceVec [I]));
            if (Opt.Profile())
                ApplyProfile (*pModule, Opt.ProfileName());
            if (Opt.Dhcp())
                UseDhcp (*pModule, Opt.IfaceName());
        }

    } // CLICore()

} // anonymous namespace



int main (int argc, char* argv [])
{
    try
    {
        nsCore::CCLIOpt Opt (argc, argv);
        if (Opt.Help())
            std::cout << "Usage :\n    rutilt [IFACE_NAME] [OPTIONS...]\n"
"Short options can be combined, options with an argument must appear last in"
" this\ncase. Mandatory arguments to long options are mandatory for short"
" options too.\nIf IFACE_NAME is specified, rutilt will try to use this network"
" interface and\nwill not prompt the interface chooser.\nOptions :\n"
"    -h, --help\n        display this help and exit\n-v, --version\n"
"        output version information and exit\n    -t, --hide\n"
"        force to start hidden in tray\n-p, --profile PROFILE_NAME\n"
"        apply PROFILE_NAME on start (quote PROFILE_NAME if it contains"
" spaces),\n        the IP settings of that profile will not be applied.\n"
"-d, --dhcp\n        use dhcp (run the set_ip.sh script) on start\n-e, --exit\n"
"        exit after processing the command line options (do not start the"
" gui)\n        IFACE_NAME must be specified when using this options with"
" --dhcp or\n        --profile and the application executed as root.\n"
"All Gtk+ specific options are also supported, see the Gtk+ documentation."
                      << std::endl;
        else if (Opt.Version())
            std::cout << "RutilT v0.15\nCopyright (C) 2005-2007 Romain BONDUE\n"
"This is free software.  You may redistribute copies of it under the terms of\n"
"the GNU General Public License <http://www.gnu.org/licenses/gpl.html>.\n"
"There is NO WARRANTY, to the extent permitted by law.\n\n"
"Written by Romain BONDUE, see the AUTHORS file or the about dialog for all\n"
"contributors." << std::endl;
        else if (Opt.Exit())
            CLICore (Opt);
        else
        {
            ::gtk_init (&argc, &argv);
            nsGUI::CGtkGUI UI;
            UI.Run (Opt);
        }
    }
    catch (const nsErrors::CException& Exc) // Handle CSystemExc too.
    {
        std::cerr << Exc.GetMsg() << "\nCode : " << Exc.GetCode() << std::endl;
        return Exc.GetCode();
    }
    catch (const std::exception& Exc)
    {
        std::cerr << "Exception : " << Exc.what() << std::endl;
        return nsErrors::UnknownExc;
    }
    catch (...)
    {
        std::cerr << "Unknown exception." << std::endl;
        return nsErrors::UnknownExc;
    }

    return 0;

} // main()
