/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file Parameters.h
    \author Romain BONDUE
    \date 07/07/2005 */
#ifndef __PARAMETERS_H__
#define __PARAMETERS_H__

#include <string>
#include <stdexcept> // std::bad_alloc

extern "C"{
    // Compilation problems if socket.h is not included (or after wireless.h)
#include <sys/socket.h>
    // Those 3 headers were included by wireless.h until WE21.
#include <linux/types.h>		/* for "caddr_t" et al		*/
#include <linux/socket.h>		/* for "struct sockaddr" et al	*/
#include <linux/if.h>			/* for IFNAMSIZ and co... */
#include <linux/wireless.h> // IW_MODE_AUTO, IW_MODE_ADHOC etc...
}

#include "Exceptions.h" // CRTExc



namespace nsWireless
{
    enum Mode_e {Auto = IW_MODE_AUTO, AdHoc = IW_MODE_ADHOC,
                 Managed = IW_MODE_INFRA, Master = IW_MODE_MASTER,
                 Repeater = IW_MODE_REPEAT, Secondary = IW_MODE_SECOND,
                 Monitor = IW_MODE_MONITOR, Unknown};

    const std::string& GetModeName (Mode_e) throw();

    Mode_e GetModeFromName (const std::string&) throw();

    Mode_e GetMatchingMode (Mode_e) throw();


    class CQuality
    {
      public :
        /** \param LinkQuality in pourcentage
         *  \param SignalLevel in dBm
         *  \param NoiseLevel in dBm */
        CQuality (unsigned short LinkQuality = 0, short SignalLevel = -256,
                  short NoiseLevel = -256) throw();

        unsigned short GetLinkQuality() const throw();
        short GetSignalLevel () const throw();
        short GetNoiseLevel () const throw();


      private :
        unsigned short m_LinkQuality;
        short m_SignalLevel;
        short m_NoiseLevel;

    }; // CQuality


    class CDeviceInfo
    {
      public :
        CDeviceInfo (const std::string& DeviceName, const std::string&)
                                                                    throw();

        const std::string& GetDeviceName () const throw();
        const std::string& GetProtoName () const throw();


      private :
        std::string m_DeviceName;
        std::string m_ProtoName;

    }; // CDeviceInfo


    class CHexaKey
    {
      public :
        class CBadFormatExc : public nsErrors::CRTExc<CBadFormatExc> {};


        CHexaKey () throw();
        CHexaKey (const char*, unsigned Size) throw (std::bad_alloc);
        /** \param Str Must only be composed of hexadecimal caracters. It can
         *             also contain these separators : ' ', ':' and '-'. */
        explicit CHexaKey (const std::string& Str) throw (CBadFormatExc);
        CHexaKey (const CHexaKey&) throw (std::bad_alloc);
        ~CHexaKey () throw();

        CHexaKey& operator = (const CHexaKey&) throw (std::bad_alloc);

        bool Empty () const throw();
        void Clear () throw();

        const char* Get () const throw (std::bad_alloc);
        unsigned Size () const throw();
        const std::string& GetStr () const throw();
        void Set (const char*, unsigned) throw (std::bad_alloc);
        /** \param Str Must only be composed of hexadecimal caracters. It can
         *             also contain these separators : ' ', ':' and '-'. */
        void Set (std::string) throw (CBadFormatExc);

        void SetFromASCII (const std::string&) throw (std::bad_alloc);

        bool operator == (const CHexaKey&) const throw();


      private :
        mutable unsigned m_Size;
        mutable char* m_Key;
        mutable std::string m_StrKey;

    }; // CHexaKey


        // Only for implementation purpose, there's no useful abstraction here.
    class CMacAddress : public CHexaKey
    {
      public :
        CMacAddress () throw();
        /** \param Address Must be at least 6 bytes long, everything after the
         *                 6th byte will be ignored. */
            // explicit to don't mess with NTCTS, use a string instead.
        explicit CMacAddress (const char* Address) throw (std::bad_alloc);
        /** \param Address Must be at least 12 hexadecimal caracters long, It
                   can also contain these separators : ' ', ':' and '-'. */
        CMacAddress (const std::string& Address) throw (CBadFormatExc);

        /** \param Address Must be at least 6 bytes long, everything after the
         *                 6th byte will be ignored. */
        void Set (const char* Address) throw (std::bad_alloc);

        /** \param Address Must be at least 12 hexadecimal caracters long, It
                   can also contain these separators : ' ', ':' and '-'. */
        void Set (const std::string& Address) throw (CBadFormatExc);

        bool Empty () const throw();

        static const unsigned Size = 6;


      private :
        void Set (const char*, unsigned) throw (std::bad_alloc);
        void SetFromASCII (const std::string&) throw (std::bad_alloc);

        void Check () throw (CBadFormatExc);

    }; // CMacAddress
 

    enum EncryptType_e {None, WEP, AES, TKIP, UnknownEnc};
    enum AuthType_e {Open, Shared, WPAPSK, WPANONE, WPA2PSK, WPA, WPA2,
                     UnknownAuth};

    const std::string& GetEncryptName (EncryptType_e) throw();

    EncryptType_e GetEncryptFromName (const std::string&) throw();

    const std::string& GetAuthName (AuthType_e) throw();

    AuthType_e GetAuthFromName (const std::string&) throw();


    class CEncryptionD // D for "Descriptor"
    {
      public :
        CEncryptionD (const CHexaKey& = CHexaKey(),
                      AuthType_e = Open, EncryptType_e = None,
                      unsigned DefaultKey = 0) throw (std::bad_alloc);
        CEncryptionD (const CEncryptionD&) throw (std::bad_alloc);
        CEncryptionD& operator = (const CEncryptionD&) throw (std::bad_alloc);

        EncryptType_e GetEncrypt () const throw();
        AuthType_e GetAuth () const throw();
        /** \warning No bound check! */
        const CHexaKey& GetKey (unsigned = 0) const throw();

        void SetEncrypt (EncryptType_e) throw();
        void SetAuth (AuthType_e) throw();
        /** \warning No bound check! */
        void SetKey (const CHexaKey&, unsigned = 0) throw (std::bad_alloc);

            // Should be between 0 and MaxNbKey, UNCHECKED.
        void SetDefaultKey (unsigned) throw();
        unsigned GetDefaultKey () const throw();

        bool operator == (const CEncryptionD&) const throw();

        static const unsigned MaxNbKey = 4;


      private :
        AuthType_e m_Auth;
        EncryptType_e m_Encrypt;
        CHexaKey m_KeyTab [MaxNbKey];
        unsigned m_DefaultKey;

    }; // CEncryptionD


    class CCell
    {
      public :
        CCell () throw();
        CCell (const CMacAddress&, Mode_e, const std::string&,
               const CEncryptionD&, unsigned, const CQuality&, unsigned)
                                                      throw (std::bad_alloc);

        /* If the fields are empty or equal to 0 (except the Mode),
         * they haven't been set, I'm not sure there is any guaranty. */ 

        const CMacAddress& GetAPMacAddress () const throw();
        Mode_e GetMode () const throw();
        const std::string& GetSSID () const throw();
        void SetSSID (const std::string&) throw();
        const CEncryptionD& GetEncryptionD () const throw();
        CEncryptionD& GetEncryptionD () throw();
        unsigned GetChannel () const throw();
        const CQuality& GetQuality () const throw();
        unsigned GetTxRate () const throw();


      private :
            // FIXME ** AWFUL **
        friend class CWE17Driver; // To set the fields efficiently.
        CCell (const CMacAddress&) throw(); // Used by IDriver.

        CMacAddress m_APAddr;
        Mode_e m_Mode;
        std::string m_SSID;
        CEncryptionD m_EncDescriptor;
        unsigned m_Channel;
        CQuality m_Quality;
        unsigned m_Rate;

    }; // CCell


    class CFreq
    {
      public :
        CFreq (unsigned, double) throw();
        unsigned GetChannel () const throw();
        double GetFrequency () const throw();


      private :
        unsigned m_Channel;
        double m_Frequency;

    }; // CFreq

} // namespace nsWireless

#include "private/Parameters.hxx"

#endif // __PARAMETERS_H__
